#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>

#include "config.h"
#include "prngd.h"

int randsavefd;
int debug = 0;

int devnull = -1;

static void write_seedfile(void);

static void open_seedfile(const char *seedfile)
{
  int bytes, bytes_read_total;
  unsigned char buffer[1024];

  randsavefd = open(seedfile, O_RDWR | O_CREAT | O_SYNC | O_RSYNC, 400);
  if (randsavefd < 0)
  {
    fprintf(stderr, "Could not open %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * The contents of the seedfile may be sensitive, since we start seeding
   * from the file.
   */
  if (fchmod(randsavefd, S_IRUSR | S_IWUSR) < 0)
  {
    fprintf(stderr, "Could not chmod %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Do some initial seeding stuff to pre-stir the PRNG.
   */
  seed_internal();

  /*
   * Now read as much from the file as we can get. 
   */
  bytes_read_total = 0;
  while ((bytes = read(randsavefd, buffer, 1024)) > 0)
  {
    rand_add(buffer, bytes, (double)bytes);
    bytes_read_total += bytes;
  }
  if (bytes < 0)
  {
    fprintf(stderr, "Error reading %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Done, let's stir around some more
   */
  seed_internal();

  if (debug)
    fprintf(stdout, "Read %d bytes\n", bytes_read_total);
  /*
   * Ok, we don't want to reuse the same seed again, so we will immediately
   * replace the contents of the seed-file.
   */
  write_seedfile();
}

static void write_seedfile(void)
{
  int bytes;
  unsigned char buffer[1024];

  bytes = lseek(randsavefd, 0, SEEK_SET);
  if (bytes < 0)
  {
    fprintf(stderr, "Error writing seedfile: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  bytes = rand_bytes(buffer, 1024);
  if (bytes == 0)
  {
    fprintf(stderr, "Could not retrieve rand_bytes\n");
  }
  else
  {
    bytes = write(randsavefd, buffer, 1024);
    if (bytes < 0)
    {
      fprintf(stderr, "Error writing seedfile: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
  }
  if (debug)
    fprintf(stdout, "Wrote %d bytes back to seedfile\n", bytes);
  bytes = ftruncate(randsavefd, bytes);
}

void close_seedfile(void)
{
  write_seedfile();
  close(randsavefd);
}


static void killpeer(int service_socket)
{
  unsigned char buffer[256];
  int num;
  pid_t peer_id;

  buffer[0] = 0x04;
  if (write(service_socket, buffer, 1) < 1)
  {
    fprintf(stderr, "Cannot contact peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (read(service_socket, buffer, 1) < 1)
  {
    fprintf(stderr, "Cannot read from peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  num = buffer[0];
  if (read(service_socket, buffer, num) < num)
  {
    fprintf(stderr, "Cannot read from peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  buffer[num] = '\0';
  peer_id = atoi((char *) buffer);
  fprintf(stderr, "Killing: %d\n", peer_id);
  if (kill(peer_id, SIGHUP) < 0)
  {
    fprintf(stderr, "Cannot kill %d: %s\n", peer_id, strerror(errno));
    exit(EXIT_FAILURE);
  }
  exit(EXIT_SUCCESS);
}

static void usage(const char *progname)
{
  fprintf(stderr,
	  "Usage: %s [options] /tmp/entropy\nOptions:\n\t-d: debugging on\n\t-c cmdpath: use cmdpath for entropy commands [%s]\n\t-s seedpath: use seedpath as seedfile [%s]\n\t--kill: kill daemon on other side with HUP\n",
	  progname, CONFIGFILE, RANDSAVENAME);
}

int main(int argc, char *argv[])
{
  int argp;
  int ret;
  int flags;
  int killmode = 0;
  time_t seed_stat_interval = SEED_STAT_INTERVAL;
  time_t seed_ext_interval = SEED_EXT_INTERVAL;
  char *randsock = NULL;
  char *cmdpath = CONFIGFILE;
  char *seedpath = RANDSAVENAME;
  int service_socket;
  struct sockaddr_un sockun;
  entropy_source_t *entropy_source;

  /*
   * Process arguments 
   */
  argp = 1;
  while (argp < argc)
  {
    if (!strcmp(argv[argp], "-d"))
      debug = 1;
    else if (!strcmp(argv[argp], "-c") && (argp + 1 < argc))
      cmdpath = argv[++argp];
    else if (!strcmp(argv[argp], "-s") && (argp + 1 < argc))
      seedpath = argv[++argp];
    else if (!strcmp(argv[argp], "--kill"))
      killmode = 1;
    else if ((argv[argp][0] != '-') && (randsock == NULL))
      randsock = argv[argp];
    else
    {
      usage(argv[0]);
      exit(EXIT_FAILURE);
    }
    argp++;
  }
  if (!randsock) {
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  if (!killmode)
    parse_configfile(cmdpath, &entropy_source);

  if (!randsock)
  {
    fprintf(stderr, "No socket name given, exiting!\n");
    exit(EXIT_FAILURE);
  }
  if (debug)
    fprintf(stdout, "Debugging enabled\n");

  /*
   * Open /dev/null
   */
  devnull = open("/dev/null", O_RDWR);
  if (devnull == -1) {
    fprintf(stderr, "Couldn't open /dev/null: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Start by reading back the seed saved from former runs. 
   */
  if (!killmode)
    open_seedfile(seedpath);

  /*
   * Set up the socket to service entropy requests
   */
  service_socket = socket(AF_UNIX, SOCK_STREAM, 0);
  if (service_socket < 0)
  {
    fprintf(stderr, "Could not create socket: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  /*
   * Bind the socket to the actual socket-file
   */
  sockun.sun_family = AF_UNIX;
  if (sizeof(sockun.sun_path) < strlen(randsock) + 1)
  {
    fprintf(stderr, "Sockfilename too long: %s, maxlen=%d\n", randsock,
	    sizeof(sockun.sun_path));
    exit(EXIT_FAILURE);
  }

  strcpy(sockun.sun_path, randsock);

  /*
   * Check, whether there already is a server running at this socket.
   * In this case: don't touch the socket.
   */
  if (connect(service_socket, (struct sockaddr *) &sockun,
	      sizeof(struct sockaddr_un)) < 0)
  {
    if (killmode)
    {
      fprintf(stderr, "Cannot kill peer at %s: %d\n", randsock,
	      service_socket);
      exit(EXIT_FAILURE);
    }
  }
  else if (killmode)
  {
    killpeer(service_socket);
  }
  else
  {
    fprintf(stderr, "Socket %s already used by another process: %s\n",
	    randsock, strerror(errno));
    exit(EXIT_FAILURE);
  }

  remove(randsock);
  if (bind(service_socket, (struct sockaddr *) &sockun,
	   sizeof(struct sockaddr_un)) < 0)
  {
    fprintf(stderr, "Could not bind socket to %s: %s\n", randsock,
	    strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (listen(service_socket, SOMAXCONN) < 0)
  {
    fprintf(stderr, "Could not listen on socket: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  /*
   * Switch the service_socket to non-blocking mode (otherwise it might
   * hang inside accept() if a connection is closed between select() and
   * accept().
   */
  if ((flags = fcntl(service_socket, F_GETFL, 0)) < 0)
  {
    fprintf(stderr, "Could not get socket file mode: %s\n",
	    strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (fcntl(service_socket, F_SETFL, flags | O_NONBLOCK) < 0)
  {
    fprintf(stderr, "Could not set socket file mode: %s\n",
	    strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * If not in debugging mode, make myself a daemon 
   */
  if (!debug)
  {
    ret = daemon(0, 0);
    if (ret < 0)
    {
      fprintf(stderr, "Could not daemonize: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
  }

  /*
   * Maybe we have changed our pid, let's add some more of it. 
   */
  seed_internal();

  /* Never returns */
  main_loop(service_socket, sockun, seed_stat_interval, seed_ext_interval,
	    entropy_source, MAX_GATHERER_BYTES);

  /*
   * We never get here, but want to make the compiler happy...
   */
  return(EXIT_SUCCESS);
}
