#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <assert.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>

#include "prngd.h"

int randsavefd = -1;
int debug = 0;

int devnull = -1;

static void write_seedfile(void);

static void open_seedfile(const char *seedfile)
{
  int bytes, bytes_read_total;
  unsigned char buffer[PRNGD_STATE_SIZE];

  randsavefd = open(seedfile, O_RDWR | O_CREAT | O_SYNC | O_RSYNC, 400);
  if (randsavefd < 0)
  {
    fprintf(stderr, "Could not open %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * The contents of the seedfile may be sensitive, since we start seeding
   * from the file.
   */
  if (fchmod(randsavefd, S_IRUSR | S_IWUSR) < 0)
  {
    fprintf(stderr, "Could not chmod %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Do some initial seeding stuff to pre-stir the PRNG.
   */
  seed_internal(NULL);

  /*
   * Now read as much from the file as we can get. 
   */
  bytes_read_total = 0;
  while ((bytes = read(randsavefd, buffer, PRNGD_STATE_SIZE)) > 0)
  {
    rand_add(buffer, bytes, (double)bytes);
    bytes_read_total += bytes;
  }
  if (bytes < 0)
  {
    fprintf(stderr, "Error reading %s: %s\n", seedfile, strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Done, let's stir around some more
   */
  seed_internal(NULL);

  if (debug)
    fprintf(stdout, "Read %d bytes\n", bytes_read_total);
  /*
   * Ok, we don't want to reuse the same seed again, so we will immediately
   * replace the contents of the seed-file.
   */
  write_seedfile();
}

static void write_seedfile(void)
{
  int bytes;
  unsigned char buffer[PRNGD_STATE_SIZE];

  bytes = lseek(randsavefd, 0, SEEK_SET);
  if (bytes < 0)
  {
    fprintf(stderr, "Error writing seedfile: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  bytes = rand_bytes(buffer, PRNGD_STATE_SIZE);
  if (bytes == 0)
  {
    fprintf(stderr, "Could not retrieve rand_bytes\n");
  }
  else
  {
    bytes = write(randsavefd, buffer, PRNGD_STATE_SIZE);
    if (bytes < 0)
    {
      fprintf(stderr, "Error writing seedfile: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
  }
  if (debug)
    fprintf(stdout, "Wrote %d bytes back to seedfile\n", bytes);
  bytes = ftruncate(randsavefd, bytes);
}

void close_seedfile(void)
{
  if (randsavefd >= 0) {
    write_seedfile();
    close(randsavefd);
  }
}


static void killpeer(int service_socket)
{
  unsigned char buffer[256];
  int num;
  pid_t peer_id;

  buffer[0] = 0x04;
  if (write(service_socket, buffer, 1) < 1)
  {
    fprintf(stderr, "Cannot contact peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (read(service_socket, buffer, 1) < 1)
  {
    fprintf(stderr, "Cannot read from peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  num = buffer[0];
  if (read(service_socket, buffer, num) < num)
  {
    fprintf(stderr, "Cannot read from peer: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  buffer[num] = '\0';
#ifdef PID_T_IS_LONG_INT
  peer_id = atol((char *) buffer);
  fprintf(stderr, "Killing: %ld\n", peer_id);
#else
  peer_id = atoi((char *) buffer);
  fprintf(stderr, "Killing: %d\n", peer_id);
#endif

  if (kill(peer_id, SIGHUP) < 0)
  {
#ifdef PID_T_IS_LONG_INT
    fprintf(stderr, "Cannot kill %ld: %s\n", peer_id, strerror(errno));
#else
    fprintf(stderr, "Cannot kill %d: %s\n", peer_id, strerror(errno));
#endif
    exit(EXIT_FAILURE);
  }
  exit(EXIT_SUCCESS);
}

static void usage(const char *progname)
{
  fprintf(stderr,
	  "Usage: %s [options] /path/to/socket1 [/path/to/socket2 ...]\nOptions:\n\t-d/--debug: debugging on\n\t-c/--cmdfile cmdpath: use cmdpath for entropy commands [%s]\n\t-s/--seedfile seedpath: use seedpath as seedfile [%s]\n\t-n/--no-seedfile: no seedfile, keep pool in memory only\n\t-k/--kill: kill daemon on other side\n",
	  progname, CONFIGFILE, RANDSAVENAME);
}

int main(int argc, char *argv[])
{
  int argp;
  int ret;
  int flags;
  int statval;
  int killmode = 0;
  time_t seed_stat_interval = SEED_STAT_INTERVAL;
  time_t seed_ext_interval = SEED_EXT_INTERVAL;
  int i, numsock = 0;
  char **randsock = NULL;
  char *cmdpath = CONFIGFILE;
  char *seedpath = RANDSAVENAME;
  int *service_socket, kill_socket;
  struct sockaddr_un sockun, sockun_kill;
  struct stat randsock_stat;
  entropy_source_t *entropy_source;

  /*
   * Process arguments 
   */
  argp = 1;
  while (argp < argc)
  {
    if (!strcmp(argv[argp], "-d") || !strcmp(argv[argp], "--debug"))
      debug = 1;
    else if ((!strcmp(argv[argp], "-c") || !strcmp(argv[argp], "--cmdfile"))
	     && (argp + 1 < argc))
      cmdpath = argv[++argp];
    else if ((!strcmp(argv[argp], "-s") || !strcmp(argv[argp], "--seedfile"))
	     && (argp + 1 < argc))
      seedpath = argv[++argp];
    else if (!strcmp(argv[argp], "-n") || !strcmp(argv[argp], "--no-seedfile"))
      seedpath = NULL;
    else if (!strcmp(argv[argp], "-k") || !strcmp(argv[argp], "--kill"))
      killmode = 1;
    else if (argv[argp][0] != '-') {
      if (numsock == 0)
        randsock = malloc(sizeof(char *));
      else
	randsock = realloc(randsock, (numsock + 1) * sizeof(char *));
      if (randsock == NULL) {
	fprintf(stderr, "Could not allocate memory: %s\n", strerror(errno));
	exit(EXIT_FAILURE);
      }
      randsock[numsock++] = argv[argp];
    }
    else
    {
      usage(argv[0]);
      exit(EXIT_FAILURE);
    }
    argp++;
  }
  if (!numsock) {
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  if (!killmode)
    parse_configfile(cmdpath, &entropy_source);

  if (!numsock)
  {
    fprintf(stderr, "No socket name given, exiting!\n");
    exit(EXIT_FAILURE);
  }
  if (debug)
    fprintf(stdout, "Debugging enabled\n");

  /*
   * Open /dev/null
   */
  devnull = open("/dev/null", O_RDWR);
  if (devnull == -1) {
    fprintf(stderr, "Couldn't open /dev/null: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Start by reading back the seed saved from former runs, if usage of a
   * seedfile is wanted. 
   */
  if (!killmode && seedpath)
    open_seedfile(seedpath);

  if (!(service_socket = malloc(numsock * sizeof(int)))) {
    fprintf(stderr, "Could not allocate memory: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }

  /*
   * Prepare the socket used for killing or testing for a running
   * daemon.
   */
  kill_socket = socket(AF_UNIX, SOCK_STREAM, 0);
  if (kill_socket < 0) {
    fprintf(stderr, "Could not create socket: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
  }
  /*
   * Set up the sockets to service entropy requests
   */
  for (i = 0; i < numsock; i++) {

    /*
     * Check out what is already available at the place of the future
     * egd-socket. If we cannot reach the place to be with "stat()",
     * do not continue, as something may be seriously wrong. If the
     * entry does already exist, remove() it, but only if it is of type
     * socket! If removal fails for whatever reason, stop, as something
     * may be seriously wrong.
     */
    if ((statval = stat(randsock[i], &randsock_stat) < 0) &&
	(errno != ENOENT)) {
      fprintf(stderr, "Cannot stat socket position %s: %s\n", randsock[i],
	      strerror(errno));
      exit(EXIT_FAILURE);
    }
    else if (!statval) {
      if (!S_ISSOCK(randsock_stat.st_mode)) {
	fprintf(stderr, "Will not touch %s: no socket\n", randsock[i]);
	exit(EXIT_FAILURE);
      } else {
	/*
	 * Now that we have found a socket, we must check whether there
	 * is a daemon listining on it. When in kill mode, this is great
	 * and we are going to kill it. When not in kill mode, we cannot
	 * start a new daemon at this address and exit with error.
	 */
	memset(&sockun, 0, sizeof(sockun_kill));
	sockun_kill.sun_family = AF_UNIX;
	if (sizeof(sockun_kill.sun_path) < strlen(randsock[i]) + 1) {
#ifdef SIZEOF_IS_LONG_INT
	  fprintf(stderr, "Sockfilename too long: %s, maxlen=%ld\n",
		  randsock[i], sizeof(sockun_kill.sun_path));
#else
	  fprintf(stderr, "Sockfilename too long: %s, maxlen=%d\n",
		  randsock[i], sizeof(sockun_kill.sun_path));
#endif /* SIZEOF_IS_LONG_INT */
	  exit(EXIT_FAILURE);
        }

	strcpy(sockun_kill.sun_path, randsock[i]);
	if (connect(kill_socket, (struct sockaddr *) &sockun_kill,
		    sizeof(sockun_kill)) < 0) {
	  if (killmode) {
	    fprintf(stderr, "Cannot kill peer at %s: no daemon\n", randsock[i]);
	    /*
	     * Do no exit because there may be more work to do!
	     */
	    continue;
	  }
	}
	else if (killmode) {
	  killpeer(kill_socket);
	  shutdown(kill_socket, SHUT_RDWR);
	  continue;
	}
	else {
	  fprintf(stderr,
		  "Socket %s already used by another process, exiting.\n",
		  randsock[i]);
	  exit(EXIT_FAILURE);
	}
	shutdown(kill_socket, SHUT_RDWR);

	/*
         * Now there is a socket without service listening on it: we can
	 * remove it.
	 */
	if (remove(randsock[i]) < 0) {
	  fprintf(stderr, "Cannot remove already existing entry %s: %s\n",
		  randsock[i], strerror(errno));
	  exit(EXIT_FAILURE);
	}
      }
    }

    /*
     * We can never reach this point in killmode, as for whatever condition
     * not forcing immediate abort the loop was "continue"ed before.
     */
    assert(!killmode);

    service_socket[i] = socket(AF_UNIX, SOCK_STREAM, 0);
    if (service_socket[i] < 0)
    {
      fprintf(stderr, "Could not create socket: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
    /*
     * Bind the socket to the actual socket-file
     */
    memset(&sockun, 0, sizeof(sockun));
    sockun.sun_family = AF_UNIX;
    strcpy(sockun.sun_path, randsock[i]);
    if (bind(service_socket[i], (struct sockaddr *) &sockun,
	     sizeof(sockun)) < 0)
    {
      fprintf(stderr, "Could not bind socket to %s: %s\n", randsock[i],
	      strerror(errno));
      exit(EXIT_FAILURE);
    }
    if (listen(service_socket[i], SOMAXCONN) < 0)
    {
      fprintf(stderr, "Could not listen on socket: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
    /*
     * Switch the service_socket to non-blocking mode (otherwise it might
     * hang inside accept() if a connection is closed between select() and
     * accept().
     */
    if ((flags = fcntl(service_socket[i], F_GETFL, 0)) < 0)
    {
      fprintf(stderr, "Could not get socket file mode: %s\n",
	      strerror(errno));
      exit(EXIT_FAILURE);
    }
    if (fcntl(service_socket[i], F_SETFL, flags | O_NONBLOCK) < 0)
    {
      fprintf(stderr, "Could not set socket file mode: %s\n",
	      strerror(errno));
      exit(EXIT_FAILURE);
    }
  }

  if (killmode) {
    /*
     * Everything has been done!
     */
    exit(EXIT_SUCCESS);
  }

  /*
   * If not in debugging mode, make myself a daemon 
   */
  if (!debug)
  {
    ret = daemon(0, 0);
    if (ret < 0)
    {
      fprintf(stderr, "Could not daemonize: %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
  }

  /*
   * Maybe we have changed our pid, let's add some more of it. 
   */
  seed_internal(NULL);

  /* Never returns */
  main_loop(service_socket, numsock, sockun, seed_stat_interval,
	    seed_ext_interval, entropy_source, MAX_GATHERER_BYTES);

  /*
   * We never get here, but want to make the compiler happy...
   */
  return(EXIT_SUCCESS);
}
