#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>

#include "config.h"
#include "prngd.h"

static seed_t gatherer_seed;
static pid_t gatherer_pid = -1;
static int gatherer_status;

typedef enum
{
  wait_command, wait_second_byte, wait_third_byte, wait_fourth_byte,
  wait_read, wait_write, wait_entropy, to_be_closed
}
querystate_t;

typedef struct
{
  int fd;
  int state;
  int to_process;
  int yet_processed;
  int entropy_bits;
  unsigned char command, secbyte;
  unsigned char buffer[256];
} egd_query_t;


static void termination_handler(int received_signal)
{
  if (debug)
    fprintf(stderr, "Caught SIG*, shutting down\n");
  close_seedfile();
  exit(EXIT_SUCCESS);
}

static void child_handler(int received_signal)
{
  pid_t pid;

  if (debug)
    fprintf(stderr, "Caught SIGCHLD\n");

  if ((pid = waitpid(-1, &gatherer_status, WNOHANG)) == -1)
#ifdef PID_T_IS_LONG_INT
    fprintf(stderr, "Couldn't wait for child %ld: %s\n", pid, strerror(errno));
#else
    fprintf(stderr, "Couldn't wait for child %d: %s\n", pid, strerror(errno));
#endif
  else if ((pid == gatherer_pid) && WIFEXITED(gatherer_status))
     gatherer_pid = -1;

  /*
   * We should add the gatherer_status to the PRNG pool at this point, but
   * we must not add entropy now because another rand_add() operation may be
   * running just now. We hence save the data and add it later synchronized
   * in the main_loop(). We don't need to care about the contents of
   * gatherer_status et al and what happens if the connection is shut down
   * from PRNGD because too many bytes were sent, since we don't care about
   * the data at all, we just want to add to the pool as much as we can.
   */
  seed_internal(&gatherer_seed);
}

void main_loop(int *service_socket, int numsock, struct sockaddr_un sockun,
		      time_t seed_stat_interval, time_t seed_ext_interval,
		      entropy_source_t *entropy_source, int max_gatherer_bytes)
{
  int i;
  int accept_fd;
  int sockunlen;
  int flags;
  int num, numbytes, error;
  int pipedes[2];
  time_t last_seed_stat = 0, last_seed_ext = 0;
  int max_query = 0;
  int alloc_query = 0;
  egd_query_t *egd_query = NULL;
  egd_query_t *egd_query_old;
  int max_fd;
  int gather_fd;
  int gatherer_entry;
  int gatherer_bytes;
  int first_gather = 1;
  double est = 0.0;
  time_t gather_start;
  unsigned char gatherbuf[10240];
  struct sigaction new_action;
  fd_set rfds, wfds, efds;
  struct timeval timeout;

  gather_fd = -1;
  gatherer_entry = 0;

  gatherer_bytes = 0;		/* Initialize to make gcc happy */
  gather_start = time(NULL);	/* Initialize to make gcc happy */

  /*
   * Set up the signal handling. TERM and HUP signals shall lead to clean
   * shutdown of prngd (save the seed and exit).
   * SIGPIPE might occur when a peer shuts down the connection. Of course,
   * we don't want prngd to be affected in any way (it should simply close
   * the connection), so SIGPIPE shall be ignored.
   */
  memset(&new_action, 0, sizeof(struct sigaction));
  new_action.sa_handler = termination_handler;
  sigemptyset(&new_action.sa_mask);
  new_action.sa_flags |= SA_RESTART;
  sigaction(SIGTERM, &new_action, NULL);
  sigaction(SIGHUP, &new_action, NULL);

  memset(&new_action, 0, sizeof(struct sigaction));
  new_action.sa_handler = child_handler;
  sigemptyset(&new_action.sa_mask);
  new_action.sa_flags |= SA_RESTART;
  sigaction(SIGCHLD, &new_action, NULL);

  memset(&new_action, 0, sizeof(struct sigaction));
  new_action.sa_handler = SIG_IGN;
  sigaction(SIGPIPE, &new_action, NULL);

  for (;;)
  {
    FD_ZERO(&rfds);
    FD_ZERO(&wfds);
    FD_ZERO(&efds);
    max_fd = 0;
    for (i = 0; i < numsock; i++)
    {
      FD_SET(service_socket[i], &rfds);
      if (service_socket[i] > max_fd)
	max_fd = service_socket[i];
    }
    if (gather_fd != -1)
    {
      FD_SET(gather_fd, &rfds);
      FD_SET(gather_fd, &efds);
      if (gather_fd > max_fd)
	max_fd = gather_fd;
    }
    for (i = 0; i < max_query; i++)
    {
      if (egd_query[i].fd > max_fd)
	max_fd = egd_query[i].fd;
      FD_SET(egd_query[i].fd, &efds);
      if ((egd_query[i].state == wait_command) ||
	  (egd_query[i].state == wait_second_byte) ||
	  (egd_query[i].state == wait_third_byte) ||
	  (egd_query[i].state == wait_fourth_byte) ||
	  (egd_query[i].state == wait_read))
	FD_SET(egd_query[i].fd, &rfds);
      else
	FD_SET(egd_query[i].fd, &wfds);

    }
    /*
     * If we just entered the loop and gather for the first time, we
     * don't want to wait.
     */
    if (first_gather || (rand_pool() < (ENTROPY_THRESHOLD))) {
      timeout.tv_sec = 0;
      timeout.tv_usec = 10;
    }
    else {
      timeout.tv_sec = (seed_stat_interval < seed_ext_interval ?
			 seed_stat_interval : seed_ext_interval);
      timeout.tv_usec = 0;
    }
    num = select(max_fd + 1, &rfds, &wfds, &efds, &timeout);

    if (time(0) >= last_seed_stat + seed_stat_interval) {
      seed_stat();
      last_seed_stat = time(0);
    }

    if (gather_fd != -1) {
      if (FD_ISSET(gather_fd, &efds)) {
	if (debug)
	  fprintf(stderr, "Closing connection to gatherer (efds)\n");
	close(gather_fd);
	gather_fd = -1;
      }
      else if (FD_ISSET(gather_fd, &rfds)) {
	/*
	 * Bytes are coming back from the gatherer process. Read chunks of
	 * up to 10240 bytes and feed them into the PRNG
	 */
	numbytes = read(gather_fd, gatherbuf, 10240);
	rand_add(gatherbuf, numbytes, est * numbytes);
	if (debug && numbytes > 0)
	  fprintf(stderr, "Read %d bytes from gatherer\n", numbytes);
	gatherer_bytes += numbytes;
	if ((numbytes <= 0) || (gatherer_bytes >= max_gatherer_bytes)) {
	  if (debug)
	    fprintf(stderr, "Closing connection to gatherer\n");
	  close(gather_fd);
	  rand_add(&gatherer_status, sizeof(gatherer_status), 0.0);
	  rand_add(&gatherer_seed, sizeof(seed_t), 0.25);
	  gather_fd = -1;
	}
      }
      if (gather_fd == -1) {
	/*
	 * The formerly running gatherer has just been shut down for normal
	 * or error conditions. Choose the next one.
	 */
	gatherer_entry++;
        if (entropy_source[gatherer_entry].path == NULL) {
	  first_gather = 0;	/* done */
	  gatherer_entry = 0;
	}
      }
    }

    /*
     * If a gatherer has not finished in time, kill it!!
     */
    if ((gatherer_pid > 0) &&
	(difftime(time(NULL), gather_start) > seed_ext_interval / 2 + 1)) {
      kill(gatherer_pid, SIGKILL);
      gatherer_pid = -1;
      if (gather_fd != -1) {
	close(gather_fd);
	gather_fd = -1;
	gatherer_entry++;
	if (entropy_source[gatherer_entry].path == NULL) {
	  first_gather = 0;	/* done */
	  gatherer_entry = 0;
	}
      }
    }

    /*
     * Call a gatherer, if it is time to do so. When starting up, call
     * all gatherers once to obtain initial seeding.
     */
    if (((time(0) >= last_seed_ext + seed_ext_interval) || first_gather ||
	(rand_pool() < (ENTROPY_THRESHOLD))) &&
	(gather_fd == -1))
    {
      last_seed_ext = time(0);
      /*
       * time to call another gatherer 
       */
      if (pipe(pipedes) < 0)
	continue;		/*
				 * We cannot do anything about it 
				 */

      /*
       * Select the next gatherer to be used
       */
      if ((gatherer_entry == 0) &&
		(entropy_source[gatherer_entry].path == NULL)) {
        first_gather = 0;	/* no gatherer available */
        continue;
      }

      est = entropy_source[gatherer_entry].rate; /* for later use */
      gatherer_bytes = 0;

      gatherer_pid = fork();
      if (gatherer_pid == -1) {
	if (debug)
	  fprintf(stderr, "Could not fork: %s\n", strerror(errno));
	close(pipedes[0]);
	close(pipedes[1]);
	continue;
      }
      if (gatherer_pid == 0) {
	/*
         * Child
	 */
	dup2(devnull, STDIN_FILENO);
        dup2(pipedes[1], STDOUT_FILENO);
        dup2(pipedes[1], STDERR_FILENO);
        close(pipedes[0]);
        close(pipedes[1]);
        close(devnull); 
	execl(entropy_source[gatherer_entry].path,
	      entropy_source[gatherer_entry].args[0],
	      entropy_source[gatherer_entry].args[1],
	      entropy_source[gatherer_entry].args[2],
	      entropy_source[gatherer_entry].args[3],
	      entropy_source[gatherer_entry].args[4],
	      NULL);
	_exit(0);
      }
      else {
	/*
	 * Parent
	 */
	close(pipedes[1]);
	gather_fd = pipedes[0];
        if ((flags = fcntl(gather_fd, F_GETFL, 0)) < 0) {
	  if (debug)
	    fprintf(stderr, "Could not get socket file mode: %s\n",
		  strerror(errno));
	  close(gather_fd);
        }
        if (fcntl(gather_fd, F_SETFL, flags | O_NONBLOCK) < 0) {
	  if (debug)
	    fprintf(stderr, "Could not set socket file mode: %s\n",
		  strerror(errno));
	  close(gather_fd);
        }
	gather_start = time(NULL);
	fprintf(stderr, "Spawned gatherer: %s (%s",
		entropy_source[gatherer_entry].path,
		entropy_source[gatherer_entry].args[0]);
	for (i = 1; i < 5; i++)
	  if (entropy_source[gatherer_entry].args[i])
	    fprintf(stderr, " %s", entropy_source[gatherer_entry].args[i]);
	fprintf(stderr, ")\n");
      }
      continue;

    }
    /*
     * Do service the active queries first, because accepting a new query
     * will create a new fd. First step: check errors reported and mark
     * the corresponding descriptor "to_be_closed".
     */
    for (i = 0; i < max_query; i++)
    {
      if (FD_ISSET(egd_query[i].fd, &efds))
      {
	if (debug)
	  fprintf(stderr, "Error flagged for service fd %d\n", egd_query[i].fd);
	egd_query[i].state = to_be_closed;
      }
      else if (FD_ISSET(egd_query[i].fd, &rfds) ||
	       FD_ISSET(egd_query[i].fd, &wfds) ||
	       (egd_query[i].state == wait_entropy)) {
      switch (egd_query[i].state)
      {
      case wait_entropy:
	  if (rand_bytes(egd_query[i].buffer, egd_query[i].secbyte) > 0)
	    egd_query[i].state = wait_write;
	break;
      case wait_command:
	error = read(egd_query[i].fd, &(egd_query[i].command), 1);
	if ((error == -1) && (errno == EAGAIN))
	  break;
	if (error != 1)
	  egd_query[i].state = to_be_closed;
	else if (egd_query[i].command > 4)
	  egd_query[i].state = to_be_closed;
	else if (egd_query[i].command == 0)
	{
	  numbytes = rand_pool();
	  egd_query[i].buffer[0] = numbytes / 16777216;
	  egd_query[i].buffer[1] = (numbytes % 16777216) / 65536;
	  egd_query[i].buffer[2] = (numbytes % 65536) / 256;
	  egd_query[i].buffer[3] = numbytes % 256;
	  egd_query[i].to_process = 4;
	  egd_query[i].yet_processed = 0;
	  egd_query[i].state = wait_write;
	}
	else if ((egd_query[i].command == 1) ||
		 (egd_query[i].command == 2) ||
		 (egd_query[i].command == 3))
	  egd_query[i].state = wait_second_byte;
	else if (egd_query[i].command == 4)
	{
	  egd_query[i].buffer[0] = 12;
#ifdef PID_T_IS_LONG_INT
	  sprintf((char *) (egd_query[i].buffer + 1), "%12ld", getpid());
#else
	  sprintf((char *) (egd_query[i].buffer + 1), "%12d", getpid());
#endif
	  egd_query[i].to_process = 13;
	  egd_query[i].yet_processed = 0;
	  egd_query[i].state = wait_write;
	}
	break;
      case wait_second_byte:
	error = read(egd_query[i].fd, &(egd_query[i].secbyte), 1);
	if (error != 1)
	  egd_query[i].state = to_be_closed;
	else if (egd_query[i].command == 1)
	{
	  if (rand_bytes(egd_query[i].buffer + 1, egd_query[i].secbyte))
	    numbytes = egd_query[i].secbyte;
	  else
	    numbytes = 0;
	  egd_query[i].buffer[0] = numbytes;
	  egd_query[i].to_process = numbytes + 1;
	  egd_query[i].yet_processed = 0;
	  egd_query[i].state = wait_write;
	}
	else if (egd_query[i].command == 2)
	{
	  egd_query[i].to_process = egd_query[i].secbyte;
	  egd_query[i].yet_processed = 0;
	  if (rand_bytes(egd_query[i].buffer, egd_query[i].secbyte) > 0)
	    egd_query[i].state = wait_write;
	  else
	    egd_query[i].state = wait_entropy;
	}
	else if (egd_query[i].command == 3)
	{
	  egd_query[i].entropy_bits = egd_query[i].secbyte * 256;
	  egd_query[i].state = wait_third_byte;
	}
	break;
      case wait_third_byte:
	error = read(egd_query[i].fd, &(egd_query[i].secbyte), 1);
	if (error != 1)
	  egd_query[i].state = to_be_closed;
	else
	{
	  egd_query[i].entropy_bits += egd_query[i].secbyte;
	  egd_query[i].state = wait_fourth_byte;
	}
	break;
      case wait_fourth_byte:
	error = read(egd_query[i].fd, &(egd_query[i].secbyte), 1);
	if (error != 1)
	  egd_query[i].state = to_be_closed;
	else
	{
	  egd_query[i].to_process = egd_query[i].secbyte;
	  egd_query[i].yet_processed = 0;
	  egd_query[i].state = wait_read;
	}
	break;
      case wait_read:
	num = egd_query[i].to_process - egd_query[i].yet_processed;
	error = read(egd_query[i].fd, egd_query[i].buffer, num);
	if (error <= 0)
	  egd_query[i].state = to_be_closed;
	else
	{
	  rand_add(egd_query[i].buffer, error,
		   (double) error * egd_query[i].entropy_bits / 8 /
		   egd_query[i].to_process);
	  egd_query[i].yet_processed += error;
	  if (egd_query[i].yet_processed >= egd_query[i].to_process)
	    egd_query[i].state = wait_command;
	}
	break;
      case wait_write:
	error = write(egd_query[i].fd,
		      &(egd_query[i].buffer[egd_query[i].yet_processed]),
		      egd_query[i].to_process -
		      egd_query[i].yet_processed);
	if (error < 1)
	  egd_query[i].state = to_be_closed;
	else
	  egd_query[i].yet_processed += error;
	if (egd_query[i].yet_processed == egd_query[i].to_process) {
	  egd_query[i].state = wait_command;
	  fprintf(stderr, "Sent %d bytes to service %d\n",
		egd_query[i].to_process, egd_query[i].fd);
		}
	break;
      default:
	break;
      }
      }
    }
    for (i = 0; i < max_query; i++) {
      if (egd_query[i].state == to_be_closed) {
	if (debug)
	  fprintf(stderr, "Closing service fd %d\n", egd_query[i].fd);
	close(egd_query[i].fd);
	if (i < max_query - 1)
	{
	  memmove(egd_query + i, egd_query + max_query - 1,
		  sizeof(egd_query_t));
	  i--;
	}
	max_query--;
      }
    }

    for (i = 0; i < numsock; i++)
    {
      if (FD_ISSET(service_socket[i], &rfds))
      {
        sockunlen = sizeof(sockun);
        accept_fd = accept(service_socket[i], (struct sockaddr *) &sockun,
			   &sockunlen);
        if (accept_fd < 0)
	  continue;		/*
				 * Ignore and stay alive 
				 */

        if ((flags = fcntl(accept_fd, F_GETFL, 0)) < 0)
        {
	  if (debug)
	    fprintf(stderr, "Could not get socket file mode: %s\n",
		    strerror(errno));
	  close(accept_fd);
	  continue;
        }
        if (fcntl(accept_fd, F_SETFL, flags | O_NONBLOCK) < 0)
        {
	  if (debug)
	    fprintf(stderr, "Could not set socket file mode: %s\n",
		    strerror(errno));
	  close(accept_fd);
	  continue;
        }

        /*
         * Create a new structure for the incoming request 
         */
        if (max_query + 1 > alloc_query)
        {
	  egd_query_old = egd_query;
	  if (alloc_query == 0)
	    egd_query = malloc((max_query + 1) * sizeof(egd_query_t));
	  else
	    egd_query =
	        realloc(egd_query_old,
		        (max_query + 1) * sizeof(egd_query_t));
	  if (!egd_query)
	  {
	    close(accept_fd);	/*
				 * Better this than complete shut down 
				 */
	    egd_query = egd_query_old;
	    continue;
	  }
	  alloc_query = max_query + 1;
        }
        /*
         * This routine is safe here, as in case of error we do "continue;"
         * in case of problems. So we have succesfully reached this point.
         */
        max_query++;
        egd_query[max_query - 1].fd = accept_fd;
        egd_query[max_query - 1].state = wait_command;
        fprintf(stderr, "Opening service fd %d\n", egd_query[max_query-1].fd);
      }
    }

  }

}
