/* scrypt.c
 *
 * jojo@farm9.com -- file encryption using Serpent, 160 bit key
 *
 *	compartmentilized and unrolled by <mephis5@softhome.net> 07/25/2001
 *		exchanged rijndael for serpent 
 *		adn crapy case insensitive key
 *		for sha1 hashed key to produce 160 bit key
 *
 *   Handles odd file sizes using padding, creates a header that
 *   has file size and some random numbers.  Why?  Encryption blocks
 *   are 16 bytes, so resulting file size always a multiple of 16
 *   bytes.  Weird side effect of header -- encrypting same
 *   file with same key results in totally different output.
 *
 *   Hardcoded ECB mode with 160 bit key.  
 *
 *  6 Apr 2001 -- added features suggested by Mike Cronnelly, allowing
 *    input from stdin (so you can pipe input)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>

#include "random2.h"
#include "serpent.h"
#include "sha1.h"

// 
// taken directly from rijndaeltest-fst.c
//
void rand_init(void)
{
	rand_seed(time(NULL), time(NULL) | ~0x0f);
}

#ifndef word32
#define word32  unsigned int
#endif

word32 rand_word32(void)
{
    return
        ((word32)(random_mwc() >> 7) << 24) | 
        ((word32)(random_mwc() >> 7) << 16) | 
        ((word32)(random_mwc() >> 7) << 8)  | 
         (word32)(random_mwc() >> 7);
}


static void usage() 
{
	printf( "To encrypt a file:\n" );
  	printf( "       scrypt -e <key> [<infile> [<outfile>]]\n" );
  	printf( "To decrypt a file:\n" );
  	printf( "       scrypt -d <key> [<infile> [<outfile>]]\n" );
}


BYTE establish_direction(int arg_count, char **arg_value)
{
	BYTE direction;
	
	if ((arg_count < 3) || (arg_count > 5)) 
    {
        usage();
        exit(1);
    }
  
	if (strcmp(arg_value[1], "-e") == 0)
    	direction = DIR_ENCRYPT;
    else if (strcmp(arg_value[1], "-d") == 0) 
        direction = DIR_DECRYPT;
	else 
	{
        usage();
        exit(1);
    }
	
	return direction;
}


void scrypt_encrypt_file(int argc, 
					char **argv, FILE *fpin, 
					FILE *fpout, keyInstance keyInst, 
					cipherInstance cipherInst){
	
	struct stat statbuf;
	BYTE header[100];
	BYTE block[4 * 4];
	int size = 0;
	int prevsize = 0;
	int r;
	
	// make header <file size> <random number>
	memset(block, 0, 16);
    rand_init();

    if ((argc > 3) && ((stat(argv[3], &statbuf) == -1) || (fpin == NULL))) 
	{
    	printf("**ERROR, file '%s' not found **\n", argv[3]);
        exit(1);
    } 
	else if (argc > 3) 
    	sprintf(header, "%d %d %d %d", (int)statbuf.st_size, rand_word32(), 
        rand_word32(), rand_word32());
    else 
    	sprintf(header, "zzz %d %d %d", rand_word32(), rand_word32(), 
			rand_word32());
		
    memcpy(&block[0], &header[0], 16);
    
	r = blockEncrypt(&cipherInst, &keyInst, block, 128, block);
    
	if (fwrite(block, 16, 1, fpout) != 1) 
	{
    	printf("**ERROR writing output header\n");
        exit(1);
    }
    
	// now we have that header out of the way, 
    // read in 16 byte blocks, encrypt each one
    while (1) 
	{
    	memset(block, 0, 16);
        size = fread(block, 1, 16, fpin);
        
		if (size > 0) 
		{
        	prevsize = size;
            blockEncrypt(&cipherInst, &keyInst, block, 128, block);
            
			if (fwrite(block, 16, 1, fpout) != 1) 
			{
            	printf("** ERROR writing data\n");
                exit(1);
            }
            
         }
         else 
         	break;
	}
    
	if (argc > 3) 
    	fclose( fpin );
    else 
	{
    	memset(block, 0, 16);
        sprintf(block, "%d ", prevsize);
        blockEncrypt(&cipherInst, &keyInst, block, 128, block);
            
        if (fwrite(block, 16, 1, fpout ) != 1) 
		{
        	printf("** ERROR writing data\n");
            exit(1);
        }
     }
         
     if (argc > 4) 
     	fclose(fpout);
}



void scrypt_decrypt_file(int argc, FILE *fpin, 
						 FILE *fpout, keyInstance keyInst, 
						 cipherInstance cipherInst){
	int hasprevblock = 0;
	int hasprevblock_next = 0;
	int writesize;
	int fsize;
	int size = 0;
	int r;
	BYTE prevblock[4 * 4];
	BYTE block[4 * 4];
	BYTE prevblock_next[4 * 4];
	
	
	// decrypt header which is <file size> <random number>
	if (fread( block, 16, 1, fpin ) != 1) 
	{
		printf( "** ERROR reading input\n");
		exit(1);
	}
        
	r = blockDecrypt(&cipherInst, &keyInst, block, 128, block);
	fsize = atoi( block );
	block[15] = 0;
		/* now we have that header out of the way, 
             read in 16 byte blocks, encrypt each one
        	 If fsize is zero, that means the file may have been
        	 encrypted from a command line pipe, where we did not
        	 know the size initially.  In that case we ignore the
        	 fsize of zero, and rely on the last block to tell us
        	 the actual size of the file (see above) */
	if (fsize == 0) 
	{
		while (1) 
		{
			memset(block, 0, 16);
			size = fread(block, 1, 16, fpin);
                
        	if (size == 16) 
			{
             	blockDecrypt(&cipherInst, &keyInst, block, 128, block);
                
            	if (hasprevblock_next) 
				{
                	if (fwrite(prevblock_next, 16, 1, fpout) != 1) 
					{
                    	printf("** ERROR writing data\n");
                     	exit(1);
					}
				}
                
				if (hasprevblock) 
				{
					memcpy(prevblock_next, prevblock, 16);
                	hasprevblock_next = 1;
				}
                
				memcpy(prevblock, block, 16);
                hasprevblock = 1;
			} 
			else 
			{
				blockDecrypt(&cipherInst, &keyInst, block, 128, block);
            	size = atoi(prevblock);
                
				if (hasprevblock_next) 
				{
                	if (fwrite( prevblock_next, size, 1, fpout ) != 1) 
					{
                    	printf("** ERROR writing data\n");
                        exit(1);
                    }
                }
                break;
            }
		}
	}
	else 
	{
    	while (fsize > 0) 
		{
        	memset(block, 0, 16);
            size = fread(block, 1, 16, fpin);
            
            if (size > 0) 
			{
            	blockDecrypt(&cipherInst, &keyInst, block, 128, block);
                
                if (fsize > 16) 
                	writesize = 16;
                else
                	writesize = fsize;
                
                if (fwrite(block, writesize, 1, fpout) != 1) 
				{
                	printf("** ERROR writing data\n");
                    exit(1);
                }
                
				if (writesize != 16) 
                	break;
							
                fsize -= size;
			}
			else 
            	break;
		}
	}
    
	if (argc > 3)
    	fclose(fpin);
    
	if (argc > 4)
    	fclose(fpout);
}


static BYTE convertKey(char k) 
{
	int kval = k % 16; 
  	
	if (kval < 10) 
		return( '0' + kval );
  	else
    	return( kval - 10 + 'A' );                                                                                
}        

          
static void populateKeyMaterial(char* text, BYTE* key) 
{ 
	char* textscanner = text;
	int i;                                                                                
	
	for (i = 0; i < MAX_KEY_SIZE; i++) 
	{
		*key++ = convertKey(*textscanner);
		textscanner++;
		
		if (*textscanner == 0)
			textscanner = text;  
	}     
	
	*key = 0;
}


int main(int argc, char* argv[]) 
{
    int r;
    keyInstance keyInst;
    cipherInstance cipherInst;
    BYTE direction;
    BYTE keyMaterial[320];
	FILE* fpin = stdin;
    FILE* fpout = stdout;
	
	if (argc > 3) 
	{
        fpin = fopen(argv[3], "rb");
    
		if (fpin == NULL)
            printf("**ERROR: cannot read input file\n" );
    }
  
	if (argc == 5) 
	{
        fpout = fopen(argv[4], "wb");
        
        if (fpout == NULL) 
		{
            printf("**ERROR: cannot open output file '%s'\n", argv[4]);
            exit(1);
        }
    }
	
	direction = establish_direction(argc, argv);
	populateKeyMaterial(hash_string_with_sha1(argv[2]), &keyMaterial[0]);
	
	r = makeKey(&keyInst, direction, 256, keyMaterial);
	if (r != TRUE) 
	{
        printf("**ERROR on makeKey: %d\n", r);
        exit(1);
    }
  
	r = cipherInit(&cipherInst, MODE_ECB, NULL);
    if (r != TRUE) 
	{
        printf("**ERROR on cipherInit: %d\n", r);
        exit(1);
    }
	
	if (direction == DIR_ENCRYPT) 
		scrypt_encrypt_file(argc, argv, fpin, fpout, keyInst, cipherInst);
		
    if (direction == DIR_DECRYPT)
        scrypt_decrypt_file(argc, fpin, fpout, keyInst, cipherInst);
	
    return 0;
}
