/* @(#)t16ref.c	1.4 (QUALCOMM) 03/21/01 */

/* Header for t16:  16 bit SOBER stream cipher. */
/*
This software is free for commercial and non-commercial use in
non-embedded applications subject to the following conditions.

Copyright remains vested in QUALCOMM Incorporated, and Copyright
notices in the code are not to be removed.  If this package is used in
a product, QUALCOMM should be given attribution as the author of the
SOBER encryption algorithm. This can be in the form of a textual
message at program startup or in documentation (online or textual)
provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the
   distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:  This product
   includes software developed by QUALCOMM Incorporated.

4. The software is not embedded in a device intended for a specific
   application, for example but not limited to communication devices
   (including mobile telephones), smart cards, cash registers, teller
   machines, network routers.

THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

The license and distribution terms for any publically available version
or derivative of this code cannot be changed, that is, this code cannot
simply be copied and put under another distribution license including
the GNU Public License.
*/

#define N 17
#define CONST_C 0x6996
#define WORDSIZE 16
#define WORD unsigned short

/* multiplication tables */
#include "multab.h"
#include "sbox.h"

WORD    Initial[N];   /* Initial contents of shift register -- key schedule */
WORD    R[N];         /* Working storage for the shift register */
unsigned char stcnt;  /* stutter count -- when == 0 next output stored */
WORD    stctrl;       /* used 2 bits at a time */
WORD    konst;        /* key dependent non-linear function */
int     r;            /* current offset in circular buffer */

/* external interface declarations */
void t16_key(unsigned char key[], int keylen);
void t16_genbytes(unsigned long frame, unsigned char buf[], int nbytes);

/*
 * FOLD is how many register cycles need to be performed after combining the
 * last byte of key and non-linear feedback.
 */
#define FOLD N        /* how many iterations of folding to do */
#define KEYP 15       /* where to insert key words */
#define FOLDP 4       /* where to insert non-linear feedback */

/* end of SOBER header */

/* cycle the contents of the shift register */

#define modN(a) ((a) < N ? (a) : (a)-N)
static void cycle(void)
{
    R[r] = MUL0xE382(R[modN(r+15)]) ^ R[modN(r+4)] ^ MUL0x673C(R[r]);
    if (++r == N)
        r = 0;
}

/* Return a non-linear function of some parts of the register.
 * The positions of the state bytes form a maximal span full positive
 * difference set, and are 0, 1, 6, 13, 16.
 */

static WORD
nltap(void)
{
    WORD   t;

    t = R[modN(r+0)] + R[modN(r+16)];
    t = SBox[t >> 8] ^ (t & 0xFF);
    t = (t + R[modN(r+1)] + R[modN(r+6)]) ^ konst;
    return (WORD) (t + R[modN(r+13)]);
}

/* load some key material into the register */

static void
loadkey(unsigned char key[], int keylen)
{
    int i;

    /* start folding in key, odd byte first if there is one */
    if ((keylen & 1) != 0)
    {
        R[modN(r+KEYP)] += key[0];
        cycle();
        R[modN(r+FOLDP)] ^= nltap();
    }
    for (i = keylen & 1; i < keylen; i += 2)
    {
        R[modN(r+KEYP)] += (key[i] << 8) + key[i+1];
        cycle();
        R[modN(r+FOLDP)] ^= nltap();
    }

    /* also fold in the length of the key */
    R[modN(r+KEYP)] += keylen;

    /* now diffuse */
    for (i = 0; i < FOLD; ++i)
    {
        cycle();
        R[modN(r+FOLDP)] ^= nltap();
    }
}

/* calculate initial contents of the shift register */

void
t16_key(unsigned char key[], int keylen)
{
    int i;

    /* fill the register with fibonacci numbers */
    R[0] = R[1] = 1;
    for (i = 2; i < N; i++)
        R[i] = R[i-1] + R[i-2];

    /* initialise the pointers and start folding in key */
    r = 0;
    konst = 0;
    loadkey(key, keylen);

    /* save state and key word for nonlinear function */
    cycle();
    konst = nltap();
    for (i = 0; i < N; i++)
        Initial[i] = R[modN(r+i)];
    stcnt = 0;
}

/* Fold in the per-frame key */

void
t16_seckey(unsigned char seckey[], int seckeylength)
{
    register int    i;

    /* copy initial contents */
    for (i = 0; i < N; i++)
        R[i] = Initial[i];
    r = 0;

    loadkey(seckey, seckeylength);

    stcnt = 0;
}

/* XOR pseudo-random bytes into buffer */

#define XORWORD(p,v) (p[0] ^= ((v) >> 8), p[1] ^= (v), p += 2)
void
t16_gen(unsigned char *buf, int nbytes)
{
    unsigned char   *endbuf;
    WORD        t = 0;

    /* assert((nbytes & 1) == 0) */
    endbuf = &buf[nbytes];
    while (buf < endbuf)
    {
        stctrl >>= 2;

        /* reprime stuttering if necessary */

        if (stcnt == 0)
        {
            stcnt = WORDSIZE/2;
            cycle();
            stctrl = nltap();
        }
        stcnt--;

        cycle();
        switch (stctrl & 0x3) {

        case 0: /* just waste a cycle and loop */
            continue;

        case 1: /* use the first output from two cycles */
            t = nltap() ^ CONST_C;
            cycle();
            break;

        case 2: /* use the second output from two cycles */
            cycle();
            t = nltap();
            break;

        case 3: /* return from one cycle */
            t = nltap() ^ ~CONST_C;
            break;

        }
        XORWORD(buf, t);
    }
}

/* encrypt/decrypt a frame of data */

void
t16_genbytes(unsigned long frame, unsigned char *buf, int nbytes)
{
    unsigned char   framebuf[4];

    framebuf[0] = (frame >> 24) & 0xFF;
    framebuf[1] = (frame >> 16) & 0xFF;
    framebuf[2] = (frame >>  8) & 0xFF;
    framebuf[3] = (frame) & 0xFF;
    t16_seckey(framebuf, 4);
    t16_gen(buf, nbytes);
}

#ifdef TEST
#include <stdio.h>
#include <string.h>
#include "hexlib.h"

/* test vectors */
typedef unsigned char   uchar;
typedef unsigned long   word32;
uchar   *testkey = (uchar *)"test key 128bits";
word32  testframe = 1L;
uchar   testbuf[20];
WORD	testkonst = 0x4b09;
char    *testout =
    "36 c3 df aa d1 63 41 46 01 34 05 13 8f 69 da 5e 21 45 44 fe";
char    *zeros =
    "00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00";
#define ITERATIONS 999999
char    *iterout =
    "11 31 c2 5c d6 37 58 8f 6f b1 77 65 04 2d f4 69 47 f2 e5 70";
uchar   bigbuf[1024*1024];

void
test_t16(void)
{
    long	i;

    /* test encryption */
    t16_key(testkey, strlen((char *)testkey));
    printf("%14s: %04x\n", "konst", konst);
    if (konst != testkonst) 
	printf("Expected %04x, got %04x.\n", testkonst, konst);
    t16_genbytes(testframe, testbuf, sizeof testbuf);
    hexprint("testbuf", testbuf, sizeof testbuf);
    hexcheck(testbuf, testout, sizeof testbuf);
    /* test decryption */
    t16_key(testkey, strlen((char *)testkey));
    t16_genbytes(testframe, testbuf, sizeof testbuf);
    hexprint("decrypted", testbuf, sizeof testbuf);
    hexcheck(testbuf, zeros, sizeof testbuf);
    /* test iteration */
    t16_key(testkey, strlen((char *)testkey));
    t16_genbytes(testframe, testbuf, sizeof testbuf);
    for (i = 0; i < ITERATIONS; ++i) {
	t16_key(testbuf, 16);
	t16_gen(testbuf, sizeof testbuf);
    }
    hexprint("iterated", testbuf, sizeof testbuf);
    hexcheck(testbuf, iterout, sizeof testbuf);
}

void
time_t16(void)
{
    word32      i;

    t16_key(testkey, strlen((char *)testkey));
    for (i = 0; i < 100; ++i)
        t16_genbytes(i, bigbuf, sizeof bigbuf);
}

#ifndef KEYLOAD
int
main(int ac, char **av)
{
    int         n;
    uchar       key[16];
    int         keysz;
    word32      hook;

    if (ac == 2 && strcmp(av[1], "-test") == 0) {
        test_t16();
        return nerrors;
    }
    if (ac == 2 && strcmp(av[1], "-time") == 0) {
        time_t16();
        return 0;
    }

    if (ac >= 2)
        hexread(key, av[1], keysz = strlen(av[1]) / 2);
    else
        hexread(key, "0000000000000000", keysz = 8);
    sscanf(ac >= 3 ? av[2] : "00000000", "%lx", &hook);
    sscanf(ac >= 4 ? av[3] : "10000", "%d", &n);

    t16_key(key, keysz);
    if (n > sizeof bigbuf) n = sizeof bigbuf;
    t16_genbytes(hook, bigbuf, n);
    hexbulk(bigbuf, n);
    return 0;
}
#else
#include "stdlib.h"
main(int ac, char **av) {
    int		i, j;
    int		n;
    char	c;
    WORD	state1, state2;
    uchar       key[16];

    sscanf(ac > 1 ? av[1] : "0^", "%x%c", &n, &c);
    for (i = 0; i < 2500; ++i) {
	/* random key */
	for (j = 0; j < sizeof key; ++j)
	    key[j] = random() & 0xFF;
	t16_key(key, sizeof key);
	t16_seckey("\0\0\0\0", 4);
	state1 = R[(r+n)%N];
	t16_seckey("\0\0\0\1", 4);
	state2 = R[(r+n)%N];
	printf("0x%04x\n",
	    c == '^' ?
		state1 ^ state2
		:
		(state1 - state2) & 0xFFFF);
    }
}
#endif
#endif
