/*
 * bc_proghandler.c
 *
 * Copyright (C) 2004-2006, J. Salvatore Testa II
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "bc_proghandler.h"
#include "bc_ui.h"
#include "debug.h"

#ifdef _WIN32
BOOL CALLBACK EnumWindowsProc(HWND handle, LPARAM lParam);
BOOL CALLBACK EnumWindowsProc2(HWND handle, LPARAM lParam);
extern TCHAR bc_initial_current_dir[ MAX_PATH ];
HANDLE hProcess = NULL;
HANDLE hTorProcess = NULL;

/* Tor */
gint tor_fd_stdin = 0;
gint tor_fd_stdout = 0;
gint tor_fd_stderr = 0;
GPid tor_pid = 0;
#endif

/* Crypto module. */
gint fd_stdin = 0;
gint fd_stdout = 0;
gint fd_stderr = 0;
GPid the_pid = 0;


guint event_source_id = 0;


GMutex *prog_mutex = NULL;
extern gboolean ui_state_func(GIOChannel *ioc,
                              GIOCondition cond,
                              gpointer data);
GIOChannel *ioc = NULL;

gchar *debug_file = NULL;


#ifdef _WIN32
int bc_spawn_tor(void) {
  GError *err = NULL;
  gchar **cmdline = NULL;
  GString *tor_exe_path = NULL;
  GString *tor_config_path = NULL;
  HANDLE hProcessSnap = NULL;
  PROCESSENTRY32 pe;
  unsigned int num_quarter_secs = 0;
  int hidden = 0;

  memset(&pe, 0, sizeof(pe));
  pe.dwSize = sizeof(PROCESSENTRY32);


  /* If Tor is already running, do not run it again. */
  if (hTorProcess != NULL)
    return -1;

  cmdline = g_new(gchar *, 4);
  tor_exe_path = g_string_new(bc_initial_current_dir);
  tor_exe_path = g_string_append(tor_exe_path, "\\");
  tor_exe_path = g_string_append(tor_exe_path, "tor.exe");

  tor_config_path = g_string_new(bc_initial_current_dir);
  tor_config_path = g_string_append(tor_config_path, "\\");
  tor_config_path = g_string_append(tor_config_path, "torrc.conf");

  cmdline[ 0 ] = tor_exe_path->str;
  cmdline[ 1 ] = "-f";
  cmdline[ 2 ] = tor_config_path->str;
  cmdline[ 3 ] = NULL;

  if (!g_spawn_async_with_pipes(NULL, cmdline, NULL, (GSpawnFlags)G_SPAWN_SEARCH_PATH, NULL, NULL, &tor_pid, &tor_fd_stdin, &tor_fd_stdout, &tor_fd_stderr, &err)) {
    bc_dialog("Could not execute tor!", GTK_MESSAGE_ERROR, NULL);
    bc_dialog(cmdline[ 0 ], GTK_MESSAGE_ERROR, NULL);
    exit(1);
  }

  g_string_free(tor_exe_path, TRUE);     tor_exe_path = NULL;
  g_string_free(tor_config_path, TRUE);  tor_config_path = NULL;
  g_free(cmdline);  cmdline = NULL;


  while ((hidden == 0) && (num_quarter_secs < 20)) {
    if (EnumWindows(EnumWindowsProc2, (LPARAM)&hidden) == 0)
      bc_dialog("EnumWindows failed!", GTK_MESSAGE_ERROR, NULL);

    if (hidden == 0)
      Sleep(250);

    num_quarter_secs++;
  }


  /* Open a handle to our newly created process... */
  hProcessSnap = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  if (hProcessSnap == INVALID_HANDLE_VALUE) {
    bc_dialog("CreateToolhelp32Snapshot failed.", GTK_MESSAGE_ERROR, NULL);
    return -1;
  }


  if (Process32First(hProcessSnap, &pe) == FALSE) {
    bc_dialog("Process32First failed.", GTK_MESSAGE_ERROR, NULL);
    CloseHandle (hProcessSnap);
    return -1;
  }

  hTorProcess = NULL;
  do {
    if (stricmp(pe.szExeFile, "tor.exe") == 0) {
      hTorProcess = OpenProcess(PROCESS_TERMINATE, FALSE, pe.th32ProcessID);
      if (hTorProcess == NULL)
        bc_dialog("OpenProcess failed.", GTK_MESSAGE_ERROR, NULL);
    }
  } while((hTorProcess == NULL) && Process32Next(hProcessSnap, &pe));

  CloseHandle(hProcessSnap);
  return 1;
}

void tor_kill() {
  if (tor_pid != 0) {
    g_spawn_close_pid(tor_pid);  tor_pid = 0;
  }
  if (hTorProcess != NULL) {
    TerminateProcess(hTorProcess, 666);  hTorProcess = NULL;
  }
}

BOOL CALLBACK EnumWindowsProc2(HWND handle, LPARAM lparam) {
  int *hidden = (int *)lparam;
  char windowtext[ 256 ];

  memset(windowtext, 0, sizeof(windowtext));

  GetWindowText(handle, windowtext, sizeof(windowtext) - 8);
  /*bc_dialog(windowtext, GTK_MESSAGE_INFO, NULL);*/
  if (strstr(windowtext, "tor.exe") != NULL) {
    ShowWindow(handle, SW_HIDE);
    /**hidden = 1;*/
  }

  return TRUE;
}

#endif


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int bc_spawn_crypto_module(void) {
  GError *err = NULL;
  gchar **cmdline = g_new(gchar *, 4);
  GString *bc_path = NULL;
#ifdef _WIN32
  unsigned int num_quarter_secs = 0;
  int hidden = 0;

  HANDLE hProcessSnap = NULL;
  PROCESSENTRY32 pe;

  memset(&pe, 0, sizeof(pe));
  pe.dwSize = sizeof(PROCESSENTRY32);
#endif

  bc_path = g_string_new("");
#ifdef _WIN32
  bc_path = g_string_append(bc_path, bc_initial_current_dir);
  bc_path = g_string_append(bc_path, "\\");
  bc_path = g_string_append(bc_path, "scatterchatmod.exe");
#else
  bc_path = g_string_append(bc_path, "scatterchatmod");
#endif

  cmdline[ 0 ] = bc_path->str;
  if (debug_file != NULL) {
    cmdline[ 1 ] = "--log";
    cmdline[ 2 ] = debug_file;
    cmdline[ 3 ] = NULL;
  } else
    cmdline[ 1 ] = NULL;


  if (!g_spawn_async_with_pipes(NULL, cmdline, NULL, (GSpawnFlags)G_SPAWN_SEARCH_PATH, NULL, NULL, &the_pid, &fd_stdin, &fd_stdout, &fd_stderr, &err)) {
    bc_dialog("Could not execute crypto module!", GTK_MESSAGE_ERROR, NULL);
    bc_dialog(cmdline[ 0 ], GTK_MESSAGE_ERROR, NULL);
    exit(1);
  }


#ifdef _WIN32

  while ((hidden == 0) && (num_quarter_secs < 20)) {
    if (EnumWindows(EnumWindowsProc, (LPARAM)&hidden) == 0)
      bc_dialog("EnumWindows failed!", GTK_MESSAGE_ERROR, NULL);

    if (hidden == 0)
      Sleep(250);

    num_quarter_secs++;
  }

  /* 20 quarter-seconds = 5 seconds */
  /*while ((handle == NULL) && (num_quarter_secs < 20)) {
    handle = FindWindowEx(NULL, NULL, NULL, bc_path->str);
    if (handle == NULL)
      handle = FindWindowEx(NULL, NULL, NULL, "blackchatmod.exe");

    if (handle == NULL)
      Sleep(250);

    num_quarter_secs++;
  }

  if (handle != NULL)
  ShowWindow(handle, SW_HIDE);*/


  /* Open a handle to our newly created process... */
  hProcessSnap = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  if (hProcessSnap == INVALID_HANDLE_VALUE) {
    bc_dialog("CreateToolhelp32Snapshot failed.", GTK_MESSAGE_ERROR, NULL);
    return -1;
  }


  if (Process32First(hProcessSnap, &pe) == FALSE) {
    bc_dialog("Process32First failed.", GTK_MESSAGE_ERROR, NULL);
    CloseHandle (hProcessSnap);
    return -1;
  }

  hProcess = NULL;
  do {
    /*bc_dialog(pe.szExeFile, GTK_MESSAGE_ERROR, NULL);*/

    if (stricmp(pe.szExeFile, "scatterchatmod.") == 0) {
      hProcess = OpenProcess(PROCESS_TERMINATE, FALSE, pe.th32ProcessID);
      if (hProcess == NULL)
        bc_dialog("OpenProcess failed.", GTK_MESSAGE_ERROR, NULL);
      /*bc_dialog("Opened handle!", GTK_MESSAGE_INFO, NULL);*/
    }
  } while((hProcess == NULL) && Process32Next(hProcessSnap, &pe));

  CloseHandle(hProcessSnap);

#endif
  if (debug_file != NULL) {
    g_free(debug_file);  debug_file = NULL;
  }
  g_free(cmdline);  cmdline = NULL;
  g_string_free(bc_path, TRUE);  bc_path = NULL;

  sync_stderr();

  if (ioc == NULL) {
    ioc = g_io_channel_unix_new(fd_stderr);
    event_source_id = g_io_add_watch(ioc, G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL, ui_state_func, NULL);
  }

  return 1;
}


#ifdef _WIN32
/* On Windows, the g_async_spawn_with_pipes will make the crypto module window
 * appear, and we need to hide it.  This function will be called by the Win32
 * API EnumWindows() function.  We will search for our spawned window, and
 * hide it if found. */
BOOL CALLBACK EnumWindowsProc(HWND handle, LPARAM lparam) {
  int *hidden = (int *)lparam;
  char windowtext[ 256 ];

  memset(windowtext, 0, sizeof(windowtext));

  GetWindowText(handle, windowtext, sizeof(windowtext) - 8);
  /*bc_dialog(windowtext, GTK_MESSAGE_INFO, NULL);*/
  if (strstr(windowtext, "scatterchatmod") != NULL) {
    ShowWindow(handle, SW_HIDE);
    /**hidden = 1;*/
  }

  return TRUE;
}
#endif


/* Reads all the sizes in from the module. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int get_sizes(size_t *size_array, size_t size_array_len) {
  unsigned int i = 0;
  unsigned int j = 0;
  unsigned int k = 0;
  int ret = 0;

  char line[ 256 ];
  char text_size[ 16 ];

  memset(line, 0, sizeof(line));
  memset(text_size, 0, sizeof(text_size));


  /* Read in one line from the module.  This line will have a space-delimited
   * list of sizes. */
  mod_readline(line, sizeof(line) - 1);


  /* Go through each character in the line and parse out the groups of numbers
   * into the 'text_size' array. */
  for(i = 0; i < strlen(line); i++) {

    /* Continue putting characters into text_size as long as there is still
     * room and the space delimiter is not yet found. */
    if ((line[ i ] != ' ') && (j < sizeof(text_size) - 1)) {
      text_size[ j ] = line[ i ];
      j++;
    } else if ((line[ i ] == ' ') && (k < size_array_len - 1)) {
      size_array[ k ] = (size_t)atoi(text_size);
      k++;

      /*printf("text_size: [%s]\n", text_size);*/
      memset(text_size, 0, sizeof(text_size));
      j = 0;
    }

  }

  /* Return the sum of all the sizes. */
  for (i = 0; i < k; i++)
    ret += size_array[ i ];

  return ret;
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void mod_kill() {
  if (the_pid != 0) {
    g_spawn_close_pid(the_pid);  the_pid = 0;
  }
#ifdef _WIN32
  if (hProcess != NULL) {
    TerminateProcess(hProcess, 666);  hProcess = NULL;
  }
#endif
}


/* Reads a specified about of bytes from the module.  Returns the number of
 * bytes actually read, or -1 on error. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int mod_readdata(char *buffer, size_t buffer_len, size_t bytes_to_read) {
   int ret = 0;

   if (bytes_to_read > buffer_len)
      return -1;

   ret = read(fd_stdout, buffer, bytes_to_read);
   while ((ret >= 0) && (ret < bytes_to_read)) {
     int n = read(fd_stdout, buffer + ret, bytes_to_read - ret);
     if (n < 0)
       return -1;
     ret += n;
   }


   /* There seems to be an elusive bug where the handshake setup fails because
    * not all the bytes make it out of the module.  This might help us see
    * what's causing it... */
   if (ret != bytes_to_read) {
     GString *str = NULL;
     str = g_string_new("mod_readdata: ");
     g_string_append_printf(str, "asked for %u bytes, returning %d",
			    (unsigned int)bytes_to_read, ret);
     bc_dialog(str->str, GTK_MESSAGE_INFO, NULL);
     g_string_free(str, TRUE);
   }

   return ret;
}


/* Shuts down the underlying module. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void mod_quit(void) {
  char buffer[ 256 ];

  memset(buffer, 0, sizeof(buffer));

  mod_writeln("SHUTDOWN");
  if (mod_readline(buffer, sizeof(buffer) - 8) < 0)
    mod_kill();

  close(fd_stdout);
  close(fd_stderr);
  close(fd_stdin);

  mod_kill();
  g_spawn_close_pid(the_pid);
}


/* Reads one line in from the module. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int mod_readline(char *buffer, size_t buffer_len) {
  char one_char_buf[ 2 ] = { '\0', '\0' };
  unsigned int i = 0;
  unsigned int crlf_found = 0;

  while(!crlf_found && (i < buffer_len - 1)) {

    read(fd_stdout, one_char_buf, 1);

    buffer[ i ] = one_char_buf[ 0 ];
    i++;

    if ((i >= 2) && (buffer[ i - 2 ] == '\r') && (buffer[ i - 1 ] == '\n')) {
      crlf_found = 1;
    }

  }

  buffer[ i ] = '\0';

  /* If a CRLF was read in, truncate it. */
  if (crlf_found == 1) {
    trim_crlf(buffer);
    return i - 3;
  } else
    return -1;
}


/* Reads one line in from the module's stderr. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int mod_stderr_readline(char *buffer, size_t buffer_len) {
  char one_char_buf[ 2 ] = { '\0', '\0' };
  unsigned int i = 0;
  unsigned int crlf_found = 0;

  while((crlf_found == 0) && (i < buffer_len - 1)) {
    read(fd_stderr, one_char_buf, 1);

    buffer[ i ] = one_char_buf[ 0 ];
    i++;

    if ((i >= 2) && (buffer[ i - 2 ] == '\r') && (buffer[ i - 1 ] == '\n')) {
      crlf_found = 1;
    }
  }

  buffer[ i ] = '\0';

  /* If a CRLF was read in, truncate it. */
  if (crlf_found == 1) {
    trim_crlf(buffer);
    return i - 3;
  } else
    return -1;
}



/* Writes a string to the underlying module. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void mod_write(char *string) {
  write(fd_stdin, string, strlen(string));
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void mod_writeln(char *string) {
  write(fd_stdin, string, strlen(string));
#ifdef _WIN32
  write(fd_stdin, "\r\n", strlen("\r\n"));
#else
  write(fd_stdin, "\n", strlen("\n"));
#endif
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int parseOK(char *line, unsigned int *type, unsigned int *num_results) {
  char *space_pos = NULL;

  /* If we got a plain "OK" with no type or number of results, set them
   * to zero and simply return. */
  if (strcmp(line, "OK") == 0) {
    *type = 0;
    *num_results = 0;
    return 1;
  }


  if ((strstr(line, "OK ") != line) || (strlen(line) < 4))
    return -1;

  space_pos = strchr(line + 3, ' ');
  if (space_pos == NULL)
    return -1;

  *space_pos = '\0';
  *type = atoi((const char *)line + 3);
  *num_results = atoi(space_pos + 1);

  *space_pos = ' ';  // Restore the original input.
  return 1;
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void trim_crlf(char *buffer) {
  size_t buffer_len = strlen(buffer);

  if (buffer_len < 2)
    return;

  if ((buffer[ buffer_len - 2 ] == '\r') &&
      (buffer[ buffer_len - 1 ] == '\n')) {
    buffer[ buffer_len - 2 ] = '\0';
  }
}


void mod_lock(void) {
  g_mutex_lock(prog_mutex);
}


void mod_unlock(void) {
  g_mutex_unlock(prog_mutex);
}


void sync_stderr(void) {
  unsigned int continue_flag = 0;

  char buffer[ 32 ];

  memset(buffer, 0, sizeof(buffer));

  /*printf("about to loop...\n");*/
  continue_flag = 1;
  while(continue_flag == 1) {
    mod_stderr_readline(buffer, sizeof(buffer) - 1);
    /*printf("stderr: [%s]\n", buffer);*/
    /*MessageBox(NULL, buffer, "stderr", MB_OK);*/
    if (strstr(buffer, "START_STDERR") != NULL) {
      continue_flag = 0;
      /*MessageBox(NULL, "START_STDERR FOUND!", "X", MB_OK);*/
    }
  }
}

void write_tor_config(gchar *config) {
#ifdef _WIN32
  FILE *hFile = NULL;
  GString *my_config = NULL;
  gchar *ptr = NULL;
  GString *config_path = g_string_new(bc_initial_current_dir);


  ptr = strstr((char *)config, "$datadir");
  if (ptr != NULL) {
    my_config = g_string_new(NULL);
    my_config = g_string_append_len(my_config, config, ptr - config);
    my_config = g_string_append(my_config, bc_initial_current_dir);
    my_config = g_string_append(my_config, "\\tdata\\");
    my_config = g_string_append(my_config, ptr + 8);
  } else
    my_config = g_string_new(config);


  config_path = g_string_append(config_path, "\\");
  config_path = g_string_append(config_path, "torrc.conf");

  hFile = fopen(config_path->str, "w");
  fwrite(my_config->str, sizeof(gchar), my_config->len, hFile);
  fclose(hFile);  hFile = NULL;


  g_string_free(config_path, TRUE);  config_path = NULL;
  g_string_free(my_config, TRUE);  my_config = NULL;
#endif
}
