/*
 * bc_main.c
 *
 * Copyright (C) 2004-2005 J. Salvatore Testa II
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "bc_main.h"

/**
 * Holds the pending messages until connections are either established or
 * rejected.  Keys are normalized recipient screen names; values are the
 * queued message strings.
 */
GHashTable *pending_messages = NULL;
GMutex *pending_messages_mutex = NULL;

extern gchar *encrypt_state_encrypted;
extern gchar *encrypt_state_transition_1;
extern gchar *encrypt_state_transition_2;
extern gchar *encrypt_state_plaintext;

static gboolean bc_plugin_load(GaimPlugin *plugin);

static GaimPluginInfo bc_plugin_info = {
  GAIM_PLUGIN_MAGIC,
  GAIM_MAJOR_VERSION,
  GAIM_MINOR_VERSION,
  GAIM_PLUGIN_STANDARD,  /* type */
  GAIM_GTK_PLUGIN_TYPE,  /* ui_requirement */
  0,                     /* flags */
  NULL,                  /* dependencies */
  GAIM_PRIORITY_DEFAULT, /* priority */
  "scatterchat",         /* id */
  "scatterchat",         /* name */
  VERSION,               /* version */
  "Provides secure, end-to-end encryption",  /* summary */
  "Scatter Chat r0x0rz!\n",                     /* description */
  "J. Salvatore Testa II",                      /* author */
  "http://www.scatterchat.com/",   /* homepage */
  bc_plugin_load,        /* load */
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL
};


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
static gboolean bc_plugin_load(GaimPlugin *plugin) {
  void *conv_handle = NULL;
  void *conn_handle = NULL;
  void *blist_handle = NULL;
  int ret = 0;
  /*  unsigned int continue_flag = 0;*/

  char buffer[ 32 ];

  memset(buffer, 0, sizeof(buffer));

  conv_handle = gaim_conversations_get_handle();
  conn_handle = gaim_connections_get_handle();
  blist_handle = gaim_blist_get_handle();

  gaim_signal_connect(conv_handle, "sending-im-msg", plugin,
		      GAIM_CALLBACK(send_handler), NULL);
  gaim_signal_connect(conv_handle, "receiving-im-msg", plugin,
  		      GAIM_CALLBACK(get_handler), NULL);
  gaim_signal_connect(conv_handle, "conversation-created", plugin,
                       GAIM_CALLBACK(new_conversation), NULL);
  gaim_signal_connect(conv_handle, "deleting-conversation", plugin, 
                       GAIM_CALLBACK(del_conversation), NULL);
  gaim_signal_connect(conn_handle, "signing-on", plugin,
		      GAIM_CALLBACK(signon_handler), NULL);
  gaim_signal_connect(conn_handle, "signed-off", plugin,
		      GAIM_CALLBACK(signoff_handler), NULL);
  gaim_signal_connect(blist_handle, "buddy-signed-off", plugin,
		      GAIM_CALLBACK(buddy_signoff_handler), NULL);
  gaim_signal_connect(gaim_get_core(), "quitting", plugin,
		      GAIM_CALLBACK(quitting_handler), NULL);

  ret = bc_spawn_crypto_module();
  if (ret == 0) {
     bc_dialog("Error while loading encryption module!  Program terminating...", GTK_MESSAGE_ERROR, NULL);
     exit(666);
  }


  /* The first line in the output has the module's version... */
  mod_readline(buffer, sizeof(buffer) - 1);
  /*printf("Using scatterchat module version %s (API version ", buffer);*/
  gaim_debug(GAIM_DEBUG_INFO, "bc_main.c", "Using scatterchat module version %s (API version ", buffer);

  /* ... and the second line has the API version. */
  mod_readline(buffer, sizeof(buffer) - 1);
  /*printf("%s).\n", buffer);*/
  gaim_debug(GAIM_DEBUG_INFO, "bc_main.c", "%s).\n", buffer);

  /* We write a NOOP first and see that we get back an OK as a sanity check. */
  mod_writeln("NOOP");
  mod_readline(buffer, sizeof(buffer) - 1);
  if (strcmp(buffer, "OK") != 0) {
     bc_dialog("Error while loading encryption module: NOOP failed.", GTK_MESSAGE_ERROR, NULL);
     exit(666);
  }

  bc_register_icons();

  pending_messages_mutex = g_mutex_new();
  pending_messages = g_hash_table_new_full(g_str_hash,
                                           g_str_equal,
                                           g_free,
                                           g_free);

  return TRUE;
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void bc_plugin_init(void) {
  GaimPlugin *plugin = NULL;

  plugin = gaim_plugin_new(TRUE, NULL);
  plugin->info = &bc_plugin_info;
  gaim_plugin_register(plugin);
  gaim_plugin_load(plugin);

  show_um_init_window(1);

}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void send_handler(GaimAccount *acct, char *who, char **message, void *m) {
  unsigned int i = 0;
  char *data_ptr = NULL;
  size_t total_size = 0;
  unsigned int offset = 0;
  unsigned int type = 0;
  unsigned int num_results = 0;
  GString *chunk_ptr = NULL;
  GaimConversation *c = NULL;
  GaimConvIm *im = NULL;

  char line[ 2048 ];
  char text_len[ 16 ];
  size_t size_array[ 4 ];

  memset(line, 0, sizeof(line));
  memset(text_len, 0, sizeof(text_len));
  memset(size_array, 0, sizeof(size_array));


  c = gaim_find_conversation_with_account(
        (char *)gaim_normalize(NULL, who), acct);
  if (c == NULL) {
    /*c = gaim_conversation_new(GAIM_CONV_IM, acct->gc->account,
      (unsigned char *)gaim_normalize(NULL, who));*/
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Error: conversation window not found.\n");
  }


  /* If we're not in plaintext mode, then we can't send more than 500
   * characters per message. */
  if (!bc_is_state_plaintext(c) && (strlen(*message) > 500)) {
    GString *s = g_string_new("Error:  You may not send more than 500 characters per IM.  You exceed the 500-character limit by ");
    g_string_append_printf(s, "%u characters.", (unsigned int)(strlen(*message) - 500));

    bc_dialog(s->str, GTK_MESSAGE_ERROR, NULL);
    g_string_free(s, TRUE);  s = NULL;

    g_free(*message); *message = NULL;
    return;
  }


  /* If state is encrypted or plaintext, then write the message into the
   * IM window directly.  Otherwise, queue the message for display later on
   * when the connection is established or rejected. */
  if (bc_is_state_encrypted(c) || bc_is_state_plaintext(c)) {
    im = GAIM_CONV_IM(c);
    gaim_conv_im_write(im, NULL, *message, GAIM_MESSAGE_SEND, time(NULL));
  } else {
    gchar *key = (gchar *)gaim_normalize(c->account, c->name);

    g_mutex_lock(pending_messages_mutex);
    if (g_hash_table_lookup(pending_messages, (gconstpointer)key) == NULL) {
      g_hash_table_insert(pending_messages, g_strdup(key), g_strdup(*message));
    } else {
      bc_system_message_with_conv(c, "Error: cannot send additional message while waiting for session to begin.");
    }
    g_mutex_unlock(pending_messages_mutex);
  }

  /*encrypt_state = g_hash_table_lookup(c->data, "bc_encrypt_state");
  if (encrypt_state == NULL) {
    * This is a fatal error.  The program must abort since it is not acting
     * sane. *
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Big oops!\n");
    printf("Big oops!\n");
    exit(666);
  }*/


  /* If the encryption state is FALSE, then send the IM plainly and
   * directly. */
  if (bc_is_state_plaintext(c)) {
    /*if (g_ascii_strncasecmp((gchar *)encrypt_state, _("FALSE"), 5) == 0) {*/
    /*printf("Sending plaintext message...%d\n", g_ascii_strncasecmp((gchar *)encrypt_state, _("FALSE"), 5));*/
    serv_send_im(acct->gc, who, *message, 0);
    return;
  /* If the encryption state is not FALSE, and its not TRUE, then we have a
   * serious error... */
  }

  bc_ensure_state_valid(c);


  /* Now that we sent an encrypted message, lock the state so that it can't
   * be changed back to plaintext (either accidentally or maliciously). */
  /*bc_lock_encryption_state(c);*/

  /* mingw doesn't like snprintf... */
  if (sizeof(text_len) >= 16)
    sprintf(text_len, "%u", (unsigned int)strlen(*message));
  else
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "text_len not big enough!\n");

  /* Pass the cleartext message that we wish to encrypt. */
  mod_write("SEND ");
  mod_write(who);
  mod_write("/");
  mod_write(acct->protocol_id);
  mod_write("|");
  mod_writeln(text_len);
  mod_writeln(*message);

  /* Now that the message has been passed to the crypto module, we don't need
   * it anymore, so clear it. */
  g_free(*message);  *message = NULL;

  mod_readline(line, sizeof(line) - 8);
  if (strstr(line, "ERROR") == line) {
    /*printf("Error: %s\n", line);*/
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Error: %s\n", line);
    return;
  }

  /*printf("send_handler: read line: [%s]\n", line);*/

  /* Since we know now that an error wasn't returned by the module, we can
   * parse the "OK" line and find its type and number of results. */
  parseOK(line, &type, &num_results);

  /* Add up all the result sizes to see how much we need to read in. */
  total_size = get_sizes(size_array, sizeof(size_array));

  data_ptr = (char *)calloc(total_size + 8, sizeof(char));
  if (data_ptr == NULL) {
    fprintf(stderr, "Can't calloc!\n");
    exit(1);
  }

  /* Read in all the data at once. */
  mod_readdata(data_ptr, total_size, total_size);
  mod_readline(line, sizeof(line) - 1);
  /*printf("send_handler: readdata: [%s]\n", data_ptr);*/

  /* Send each of the results to the buddy. */
  offset = 0;
  for(i = 0; i < num_results; i++) {
    chunk_ptr = g_string_sized_new(size_array[ i ]);
    chunk_ptr = g_string_append_len(chunk_ptr, data_ptr + offset,
				    size_array[ i ]);
    serv_send_im(acct->gc, who, chunk_ptr->str, 0);
    g_string_free(chunk_ptr, TRUE);  chunk_ptr = NULL;

    offset += size_array[ i ];
  }

  free(data_ptr);  data_ptr = NULL;
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
gboolean get_handler(GaimAccount *acct, char **who, char **message,
		     GaimConvImFlags flags, void *m) {
  unsigned int type = 0;
  unsigned int num_results = 0;
  unsigned int total_size = 0;
  unsigned int offset = 0;
  unsigned int i = 0;
  unsigned int message_len = 0;
  char *data_ptr = NULL;
  GString *chunk_ptr = NULL;
  GaimConversation *c = NULL;
  char *font_tag = NULL;
  char *end_tag_pos = NULL;
  unsigned int parsed = 0;
  char *tag = NULL;
  char *original_message = NULL;

  char line[ 256 ];
  char text_len[ 16 ];
  size_t size_array[ 4 ];

  memset(line, 0, sizeof(line));
  memset(text_len, 0, sizeof(text_len));
  memset(size_array, 0, sizeof(size_array));

  /*printf("got from %s: [%s][%u]\n", *who, *message, strlen(*message));*/

  message_len = strlen(*message);

  /* This happens on Yahoo! for some reason... */
  if (message_len == 0) {
    /*printf("Message of size 0 found!  Returning...\n");*/
    return TRUE;
  }


  /***************************************************************************/


  /* Save an unmodified copy of the original message so that we can print it
   * in the case that we have a plaintext message and we're in plaintext mode
   * (we won't know this until after the crypto module parses it. */
  original_message = g_strdup(*message);


  /* MSN will give us back a couple <FONT> tags.  We'll parse them out here. */
  font_tag = strstr(*message, "<FONT");
  while (font_tag != NULL) {
    end_tag_pos = strchr(*message, '>');
    if (end_tag_pos != NULL) {
      g_strlcpy(*message, end_tag_pos + 1, message_len);
      /*printf("parsed: [%s]\n", *message);*/
      font_tag = strstr(*message, "<FONT");
    } else
      font_tag = NULL;

    /*printf("looping...\n");*/
    parsed = 1;
  }

  font_tag = strstr(*message, "</FONT>");
  /*printf("ending tag: [%s]\n", font_tag);*/
  if (font_tag != NULL) {
    *font_tag = '\0';
    /*printf("DONE: [%s]\n", *message);*/
    parsed = 1;
  }


  /* Jabber parsing... */
  tag = strstr(*message, "<body");
  if (tag != NULL) {
    end_tag_pos = strchr(tag, '>');
    if (end_tag_pos != NULL) {
      g_strlcpy(*message, end_tag_pos + 1, message_len);
      parsed = 1;
    }
  }

  tag = strstr(*message, "</body>");
  if (tag != NULL) {
    *tag = '\0';
    parsed = 1;
  }

  /***************************************************************************/


  /* Since some parsing was done on the message, update its size. */
  if (parsed == 1)
    message_len = strlen(*message);



  c = gaim_find_conversation_with_account((char *)gaim_normalize(NULL, *who), acct);
  if (c == NULL) {
    c = gaim_conversation_new(GAIM_CONV_IM, acct,
			      (char *)gaim_normalize(NULL, *who));
    /*printf("creating new conversation!\n");*/
  }
  

  /*printf("\t\tGET_MESSAGE: [%s]\n\n", raw_message);*/

  /* mingw doesn't like snprintf... */
  if (sizeof(text_len) >= 16)
    sprintf(text_len, "%u", message_len);
  else
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "text_len not big enough!\n");
  

  mod_write("PROCESS ");
  mod_write(*who);
  mod_write("/");
  mod_write(acct->protocol_id);
  mod_write("|");
  mod_writeln(text_len);
  mod_writeln(*message);

  g_free(*message);  *message = NULL;

  mod_readline(line, sizeof(line) - 8);
  if (strstr(line, "ERROR") == line) {
    GString *str = g_string_new(NULL);

    g_string_append_printf(str, "Internal error: %s", line);
    bc_system_message(c->name, str->str);
    g_string_free(str, TRUE);  str = NULL;

    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Error: %s!\n", line);
    g_free(original_message);  original_message = NULL;
    return FALSE;
  }


  parseOK(line, &type, &num_results);


  /* If we didn't get a plaintext message (ie: an encrypted message, or
   * some kind of crypto handshake info), then we need to switch to
   * encryption mode and lock it so that it cannot be downgraded.  No
   * harm is done here if the state was already encrypted & locked. */
  /*if (type != BC_PROCESS_MESSAGE_CLEARTEXT) {
    bc_enable_encryption_state(c);
    bc_lock_encryption_state(c);
    }*/


  /* If some kind of non-actionable message was received, then do nothing. */
  if (type == BC_PROCESS_MESSAGE_NONACTIONABLE_HANDSHAKE) {
    /*printf("Got type 4.\n");*/
    g_free(*message);  *message = NULL;
    g_free(original_message);  original_message = NULL;
    return FALSE;
  }

  /* Count up the total size of all the messages returned from the module so
   * we can read them in all at once. */
  total_size = get_sizes(size_array, sizeof(size_array));

  data_ptr = (char *)calloc(total_size + 8, sizeof(char));
  if (data_ptr == NULL) {
    fprintf(stderr, "Can't calloc!\n");
    exit(1);
  }

  if (mod_readdata(data_ptr, total_size, total_size) < 0) {
    GString *str = g_string_new(NULL);

    g_string_append_printf(str, "Internal error!: could not read data of size %d!", total_size);
    bc_system_message(c->name, str->str);
    g_string_free(str, TRUE);  str = NULL;
  }

  mod_readline(line, sizeof(line) - 1);

  /* If we got the public key from the buddy, we must handle it... */
  if (type == BC_PROCESS_MESSAGE_PUBLICKEY) {
    /*printf("fp: %s\n", data_ptr);*/

    /* If the dialog timed out (or another error occurred), then explicitly
     * reject the key.  Note that the channel may already be expired, but if
     * not, this will make sure it no longer exists. */
    if (!handle_fingerprint(acct, c, *who, data_ptr)) {
      mod_write("KEY_REJECT ");
      mod_write(*who);
      mod_write("/");
      mod_writeln(acct->protocol_id);

      mod_readline(line, sizeof(line) - 8);
      if (strstr(line, "ERROR") == line) {
	GString *str = g_string_new(NULL);

	g_string_append_printf(str, "Error: %s", line);
	bc_system_message(c->name, str->str);
	g_string_free(str, TRUE);  str = NULL;

	gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::get_handler",
		   "Error: %s\n", line);
      }
    }


  /* If we got an actionable handshake message, we must send each result back
   * to the buddy in seperate IMs. */
  } else if (type == BC_PROCESS_MESSAGE_ACTIONABLE_HANDSHAKE) {
    /*send(socket, data_ptr, size_array[ i ], 0);*/
    for(i = 0; i < num_results; i++) {
      /*int k = 0;
      printf("\t\tblock: ");
	for (k = 0; k < size_array[ i ]; k++)
	printf("%c", *(data_ptr + offset + k));
	printf("\n\n");*/

      chunk_ptr = g_string_sized_new(size_array[ i ]);
      chunk_ptr = g_string_append_len(chunk_ptr, data_ptr + offset,
				      size_array[ i ]);
      serv_send_im(acct->gc, *who, chunk_ptr->str, 0);
      g_string_free(chunk_ptr, TRUE);

      /*send(socket, data_ptr + offset, size_array[ i ], 0);*/
      /*usleep(333333);*/  /* Sleep 1/3rd of a second. */
      /*usleep(500000);*/
      offset += size_array[ i ];
    }

  /* If we received an encrypted message, we write it into the IM window. */
  } else if (type == BC_PROCESS_MESSAGE_ENCRYPTED) {
    char *umid = NULL;

    /*printf("crypted message: [%s]\n", data_ptr);*/

    umid = get_umid(*who, acct);
    gaim_conv_im_write(GAIM_CONV_IM(c), umid, data_ptr,
		       GAIM_MESSAGE_RECV, time(NULL));
    free(umid);  umid = NULL;

  /* A plaintext message was sent to us! */
  } else if (type == BC_PROCESS_MESSAGE_CLEARTEXT) {

    /* If we got a plaintext message while we're in encryption mode, then
     * we must reject the message.  Locked mode makes no difference, because
     * the user may have just switched to encrypted mode, and was just about
     * to send an encrypted message while this one came in.  The message
     * should still be rejected in this case since the user might get
     * confused if we switch back to plaintext (they may send sensitive
     * information in the clear!) */
    if (bc_is_state_encrypted(c)) {
      GString *temp = g_string_new("");
      g_string_append_printf(temp, "WARNING: You have received a "
			     "plaintext message from %s while you are "
			     "in Encrypted mode.  This may have "
			     "happened because of one of three "
			     "reasons:\n\n1.) %s does not have the "
			     "capability to send encrypted messages.\n"
			     "2.) %s is trying to TRICK you into "
			     "sending information back to him/her "
			     "without using encryption.\n"
			     "3.) A legitimate communications error "
			     "has occured.\n\nThe message received "
			     "from %s is: [%s]", *who, *who, *who, *who,
			     data_ptr);
      bc_system_message(c->name, temp->str);
      g_string_free(temp, TRUE);  temp = NULL;

   /* We are not in the encrypted state, so it is safe to write the IM
    * message into the window. */
    } else {
      /*printf("plaintext message: [%s]\n", data_ptr);*/
      /* Note that we will print the original message since we might have
       * parsed out legitimate HTML tags before processing. */
      gaim_conv_im_write(GAIM_CONV_IM(c), *who, original_message,
			 GAIM_MESSAGE_RECV, time(NULL));
    }

  }

  free(data_ptr);  data_ptr = NULL;
  g_free(*message);  *message = NULL;
  g_free(original_message);  original_message = NULL;
  return TRUE;
}


/* Displays a key fingerprint to the user for them to accept or reject.. */
/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
int handle_fingerprint(GaimAccount *acct, GaimConversation *c,
			char *who, char *data) {
  unsigned int type = 0;
  unsigned int num_results = 0;
  unsigned int total_size = 0;
  unsigned int i = 0;
  unsigned int offset = 0;
  char *data_ptr = NULL;
  GString *chunk_ptr = NULL;
  char *prefix_pos = NULL;

  size_t size_array[ 4 ];
  char line[ 256 ];
  struct NEW_FP_STRUCT new_fp_struct;

  memset(size_array, 0, sizeof(size_array));
  memset(line, 0, sizeof(line));
  memset(&new_fp_struct, 0, sizeof(struct NEW_FP_STRUCT));


  /* Fill in the ID & fingerprint so that the UI can display it. */
  g_strlcpy(new_fp_struct.umid, (const gchar *)gaim_normalize(NULL, who),
	    sizeof(new_fp_struct.umid) - 8);
  g_strlcpy(new_fp_struct.fingerprint, data,
	    sizeof(new_fp_struct.fingerprint) - 8);

  g_strlcpy(new_fp_struct.protocol, acct->protocol_id,
	    sizeof(new_fp_struct.protocol) - 8);


  /* If the protocol begins with the prefix, parse it out. */
  prefix_pos = strstr(new_fp_struct.protocol, "prpl-");
  if (prefix_pos != NULL) {
    g_strlcpy(new_fp_struct.protocol, new_fp_struct.protocol + 5,
	      sizeof(new_fp_struct.protocol) - 8);
  }


  /* Prompt the user with a UI window.  The NEW_FP_STRUCT argument will hold
   * the user's choice upon return. */
  show_fingerprint_window(&new_fp_struct);

  /*printf("bc_main: trigger is: %u\n", new_fp_struct.trigger);*/

  /* If the dialog timed out before the user chose something, just return. */
  if (new_fp_struct.trigger == 1) {
    /*mod_write("STATUS ");
    mod_writeln(who);
    mod_readline(line, sizeof(line) - 8);

    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::handle_fingerprint",
    "Key rejected due to timeout [%s]\n", line);*/
    /*printf("RETURNING...\n");*/
    return 0;
  }

  /*printf("CONTINUING...[%u]\n", new_fp_struct.action);*/


  /* If the user chose to accept the key fingerprint permanently, tell the
   * module to add it to the key database. */
  if (new_fp_struct.action == ACTION_ACCEPT_PERMANENTLY) {
    GString *string = g_string_new("");

    /*printf("User accepted permanently, adding to DB...\n");*/
    mod_write("KEY_ACCEPT_PERMANENTLY ");
    mod_write(who);
    mod_write("/");
    mod_write(acct->protocol_id);
    mod_write("|");
    mod_writeln(new_fp_struct.umid);

    g_string_append_printf(string, "You have permanently accepted %s's (%s) "
			   "key, identified by its fingerprint, [%s], and "
			   "added it to your key database.  If you "
			   "selected this option without properly "
			   "verifying the fingerprint through an alternate "
			   "communication mechanism (such as over the "
			   "telephone or in person), sign off immediately "
			   "and delete it from your key database in the "
			   "Encryption Initialization window.  See the "
			   "official documentation for details.",
			   new_fp_struct.umid, who, data);

    bc_system_message_with_conv(c, string->str);
    g_string_free(string, TRUE);  string = NULL;

  /* If the user chose to accept the key fingerprint temporarily, we notify
   * the module and that's it. */
  } else if (new_fp_struct.action == ACTION_ACCEPT_TEMPORARILY) {
    /*printf("User accepted temporarily, not adding to DB...\n");*/
    GString *string = g_string_new("");

    mod_write("KEY_ACCEPT_TEMPORARY ");
    mod_write(who);
    mod_write("/");
    mod_write(acct->protocol_id);
    mod_write("|");
    mod_writeln(new_fp_struct.umid);

    g_string_append_printf(string, "You have accepted %s's (%s) key "
			   "temporarily.  Its fingerprint is:\n[%s].\nWrite "
			   "down this fingerprint and confirm it with your "
			   "buddy through some other communication mechanism "
			   "(for example, in person or over a telephone). "
			   "After it has been verified, you may permanently "
			   "add the key in the future, or manually add the "
			   "fingerprint to you database in the Encryption "
			   "Initialization window.",
			   new_fp_struct.umid, who, data);

    bc_system_message_with_conv(c, string->str);
    g_string_free(string, TRUE);  string = NULL;

  /* If the user rejected the key... */
  } else if (new_fp_struct.action == ACTION_REJECT) {
    GString *string = g_string_new("");

    /*printf("User rejected key.\n");*/
    mod_write("KEY_REJECT ");
    mod_write(who);
    mod_write("/");
    mod_writeln(acct->protocol_id);


    g_string_append_printf(string, "You have rejected %s's key, identified "
			   "by fingerprint\n[%s].\nA secure communications "
			   "channel will not be established and your key "
			   "database will not be modified.", who, data);
    bc_system_message_with_conv(c, string->str);
    g_string_free(string, TRUE);  string = NULL;

  } else {
    /* If the user didn't choose one of the three cases above, then we have
     * a major internal error! */
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::handle_fingerprint",
	       "Internal error.\n");
    printf("Internal error!\n");
    exit(666);
  }

  mod_readline(line, sizeof(line) - 8);
  if (strstr(line, "ERROR") == line) {
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::handle_fingerprint",
	       "Error: %s\n", line);
    return 0;
  }

  if (parseOK(line, &type, &num_results) < 0)
    goto handle_fingerprint_error;

  if (num_results > 0) {
    total_size = get_sizes(size_array, sizeof(size_array));
    /*printf("total_size: %u\n", total_size);*/
  
    data_ptr = (char *)calloc(total_size + 8, sizeof(char));
    if (data_ptr == NULL) {
      fprintf(stderr, "Can't calloc!\n");
      exit(1);
    }

    mod_readdata(data_ptr, total_size, total_size);
    mod_readline(line, sizeof(line) - 1);

    /* There may be actionable messages to send to the buddy, like if their
     * key was accepted... */
    for(i = 0; i < num_results; i++) {
      chunk_ptr = g_string_sized_new(size_array[ i ]);
      chunk_ptr = g_string_append_len(chunk_ptr, data_ptr + offset,
				      size_array[ i ]);
      serv_send_im(acct->gc, who, chunk_ptr->str, 0);
      g_string_free(chunk_ptr, TRUE);  chunk_ptr = NULL;

      offset += size_array[ i ];
    }

    free(data_ptr);  data_ptr = NULL;
  }
   
  return 1;

 handle_fingerprint_error:
  free(data_ptr);  data_ptr = NULL;
  gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::handle_fingerprint",
	       "handle_fingerprint_error\n");

  bc_dialog("Error in handle_fingerprint()!", GTK_MESSAGE_ERROR, NULL);
  return 0;
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void signon_handler(GaimConnection *gc, void *data) {
  /*printf("signing on...\n");*/
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void signoff_handler(GaimConnection *gc, void *data) {
  char line[ 256 ];

  memset(line, 0, sizeof(line));

  /*printf("signing off...\n");*/
  mod_writeln("RESET");
  mod_readline(line, sizeof(line) - 8);
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void buddy_signoff_handler(GaimBuddy *b, void *data) {
  /*GaimConversation *c = NULL;*/

  char line[ 256 ];

  memset(line, 0, sizeof(line));

  /*printf("%s signed off.\n", b->name);*/
  mod_write("WIPE ");
  mod_write(b->name);
  mod_write("/");
  mod_writeln(b->account->protocol_id);

  mod_readline(line, sizeof(line) - 8);
  /*printf("line: [%s]\n", line);*/

  /*c = gaim_find_conversation(b->name);
  if (c != NULL)
  bc_set_state_plaintext(c);*/

}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
int show_crypto_reset(GaimBuddy *b) {
  return 1;
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void reset_crypto_cb(GtkWidget *w, GaimBuddy *b) {
  /* b->name gives screen name. */
  /*printf("name: %s\n", b->name);*/
}




/* Resolves a buddy's screen name into their UMID.  NULL is returned if the
 * buddy is not known to the underlying crypto module. */
/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
char *get_umid(char *who, GaimAccount *acct) {
  char line[ 256 ];

  memset(line, 0, sizeof(line));

  mod_write("LOOKUP_NAME ");
  mod_write(who);
  mod_write("/");
  mod_writeln(acct->protocol_id);

  mod_readline(line, sizeof(line) - 8);
  if (strstr(line, "ERROR") == line)
    return NULL;
  else
    return strdup(line);
}


/* Generates a key given an ID argument, which serves as a basis to create
 * its filename. Returns 1 and fills 'fingerprint_buffer' with the
 * generated fingerprint upon success.  On failure, -1 is returned and
 * 'fingerprint_buffer' holds the error message. */
/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
int bc_generate_key(const char *id, gchar *id_pw,
		    char *fingerprint_buffer,
		    size_t fingerprint_buffer_len) {
  unsigned int type = 0;
  unsigned int num_results = 0;

  char buf[ 256 ];

  memset(buf, 0, sizeof(buf));


  mod_write("GENERATE_KEY ");
  mod_write((char *)id);
  mod_write("|");
  mod_writeln(id_pw);

  mod_readline(buf, sizeof(buf) - 8);
  if (parseOK(buf, &type, &num_results) < 0) {
    /*printf("bc_generate_key ERROR: [%s]\n", buf);*/
    /*g_strlcpy(fingerprint_buffer, buf, fingerprint_buffer_len - 8);*/
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c::bc_generate_key",
	       "buf: %s\n", buf);
    return -1;
  }

  mod_readline(buf, sizeof(buf) - 8);
  g_strlcpy(fingerprint_buffer, buf, fingerprint_buffer_len - 8);
  return 1;
}


/* Returns 1 if an ID to sign on with has been selected yet, and 0 if it has
 * not. */
/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
int is_encryption_id_selected(void) {
  char buf[ 64 ];

  memset(buf, 0, sizeof(buf));

  mod_writeln("IS_USER_SELECTED");
  mod_readline(buf, sizeof(buf) - 8);

  if (strcmp(buf, "YES") == 0)
    return 1;
  else if (strcmp(buf, "NO") == 0)
    return 0;
  else {
    /* This will never happen. */
    return 0;
  }
}


/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
void quitting_handler(void *x, void *y) {
#ifdef _WIN32
  tor_kill();
#endif
  mod_quit();
}


int bc_encrypt_file(char *who, GaimAccount *acct, char *original_filename,
		    char *new_filename, size_t new_filename_size) {
  char buffer[ 256 ];

  memset(buffer, 0, sizeof(buffer));

  bc_get_temporary_filename(new_filename, new_filename_size - 8);

  /*printf("temporary filename: [%s]\n", new_filename);*/

  mod_write("ENCRYPT_FILE ");
  mod_write(who);
  mod_write("/");
  mod_write(acct->protocol_id);
  mod_write("|");
  mod_write(original_filename);
  mod_write("|");
  mod_writeln(new_filename);

  mod_readline(buffer, sizeof(buffer) - 8);
  if (strstr(buffer, "ERROR") == buffer) {
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c",
	       "Error while encrypting file: %s\n", buffer);
    return -1;
  }

  return 1;
}


int bc_decrypt_file(char *who, GaimAccount *acct, char *original_filename) {
  char tempfile[ 512 ];
  char buffer[ 256 ];

  memset(tempfile, 0, sizeof(tempfile));
  memset(buffer, 0, sizeof(buffer));

  bc_get_temporary_filename(tempfile, sizeof(tempfile) - 8);
  /*printf("decrypting into temporary file: [%s]\n", tempfile);*/

  mod_write("DECRYPT_FILE ");
  mod_write(who);
  mod_write("/");
  mod_write(acct->protocol_id);
  mod_write("|");
  mod_write(original_filename);
  mod_write("|");
  mod_writeln(tempfile);

  mod_readline(buffer, sizeof(buffer) - 8);
  if (strstr(buffer, "ERROR") == buffer) {
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c",
	       "Error while decrypting file: %s\n", buffer);
    return -1;
  }

  /*printf("Renaming [%s] to [%s]...\n", tempfile, original_filename);*/
  if (rename(tempfile, original_filename) < 0) {
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c",
	       "Error while renaming decrypted file.\n");
    return -1;
  }

  return 1;
}

int bc_do_tor_stuff(void) {
  char buffer[ 16 ];

  memset(buffer, 0, sizeof(buffer));

  if (bc_prefs_is_yes(BC_PREFS_TOR_ENABLED)) {
#ifdef _WIN32
    if (bc_prefs_is_yes(BC_PREFS_TOR_USE_BUILTIN)) {
      if (bc_spawn_tor() < 0) {
        /* If the user wanted to connect via Tor and it cannot be started,
         * then this is a fatal error. */
        exit(1);
      }
    }
#endif

    if (bc_tor_set_proxy_settings() < 0)
      return -1;

    // TODO: Wait for proxy.
  }
  

  return 1;
}

int bc_tor_set_proxy_settings(void) {
  GaimProxyInfo *p = NULL;
  GaimProxyType new_type, cur_type;
  char *new_host, *cur_host;
  int new_port, cur_port;
  char *new_username, *cur_username;
  char *new_password, *cur_password;

  char buffer[ 128 ];

  memset(buffer, 0, sizeof(buffer));

  p = gaim_global_proxy_get_info();

  new_type = GAIM_PROXY_SOCKS5;
  bc_prefs_get(BC_PREFS_TOR_PORT, buffer, sizeof(buffer) - 8);
  new_host = "127.0.0.1";
  new_port = atoi(buffer);
  new_username = NULL;
  new_password = NULL;

  cur_type = gaim_proxy_info_get_type(p);
  cur_host = (char *)gaim_proxy_info_get_host(p);
  cur_port = gaim_proxy_info_get_port(p);
  cur_username = (char *)gaim_proxy_info_get_username(p);
  cur_password = (char *)gaim_proxy_info_get_password(p);

  if ((new_type != cur_type) ||
      (strcmp(new_host, cur_host) != 0) ||
      (new_port != cur_port) ||
      (bc_strcmp_null(new_username, cur_username) != 0) ||
      (bc_strcmp_null(new_password, cur_password) != 0)) {

    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY, BC_PREFS_YES);

    GString *s = g_string_new(NULL);
    g_string_append_printf(s, "%d", cur_type);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_TYPE, s->str);
    g_string_free(s, TRUE);  s = NULL;

    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_HOST, cur_host);

    s = g_string_new(NULL);
    g_string_append_printf(s, "%d", cur_port);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PORT, s->str);
    g_string_free(s, TRUE);  s = NULL;

    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_USERNAME, cur_username);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PASSWORD, cur_password);
  } else {
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY, BC_PREFS_NO);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_TYPE,
                 BC_PREFS_TOR_SAVED_PROXY_TYPE_DEFAULT);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_HOST,
                 BC_PREFS_TOR_SAVED_PROXY_HOST_DEFAULT);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PORT,
                 BC_PREFS_TOR_SAVED_PROXY_PORT_DEFAULT);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_USERNAME,
                 BC_PREFS_TOR_SAVED_PROXY_USERNAME_DEFAULT);
    bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PASSWORD,
                 BC_PREFS_TOR_SAVED_PROXY_PASSWORD_DEFAULT);
  }

  if (bc_prefs_save() < 0)
    return -1;

  gaim_proxy_info_set_type(p, GAIM_PROXY_SOCKS5);
  gaim_proxy_info_set_host(p, "127.0.0.1");
  gaim_proxy_info_set_port(p, new_port);
  gaim_proxy_info_set_username(p, NULL);
  gaim_proxy_info_set_password(p, NULL);

  return 1;
}

int bc_tor_restore_proxy_settings(void) {
  if (bc_prefs_is_yes(BC_PREFS_TOR_SAVED_PROXY)) {
    GaimProxyInfo *p = gaim_global_proxy_get_info();
    char buffer[ 256 ];

    memset(buffer, 0, sizeof(buffer));


    bc_prefs_get(BC_PREFS_TOR_SAVED_PROXY_TYPE, buffer, sizeof(buffer) - 8);
    gaim_proxy_info_set_type(p, atoi(buffer));

    bc_prefs_get(BC_PREFS_TOR_SAVED_PROXY_HOST, buffer, sizeof(buffer) - 8);
    gaim_proxy_info_set_host(p, buffer);

    bc_prefs_get(BC_PREFS_TOR_SAVED_PROXY_PORT, buffer, sizeof(buffer) - 8);
    gaim_proxy_info_set_port(p, atoi(buffer));

    bc_prefs_get(BC_PREFS_TOR_SAVED_PROXY_USERNAME, buffer,sizeof(buffer) - 8);
    if (strcmp(buffer, "") != 0)
      gaim_proxy_info_set_username(p, NULL);
    else
      gaim_proxy_info_set_username(p, buffer);

    bc_prefs_get(BC_PREFS_TOR_SAVED_PROXY_PASSWORD, buffer,sizeof(buffer) - 8);
    if (strcmp(buffer, "") != 0)
      gaim_proxy_info_set_password(p, NULL);
    else
      gaim_proxy_info_set_password(p, buffer);
  }

  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY, BC_PREFS_NO);
  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_TYPE,
               BC_PREFS_TOR_SAVED_PROXY_TYPE_DEFAULT);
  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_HOST,
               BC_PREFS_TOR_SAVED_PROXY_HOST_DEFAULT);
  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PORT,
               BC_PREFS_TOR_SAVED_PROXY_PORT_DEFAULT);
  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_USERNAME,
               BC_PREFS_TOR_SAVED_PROXY_USERNAME_DEFAULT);
  bc_prefs_set(BC_PREFS_TOR_SAVED_PROXY_PASSWORD,
               BC_PREFS_TOR_SAVED_PROXY_PASSWORD_DEFAULT);

  /* We won't call bc_prefs_save because the calling function will later. */
  return 1;
}
