/**
 * bc_utils.c
 *
 * Copyright (C) 2003-2005, J. Salvatore Testa II
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "bc_utils.h"


/* Creates a filesystem directory. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void bc_create_dir(char *dir) {
   if (bc_does_dir_exist(dir) == 0) {
#ifdef _WIN32
      mkdir(dir);
#else
      mkdir(dir, 0700);
#endif
   }
}


/* Checks to see if a directory exists.  1 is returned if it does, 0 if it does
 * not, and -1 if the path represents a link or other kind of object. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int bc_does_dir_exist(char *dir) {
#ifdef _WIN32
   if (GetFileAttributes(dir) == FILE_ATTRIBUTE_DIRECTORY)
      return 1;
   else
      return 0;
#else
  struct stat dirstat;

  memset(&dirstat, 0, sizeof(dirstat));

  if (lstat(dir, &dirstat) < 0)
    return 0;

  if ((dirstat.st_mode & S_IFMT) == S_IFDIR)
    return 1;
  else
    return -1;
#endif
}


/* Retrieves the default directory for the keys & database. Returns 1 on
 * success, 0 on error. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
unsigned int bc_get_default_dir(char *buffer, size_t buffer_size) {
  GString *string = NULL;

  char home_dir[ 256 ];

  memset(home_dir, 0, sizeof(home_dir));

  if (bc_get_home_directory(home_dir, sizeof(home_dir))) {
    string = g_string_new("");
    string = g_string_append(string, home_dir);
#ifdef _WIN32
    string = g_string_append(string, "\\.scatterchat");
#else
    string = g_string_append(string, "/.scatterchat");
#endif

    /* If the path is a valid directory, or if nothing exists there, then
     * its good.  If it is a link or other object (-1), then we return an
     * error. */
    if (bc_does_dir_exist(string->str) >= 0) {
      g_strlcpy(buffer, string->str, buffer_size);
      g_string_free(string, TRUE);  string = NULL;
      return 1;
    } else {
      g_string_free(string, TRUE);  string = NULL;
      return 0;
    }

  } else
    return 0;
}


/* Removes an empty directory. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
void bc_remove_dir(char *dir) {
  if (bc_does_dir_exist(dir) == 1)
    rmdir(dir);
}


/* Retrieves the user's home directory. Returns 1 on success, 0 on error. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
unsigned int bc_get_home_directory(char *buffer, size_t buffer_size) {
#ifdef _WIN32
   HRESULT hRes = 0;
   char default_path[ MAX_PATH ];

   memset(default_path, 0, MAX_PATH);

   hRes = SHGetFolderPath(NULL, CSIDL_PERSONAL, NULL,
                          0, default_path);

   if ((hRes == S_FALSE) || (hRes == E_FAIL) || (hRes == E_INVALIDARG)) {
      return 0;
   }

   g_strlcpy(buffer, default_path, buffer_size);
   return 1;
#else
  struct passwd *p = NULL;

  setpwent();
  p = getpwuid(getuid());
  if ((p != NULL) && (p->pw_dir != NULL)) {
    g_strlcpy(buffer, p->pw_dir, buffer_size);
    endpwent();
    return 1;
  } else
    return 0;
#endif
}


/* Determines whether or not the specified key fingerprint is valid (IE:
 * it consists of 20 hexidecimal bytes seperated by spaces.) */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
unsigned int bc_is_key_fingerprint_valid(char *fp) {
  unsigned int i = 0;

  /* If the fingerprint isn't 59 characters long, its bad. */
  if (strlen(fp) != 59)
    return 0;

  /* Go through the first 19 bytes... */
  for (i = 0; i < 19; i++) {
    if (bc_is_char_hexidecimal(fp[ i * 3 ]) &&
	bc_is_char_hexidecimal(fp[ (i * 3) + 1 ]) &&
	(fp[ (i * 3) + 2 ] == ' ')) {
      /* This byte is good. */
    } else
      return 0;
  }

  /* Now check the last byte... */
  if (bc_is_char_hexidecimal(fp[ 57 ]) && bc_is_char_hexidecimal(fp[ 58 ])) {
    /* The last byte is good. */
  } else
    return 0;

  return 1;
}


/* Determines if a character is a valid hexidecimal digit. */
/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
unsigned int bc_is_char_hexidecimal(char a_char) {
  switch (a_char) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'A':
  case 'B':
  case 'C':
  case 'D':
  case 'E':
  case 'F':
    return 1;

  default:
    return 0;
  }
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int bc_parse_version(char *version, int *major, int *minor, int *revision) {
  char *dot_pos = NULL;

  char *my_version = strdup(version);
  char *ptr = my_version;

  dot_pos = strchr(ptr, '.');
  if (dot_pos == NULL) {
    free(my_version);  my_version = NULL;
    return 0;
  }

  *dot_pos = '\0';
  *major = atoi(ptr);
  ptr = dot_pos + 1;

  dot_pos = strchr(ptr, '.');
  if (dot_pos == NULL) {
    free(my_version);  my_version = NULL;
    return 0;
  }

  *dot_pos = '\0';
  *minor = atoi(ptr);
  ptr = dot_pos + 1;

  *revision = atoi(ptr);

  free(my_version);  my_version = NULL;
  return 1;
}


/* Security audit trail:
 *    - Joe Testa: January 2nd, 2005
 */
int bc_check_new_version(char *buf, size_t len) {
  int s_major = 0;
  int s_minor = 0;
  int s_revision = 0;

  int l_major = 0;
  int l_minor = 0;
  int l_revision = 0;

  if (!bc_parse_version(buf, &s_major, &s_minor, &s_revision))
    return -1;

  if (!bc_parse_version(VERSION, &l_major, &l_minor, &l_revision))
    return -1;

  if (s_major > l_major)
    return 0;
  else if (s_major == l_major) {
    if (s_minor > l_minor)
      return 0;
    else if (s_minor == l_minor) {
      if (s_revision > l_revision)
	return 0;
      else
	return 1;
    }
  }

  return 1;
}

int bc_parse_stderr_line(char *line,
			 char *screenname, size_t screenname_size,
			 char *umid, size_t umid_size,
			 unsigned int *operation,
			 unsigned int *oldstate,
			 unsigned int *newstate) {
  char *ptr = NULL;
  char *pipe_pos = NULL;

  ptr = line;

  pipe_pos = strchr(ptr, '|');
  if (pipe_pos == NULL)
    goto bc_parse_stderr_line_error;

  *pipe_pos = '\0';
  g_strlcpy(screenname, ptr, screenname_size);
  ptr = pipe_pos + 1;

  pipe_pos = strchr(ptr, '|');
  if (pipe_pos == NULL)
    goto bc_parse_stderr_line_error;

  *pipe_pos = '\0';
  g_strlcpy(umid, ptr, umid_size);
  ptr = pipe_pos + 1;
  
  if (sscanf(ptr, "%u %u %u", operation, oldstate, newstate) <= 0)
    goto bc_parse_stderr_line_error;
  return 1;


 bc_parse_stderr_line_error:
  return 0;
}


char *bc_parse_state_into_text(unsigned int state) {

  switch(state) {

  case BUDDY_RESET:
    return "Encrypted conversation has been reset.";

  case SENT_PUBKEYS:
    return "Sent public keys to buddy.";

  case SENT_SESSION_KEY:
    return "Sent session key to buddy.";

  case AWAITING_PUBKEYS_1:
    return "Awaiting signature key from buddy.";

  case AWAITING_PUBKEYS_2:
    return "Awaiting encryption key from buddy.";

  case AWAITING_SESSION_KEY:
    return "Awaiting session key from buddy.";

  case SIGNKEY_FINGERPRINT_VALID:
    return "Buddy's signature key fingerprint is valid.";

  case ENCRYPTKEY_VALID:
    return "Buddy's encryption key is valid.";

  case SESSION_KEY_VALID:
    return "Session key is valid.";

  case CONNECTED:
    return "Encrypted conversation successfully started.";

  case TIMEDOUT:
    return "Encrypted conversation failed to begin due to timeout.";

  default:
    return "Unknown state.";
  }
}


/* This function was copied from Gaim sources (conversation.c), and modified
 * to handle a different normalization.  Copyright (C) 1998-2005 the Gaim
 * team. */
GaimConversation *bc_find_conversation(char *name){
  GaimConversation *c = NULL;
  char *cuser = NULL;
  GList *cnv = NULL;
  /* UM: workaround for Gaim bug (name arg gets changed somehow!). */
  char *saved_name = g_strdup(name);

  g_return_val_if_fail(name != NULL, NULL);

  cuser = g_strdup(gaim_normalize(NULL, name));
  
  for (cnv = gaim_get_conversations(); cnv != NULL; cnv = cnv->next) {
    c = (GaimConversation *)cnv->data;

    /*printf("Found conversation with name [%s][%s]\n", cuser, gaim_normalize(NULL, gaim_conversation_get_name(c)));*/

    if (!gaim_utf8_strcasecmp(cuser, gaim_normalize(NULL, gaim_conversation_get_name(c))))
      break;

    c = NULL;
  }

  g_free(cuser);  cuser = NULL;
  strcpy(name, saved_name);  g_free(saved_name);  saved_name = NULL;

  return c;
}


/* Returns the status of a particular buddy, or NULL if the buddy is not known
 * to the underlying crypto module. */
/* Security audit trail:
 *    - Joe Testa: January 1st, 2005.
 */
char *get_status(char *who, GaimAccount *acct) {
  char line[ 256 ];

  memset(line, 0, sizeof(line));

  mod_write("STATUS ");
  mod_write(who);
  mod_write("/");
  mod_writeln(acct->protocol_id);

  mod_readline(line, sizeof(line) - 8);
  if (strstr(line, "ERROR") == line)
    return NULL;
  else
    return strdup(line);
}


int bc_is_secure_channel_established(char *who, GaimAccount *acct) {
  unsigned int retVal = 0;

#ifdef USING_ENCRYPTION
  char *status = get_status(who, acct);

  if ((status != NULL) && (strcmp(status, "Connected.") == 0))
    retVal = 1;

  if (status != NULL) {
    free(status);  status = NULL;
  }
#endif

  return retVal;
}

#include "debug.h"
void bc_get_temporary_filename(char *tempfile, size_t tempfile_size) {
#ifndef _WIN32
  GString *str = NULL;
  int fd = 0;
  mode_t old_umask = 0;
#else
  TCHAR tempPath[ 256 ];
  TCHAR tempFilename[ MAX_PATH ];
#endif 

#ifndef _WIN32
  srand(time(NULL));

  str = g_string_new("/tmp/");
  g_string_append_printf(str, "%uXXXXXXX", rand());

  old_umask = umask(0700);
  fd = mkstemp(str->str);
  if (fd < 0) {
    umask(old_umask);
    g_string_free(str, TRUE);  str = NULL;
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Could not create temporary file!\n");
    return;
  }

  umask(old_umask);

  close(fd);
  if (bc_delete_file(str->str) < 0) {
    g_string_free(str, TRUE);  str = NULL;
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Could not delete temporary file!\n");
    return;
  }

  g_strlcpy(tempfile, str->str, tempfile_size);
  g_string_free(str, TRUE);  str = NULL;

#else

  memset(tempPath, 0, sizeof(tempPath));
  memset(tempFilename, 0, sizeof(tempFilename));

  GetTempPath(sizeof(tempPath) - 8, tempPath);
  if (GetTempFileName(tempPath, "gaim", 0, tempFilename) == 0) {
    gaim_debug(GAIM_DEBUG_ERROR, "bc_main.c", "Could not create temporary file!\n");
    return;
  }

  DeleteFile(tempFilename);
  g_strlcpy(tempfile, tempFilename, tempfile_size);

#endif
}


gchar *bc_get_debugging_file(char *path) {
  gchar *retval = NULL;
  GString *debug_file_path = NULL;
#ifdef _WIN32
  DWORD read_len = 0;
  HANDLE hFile = NULL;
#else
  int read_len = 0;
  FILE *hFile = NULL;
#endif
  char buffer[ 256 ];

  memset(buffer, 0, sizeof(buffer));

  debug_file_path = g_string_new(path);
  debug_file_path = g_string_append(debug_file_path, "debug");

#ifdef _WIN32
  hFile = CreateFile(debug_file_path->str, GENERIC_READ, FILE_SHARE_READ, NULL,
                     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile != INVALID_HANDLE_VALUE) {
    if (ReadFile(hFile, buffer, sizeof(buffer) - 8, &read_len, NULL))
      retval = g_strdup(buffer);

    CloseHandle(hFile);  hFile = NULL;
  }
#else
  hFile = fopen(debug_file_path->str, "r");
  if (hFile != NULL) {
    if ((read_len = fread(buffer, sizeof(char), sizeof(buffer) - 8, hFile)) > 0)
      retval = g_strdup(buffer);

    fclose(hFile);  hFile = NULL;
  }
#endif

  g_string_free(debug_file_path, TRUE);  debug_file_path = NULL;
  return retval;
}


/* This function returns 1 if the specified path points to a regular file,
 * returns 0 if the path does not exist, and returns -1 if otherwise (for
 * symlinks, directories, etc). */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
int bc_is_regular_file(char *path) {
#ifndef _WIN32
  struct stat thestat;
 
  memset(&thestat, 0, sizeof(thestat));
 
  if (lstat(path, &thestat) != -1) {
    if (S_ISREG(thestat.st_mode))
      return 1;
    else
      return -1;
  } else
    return 0;
#else
  WIN32_FILE_ATTRIBUTE_DATA stat;
 
  memset(&stat, 0, sizeof(stat));
 
  if (GetFileAttributesEx(path, GetFileExInfoStandard, &stat)) {
    if (stat.dwFileAttributes & (FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_ARCHIVE))
      return 1;
    else
      return -1;
  } else
    return 0;
#endif
}

int bc_delete_file(char *file) {
  if (bc_is_regular_file(file) != 1)
    return -1;

#ifdef _WIN32
  if (DeleteFile(file) != 0)
    return -1;
#else
  if (unlink(file) < 0)
    return -1;
#endif
  return 1;
}


/* Sets a peer's autoencrypt status. */
int bc_autoencrypt(char *who, gboolean which) {
  char line[ 16 ];
  unsigned int i = 0;

  memset(line, 0, sizeof(line));

  for (i = 0; i < 4; i++) {
    mod_write("AUTO_FEEDBACK ");
    mod_write(who);
    mod_write("|");
    if (which == TRUE)
      mod_writeln("C");
    else
      mod_writeln("P");

    mod_readline(line, sizeof(line) - 8);
    if (strstr(line, "OK") != line)
      return -1;
  }

  return 1;
}

/* Returns the size of a file. */
unsigned int bc_get_filesize(char *path) {
  unsigned int ret = 0;
#ifdef _WIN32
  HANDLE hFile = NULL;

  hFile = CreateFile(path, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile != INVALID_HANDLE_VALUE) {
    ret = (unsigned int)GetFileSize(hFile, NULL);
    CloseHandle(hFile); hFile = NULL;
  }
  
#else
  struct stat s;

  memset(&s, 0, sizeof(struct stat));

  stat(path, &s);
  ret = s.st_size;
#endif
  return ret;
}

int bc_strcmp_null(char *s1, char *s2) {
  if (s1 == s2)
    return 0;

  if ((s1 == NULL) && (s2 != NULL))
    return -1;

  if ((s1 != NULL) && (s2 == NULL))
    return 1;

  return strcmp(s1, s2);
}
