/**
 * console.c
 *
 * Copyright (C) 2004-2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif


#ifdef BUILD_EXECUTABLE

#include "console.h"
#ifndef _WIN32
GCRY_THREAD_OPTION_PTHREAD_IMPL;
#endif


#ifdef DMALLOC
  #include "dmalloc.h"
#endif

BC_MUTEX *state_output_mutex = NULL;

extern gcry_sexp_t *my_signing_keypair;
extern gcry_sexp_t *my_encrypting_keypair;
extern BC_THREAD *timeout_thread;
extern BC_MUTEX *pending_list_mutex;
extern gcry_mpi_t mpi_one;
extern struct PENDING_MESSAGE *pending_messages;
extern int logfile_fd;
extern char logfile_path[ LOGFILE_PATH_SIZE ];
extern char um_database_path[ MAX_PATH_LEN ];
extern char um_load_save_dir[ MAX_PATH_LEN ];

char *ERROR_ARGUMENTS_INVALID = "ERROR ARGUMENTS INVALID";
char *ERROR_ARGUMENTS_MISSING = "ERROR ARGUMENTS MISSING";
char *ERROR_LOAD_SAVE_NOT_SET = "MUST SET LOAD/SAVE DIRECTORY FIRST";


/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
int main(int ac, char **av) {

  /* We may be running as ROOT, so we want to call 'module_init()' right
   * away so it can do its thing and drop the privileges. */
  module_init();

#ifndef _WIN32
  /* Make super-sure that we don't have privileges anymore. */
  setuid(getuid());
#endif

  memset(logfile_path, 0, sizeof(logfile_path));

  process_args(ac, av);
  handle_commands();

  return 0;
}


/* Writes a string to the console. */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void bc_console_write(char *stuff) {
  printf("%s", stuff);
}


/* Writes a newline-terminated string to the console. */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void bc_console_writeln(char *stuff) {
#ifdef _WIN32
  printf("%s\n", stuff);
#else
  printf("%s\r\n", stuff);
#endif
  fflush(stdout);
}


/* Writes a state change to stderr. */
/* Security audit trail:
 *    - Joe Testa: March 6th, 2005
 */
void change_state(struct UM_BUDDY *buddy, unsigned int operation,
		  unsigned int new_state) {
  char *screenname = buddy->screenname;
  char *um_id = buddy->um_id;

  if (screenname == NULL)
    return;

  /* Just in case we re-set a null state. */
  if (buddy->state == new_state)
    return;

  if (um_id == NULL)
    um_id = "";


  /* If any a newline, carriage return, or pipe is in either of the strings,
   * then we CANNOT send the state over stderr, otherwise an attacker who
   * can inject these characters could confuse the calling program into
   * seeing a state change that did not really occur. */
  if ((strchr(screenname, '\n') != NULL) ||
      (strchr(screenname, '\r') != NULL) ||
      (strchr(screenname, '|') != NULL) ||
      (strchr(um_id, '\n') != NULL) ||
      (strchr(um_id, '\r') != NULL) ||
      (strchr(um_id, '|') != NULL)) {
    return;
  }


  bc_mutex_lock(state_output_mutex);
#ifdef _WIN32
  fprintf(stderr, "%s|%s|%u %u %u\n", screenname, um_id, operation, buddy->state, new_state);
  fflush(stderr);
#else
  fprintf(stderr, "%s|%s|%u %u %u\r\n", screenname, um_id, operation, buddy->state, new_state);
#endif
  bc_mutex_unlock(state_output_mutex);
}


/* Provides the loop to handle user input via stdin. */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void handle_commands(void) {
  char line[ 2048 ];

  memset(line, 0, sizeof(line));

  bc_log("handle_commands", BC_LOG_DEBUG, "called.");

  /* The first thing the program outputs is the version of this module,
   * followed by the version of the API it understands (each on its own
   * line). */
  bc_console_writeln(VERSION);
  bc_console_writeln(API_VERSION);

#ifdef _WIN32
  fprintf(stderr, "START_STDERR\n");
  fflush(stderr);
#else
  fprintf(stderr, "START_STDERR\r\n");
#endif

  /* Read in the line, as long as there is one to read. */
  while(fgets(line, sizeof(line) - 1, stdin) != NULL) {

    /* Trim the trailing newline off. */
    trim_line(line);

    if (strcmp(line, "RESET") == 0)
      module_reset();
    else if ((strcmp(line, "SHUTDOWN") == 0) ||
	     (strcmp(line, "QUIT") == 0) ||
	     (strcmp(line, "EXIT") == 0)) {
      module_shutdown();
    } else if (bc_startswith(line, "INIT"))
      module_dirinit(line);
    else if (bc_startswith(line, "SELECT_USER"))
      module_selectuser(line);
    else if (bc_startswith(line, "ADD_KEY"))
      module_addkey(line);
    else if (bc_startswith(line, "DEL_KEY"))
      module_delkey(line);
    else if (bc_startswith(line, "GENERATE_KEY"))
      module_generatekey(line);
    else if (bc_startswith(line, "DEL_ID"))
      module_delid(line);
    else if (bc_startswith(line, "SEND"))
      module_sendmessage(line);
    else if (bc_startswith(line, "PROCESS"))
      module_processmessage(line);
    else if (bc_startswith(line, "KEY_ACCEPT_PERMANENTLY"))
      module_keyacceptpermanently(line);
    else if (bc_startswith(line, "KEY_ACCEPT_TEMPORARY"))
      module_keyaccepttemporary(line);
    else if (bc_startswith(line, "KEY_REJECT"))
      module_keyreject(line);
    else if (bc_startswith(line, "WIPE"))
      module_wipe(line);
    else if (bc_startswith(line, "STATUS"))
      module_status(line);
    else if (bc_startswith(line, "LOOKUP_NAME"))
      module_lookupname(line);
    else if (bc_startswith(line, "ENCRYPT_FILE"))
      module_encryptfile(line);
    else if (bc_startswith(line, "DECRYPT_FILE"))
      module_decryptfile(line);
    else if (bc_startswith(line, "AUTO_QUERY"))
      module_autoquery(line);
    else if (bc_startswith(line, "AUTO_FEEDBACK"))
      module_autofeedback(line);
    else if (strcmp(line, "IS_LOADSAVE_INITIALIZED") == 0) {
      if (strlen(um_load_save_dir) > 0)
	bc_console_writeln("YES");
      else
	bc_console_writeln("NO");
    } else if (strcmp(line, "IS_USER_SELECTED") == 0) {
      if ((my_signing_keypair != NULL) || (my_encrypting_keypair != NULL))
	bc_console_writeln("YES");
      else
	bc_console_writeln("NO");
    } else if (strcmp(line, "NOOP") == 0)
      bc_console_writeln("OK");
    /*else if (bc_startswith(line, "BASE64_PADDING"))
      module_b64_padding(line);*/
    else if (strcmp(line, "") == 0) {
      // Do nothing.
    } else {
      BCString *tmp = bc_string_new("ERROR COMMAND NOT RECOGNIZED: ");
      bc_string_append_printf(tmp, "%s", line);
      bc_console_writeln(tmp->str);
      bc_string_free(&tmp, 1);
    }
  }

}


/**
 * Add a key to the buddy key database.
 *
 * SYNTAX: ADD_KEY <name>|<fingerprint>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 *    - Joe Testa: October 17th, 2004
 */
void module_addkey(char *line) {
  unsigned int line_len = strlen(line);
  char *name = NULL;
  char *fp = NULL;
  char *pipe_pos = NULL;

  if (line_len < 10) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 8;

  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  fp = pipe_pos + 1;
  *pipe_pos = '\0';

  if ((strlen(name) == 0) || (strlen(fp) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if (strlen(um_database_path) == 0) {
    bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
    return;
  }

  bc_log("module_addkey", BC_LOG_DEBUG, "Adding key to database: [%s][%s]",
	 name, fp);

  key_database_add(name, fp, um_database_path);
  bc_console_writeln("OK");
}


/**
 * Gives feedback about sessions with a peer.
 *
 * SYNTAX: AUTO_FEEDBACK <NAME>|["C" | "P"]<CRLF>
 */
/* Security audit trail:
 *
 */

void module_autofeedback(char *line) {
   unsigned int line_len = strlen(line);
   char *pipe_pos = NULL;
   char *name = NULL;
   char *cp = NULL;


   if (strlen(um_load_save_dir) == 0) {
     bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
     return;
   }

   if (line_len < 15) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
   }

   name = line + 14;


   pipe_pos = strchr(name, '|');
   if (pipe_pos == NULL) {
     bc_console_writeln(ERROR_ARGUMENTS_MISSING);
     return;
   }

   *pipe_pos = '\0';
   cp = pipe_pos + 1;

   if ((strlen(name) == 0) || (strlen(cp) == 0)) {
     bc_console_writeln(ERROR_ARGUMENTS_MISSING);
     return;
   }

   if ((strcmp(cp, "C") != 0) && (strcmp(cp, "P") != 0)) {
     bc_console_writeln(ERROR_ARGUMENTS_INVALID);
     return;
   }

   /*printf("[%s][%s]\n", name, cp);*/
   if (auto_feedback(name, cp))
     bc_console_writeln("OK");
   else
     bc_console_writeln("ERROR");

}


/**
 * Determines if the caller should default encryption to "on" or "off" for a
 * new session with the specified peer.
 *
 * SYNTAX: AUTO_QUERY <NAME><CRLF>
 */
/* Security audit trail:
 *
 */

void module_autoquery(char *line) {
   unsigned int line_len = strlen(line);
   /*char *pipe_pos = NULL;*/
   char *name = NULL;
   /*   char *cp = NULL;*/


   if (strlen(um_load_save_dir) == 0) {
     bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
     return;
   }

   if (line_len < 12) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
   }

   name = line + 11;



   /*printf("[%s]\n", name);*/
   if (auto_query(name))
     bc_console_writeln("YES");
   else
     bc_console_writeln("NO");

}


/**
 * Enables or disables base64 message padding.  Default is ON.
 * >>> NOT YET SUPPORTED <<<
 *
 * SYNTAX: BASE64_PADDING [ON|OFF]
 */
/* Security audit trail:
 *    - Joe Testa: January 30th, 2005
 */
void module_b64_padding(char *line) {
   unsigned int line_len = strlen(line);
   char *ptr = NULL;

   if (line_len < 17) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
   }

   ptr = line + 15;
   if (strcmp(ptr, "ON") == 0) {
     /* Not implemented. */
   } else if (strcmp(ptr, "OFF") == 0) {
     /* Not implemented. */
   } else {
     bc_console_writeln(ERROR_ARGUMENTS_INVALID);
     return;
   }

   bc_console_writeln("OK");
}


/**
 * DECRYPT_FILE <PROTOCOL-NAME>|<IN FILE>|<OUT FILE><CRLF>
 */
void module_decryptfile(char *line) {
  unsigned int line_len = strlen(line);
  char *name = NULL;
  char *infile = NULL;
  char *outfile = NULL;
  char *pipe_pos = NULL;

  if (line_len < 14) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 13;

  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  infile = pipe_pos + 1;
  *pipe_pos = '\0';

  pipe_pos = strchr(infile, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  outfile = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(name) == 0) || (strlen(infile) == 0) || (strlen(outfile) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if (decrypt_file(name, infile, outfile) == 1)
    bc_console_writeln("OK");
  else
    bc_console_writeln("ERROR");

  return;
}



/**
 * Deletes a key to the buddy key database.
 *
 * SYNTAX: DEL_KEY <name>|<fingerprint>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 *    - Joe Testa: October 17th, 2004
 */
void module_delkey(char *line) {
  unsigned int line_len = strlen(line);
  char *name = NULL;
  char *fp = NULL;
  char *pipe_pos = NULL;

  if (line_len < 10) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 8;

  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  fp = pipe_pos + 1;
  *pipe_pos = '\0';

  if ((strlen(name) == 0) || (strlen(fp) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if (strlen(um_database_path) == 0) {
    bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
    return;
  }

  if (key_database_delete(name, fp, um_database_path) == 1)
    bc_console_writeln("OK");
  else
    bc_console_writeln("ERROR NAME AND FINGERPRINT NOT FOUND");

}


/**
 * Initializes the load/save directory.
 * (Known issue:  if the fingerprint of a key cannot be loaded, the module
 *                already outputted "OK", when it should say "ERROR"...)
 *
 * SYNTAX: INIT <directory>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 *    - Joe Testa: October 17th, 2004
 */
void module_dirinit(char *line) {
  unsigned int line_len = strlen(line);
  char *unparsed_dir = NULL;
  BCSList *my_ids = NULL;
  BCSList *db_ids = NULL;
  BCSList *db_fps = NULL;
  unsigned int num_my_ids = 0;
  unsigned int num_db_ids = 0;
  unsigned int i = 0;
  BCString *string = NULL;
  BCString *string2 = NULL;
  /*BCString *entropy_pool_file = NULL;*/
  /*unsigned int error = 0;*/

  char parsed_directory[ 256 ];
  char buffer2[ 128 ];

  memset(parsed_directory, 0, sizeof(parsed_directory));
  memset(buffer2, 0, sizeof(buffer2));

  if (line_len <= 5) {
    bc_console_writeln("ERROR MUST SPECIFY LOAD/SAVE DIR");
    goto module_dirinit_error;
  }

  unparsed_dir = line + 5;

  /* Parse and normalize the directory string passed in. */
  parse_directory_string(unparsed_dir, parsed_directory,
			 sizeof(parsed_directory) - 8);

  /* Get a list of the identities (*.bc files) in the directory. */
  my_ids = load_ids_from_directory(parsed_directory);
  if (my_ids == NULL) {
    bc_console_writeln("ERROR DIRECTORY NOT FOUND");
    goto module_dirinit_error;
  }

  string = bc_string_new(parsed_directory);
  string = bc_string_append(string, UM_DATABASE_FILENAME);

  /* Load the buddy fingerprints in the database. */
  get_database_raw_list(string->str, &db_ids, &db_fps);
  bc_string_clear(string);

  bc_console_writeln("OK");

  /* -1 because the first entry will be empty. */
  num_my_ids = bc_slist_length(my_ids);
  bc_string_append_printf(string, "%u", num_my_ids - 1);

  /* Output the number of IDs in the directory. */
  bc_console_writeln(string->str);
  bc_string_clear(string);

  /* Loop through all the IDs and output their name and fingerprint in the
   * following format:  <name>|<fingerprint> */
  string2 = bc_string_new(NULL);
  for(i = 1; i < num_my_ids; i++) {
    char *my_id = bc_slist_nth_data(my_ids, i);

    string = bc_string_append(string, my_id);
    string = bc_string_append(string, "|");

    string2 = bc_string_append(string2, parsed_directory);
    string2 = bc_string_append(string2, my_id);
    string2 = bc_string_append(string2, ".bc");

    /* Load the current ID's fingerprint. */
    if (!get_fingerprint_from_keyfile(string2->str,
				      buffer2, sizeof(buffer2) - 8)) {
      goto module_dirinit_error;
    }

    string = bc_string_append(string, buffer2);

    /* Write the id and fingerprint to the console on its own line. */
    bc_console_writeln(string->str);
    bc_string_clear(string);
    bc_string_clear(string2);
  }
  bc_string_free(&string2, 1);


  num_db_ids = bc_slist_length(db_ids);
  bc_string_clear(string);
  bc_string_append_printf(string, "%u", num_db_ids);


  bc_console_writeln(string->str);
  bc_string_clear(string);

  /* Loop through all the entries in the key database.  Their names &
   * fingerprints will be output in just the same way.*/
  for(i = 0; i < num_db_ids; i++) {
    string = bc_string_append(string, bc_slist_nth_data(db_ids, i));
    string = bc_string_append(string, "|");
    string = bc_string_append(string, bc_slist_nth_data(db_fps, i));

    bc_console_writeln(string->str);
    bc_string_clear(string);
  }
  bc_string_free(&string, 1);

  /* Set the load/save directory and store the path to the key database. */
  bc_strlcpy(um_load_save_dir, parsed_directory, sizeof(um_load_save_dir) - 8);
  bc_strlcpy(um_database_path, parsed_directory, sizeof(um_database_path) - 8);
  bc_strlcat(um_database_path, UM_DATABASE_FILENAME, sizeof(um_database_path) - 8);

  if (auto_load(um_load_save_dir) != 1) {
    /* ERROR*/
  }

  check_for_debug_file(um_load_save_dir);

  /* Set the entropy pool to the "entropy.clr" file inside the load/save
   * directory. */
  /*entropy_pool_file = bc_string_new(NULL);
  bc_string_append_printf(entropy_pool_file, "%s%s", parsed_directory,
			  "entropy.clr");
  gcry_control (GCRYCTL_SET_RANDOM_SEED_FILE, entropy_pool_file->str);
  bc_string_free(&entropy_pool_file, 1);*/

  free_database_raw_list(&db_ids, &db_fps);
  bc_slist_free(&my_ids);
  return;

 module_dirinit_error:
  free_database_raw_list(&db_ids, &db_fps);
  bc_slist_free(&my_ids);

  memset(um_load_save_dir, 0, sizeof(um_load_save_dir));
  memset(um_database_path, 0, sizeof(um_database_path));
  return;
}


/**
 * ENCRYPT_FILE <PROTOCOL-NAME>|<IN FILE>|<OUT FILE><CRLF>
 */
void module_encryptfile(char *line) {
  unsigned int line_len = strlen(line);
  char *name = NULL;
  char *infile = NULL;
  char *outfile = NULL;
  char *pipe_pos = NULL;

  if (line_len < 14) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 13;

  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  infile = pipe_pos + 1;
  *pipe_pos = '\0';

  pipe_pos = strchr(infile, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  outfile = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(name) == 0) || (strlen(infile) == 0) || (strlen(outfile) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if (encrypt_file(name, infile, outfile) == 1)
    bc_console_writeln("OK");
  else
    bc_console_writeln("ERROR");

  return;
}


/**
 * Generates a keypair.
 *
 * SYNTAX: GENERATE_KEY <name>|<password>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_generatekey(char *line) {
  unsigned int line_len = strlen(line);
  char *id = NULL;
  char *pipe_pos = NULL;
  char *pw = NULL;

  char buffer[ 128 ];

  memset(buffer, 0, sizeof(buffer));


  if (line_len < 14) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  id = line + 13;
  pipe_pos = strchr(id, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  *pipe_pos = '\0';
  pw = pipe_pos + 1;

  if ((strlen(id) == 0) || (strlen(pw) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if ((strchr(id, '.') != NULL) || (strchr(id, '|') != NULL) ||
      (strchr(id, '/') != NULL) || (strchr(id, '\\') != NULL)) {
    bc_console_writeln("ERROR ARGUMENT CONTAINS ILLEGAL CHARACTERS");
    return;
  }

  if (strlen(um_load_save_dir) == 0) {
    bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
    return;
  }

  if (generate_key(um_load_save_dir, id, pw,buffer, sizeof(buffer) - 8) == 1) {
    bc_console_writeln("OK");
    bc_console_writeln(buffer);
  } else
    bc_console_writeln("ERROR UNKNOWN");

}


/**
 * Deletes an identity from the database.
 *
 * SYNTAX: DEL_ID <name>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_delid(char *line) {
  unsigned int line_len = strlen(line);
  char *id = NULL;
  BCString *string = NULL;

  char buffer[ 128 ];

  memset(buffer, 0, sizeof(buffer));

  if (strlen(um_load_save_dir) == 0) {
    bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
    return;
  }

  if (line_len < 8) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  id = line + 7;

  if (strlen(id) == 0) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  if ((strchr(id, '.') != NULL) || (strchr(id, '|') != NULL) ||
      (strchr(id, '/') != NULL) || (strchr(id, '\\') != NULL)) {
    bc_console_writeln("ERROR ARGUMENT CONTAINS ILLEGAL CHARACTERS");
    return;
  }

  string = bc_string_new(um_load_save_dir);
  string = bc_string_append(string, id);
  string = bc_string_append(string, ".bc");

  /* unlink(2) says that symbolic links themselves are removed, not the object
   * they point to.  So this is safe to do. */
  if (bc_delete_file(string->str) < 0)
    bc_console_writeln("ERROR DELETING ID");
  else
    bc_console_writeln("OK");

  bc_string_free(&string, 1);
  return;
}


/* Initializes the library for the first time.  THIS FUNCTION MAY BE RUNNING
 * AS ROOT!  BE VERY CAREFUL! */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_init(void) {
  int ret = 0;

#ifndef _WIN32
  gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_pthread);
#endif

  if (!gcry_check_version(NEEDED_GCRYPT)) {
#ifndef _WIN32
    setuid(getuid());
#endif
    fprintf(stderr, "ERROR: libgcrypt v%s required, but you only have v%s.\n",
	   NEEDED_GCRYPT, gcry_check_version(NULL));
    exit(1);
  }

  gcry_control(GCRYCTL_INIT_SECMEM, 1024 * 1024 * 3);
  gcry_control(GCRYCTL_SUSPEND_SECMEM_WARN);
  /*gcry_control(GCRYCTL_SET_VERBOSITY, 0);*/


#ifndef _WIN32
  /* Privileges are hereby DROPPED. */
  setuid(getuid());
  /**********************************/


  /* Here, we're making sure that we can't re-gain privileges.  If the
   * setuid(0) or seteuid(0) calls are successful, then we terminate! */
  setuid(0);
  seteuid(0);
  if ((getuid() == 0) || (geteuid() == 0)){
    fprintf(stderr, "(e)uid is zero!  Terminating now!...");
    exit(666);
  }
#endif


  /*g_thread_init(NULL);*/

  pending_messages = NULL;

  if (state_output_mutex == NULL)
    state_output_mutex = bc_mutex_new();

  if (pending_list_mutex == NULL)
    pending_list_mutex = bc_mutex_new();

  if (timeout_thread == NULL)
    timeout_thread = bc_thread_create(crypto_state_timeout_thread);

  mpi_one = gcry_mpi_new(8);
  ret = gcry_mpi_scan(&mpi_one, GCRYMPI_FMT_USG, (unsigned char *)"\x01", 1, NULL);
  if (ret != 0) {
    fprintf(stderr, "Error while making MPI!!\n");
    exit(1);
  }

}


/**
 * Adds a key fingerprint to the key database.
 *
 * SYNTAX: KEY_ACCEPT_PERMANENTLY <protocol name>|<umid>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_keyacceptpermanently(char *line) {
  char *protocol_name = NULL;
  char *umid = NULL;
  char *pipe_pos = NULL;
  char *a_size_str = NULL;
  char *a_message = NULL;
  unsigned int a_size_int = 0;
  unsigned int total_size = 0;
  unsigned int i = 0;
  unsigned int num_results = 0;
  struct UM_BUDDY *buddy = NULL;
  BCString *str_sizes = NULL;
  BCString *str_results = NULL;
  BCString *str_tmp = NULL;

  MXFER mxfer;
  char buddy_sign_key_fingerprint[ 192 ];

  memset(&mxfer, 0, sizeof(MXFER));
  memset(buddy_sign_key_fingerprint, 0, sizeof(buddy_sign_key_fingerprint));

  if (strlen(line) < 24) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  protocol_name = line + 23;
  pipe_pos = strchr(protocol_name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  umid = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(protocol_name) == 0) || (strlen(umid) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }



  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_console_writeln("ERROR INVALID STATE");
    return;
  }


  if ((buddy->public_sign_key == NULL) || (buddy->um_id != NULL)) {
    bc_console_writeln("ERROR INVALID STATE");
    return;
  }


  /* Assign this buddy a UMID. */
  buddy->um_id = strdup(umid);

  get_key_fingerprint_from_sexp(buddy->public_sign_key,
				buddy_sign_key_fingerprint,
				sizeof(buddy_sign_key_fingerprint) - 8);

  key_database_add(umid, buddy_sign_key_fingerprint, NULL);


  buddy_fingerprint_is_good(&mxfer, protocol_name);
  if (mxfer.error != NULL) {
    bc_console_write("ERROR ");
    bc_console_writeln(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return;
  }

  num_results = bc_slist_length(mxfer.result_sizes);

  /* parse results */

  bc_console_write("OK 3 ");

  str_tmp = bc_string_new(NULL);
  bc_string_append_printf(str_tmp, "%u", num_results);
  bc_console_writeln(str_tmp->str);
  bc_string_free(&str_tmp, 1);


  str_sizes = bc_string_new(NULL);
  str_results = bc_string_new(NULL);

  for (i = 0; i < num_results; i++) {
    a_size_str = bc_slist_nth_data(mxfer.result_sizes, i);
    a_message = bc_slist_nth_data(mxfer.results, i);

    a_size_int = (unsigned int)atoi(a_size_str);
    total_size += a_size_int;

    bc_string_append_printf(str_sizes, "%s ", a_size_str);

    str_results = bc_string_append_len(str_results, a_message, a_size_int);

    free(a_size_str);  a_size_str = NULL;
    free(a_message);   a_message = NULL;
  }
  bc_slist_free(&mxfer.results);
  bc_slist_free(&mxfer.result_sizes);

  bc_console_writeln(str_sizes->str);
  fwrite(str_results->str, sizeof(unsigned char), total_size, stdout);
  bc_console_writeln("");

  bc_string_free(&str_sizes, 1);
  bc_string_free(&str_results, 1);

}


/**
 * Accepts a key temporarily (ie: it is not added to the key database.).
 *
 * SYNTAX: KEY_ACCEPT_TEMPORARY <protocol name>|<umid>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_keyaccepttemporary(char *line) {
  BCString *str_sizes = NULL;
  BCString *str_results = NULL;
  BCString *str_tmp = NULL;
  char *a_size_str = NULL;
  char *a_message = NULL;
  unsigned int a_size_int = 0;
  unsigned int total_size = 0;
  unsigned int i = 0;
  unsigned int num_results = 0;
  struct UM_BUDDY *buddy = NULL;
  char *protocol_name = NULL;
  char *umid = NULL;
  char *pipe_pos = NULL;

  MXFER mxfer;
  char buddy_sign_key_fingerprint[ 192 ];

  memset(&mxfer, 0, sizeof(MXFER));
  memset(buddy_sign_key_fingerprint, 0, sizeof(buddy_sign_key_fingerprint));

  if (strlen(line) < 22) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  protocol_name = line + 21;
  pipe_pos = strchr(protocol_name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  umid = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(protocol_name) == 0) || (strlen(umid) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }


  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_console_writeln("ERROR INVALID STATE");
    return;
  }


  if ((buddy->public_sign_key == NULL) || (buddy->um_id != NULL)) {
    bc_console_writeln("ERROR INVALID STATE");
    return;
  }


  /* Assign this buddy a UMID. */
  buddy->um_id = strdup(umid);


  buddy_fingerprint_is_good(&mxfer, protocol_name);
  if (mxfer.error != NULL) {
    bc_console_write("ERROR ");
    bc_console_writeln(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return;
  }

  num_results = bc_slist_length(mxfer.result_sizes);


  bc_console_write("OK 3 ");

  str_tmp = bc_string_new(NULL);
  bc_string_append_printf(str_tmp, "%u", num_results);
  bc_console_writeln(str_tmp->str);
  bc_string_free(&str_tmp, 1);


  str_sizes = bc_string_new(NULL);
  str_results = bc_string_new(NULL);

  for (i = 0; i < num_results; i++) {
    a_size_str = bc_slist_nth_data(mxfer.result_sizes, i);
    a_message = bc_slist_nth_data(mxfer.results, i);

    a_size_int = (unsigned int)atoi(a_size_str);
    total_size += a_size_int;

    bc_string_append_printf(str_sizes, "%s ", a_size_str);

    str_results = bc_string_append_len(str_results, a_message, a_size_int);
    /*str_results = bc_string_append(str_results, " ");*/ /****/

    free(a_size_str);  a_size_str = NULL;
    free(a_message);   a_message = NULL;
  }
  bc_slist_free(&mxfer.results);
  bc_slist_free(&mxfer.result_sizes);

  bc_console_writeln(str_sizes->str);
  fwrite(str_results->str, sizeof(unsigned char), total_size, stdout);
  bc_console_writeln("");

  bc_string_free(&str_sizes, 1);
  bc_string_free(&str_results, 1);

}


/**
 * Rejects a key.  All state information relating to the specified protocol
 * name is reset.
 *
 * SYNTAX: KEY_REJECT <protocol name>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_keyreject(char *line) {
  struct UM_BUDDY *buddy = NULL;
  char *protocol_name = NULL;


  if (strlen(line) < 12) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  protocol_name = line + 11;

  if (strlen(protocol_name) == 0) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  /*printf("protocol_name: [%s]\n", protocol_name);*/

  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_mutex_unlock(pending_list_mutex);
    bc_console_writeln("ERROR INVALID STATE");
    return;
  }

  change_state(buddy, STATE_FLAG_SET, BUDDY_RESET);
  buddy->state = 0;
  remove_buddy_from_pending_list(buddy);

  remove_buddy_from_validated_list(buddy);
  wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  bc_mutex_unlock(pending_list_mutex);

  bc_console_writeln("OK");
}


/**
 * Resolves a protocol name into the corresponding UMID.
 *
 * SYNTAX: KEY_REJECT <protocol name>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_lookupname(char *line) {
  char *name = NULL;
  struct UM_BUDDY *buddy = NULL;
  unsigned int free_mutex = 0;

  if (strlen(line) < 13) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 12;

  buddy = find_buddy_in_validated_list(bc_normalize(name));
  if (buddy == NULL) {
    bc_mutex_lock(pending_list_mutex);
    free_mutex = 1;
    buddy = find_buddy_in_pending_list(bc_normalize(name));
  }

  if ((buddy == NULL) || (buddy->um_id == NULL))
    bc_console_writeln("ERROR NAME IS UNKNOWN");
  else
    bc_console_writeln(buddy->um_id);


  if (free_mutex == 1)
    bc_mutex_unlock(pending_list_mutex);
}


/**
 * Processes an encrypted or unencrypted message
 *
 * SYNTAX: PROCESS <name>|<size>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_processmessage(char *line) {
  char *pipe_pos = NULL;
  char *name = NULL;
  char *data_size = NULL;
  char *buffer = NULL;
  char *a_size_str = NULL;
  char *a_message = NULL;
  unsigned int a_size_int = 0;
  unsigned int total_size = 0;
  unsigned int i = 0;
  unsigned int data_size_int = 0;
  unsigned int num_results = 0;
  size_t x = 0;
  BCString *str_sizes = NULL;
  BCString *str_results = NULL;
  BCString *str_tmp = NULL;

  MXFER mxfer;

  memset(&mxfer, 0, sizeof(MXFER));


  if ((my_signing_keypair == NULL) || (my_encrypting_keypair == NULL)) {
    bc_console_writeln("ERROR USER MUST FIRST BE SELECTED");
    return;
  }

  if (strlen(line) < 9) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 8;
  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }


  data_size = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(name) == 0) || (strlen(data_size) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }


  data_size_int = (unsigned int)atoi(data_size);
  if (data_size_int > MAX_MESSAGE_SIZE) {
    bc_console_writeln("ERROR INVALID ARGUMENTS");
    return;
  }

  bc_log("module_processmessage", BC_LOG_INFO,
	 "Processing message of size %u.", data_size_int);

  buffer = (char *)calloc(data_size_int + 8, sizeof(char));
  if (buffer == NULL) {
    bc_console_writeln("ERROR INTERNAL");
    return;
  }

  clearerr(stdin);
  x = fread(buffer, sizeof(char), data_size_int, stdin);
  /*bc_log("module_processmessage", BC_LOG_INFO,
    "Actually read in %u bytes from stdin: [%s]", x, buffer);*/

  if (ferror(stdin)) {
    bc_console_writeln("ERROR INTERNAL");
    return;
  }

  /* Clear the CRLF that's still in the buffer. */
  readcrlf();


  get_handler(&mxfer, name, buffer);
  free(buffer);  buffer = NULL;

  if (mxfer.error != NULL) {
    bc_log("module_processmessage", BC_LOG_ERROR, mxfer.error->str);
    bc_console_write("ERROR ");
    bc_console_writeln(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return;
  }


  num_results = bc_slist_length(mxfer.result_sizes);


  bc_console_write("OK ");

  /* Write out the type of the result. */
  str_tmp = bc_string_new(NULL);
  bc_string_append_printf(str_tmp, "%u ", mxfer.type);
  bc_console_write(str_tmp->str);

  bc_string_clear(str_tmp);

  bc_string_append_printf(str_tmp, "%u", num_results);
  bc_console_writeln(str_tmp->str);
  bc_string_free(&str_tmp, 1);


  str_sizes = bc_string_new(NULL);
  str_results = bc_string_new(NULL);


  if (num_results != 0) {

    for (i = 0; i < num_results; i++) {
      a_size_str = bc_slist_nth_data(mxfer.result_sizes, i);
      a_message = bc_slist_nth_data(mxfer.results, i);

      a_size_int = (unsigned int)atoi(a_size_str);
      total_size += a_size_int;

      bc_string_append(str_sizes, a_size_str);
      bc_string_append(str_sizes, " ");

      str_results = bc_string_append_len(str_results, a_message, a_size_int);

      free(a_size_str);  a_size_str = NULL;
      free(a_message);   a_message = NULL;
    }
    bc_slist_free(&mxfer.results);
    bc_slist_free(&mxfer.result_sizes);

    bc_console_writeln(str_sizes->str);
    fwrite(str_results->str, sizeof(char), total_size, stdout);
    bc_console_writeln("");
  }

  bc_string_free(&str_sizes, 1);
  bc_string_free(&str_results, 1);

}


/**
 * Resets the module to its initial state; all state information is cleared.
 *
 * SYNTAX: RESET
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_reset() {
  bc_mutex_lock(pending_list_mutex);
  wipe_entire_pending_list();
  clear_all_pending_messages();
  bc_mutex_unlock(pending_list_mutex);

  wipe_entire_validated_list();

  if (auto_save(um_load_save_dir) != 1) {
    bc_log("module_reset()", BC_LOG_DEBUG, "auto_save error!");
  }


  bc_log("module_reset()", BC_LOG_DEBUG, "called.");
  bc_console_writeln("OK");
}


/**
 * Selects an ID that exists in the previously-chosen load/save directory.
 *
 * SYNTAX: SELECT_USER <id>|<password>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_selectuser(char *line) {
  int ret = 0;
  char *id = NULL;
  char *pw = NULL;
  char *pipe_pos = NULL;

  if (strlen(um_load_save_dir) == 0) {
    bc_console_writeln(ERROR_LOAD_SAVE_NOT_SET);
    return; 
  }

  if (strlen(line) < 13) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  id = line + 12;
  pipe_pos = strchr(id, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return; 
  }

  *pipe_pos = '\0';
  pw = pipe_pos + 1;

  if ((strlen(id) == 0) || (strlen(pw) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }


  if ((ret = load_database_and_keys(um_load_save_dir, id, pw)) == 1)
    bc_console_writeln("OK");
  else if (ret == 0)
    bc_console_writeln("ERROR ID IS UNKNOWN");
  else if (ret < 0)
    bc_console_writeln("PASSWORD IS BAD");

}


/**
 * Encrypts a plaintext message.
 *
 * SYNTAX: SEND <name>|<size>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_sendmessage(char *line) {
  char *name = NULL;
  char *data_size = NULL;
  unsigned int data_size_int = 0;
  char *pipe_pos = NULL;
  char *buffer = NULL;
  size_t x = 0;
  unsigned int total_size = 0;
  BCString *str_sizes = NULL;
  BCString *str_results = NULL;
  BCString *str_tmp = NULL;
  unsigned int a_size_int = 0;
  unsigned int i = 0;
  char *a_size_str = NULL;
  char *a_message = NULL;
  unsigned int num_results = 0;

  MXFER mxfer;
  unsigned char one_char_buf[ 1 ];
  char crlf[ 16 ];

  memset(&mxfer, 0, sizeof(mxfer));
  memset(one_char_buf, 0, sizeof(one_char_buf));
  memset(crlf, 0, sizeof(crlf));


  if ((my_signing_keypair == NULL) || (my_encrypting_keypair == NULL)) {
    bc_console_writeln("ERROR USER MUST FIRST BE SELECTED");
    return;
  }

  if (strlen(line) < 7) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 5;
  pipe_pos = strchr(name, '|');
  if (pipe_pos == NULL) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  data_size = pipe_pos + 1;
  *pipe_pos = '\0';


  if ((strlen(name) == 0) || (strlen(data_size) == 0)) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }


  data_size_int = (unsigned int)atoi(data_size);
  if (data_size_int > MAX_MESSAGE_SIZE) {
    bc_console_writeln("ERROR INVALID ARGUMENTS");
    return;
  }


  buffer = (char *)calloc(data_size_int + 8, sizeof(char));
  if (buffer == NULL) {
    bc_console_writeln("ERROR INTERNAL");
    return;
  }


  clearerr(stdin);
  x = fread(buffer, sizeof(char), data_size_int, stdin);
  if (ferror(stdin)) {
    free(buffer);  buffer = NULL;
    bc_console_writeln("ERROR INTERNAL");
    return;
  }


  /* Since we read the exact number of bytes of data above, we need to clear
   * the CRLF that's still in the buffer. */
  readcrlf();


  send_handler(&mxfer, name, buffer);
  free(buffer);  buffer = NULL;

  if (mxfer.error != NULL) {
    bc_log("module_sendmessage", BC_LOG_ERROR, mxfer.error->str);
    bc_console_write("ERROR ");
    bc_console_writeln(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return;
  }

  num_results = bc_slist_length(mxfer.result_sizes);

  bc_console_write("OK 3 ");
  str_sizes = bc_string_new(NULL);
  str_results = bc_string_new(NULL);

  str_tmp = bc_string_new(NULL);
  bc_string_append_printf(str_tmp, "%u", num_results);
  bc_console_writeln(str_tmp->str);
  bc_string_free(&str_tmp, 1);

  for (i = 0; i < num_results; i++) {
    a_size_str = bc_slist_nth_data(mxfer.result_sizes, i);
    a_message = bc_slist_nth_data(mxfer.results, i);

    a_size_int = atoi(a_size_str);
    total_size += a_size_int;

    bc_string_append(str_sizes, a_size_str);
    bc_string_append(str_sizes, " ");
    str_results = bc_string_append_len(str_results, a_message, a_size_int);

    free(a_size_str);  a_size_str = NULL;
    free(a_message);   a_message = NULL;
  }
  bc_slist_free(&mxfer.results);
  bc_slist_free(&mxfer.result_sizes);

  bc_console_writeln(str_sizes->str);
  fwrite(str_results->str, sizeof(char), total_size, stdout);
  bc_console_writeln("");

  bc_string_free(&str_sizes, 1);
  bc_string_free(&str_results, 1);

}


/**
 * Shuts the module down.
 *
 * SYNTAX: SHUTDOWN
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_shutdown() {
  time_t c;

  memset(&c, 0, sizeof(c));

  bc_log("module_shutdown", BC_LOG_DEBUG, "called.");

  bc_mutex_lock(state_output_mutex);
#ifdef _WIN32
  fprintf(stderr, "END_STDERR\n");
  fflush(stderr);
#else
  fprintf(stderr, "END_STDERR\r\n");
#endif
  bc_mutex_unlock(state_output_mutex);


  /* Update the entropy pool file. */
  update_entropy_file();


  kill_state_timeout_thread(&timeout_thread);

  module_reset();
  auto_clear();

  time(&c);
  bc_log("module_shutdown", BC_LOG_INFO, "Logging shutdown at %s", ctime((const time_t *)&c));
  close(logfile_fd);  logfile_fd = -1;

  bc_mutex_destroy(&state_output_mutex);
  bc_mutex_destroy(&pending_list_mutex);

  exit(0);
}


/**
 * Returns an English-readable status on a specified buddy.
 *
 * SYNTAX: STATUS <name>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_status(char *line) {
  char *name = NULL;
  BCString *retval = NULL;
  struct UM_BUDDY *buddy = NULL;

  if (strlen(line) < 8) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  name = line + 7;
  /*printf("name: [%s]\n", name);*/

  buddy = find_buddy_in_pending_list(bc_normalize(name));
  if (buddy != NULL) {
    if (buddy->state == SENT_PUBKEYS)
      retval = bc_string_append(retval, "Sent public keys; ");

    if (buddy->state == SENT_SESSION_KEY)
      retval = bc_string_append(retval, "Sent session key; ");

    if (buddy->state == AWAITING_PUBKEYS_1)
      retval = bc_string_append(retval, "Awaiting public signature key; ");

    if (buddy->state == AWAITING_PUBKEYS_2)
      retval = bc_string_append(retval, "Awaiting public encryption key; ");

    if (buddy->state == AWAITING_SESSION_KEY)
      retval = bc_string_append(retval, "Awaiting session key; ");

    if (buddy->state == SIGNKEY_FINGERPRINT_VALID)
      retval = bc_string_append(retval, "Public signature key validated; ");

    if (buddy->state == ENCRYPTKEY_VALID)
      retval = bc_string_append(retval, "Public encryption key validated; ");

    if (buddy->state == SESSION_KEY_VALID)
      retval = bc_string_append(retval, "Session key validated; ");

  } else {
    buddy = find_buddy_in_validated_list(bc_normalize(name));
    if (buddy != NULL)
      retval = bc_string_new("Connected; ");
  }


  if ((retval != NULL) && (strlen(retval->str) > 2)) {
    retval->str[ strlen(retval->str) - 2 ] = '.';
    retval->str[ strlen(retval->str) - 1 ] = '\0';
    bc_console_writeln(retval->str);
    bc_string_free(&retval, 1);
  } else
    bc_console_writeln("ERROR NAME IS UNKNOWN");

}


/**
 * Wipes a buddy's state information from memory.
 *
 * SYNTAX: WIPE <name>
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void module_wipe(char *line) {
  struct UM_BUDDY *buddy = NULL;
  char *protocol_name = NULL;

  if (strlen(line) < 6) {
    bc_console_writeln(ERROR_ARGUMENTS_MISSING);
    return;
  }

  protocol_name = line + 5;
  /*printf("pname: [%s]\n", protocol_name);*/

  buddy = find_buddy_in_validated_list(bc_normalize(protocol_name));
  if (buddy != NULL) {
    change_state(buddy, STATE_FLAG_SET, BUDDY_RESET);
    buddy->state = 0;

    remove_buddy_from_validated_list(buddy);

    bc_mutex_lock(pending_list_mutex);
    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
    bc_mutex_unlock(pending_list_mutex);

    bc_console_writeln("OK");
    return;
  }

  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_mutex_unlock(pending_list_mutex);
    bc_console_writeln("ERROR NAME IS UNKNOWN");
    return;
  }

  change_state(buddy, STATE_FLAG_SET, BUDDY_RESET);
  buddy->state = 0;

  remove_buddy_from_pending_list(buddy);
  wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  bc_mutex_unlock(pending_list_mutex);

  bc_console_writeln("OK");
}


/**
 * Processes the module's command line arguments. 
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void process_args(int ac, char **av) {
  if ((ac == 2) && (strcmp(av[ 1 ], "--help") == 0)) {
    printf("This is the help: %s\n", VERSION);
    exit(0);
  }

  if ((ac == 2) && (strcmp(av[ 1 ], "--version") == 0)) {
    printf("%s\n", VERSION);
    exit(0);
  }

  if ((ac == 2) && (strcmp(av[ 1 ], "--apiversion") == 0)) {
    printf("%s\n", API_VERSION);
    exit(0);
  }

  if ((ac == 3) && (strcmp(av[ 1 ], "--log") == 0))
    enable_logging(av[ 2 ]);

}


/**
 * Removes and discards a CLRF from stdin.
 */
/* Security audit trail:
 *    - Joe Testa: January 9th, 2005
 */
void readcrlf(void) {
  char crlf[ 4 ];

  memset(crlf, 0, sizeof(crlf));

  fgets(crlf, sizeof(crlf) - 1, stdin);
}

#endif
