/**
 * key_database.c
 *
 * Copyright (C) 2004-2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#include "key_database.h"
#ifdef DMALLOC
  #include "dmalloc.h"
#endif

extern char um_database_path[];


/* Adds a new Ultramagnetic ID/key fingerprint pair to the database. */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
void key_database_add(char *umid, char *key_fingerprint,
		      char *database_path) {

  FILE *handle = NULL;
  unsigned int i = 0;
  char achar = '\0';
  char *db_path = NULL;

  /* Ensure that umid contains only numbers & letters (and other printable
   *     chars). */
  for(i = 0; i < strlen(umid); i++) {
    achar = umid[ i ];
    if ((achar < 32) || (achar > 122)) {
      printf("INVALID CHARACTERS IN UMID!\n");
      return;
    }
  }

  if (database_path == NULL)
    db_path = um_database_path;
  else
    db_path = database_path;

  handle = fopen(db_path, "a");
  if (handle == NULL) {
    printf("ERROR ADDING TO DATABASE!\n");
    return;
  }

  fprintf(handle, "%s|%s\n", umid, key_fingerprint);
  fclose(handle);
}


/* Deletes an entry from the database.  Returns 1 on success and -1 on
 * error. */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
int key_database_delete(char *umid, char *key_fingerprint,
			char *database_path) {

  FILE *handle = NULL;
  char *buffer = NULL;
  unsigned int buffer_size = 0;
  int fd = 0;
  int deleted = -1;

  char file_line[ 256 ];
  char target_string[ 256 ];
  struct stat file_status;

  memset(file_line, 0, sizeof(file_line));
  memset(target_string, 0, sizeof(target_string));
  memset(&file_status, 0, sizeof(struct stat));

  /* TODO: replace character arrays with BCStrings. */
  bc_strlcpy(target_string, umid, sizeof(target_string) - 8);
  bc_strlcat(target_string, "|", sizeof(target_string) - 8);
  bc_strlcat(target_string, key_fingerprint, sizeof(target_string) - 8);
  bc_strlcat(target_string, "\n", sizeof(target_string) - 8);


  if (stat(database_path, &file_status) < 0) {
    bc_log("key_database_delete", BC_LOG_ERROR,
	   "ERROR WHILE GETTING FILE SIZE!");
    return -1;
  }

  buffer_size = file_status.st_size * 2;
  buffer = calloc(buffer_size + 16, sizeof(char));
  if (buffer == NULL) {
    bc_log("key_database_delete", BC_LOG_ERROR, "Out of memory!");
    return -1;
  }

  handle = fopen(database_path, "r");
  if (handle == NULL) {
    bc_log("key_database_delete", BC_LOG_ERROR, "ERROR REMOVING FROM DATABASE!");
    free(buffer);  buffer = NULL;
    return -1;
  }

  while(fgets(file_line, sizeof(file_line) - 8, handle) != NULL) {
    if (strcmp(file_line, target_string) != 0) {
      bc_strlcat(buffer, file_line, buffer_size);
    } else {
      deleted = 1;
    }
  }

  fclose(handle);  handle = NULL;

  fd = open(database_path, O_WRONLY | O_TRUNC);
  if (fd < 0) {
    bc_log("key_database_delete", BC_LOG_ERROR, "ERROR REMOVING FROM DATABASE!\n");
    free(buffer);  buffer = NULL;
    return -1;
  }

  if (strcmp(buffer, "") != 0) {
    write(fd, buffer, strlen(buffer));

    if (buffer[strlen(buffer) - 1] != '\n')
      write(fd, "\n", strlen("\n"));
  }
  close(fd);  fd = 0;

  free(buffer);  buffer = NULL;
  return deleted;

}


/* Retrieves the Ultramagnetic ID given a key fingerprint, or NULL if the
 * key does not match anything.  Returns 1 on success and -1 on error.*/
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
int key_database_lookup(char *key_fingerprint,
			char *buffer, unsigned int buffer_len) {
  int retVal = -1;
  FILE *handle = NULL;
  char *pipe_ptr = NULL;
  char *temp_umid = NULL;
  char *temp_fingerprint = NULL;

  char file_line[ 128 ];

  memset(file_line, 0, sizeof(file_line));

  handle = fopen(um_database_path, "r");
  if (handle == NULL) {
    bc_log("key_database_lookup", BC_LOG_ERROR, "ERROR WHILE OPENING DATABASE!\n");
    return -1;
  }

  while((fgets(file_line, sizeof(file_line) - 8, handle) != NULL) &&
	(retVal == -1)) {
    /*printf("file_line:  [%s]\n", file_line);*/

    if (strlen(file_line) > 0) {
      /* Remove the ending newline. */
      file_line[ strlen(file_line) - 1 ] = '\0';
    }


    pipe_ptr = strchr(file_line, '|');
    if (pipe_ptr == NULL) {
      bc_log("key_database_lookup", BC_LOG_ERROR, "Database is corrupt: no pipe: %s\n", file_line);
      return -1;
    }

    temp_umid = file_line;
    temp_fingerprint = pipe_ptr + 1;
    *pipe_ptr = '\0';

    if (strcmp(key_fingerprint, temp_fingerprint) == 0) {
      bc_strlcpy(buffer, temp_umid, buffer_len - 8);
      retVal = 1;
    }

  }

  return retVal;
}
