/**
 * library.c
 *
 * Copyright (C) 2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#ifdef BUILD_SHARED

#include "library.h"
#include "module.h"
#include <gcrypt.h>
#include <errno.h>
#include "utils.h"
#include "auto.h"
#include "debug.h"

#ifndef _WIN32
GCRY_THREAD_OPTION_PTHREAD_IMPL;
#endif

BC_MUTEX *state_output_mutex = NULL;
void (*change_state_func)(char *sn, char *umid, unsigned int operation, unsigned int old_state, unsigned int new_state) = NULL;


extern gcry_mpi_t mpi_one;
extern BC_MUTEX *pending_list_mutex;
extern BC_THREAD *timeout_thread;
extern struct PENDING_MESSAGE *pending_messages;
extern char um_load_save_dir[ MAX_PATH_LEN ];
extern int logfile_fd;
extern char um_database_path[ MAX_PATH_LEN ];
extern gcry_sexp_t *my_signing_keypair;
extern gcry_sexp_t *my_encrypting_keypair;

void change_state(struct UM_BUDDY *buddy, unsigned int operation,
                  unsigned int new_state);
BC_PROCESSED_MESSAGES *_bc_make_bpm_from_list(BCSList *list);
BC_ERROR *_bc_error_new(char *message);

void change_state(struct UM_BUDDY *buddy, unsigned int operation,
                  unsigned int new_state) {
  bc_mutex_lock(state_output_mutex);

  if (change_state_func != NULL) {
    change_state_func(buddy->screenname, buddy->um_id, operation,
		      buddy->state, new_state);
  }

  bc_mutex_unlock(state_output_mutex);
}

int bc_startup(char *bc_version, size_t bc_version_size,
	       char *bc_api_version, size_t bc_api_version_size,
	       void (*callback_func)(char *sn, char *umid,
					 unsigned int operation,
					 unsigned int old_state,
					 unsigned int new_state)) {
  int ret = 0;

  change_state_func = callback_func;
#ifndef _WIN32
  gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_pthread);
#endif
  if (!gcry_check_version(NEEDED_GCRYPT))
    return BC_ERROR_NEWER_GCRYPT_NEEDED;

  gcry_control(GCRYCTL_SUSPEND_SECMEM_WARN);
  gcry_control(GCRYCTL_INIT_SECMEM, 1024 * 1024 * 3);

  pending_messages = NULL;

  if (state_output_mutex == NULL)
    state_output_mutex = bc_mutex_new();

  if (pending_list_mutex == NULL)
    pending_list_mutex = bc_mutex_new();

  if (timeout_thread == NULL)
    timeout_thread = bc_thread_create(crypto_state_timeout_thread);

  mpi_one = gcry_mpi_new(8);
  ret = gcry_mpi_scan(&mpi_one, GCRYMPI_FMT_USG, (unsigned char *)"\x01", 1, NULL);
  if (ret != 0)
    return BC_ERROR_STARTUP_FAILED;


  bc_strlcpy(bc_version, VERSION, bc_version_size);
  bc_strlcpy(bc_api_version, API_VERSION, bc_api_version_size);
  return 1;
}

int bc_init(char *load_save_dir, BC_IDS_AND_FINGERPRINTS **db_head) {
  BCSList *my_ids = NULL;
  BCSList *db_ids = NULL;
  BCSList *db_fps = NULL;
  unsigned int i = 0;
  int error = 0;
  unsigned int num_mine = 0;
  unsigned int num_db = 0;
  BC_IDS_AND_FINGERPRINTS *ptr = NULL;

  char parsed_directory[ 256 ];
  char filepath[ 256 ];
  char buffer[ 256 ];

  memset(parsed_directory, 0, sizeof(parsed_directory));
  memset(filepath, 0, sizeof(filepath));
  memset(buffer, 0, sizeof(buffer));


  (*db_head) = (BC_IDS_AND_FINGERPRINTS *)calloc(1, sizeof(BC_IDS_AND_FINGERPRINTS));
  ptr = *db_head;

  parse_directory_string(load_save_dir, parsed_directory,
			 sizeof(parsed_directory) - 8);

  my_ids = load_ids_from_directory(parsed_directory);
  if (my_ids == NULL) {
    error = BC_ERROR_DIRECTORY_NOT_FOUND;
    goto bc_init_error;
  }

  num_mine = bc_slist_length(my_ids) - 1;
  ptr->num_mine = num_mine;
  ptr->my_ids = (char **)calloc(num_mine, sizeof(char *));
  ptr->my_fps = (char **)calloc(num_mine, sizeof(char *));


  for(i = 1; i < num_mine + 1; i++) {
    char *my_id = bc_slist_nth_data(my_ids, i);

    ptr->my_ids[ i - 1 ] = (char *)calloc(strlen(my_id) + 1, sizeof(char));
    bc_strlcpy(ptr->my_ids[ i - 1 ], my_id, strlen(my_id) + 1);


    bc_strlcpy(filepath, parsed_directory, sizeof(filepath));
    bc_strlcat(filepath, my_id, sizeof(filepath));
    bc_strlcat(filepath, ".bc", sizeof(filepath));

    /* Load the current ID's fingerprint. */
    if (!get_fingerprint_from_keyfile(filepath, buffer, sizeof(buffer) - 8)) {
      error = BC_ERROR_FINGERPRINT_ERROR;
      goto bc_init_error;
    }

    ptr->my_fps[ i - 1 ] = (char *)calloc(strlen(buffer) + 1, sizeof(char));
    bc_strlcpy(ptr->my_fps[ i - 1 ], buffer, strlen(buffer) + 1);
  }
  bc_slist_free(&my_ids);


  bc_strlcpy(filepath, parsed_directory, sizeof(filepath));
  bc_strlcat(filepath, UM_DATABASE_FILENAME, sizeof(filepath));

  /* Load the buddy fingerprints in the database. */
  get_database_raw_list(filepath, &db_ids, &db_fps);

  num_db = bc_slist_length(db_ids);

  ptr->num_db = num_db;

  ptr->db_ids = (char **)calloc(num_db, sizeof(char *));
  ptr->db_fps = (char **)calloc(num_db, sizeof(char *));

  for (i = 1; i < num_db + 1; i++) {
    char *db_id = bc_slist_nth_data(db_ids, i);
    char *db_fp = bc_slist_nth_data(db_fps, i);


    ptr->db_ids[ i - 1 ] = (char *)calloc(strlen(db_id) + 1, sizeof(char));
    bc_strlcpy(ptr->db_ids[ i - 1 ], db_id, strlen(db_id) + 1);

    ptr->db_fps[ i - 1 ] = (char *)calloc(strlen(db_fp) + 1, sizeof(char));
    bc_strlcpy(ptr->db_fps[ i - 1 ], db_fp, strlen(db_fp) + 1);
  }
  bc_slist_free(&db_ids);
  bc_slist_free(&db_fps);

  /* Set the load/save directory and store the path to the key database. */
  bc_strlcpy(um_load_save_dir, parsed_directory, sizeof(um_load_save_dir) - 8);
  bc_strlcpy(um_database_path, parsed_directory, sizeof(um_database_path) - 8);
  bc_strlcat(um_database_path, UM_DATABASE_FILENAME, sizeof(um_database_path) - 8);

  free_database_raw_list(&db_ids, &db_fps);
  return 1;

 bc_init_error:
  free_database_raw_list(&db_ids, &db_fps);
  bc_free_ids_and_fingerprints(db_head);
  return error;
}

void bc_free_ids_and_fingerprints(BC_IDS_AND_FINGERPRINTS **db_head) {
  unsigned int i = 0;
  BC_IDS_AND_FINGERPRINTS *ptr = NULL;

  if (db_head == NULL)
    return;

  ptr = *db_head;
  if (ptr == NULL)
    return;

  for (i = 0; i < ptr->num_mine; i++) {
    if (ptr->my_ids != NULL) {
      free(ptr->my_ids[ i ]);  ptr->my_ids[ i ] = NULL;
    }
    if (ptr->my_fps != NULL) {
      free(ptr->my_fps[ i ]);  ptr->my_fps[ i ] = NULL;
    }
  }
  if (ptr->my_ids != NULL) {
    free(ptr->my_ids);  ptr->my_ids = NULL;
  }
  if (ptr->my_fps != NULL) {
    free(ptr->my_fps);  ptr->my_fps = NULL;
  }


  for (i = 0; i < ptr->num_db; i++) {
    if(ptr->db_ids != NULL) {
      free(ptr->db_ids[ i ]);  ptr->db_ids[ i ] = NULL;
    }
    if (ptr->db_fps != NULL) {
      free(ptr->db_fps[ i ]);  ptr->db_fps[ i ] = NULL;
    }
  }
  if(ptr->db_ids != NULL) {
    free(ptr->db_ids);  ptr->db_ids = NULL;
  }
  if (ptr->db_fps != NULL) {
    free(ptr->db_fps);  ptr->db_fps = NULL;
  }


  free(ptr); ptr = NULL;
  *db_head = NULL;
}

int bc_select_user(char *id, char *password) {
  int ret = load_database_and_keys(um_load_save_dir, id, password);

  if (ret == 0)
    ret = BC_ERROR_UNKNOWN_ID;
  else if (ret < 0)
    ret = BC_ERROR_BAD_PASSWORD;

  return ret;
}

int bc_send(char *protocol_name, char *data,
	    BC_PROCESSED_MESSAGES **messages,
	    BC_ERROR **error) {
  BC_PROCESSED_MESSAGES *ptr = NULL;
  unsigned int num = 0;
  unsigned int i = 0;
  char *a_message = NULL;
  char *my_protocol_name = NULL;

  MXFER mxfer;

  memset(&mxfer, 0, sizeof(MXFER));

  /* Some inner functions might modify this arg, so use a copy instead. */
  /* TODO:  fix this quirk for real. */
  my_protocol_name = strdup(protocol_name);
  send_handler(&mxfer, my_protocol_name, data);
  free(my_protocol_name);  my_protocol_name = NULL;

  if (mxfer.error != NULL) {
    bc_log("bc_send", BC_LOG_ERROR, mxfer.error->str);
    *error = _bc_error_new(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return -1;
  }

  num = bc_slist_length(mxfer.result_sizes);

  ptr = (BC_PROCESSED_MESSAGES *)calloc(1, sizeof(BC_PROCESSED_MESSAGES));
  ptr->type = mxfer.type;
  ptr->num = num;
  ptr->result = (char **)calloc(num, sizeof(char *));


  for (i = 0; i < num; i++) {
    /*a_size_str = (unsigned char *)bc_slist_nth_data(mxfer.result_sizes, i);*/
    a_message = (char *)bc_slist_nth_data(mxfer.results, i);
    ptr->result[ i ] = (char *)calloc(strlen(a_message) + 1, sizeof(char *));
    bc_strlcpy(ptr->result[ i ], a_message, strlen(a_message) + 1);
  }

  *messages = ptr;
  return 1;
}

int bc_process(char *protocol_name, char *data,
	       BC_PROCESSED_MESSAGES **messages,
	       BC_ERROR **error) {
  BC_PROCESSED_MESSAGES *ptr = NULL;
  unsigned int num = 0;
  unsigned int i = 0;
  char *a_message = NULL;
  char *my_protocol_name = NULL;

  MXFER mxfer;

  memset(&mxfer, 0, sizeof(MXFER));

  /* Some inner functions might modify this arg, so use a copy instead. */
  /* TODO:  fix this quirk for real. */
  my_protocol_name = strdup(protocol_name);
  get_handler(&mxfer, my_protocol_name, data);
  free(my_protocol_name);  my_protocol_name = NULL;

  if (mxfer.error != NULL) {
    bc_log("bc_process", BC_LOG_ERROR, mxfer.error->str);
    *error = _bc_error_new(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return -1;
  }

  num = bc_slist_length(mxfer.result_sizes);

  ptr = (BC_PROCESSED_MESSAGES *)calloc(1, sizeof(BC_PROCESSED_MESSAGES));
  ptr->type = mxfer.type;
  ptr->num = num;
  ptr->result = (char **)calloc(num, sizeof(char *));


  for (i = 0; i < num; i++) {
    /*a_size_str = (unsigned char *)bc_slist_nth_data(mxfer.result_sizes, i);*/
    a_message = (char *)bc_slist_nth_data(mxfer.results, i);
    ptr->result[ i ] = (char *)calloc(strlen(a_message) + 1, sizeof(char *));
    bc_strlcpy(ptr->result[ i ], a_message, strlen(a_message) + 1);
  }

  *messages = ptr;
  return 1;
}

int bc_key_accept_temporary(char *protocol_name, char *umid,
			    BC_PROCESSED_MESSAGES **messages,
			    BC_ERROR **error) {
  struct UM_BUDDY *buddy = NULL;

  MXFER mxfer;

  memset(&mxfer, 0, sizeof(MXFER));


  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL)
    return BC_ERROR_INVALID_STATE;

  if ((buddy->public_sign_key == NULL) || (buddy->um_id != NULL))
    return BC_ERROR_INVALID_STATE;

  buddy->um_id = strdup(umid);

  buddy_fingerprint_is_good(&mxfer, protocol_name);
  if (mxfer.error != NULL) {
    *error = _bc_error_new(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return -1;
  }

  *messages = _bc_make_bpm_from_list(mxfer.results);
  return 1;
}

int bc_key_accept_permanently(char *protocol_name, char *umid,
			      BC_PROCESSED_MESSAGES **messages,
			      BC_ERROR **error) {
  struct UM_BUDDY *buddy = NULL;

  char buddy_sign_key_fingerprint[ 192 ];
  MXFER mxfer;

  memset(buddy_sign_key_fingerprint, 0, sizeof(buddy_sign_key_fingerprint));
  memset(&mxfer, 0, sizeof(MXFER));


  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL)
    return BC_ERROR_INVALID_STATE;

  if ((buddy->public_sign_key == NULL) || (buddy->um_id != NULL))
    return BC_ERROR_INVALID_STATE;

  buddy->um_id = strdup(umid);

  get_key_fingerprint_from_sexp(buddy->public_sign_key,
				buddy_sign_key_fingerprint,
				sizeof(buddy_sign_key_fingerprint) - 8);

  key_database_add(umid, buddy_sign_key_fingerprint, NULL);

  buddy_fingerprint_is_good(&mxfer, protocol_name);
  if (mxfer.error != NULL) {
    *error = _bc_error_new(mxfer.error->str);
    bc_string_free(&(mxfer.error), 1);
    return -1;
  }

  *messages = _bc_make_bpm_from_list(mxfer.results);
  return 1;
}

int bc_key_reject(char *protocol_name) {

  struct UM_BUDDY *buddy = NULL;

  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_mutex_unlock(pending_list_mutex);
    return BC_ERROR_INVALID_STATE;
  }

  change_state(buddy, STATE_FLAG_SET, BUDDY_RESET);
  buddy->state = 0;
  remove_buddy_from_pending_list(buddy);

  remove_buddy_from_validated_list(buddy);
  wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  bc_mutex_unlock(pending_list_mutex);

  return 1;
}

int bc_lookup_name(char *name_in, char *name_out, size_t name_out_size) {
  struct UM_BUDDY *buddy = NULL;
  unsigned int free_mutex = 0;
  int ret = 0;

  buddy = find_buddy_in_validated_list(bc_normalize(name_in));
  if (buddy == NULL) {
    bc_mutex_lock(pending_list_mutex);
    free_mutex = 1;
    buddy = find_buddy_in_pending_list(bc_normalize(name_in));
  }


  if ((buddy != NULL) && (buddy->um_id != NULL)) {
    bc_strlcpy(name_out, buddy->um_id, name_out_size);
    ret = 1;
  } else
    ret = BC_ERROR_UNKNOWN_ID;


  if (free_mutex == 1)
    bc_mutex_unlock(pending_list_mutex);

  return ret;
}

int bc_wipe(char *protocol_name) {
  struct UM_BUDDY *buddy = NULL;

  buddy = find_buddy_in_validated_list(bc_normalize(protocol_name));
  if (buddy != NULL) {
    buddy->state = 0;
    remove_buddy_from_validated_list(buddy);

    bc_mutex_lock(pending_list_mutex);
    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
    bc_mutex_unlock(pending_list_mutex);

    return 1;
  }

  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy == NULL) {
    bc_mutex_unlock(pending_list_mutex);
    return BC_ERROR_UNKNOWN_ID;
  }

  buddy->state = 0;
  remove_buddy_from_pending_list(buddy);
  wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  bc_mutex_unlock(pending_list_mutex);

  return 1;
}

int bc_status(char *protocol_name, char *status, size_t status_size) {
  struct UM_BUDDY *buddy = NULL;
  BCString *retval = NULL;

  buddy = find_buddy_in_pending_list(bc_normalize(protocol_name));
  if (buddy != NULL) {
    if (buddy->state == SENT_PUBKEYS)
      retval = bc_string_append(retval, "Sent public keys; ");

    if (buddy->state == SENT_SESSION_KEY)
      retval = bc_string_append(retval, "Sent session key; ");

    if (buddy->state == AWAITING_PUBKEYS_1)
      retval = bc_string_append(retval, "Awaiting public signature key; ");

    if (buddy->state == AWAITING_PUBKEYS_2)
      retval = bc_string_append(retval, "Awaiting public encryption key; ");

    if (buddy->state == AWAITING_SESSION_KEY)
      retval = bc_string_append(retval, "Awaiting session key; ");

    if (buddy->state == SIGNKEY_FINGERPRINT_VALID)
      retval = bc_string_append(retval, "Public signature key validated; ");

    if (buddy->state == ENCRYPTKEY_VALID)
      retval = bc_string_append(retval, "Public encryption key validated; ");

    if (buddy->state == SESSION_KEY_VALID)
      retval = bc_string_append(retval, "Session key validated; ");

  } else {
    buddy = find_buddy_in_validated_list(bc_normalize(protocol_name));
    if (buddy != NULL)
      retval = bc_string_new("Connected; ");
  }


  if ((retval != NULL) && (strlen(retval->str) > 2)) {
    retval->str[ strlen(retval->str) - 2 ] = '.';
    retval->str[ strlen(retval->str) - 1 ] = '\0';
    bc_strlcpy(status, retval->str, status_size);
    bc_string_free(&retval, 1);

    return 1;
  } else
    return BC_ERROR_UNKNOWN_ID;

}

void bc_reset(void) {
  bc_mutex_lock(pending_list_mutex);
  wipe_entire_pending_list();
  clear_all_pending_messages();
  bc_mutex_unlock(pending_list_mutex);

  wipe_entire_validated_list();

  if (auto_save(um_load_save_dir) != 1) {
    /* ERROR */
  }


  bc_log("bc_reset()", BC_LOG_DEBUG, "called.");
}

void bc_shutdown(void) {
  update_entropy_file();

  /* TODO: destroy the thread too! */
  bc_reset();
  close(logfile_fd);  logfile_fd = -1;

  auto_clear();

  kill_state_timeout_thread(&timeout_thread);

  bc_mutex_destroy(&state_output_mutex);
  bc_mutex_destroy(&pending_list_mutex);
}

int bc_add_key(char *id, char *fp) {
  if (strlen(um_database_path) == 0)
    return BC_ERROR_LOADSAVE_NOT_SET;

  bc_log("bc_add_key", BC_LOG_DEBUG, "Adding key to database: [%s][%s]",
	 id, fp);

  key_database_add(id, fp, um_database_path);
  return 1;
}

int bc_del_key(char *id, char *fp) {
  if (strlen(um_database_path) == 0)
    return BC_ERROR_LOADSAVE_NOT_SET;

  if (key_database_delete(id, fp, um_database_path) == 1)
    return 1;
  else
    return BC_ERROR_UNKNOWN_ID;
}

int bc_generate_key(char *id, char *password,
		    char *fingerprint, size_t fingerprint_size) {

  char buffer[ 256 ];

  memset(buffer, 0, sizeof(buffer));


  if ((strchr(id, '.') != NULL) || (strchr(id, '|') != NULL) ||
      (strchr(id, '/') != NULL) || (strchr(id, '\\') != NULL)) {
    return BC_ERROR_ILLEGAL_CHARS;
  }

  if (strlen(um_load_save_dir) == 0)
    return BC_ERROR_LOADSAVE_NOT_SET;

  if (generate_key(um_load_save_dir, id, password,
		   buffer, sizeof(buffer) - 8) == 1) {
    bc_strlcpy(fingerprint, buffer, fingerprint_size);
    return 1;
  } else
    return BC_ERROR_GENERIC;

}

int bc_is_loadsave_initialized(void) {
  if (strlen(um_load_save_dir) > 0)
    return 1;
  else
    return 0;
}

int bc_is_user_selected(void) {
  if ((my_signing_keypair != NULL) || (my_encrypting_keypair != NULL))
    return 1;
  else
    return 0;
}

int bc_base64_padding(unsigned int flag) {
  return 0;
}

/* TODO: custom error messages! */
int bc_encrypt_file(char *protocol_name, char *in_file, char *out_file) {
  if (encrypt_file(protocol_name, in_file, out_file) == 1)
    return 1;
  else
    return BC_ERROR_GENERIC;
}

int bc_decrypt_file(char *protocol_name, char *in_file, char *out_file) {
  if (decrypt_file(protocol_name, in_file, out_file) == 1)
    return 1;
  else
    return BC_ERROR_GENERIC;
}

int bc_auto_query(char *name) {
  if (strlen(um_load_save_dir) == 0)
    return BC_ERROR_LOADSAVE_NOT_SET;
  
  if (auto_query(name))
    return 1;
  else
    return 0;
}

int bc_auto_feedback(char *name, unsigned int flag) {
  char *cp = NULL;

  if (strlen(um_load_save_dir) == 0)
    return BC_ERROR_LOADSAVE_NOT_SET;

  if (flag == 0)
    cp = "P";
  else
    cp = "C";

  if (auto_feedback(name, cp))
    return 1;
  else
    return BC_ERROR_GENERIC;
}

void bc_error_free(BC_ERROR **error) {
  BC_ERROR *e = NULL;

  if (error == NULL)
    return;

  e = *error;
  if (e == NULL) {
    *error = NULL;
    return;
  }

  if (e->message != NULL) {
    free(e->message);  e->message = NULL;
  }

  free(e); e = NULL;
  *error = NULL;
}

void bc_processed_messages_free(BC_PROCESSED_MESSAGES **bpm) {
  BC_PROCESSED_MESSAGES *b = NULL;
  unsigned int i = 0;

  if (bpm == NULL)
    return;

  b = *bpm;
  if(b == NULL) {
    *bpm = NULL;
    return;
  }

  for (i = 0; i < b->num; i++) {
    free(b->result[ i ]);  b->result[ i ] = NULL;
  }
  free(b->result);  b->result = NULL;

  b->type = 0;
  b->num = 0;

  free(b); b = NULL;
  *bpm = NULL;
}

BC_ERROR *_bc_error_new(char *message) {
  BC_ERROR *retval = (BC_ERROR *)calloc(1, sizeof(BC_ERROR));
  retval->message = (char *)calloc(strlen(message) + 1, sizeof(char));
  bc_strlcpy(retval->message, message, strlen(message) + 1);

  return retval;
}

BC_PROCESSED_MESSAGES *_bc_make_bpm_from_list(BCSList *list) {
  BC_PROCESSED_MESSAGES *ptr = NULL;
  unsigned int num = 0;
  unsigned int i = 0;
  char *a_message = NULL;


  num = bc_slist_length(list);

  ptr = (BC_PROCESSED_MESSAGES *)calloc(1, sizeof(BC_PROCESSED_MESSAGES));
  ptr->num = num;
  ptr->result = (char **)calloc(num, sizeof(char *));


  for (i = 0; i < num; i++) {
    /*a_size_str = (unsigned char *)bc_slist_nth_data(mxfer.result_sizes, i);*/
    a_message = (char *)bc_slist_nth_data(list, i);
    ptr->result[ i ] = (char *)calloc(strlen(a_message) + 1, sizeof(char *));
    bc_strlcpy(ptr->result[ i ], a_message, strlen(a_message) + 1);
  }

  return ptr;
}

#endif
