/**
 * library.h
 *
 * Copyright (C) 2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef LIBRARY_H
#define LIBRARY_H

#include <sys/types.h>

/* Error codes returned by these functions. Self-explanatory. */
#define BC_ERROR_GENERIC                -1
#define BC_ERROR_NEWER_GCRYPT_NEEDED    -2
#define BC_ERROR_STARTUP_FAILED         -3
#define BC_ERROR_DIRECTORY_NOT_FOUND    -4
#define BC_ERROR_FINGERPRINT_ERROR      -5
#define BC_ERROR_UNKNOWN_ID             -6
#define BC_ERROR_BAD_PASSWORD           -7
#define BC_ERROR_INVALID_STATE          -8
#define BC_ERROR_LOADSAVE_NOT_SET       -9
#define BC_ERROR_ILLEGAL_CHARS         -10


/* State change flags that describe the operation which provoked an invokation
 * of the callback function specified in bc_startup().  These flags match
 * the 'operation' argument. */
#define BC_STATE_OPERATION_REMOVE 0
#define BC_STATE_OPERATION_ADD 1
#define BC_STATE_OPERATION_SET 2


/* State information that describes the 'new_state' argument passed to the
 * callback function specified in bc_startup(). */
#define BC_STATE_BUDDY_RESET                  0
#define BC_STATE_SENT_PUBKEYS                 1
#define BC_STATE_SENT_SESSION_KEY             2
#define BC_STATE_AWAITING_PUBKEYS_1           4
#define BC_STATE_AWAITING_PUBKEYS_2           8
#define BC_STATE_AWAITING_SESSION_KEY        16
#define BC_STATE_SIGNKEY_FINGERPRINT_VALID   32
#define BC_STATE_ENCRYPTKEY_VALID            64
#define BC_STATE_SESSION_KEY_VALID          128
#define BC_STATE_CONNECTED                  256
#define BC_STATE_TIMEDOUT                   512
#define BC_STATE_REPLAY_IMMUNE             1024


/* The BC_MESSAGE_TYPE_XXX constants are used to compare against the 'type'
 * field of the BC_PROCESSED_MESSAGES struct.  They distinguish the type of
 * results in that struct. */

/* Zero is not used. */
#define BC_MESSAGE_TYPE_UNUSED                   0

/* Signifies that the results field holds a plaintext message that was
 * originally encrypted by the peer. */
#define BC_MESSAGE_TYPE_WAS_ENCRYPTED            1

/* Signifies that the results field holds a plaintext message that was sent
 * by the peer WITHOUT encrypting & authenticating.  BE CAREFUL WHEN YOU HANDLE
 * THIS TYPE OF MESSAGE! */
#define BC_MESSAGE_TYPE_PLAINTEXT                2

/* Signifies that the results field holds messages that must be sent back to
 * the peer immediately; they are a necessary part of the initial handshake
 * protocol. */
#define BC_MESSAGE_TYPE_ACTIONABLE_HANDSHAKE     3

/* Signifies that the previously processed message was a part of the initial
 * handshake protocol and must be ignored by the calling program.  Do not
 * examine the 'results' field any further. */
#define BC_MESSAGE_TYPE_NONACTIONABLE_HANDSHAKE  4

/* Signifies that the results field holds the peer's public key fingerprint
 * and that this fingerprint is not in the internal key database (thus the
 * calling program must present it to the user for manual verification). */
#define BC_MESSAGE_TYPE_FINGERPRINT              5



struct _BC_IDS_AND_FINGERPRINTS {
  unsigned int num_mine;
  unsigned int num_db;
  char **my_ids;
  char **my_fps;
  char **db_ids;
  char **db_fps;
};
typedef struct _BC_IDS_AND_FINGERPRINTS BC_IDS_AND_FINGERPRINTS;


struct _BC_PROCESSED_MESSAGES {
  unsigned int type;
  unsigned int num;
  char **result;
};
typedef struct _BC_PROCESSED_MESSAGES BC_PROCESSED_MESSAGES;

struct _BC_ERROR {
  char *message;
};
typedef struct _BC_ERROR BC_ERROR;

int bc_startup(char *bc_version, size_t bc_version_size,
	       char *bc_api_version, size_t bc_api_version_size,
	       void (*change_state_func)(char *sn, char *umid,
					 unsigned int operation,
					 unsigned int old_state,
					 unsigned int new_state));

int bc_init(char *load_save_dir, BC_IDS_AND_FINGERPRINTS **db_head);

void bc_free_ids_and_fingerprints(BC_IDS_AND_FINGERPRINTS **db_head);

int bc_select_user(char *id, char *password);

int bc_send(char *protocol_name, char *data,
	    BC_PROCESSED_MESSAGES **messages, BC_ERROR **error);

int bc_process(char *protocol_name, char *data,
	       BC_PROCESSED_MESSAGES **messages, BC_ERROR **error);

int bc_key_accept_temporary(char *protocol_name, char *umid,
			    BC_PROCESSED_MESSAGES **messages,
			    BC_ERROR **error);

int bc_key_accept_permanently(char *protocol_name, char *umid,
			      BC_PROCESSED_MESSAGES **messages,
			      BC_ERROR **error);

void bc_free_processed_messages(BC_PROCESSED_MESSAGES **messages);

int bc_key_reject(char *protocol_name);

int bc_lookup_name(char *name_in, char *name_out, size_t name_out_size);

int bc_wipe(char *protocol_name);

int bc_status(char *protocol_name, char *status, size_t status_size);

void bc_reset(void);

void bc_shutdown(void);

int bc_add_key(char *id, char *fp);

int bc_del_key(char *id, char *fp);

int bc_generate_key(char *id, char *password,
		    char *fingerprint, size_t fingerprint_size);

int bc_is_loadsave_initialized(void);

int bc_is_user_selected(void);

int bc_base64_padding(unsigned int flag);

/* TODO: custom error messages! */
int bc_encrypt_file(char *protocol_name, char *in_file, char *out_file);

int bc_decrypt_file(char *protocol_name, char *in_file, char *out_file);

int bc_auto_query(char *name);

int bc_auto_feedback(char *name, unsigned int flag);

void bc_error_free(BC_ERROR **error);

void bc_processed_messages_free(BC_PROCESSED_MESSAGES **bpm);

#endif
