/**
 * module.c
 *
 * Copyright (C) 2003-2006 J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#include "module.h"
#ifdef DMALLOC
  #include "dmalloc.h"
#endif


extern void change_state(struct UM_BUDDY *buddy, unsigned int operation, unsigned int new_state);
#ifdef BUILD_EXECUTABLE
#include "console.h"
#else
#include "library.h"
#endif

gcry_sexp_t *my_signing_keypair = NULL;
gcry_sexp_t *my_encrypting_keypair = NULL;
gcry_mpi_t mpi_one;

BC_MUTEX *pending_list_mutex = NULL;
BC_THREAD *timeout_thread = NULL;

struct UM_BUDDY *validated_buddies_list = NULL;
struct UM_BUDDY *pending_buddies_list = NULL;

char um_load_save_dir[ MAX_PATH_LEN ] = "\0";
char um_database_path[ MAX_PATH_LEN ] = "\0";

struct PENDING_MESSAGE *pending_messages = NULL;

/* These are the fields within the generated key s-expression that represent
 * the public values. */
char *sign_key_fieldz[] = { "p", "q", "g", "y", NULL };
char *encrypt_key_fieldz[] = { "p", "g", "y", NULL };

unsigned int thread_continue_flag = 1;

/* This function free()'s all the entries inside a GList, then
 * g_list_free()'s the list itself. Call this function on GLists that are
 * created from g_strdup()'ed strings. The argument is set to NULL when
 * this code returns. */
/* Security audit trail:
 *    - Joe Testa: March 4th, 2004
 *    - Joe Testa: July 18th, 2003
 */
/*void bc_free_glist(GList **list) {

  unsigned int i = 0;

  if ((list != NULL) && (*list != NULL)) {
    for (i = 0; i < g_list_length(*list); i++) {
      * Caution:  argument is not NULLified after free()'ed! *
      free(g_list_nth_data(*list, i));
    }
    g_list_free(*list);  *list = NULL;
  }

  }*/




/* This function parses the directory path passed in and produces 
 * directory (ie: not a file and not a symlink) and that the end of
 * the path has a slash character. On error, the first character in
 * 'output_buffer' will be NULL (so strlen(return_value) will be zero). */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 *    - Joe Testa: July 18th, 2003
 */
void parse_directory_string(char *directory_path,
                            char *output_buffer,
                            unsigned int output_buffer_len) {

  unsigned int current_output_len = 0;
  char lastChar = '\0';
#ifdef _WIN32
  HANDLE hFile = 0;
#else
  DIR *dir_handle = NULL;
#endif

  struct stat st;

  memset(&st, 0, sizeof(struct stat));


  bc_strlcpy(output_buffer, directory_path, output_buffer_len);

#ifdef _WIN32
  hFile = CreateFile(output_buffer, GENERIC_READ, FILE_SHARE_READ, NULL,
		     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  /* If the file handle is not invalid, then the user clicked on a file
  *  before clicking OK, so we need to parse out the find now. */
  if (hFile != INVALID_HANDLE_VALUE) {
    char *ptr = NULL;

    CloseHandle(hFile); hFile = 0;
    ptr = strrchr(output_buffer, SLASH_CHAR);
    if (ptr != NULL) {
      *(ptr + 1) = '\0';
    } else {
      /* Apparently, what was passed in was a file, but we somehow failed to
       * parse out the file name.  I have no idea how this can happen; just
       * more proactive code... */
      output_buffer[ 0 ] = '\0';
      return;
    }
  }
#else

  /* If the directory selected by the user is a symlink or the path is invalid,
   * bail. */
  if (is_symlink(output_buffer) >= 0) {
    output_buffer[ 0 ] = '\0';
    return;
  }

  /* Attempt to open the directory.  If it cannot be opened, then the user
   * selected a file to open, so we must parse it out here. */
  dir_handle = opendir(output_buffer);
  if (dir_handle == NULL) {
    char *ptr = NULL;
    ptr = strrchr(output_buffer, SLASH_CHAR);
    if (ptr != NULL) {
      *(ptr + 1) = '\0';
    } else {
      /* Apparently, what was passed in was a file, but we somehow failed to
       * parse out the file name.  I have no idea how this can happen; just
       * more proactive code... */
      output_buffer[ 0 ] = '\0';
      return;
    }

  } else {
    /* No accidental double-closedir vulns allowed!  =P */
    closedir(dir_handle);  dir_handle = NULL;
  }

#endif

  /* Ensure one last time that the directory ends with the appropriate
   * slash character for this platform. */

  /* If the current output buffer length is zero, then we bail. (Otherwise
   * we will access outside the array's boundaries below!) */
  current_output_len = strlen(output_buffer);
  if (current_output_len == 0) {
    output_buffer[ 0 ] = '\0';
    return;
  }
  lastChar = output_buffer[ current_output_len - 1 ];

  /* +3 avoids potential off-by-ones! */
  if ((lastChar != SLASH_CHAR) &&
      (current_output_len + 3 < output_buffer_len)) {
    output_buffer[ current_output_len ] = SLASH_CHAR;
    output_buffer[ current_output_len + 1 ] = '\0';
  /* If the slash character is missing, and there is not enough room to
   * insert it, then NULLify the output buffer to signify an error. */
  } else if ((lastChar != SLASH_CHAR) &&
      !(current_output_len + 3 < output_buffer_len)) {
    output_buffer[ 0 ] = '\0';
  }

  return;
}


/* Returns a GList of all *.um files that were found given a directory path
 * that was parsed with the 'parse_directory_string' function.  Note that the
 * file contents are not checked for validity. */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 *    - Joe Testa: July 18th, 2003
 */
BCSList *load_ids_from_directory(char *dir_name) {

  BCSList *retval = NULL;
#ifdef _WIN32
  HANDLE hSearch = 0;
  unsigned char *ptr = NULL;
  unsigned int file_len = 0;

  WIN32_FIND_DATA wfd;
  unsigned char new_dir_name[ 256 ];
  unsigned char abspath[ 256 ];

  memset(&wfd, 0, sizeof(WIN32_FIND_DATA));
  memset(new_dir_name, 0, sizeof(new_dir_name));
  memset(abspath, 0, sizeof(abspath));

  retval = bc_slist_append(retval, "");
  /* XXX - Check to see if dir_name ends in a slash? */

  /* The FindFirstFile call needs a star at the end of the directory path... */
  bc_strlcpy(new_dir_name, dir_name, sizeof(new_dir_name) - 8);
  bc_strlcat(new_dir_name, "*", sizeof(new_dir_name) - 8);

  hSearch = FindFirstFile(new_dir_name, &wfd);
  if (hSearch == INVALID_HANDLE_VALUE) {
    bc_log("load_ids_from_directory", BC_LOG_ERROR, "CAN'T OPEN [%s]\n", new_dir_name);
    return NULL;
  }

  /* While there are more files in the directory... */
  while(GetLastError() != ERROR_NO_MORE_FILES) {
    /* We don't deal with the current and parent directories. */
    if ((strcmp(wfd.cFileName, ".") != 0) &&
	(strcmp(wfd.cFileName, "..") != 0)) {

      file_len = strlen(wfd.cFileName);
      ptr = strstr(wfd.cFileName, ".bc");
      if ((ptr != NULL) && (file_len > 3)) {

	/* Make sure that the ".bc" appears at the very end of the file's
	   name. */
	if (ptr == (unsigned char *)(wfd.cFileName + file_len - 3)) {
          bc_strlcpy(abspath, dir_name, sizeof(abspath) - 8);
          bc_strlcat(abspath, wfd.cFileName, sizeof(abspath) - 8);

          /* Truncate the ".bc" and add the resulting filename to the list
	   * (only if its a regular file...) */
          if (is_regular_file(abspath) == 1) {
            *ptr = '\0';
	    retval = bc_slist_append(retval, strdup(wfd.cFileName));
	  }
	}
      }
    }

    FindNextFile(hSearch, &wfd);
  }

  FindClose(hSearch);

#else
  DIR *dir_handle = NULL;
  struct dirent *dir_entry = NULL;
  char *ptr = NULL;
  unsigned int file_len = 0;

  char abspath[ 256 ];
  char file[ 256 ];


  memset(abspath, 0, sizeof(abspath));
  memset(file, 0, sizeof(file));


  retval = bc_slist_append(retval, "");

  /* If the directory can't be opened, BAIL! */
  if ((dir_handle = opendir(dir_name)) == NULL)
    return NULL;

  /* While there are more files to read from the directory... */
  while((dir_entry = readdir(dir_handle)) != NULL) {

    bc_strlcpy(file, dir_entry->d_name, sizeof(file) - 8);
    file_len = strlen(file);

    ptr = strstr(file, ".bc");
    if ((ptr != NULL) && (file_len > 3)) {

      /* Make sure that the ".bc" appears at the very end of the file's
         name. */
      if (ptr == (file + file_len - 3)) {

        bc_strlcpy(abspath, dir_name, sizeof(abspath) - 8);
        bc_strlcat(abspath, file, sizeof(abspath) - 8);

        /* Truncate the ".bc" and add the resulting filename to the list
	 * (only if its a regular file...) */
        if (is_regular_file(abspath) == 1) {
          *ptr = '\0';
	  retval = bc_slist_append(retval, strdup(file));
        }
      }

    }

  }

  closedir(dir_handle);  dir_handle = NULL;
#endif


  return retval;
}


/* Called when a buddy on the user's buddy list has signed off.  If this
 * buddy has any encryption state information stored locally, it is wiped. */
/* Audit trail:
 *     - Joe Testa: July 19, 2003
 *     - Joe Testa: March 4th, 2004
 */
/*void buddy_signoff_handler(GaimBuddy *b, void *data) {
  struct UM_BUDDY *buddy = NULL;
  char *who = NULL;

  who = b->name;

  printf("[%s] has logged on/off.\n", who);

  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list((unsigned char *)gaim_normalize(NULL, who));

  if (buddy != NULL) {
    unsigned char *pending_message = NULL;
    printf("\tRemoving [%s] from pending list...\n", who);

    remove_buddy_from_pending_list(buddy);

    * If the buddy signed off while there was still a pending message
     * for them, remove it now. *
    pending_message = get_pending_message((unsigned char *)gaim_normalize(NULL,
						who), 1);
    if (pending_message != NULL) {
      free(pending_message);  pending_message = NULL;
    }

    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  }
  bc_mutex_unlock(pending_list_mutex);


  buddy = find_buddy_in_validated_list((unsigned char *)gaim_normalize(NULL, who));
  if (buddy != NULL) {
    printf("\tRemoving [%s] from validated list...\n", who);
    remove_buddy_from_validated_list(buddy);
    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
  }

  return;
}*/



/* One of the integral pieces of encryption support.  Called when the user
 * sends a message to another buddy, but before it is actually sent.  This
 * function intercepts the message and determines what to do with it
 * (queue it and set up a new encrypted channel, or encrypt it and send it
 * on its way. */
/* Security audit trail:
 *    - Joe Testa: January 23rd, 2005
 *    - Joe Testa: March 6th, 2004
 *    - Joe Testa: July 19, 2003
 */
void send_handler(MXFER *mxfer, char *who, char *message) {
  char *screenname = NULL;
  struct UM_BUDDY *buddy = NULL;
  int ret = 0;
  char *pending_message = NULL;

  bc_log("send_handler", BC_LOG_INFO, "send_handler(MXFER *, %s, \"%s\")",
	 who, message);

  bc_mutex_lock(pending_list_mutex);



  /* If the user tried to send a message before, when a connection to the
   * buddy had not been established, then we will discard this message.
   */
  if (get_pending_message(bc_normalize(who), 0) != NULL) {
    /*um_system_error_message(who, "Error: cannot send another message until handshake is completed.");*/
    bc_mutex_unlock(pending_list_mutex);
    bc_log("send_handler", BC_LOG_ERROR, "User attempted to send another "
	   "message while one was already pending.");
    return;
  }


  /* See if the buddy is stored in the list of validated peers.  Validated
   * means that the full Ultramagnetic handshake was successfully performed.
   * If the buddy is not in this list, then he/she is in some other state of
   * the Ultramagnetic handshake (or perhaps isn't even known to us, yet).
   */
  buddy = find_buddy_in_validated_list(bc_normalize(who));
  if (buddy == NULL) {

    /* Get the buddy from the list of peers who are in the intermediate stages
     * of the Ultramagnetic handshake.  If this call returns NULL, then the
     * user is completely unknown to us at this time, and so we must initialize
     * the handshake ourselves.
     */
    buddy = find_buddy_in_pending_list(bc_normalize(who));
    if (buddy == NULL) {

      /* Reaching here means that we must initialize the handshake ourselves.*/

      unsigned int screenname_len = 0;

      bc_log("send_handler", BC_LOG_INFO,
	     "Initializing Ultramagnetic protocol with [%s]...", who);    


      /* Send the peer our public signing and encrypting keys. */
      ret = send_public_keys(mxfer, who);
      if (ret != 1) {
	bc_log("send_handler", BC_LOG_ERROR,
	       "Error while sending public keys to [%s]", who);
	goto send_handler_error;
      }


      screenname_len = strlen(who) + 1;
      screenname = calloc(screenname_len + 16, sizeof(unsigned char));
      buddy = calloc(1, sizeof(struct UM_BUDDY));
      if ((buddy == NULL) || (screenname == NULL)) {
	mxfer->type = MXFER_TYPE_ERROR;
	mxfer->error = bc_string_new("Internal error in send_handler (1): "
				     "Out of memory");
	bc_log("send_handler", BC_LOG_ERROR, mxfer->error->str);
        goto send_handler_error;
      }


      /* Construct the UM_BUDDY struct as much as we can at this point. */
      bc_strlcpy(screenname, bc_normalize(who), screenname_len);
      buddy->screenname = screenname;

      change_state(buddy, STATE_FLAG_SET, SENT_PUBKEYS);
      buddy->state = SENT_PUBKEYS;

      change_state(buddy, STATE_FLAG_ADD, AWAITING_PUBKEYS_1);
      buddy->state |= AWAITING_PUBKEYS_1;
      time(&(buddy->last_touched));
      buddy->next = NULL;

      add_to_pending_list(buddy);


      add_pending_message(strdup(bc_normalize(who)), strdup(message));
      bc_log("send_handler", BC_LOG_INFO,
	     "Added pending message to [%s]: [%s]", who, message);
    } else {
      /* Reaching here means that the local user tried to send another message
       * even though the handshake has not yet completed.
       * Update: This case is already handled above, so control should not
       * reach here. Update: No it isn't.  This case occurs when we try to
       * send a message while during a crypto handshake that the peer
       * initiated. We ought to queue this message, but unfortunately, the
       * sequence numbers get b0rked here and the message won't be sent. FIX!*/
      /*bc_log("send_handler", BC_LOG_ERROR, "FATAL ERROR!@#!");
	goto send_handler_error;*/
      add_pending_message(strdup(bc_normalize(who)), strdup(message));
      bc_log("send_handler", BC_LOG_INFO,
	     "Added pending message to [%s]: [%s]", who, message);
    }


    /* We've just sent a message to a validated buddy... */
  } else if (buddy->state == CONNECTED) {

    bc_log("send_handler", BC_LOG_INFO,
	   "calling send_encrypted_im(%s, \"%s\")", who, message);
    ret = send_encrypted_im(mxfer, buddy, message);
    if (ret != 1) {
      bc_log("send_handler", BC_LOG_ERROR,
	     "Error while sending encrypted IM to [%s]!", who);
      goto send_handler_error;
    }

  } else {
    /* Reaching here means that there is something SERIOUSLY wrong with our
     * state information.  IE:  a buddy who is not connected is in our
     * validated list! */
    bc_log("send_handler", BC_LOG_ERROR, "ERROR IN SEND HANDLER:  [%u]\n", buddy->state);
    exit(1);
  }


  bc_mutex_unlock(pending_list_mutex);
  return;


 send_handler_error:
  pending_message = get_pending_message(bc_normalize(who), 1);
  if (pending_message != NULL) {
    free(pending_message);  pending_message = NULL;
  }

  bc_mutex_unlock(pending_list_mutex);
  free(screenname);  screenname = NULL;
  free(buddy);  buddy = NULL;
  return;
}



/* One of the integral pieces of encryption support.  Called when the user
 * receives a message, but before it is displayed.  This function intercepts
 * the message and does one of three things:  1.) decrypt the message, if
 * the remote user has completed the UM handshake, 2.) continue the
 * handshake protocol, if appropriate, or 3.) discard the message if it was
 * plaintext. */
/* Security audit trail:
 *    - Joe Testa: January 23rd, 2005
 *    - Joe Testa: March 6th, 2004
 *    - Joe Testa: July 19, 2003
 */
void get_handler(MXFER *mxfer, char *who, char *message) {
  char *screenname = NULL;
  struct UM_BUDDY *buddy = NULL;
  int ret = -1;
  char *pending_message = NULL;

  bc_log("get_handler", BC_LOG_DEBUG, "Got IM from [%s]: [%s]\n",
	 who, message);

  bc_mutex_lock(pending_list_mutex);

  /* See if the buddy is stored in the list of validated peers.  Validated
   * means that the full Ultramagnetic handshake was successfully performed.
   * If the buddy is not in this list, then he/she is in some other state of
   * the Ultramagnetic handshake (or perhaps isn't even known to us, yet).
   */
  buddy = find_buddy_in_validated_list(bc_normalize(who));


  if (buddy == NULL) {

    /* Get the buddy from the list of peers who are in the intermediate stages
     * of the Ultramagnetic handshake.  If this call returns NULL, then the
     * user is completely unknown to us at this time. */
    buddy = find_buddy_in_pending_list(bc_normalize(who));
    if (buddy == NULL) {

      /* Reaching here means that another user has sent us a message and that
       * we've never encountered anything from he/she before (or that previous
       * connection attempts were denied).  We must be VERY careful that we
       * do not leak information about ourselves before the peer is validated.
       * ie:  it should be impossible for an untrusted buddy to scan us to
       * determine whether or not we're using Ultramagnetic. */

      unsigned int screenname_len = 0;

      buddy = calloc(1, sizeof(struct UM_BUDDY));
      if (buddy == NULL) {
	mxfer->type = MXFER_TYPE_ERROR;
	mxfer->error = bc_string_new("Internal error in get_handler (1): "
				     "Out of memory");
	bc_log("get_handler", BC_LOG_ERROR, mxfer->error->str);
        goto get_handler_error;
      }
      time(&(buddy->last_touched));

      /* If we don't know anything about this buddy, then the first message
       * according to the Ultramagnetic handshake protocol must be the buddy's
       * public sign key.  If -1 was returned, then an error occured while
       * reading in the key, and this buddy should be DROPPED immediately. */
      ret = get_pubkeys_1(mxfer, buddy, message);
      if (ret == 1) {

        screenname_len = strlen(who) + 1;
        screenname = calloc(screenname_len + 8, sizeof(unsigned char));
        if (screenname == NULL) {
	  change_state(buddy, STATE_FLAG_SET, 0);
          buddy->state = 0;
          wipe_buddy(buddy);  free(buddy);  buddy = NULL;
          goto get_handler_error;
        }

        bc_strlcpy(screenname, bc_normalize(who), screenname_len);
        buddy->screenname = screenname;

	change_state(buddy, STATE_FLAG_SET, AWAITING_PUBKEYS_2);
        buddy->state = AWAITING_PUBKEYS_2;

        add_to_pending_list(buddy);
      } else {
	char message_len[ 32 ];

	memset(message_len, 0, sizeof(message_len));

	/* There was some error in get_pubkeys_1, so we'll treat this 
	 * message as plaintext. */
	change_state(buddy, STATE_FLAG_SET, 0);
        buddy->state = 0;
        wipe_buddy(buddy);  free(buddy);  buddy = NULL;

	mxfer->type = MXFER_TYPE_MESSAGE_CLEAR;
	mxfer->results = bc_slist_append(mxfer->results, strdup(message));
	bc_string_free(&(mxfer->error), 1);

	sprintf(message_len, "%u", (unsigned int)strlen(message));
	mxfer->result_sizes = bc_slist_append(mxfer->result_sizes,
					     strdup(message_len));
      }


    } else if ((buddy->state & AWAITING_PUBKEYS_1) == AWAITING_PUBKEYS_1) {

      /* To get here, we apparently initialized the connection and sent our
       * public keys already.  So, we're waiting for the buddy's public keys...
       */


      /* Attempt to read in the buddy's public sign key.  If -1 is returned,
       * then an error occured while reading in the key, and this buddy should
       * be DROPPED immediately. */
      ret = get_pubkeys_1(mxfer, buddy, message);

      if (ret == 1) {
	change_state(buddy, STATE_FLAG_REMOVE, AWAITING_PUBKEYS_1);
        buddy->state &= ~AWAITING_PUBKEYS_1;  /* Turn this flag off. */

	change_state(buddy, STATE_FLAG_ADD, AWAITING_PUBKEYS_2);
	buddy->state |= AWAITING_PUBKEYS_2;   /* Turn this flag on.  */

        bc_log("get_handler", BC_LOG_INFO, "Received public signing key (%s), "
	       "awaiting encryption key...", bc_normalize(who));
      } else {
        /* ERROR, so BAIL!! */

	bc_log("get_handler", BC_LOG_ERROR, "Bad sig key from [%s]: [%s]",
	       who, message);

	change_state(buddy, STATE_FLAG_SET, 0);
        buddy->state = 0;
        remove_buddy_from_pending_list(buddy);
        wipe_buddy(buddy);  free(buddy);  buddy = NULL;
        goto get_handler_error;
      }


    } else if ((buddy->state & AWAITING_PUBKEYS_2) == AWAITING_PUBKEYS_2) {

      char buddy_sign_key_fingerprint[ 128 ];
      char umid[ 32 ];

      memset(buddy_sign_key_fingerprint, 0,
	     sizeof(buddy_sign_key_fingerprint));
      memset(umid, 0, sizeof(umid));

      /* Read in the buddy's public encryption key. */
      ret = get_pubkeys_2(mxfer, buddy, message);
      if (ret != 1) {

	bc_log("get_handler", BC_LOG_ERROR, "Bad crypto key from [%s]: [%s]",
	       who, message);

	change_state(buddy, STATE_FLAG_SET, 0);
        buddy->state = 0;
        remove_buddy_from_pending_list(buddy);
        wipe_buddy(buddy);  free(buddy);  buddy = NULL;
        goto get_handler_error;
      }

      change_state(buddy, STATE_FLAG_REMOVE, AWAITING_PUBKEYS_2);
      buddy->state &= ~AWAITING_PUBKEYS_2;

      change_state(buddy, STATE_FLAG_ADD, AWAITING_SESSION_KEY);
      buddy->state |= AWAITING_SESSION_KEY;

      bc_log("get_handler", BC_LOG_INFO,
	     "Received public encryption key (%s)...", bc_normalize(who));


      /* Check the buddy's public key with the key database. */
      get_key_fingerprint_from_sexp(buddy->public_sign_key,
			  buddy_sign_key_fingerprint,
			  sizeof(buddy_sign_key_fingerprint) - 8);

      if (key_database_lookup(buddy_sign_key_fingerprint,
			      umid, sizeof(umid)) == -1) {
	BCString *string = bc_string_new(NULL);
	bc_string_append_printf(string, "%u",
				strlen(buddy_sign_key_fingerprint));

	mxfer->type = MXFER_TYPE_MESSAGE_KEYFP;
	mxfer->results = bc_slist_append(mxfer->results,
					strdup(buddy_sign_key_fingerprint));
	mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

	bc_string_free(&string, 0);

	bc_mutex_unlock(pending_list_mutex);

	bc_log("get_handler", BC_LOG_INFO,
	       "Asking user if [%s]'s key fp is good: [%s]", who,
	       buddy_sign_key_fingerprint);

	return;
      } else {

	bc_log("get_handler", BC_LOG_INFO,
	       "According to the key db, [%s]'s key fp is good [%s]", who,
	       buddy_sign_key_fingerprint);

	/* Fingerprint is in database already, so send to the buddy what we
	 * need to send (public keys and/or session keys).  The buddy's
	 * state is already updated when this call returns. */
	if (buddy_fingerprint_is_good(mxfer, buddy->screenname) != 1)
	  goto get_handler_error;
      }


    } else if ((buddy->state & AWAITING_SESSION_KEY) == AWAITING_SESSION_KEY) {

      /* Extract the session key and check that the signature matches it.
       * This function also sets the buddy pointer with the new session key. */
      ret = get_session_key(mxfer, buddy, message);
      if (ret != 1) {
	bc_log("get_handler", BC_LOG_ERROR,
	       "Session key from [%s] was BAD!: [%s]", who, message);

	change_state(buddy, STATE_FLAG_SET, 0);
        buddy->state = 0;
        remove_buddy_from_pending_list(buddy);
        wipe_buddy(buddy);  free(buddy);  buddy = NULL;
        goto get_handler_error;
      }


      change_state(buddy, STATE_FLAG_ADD, SESSION_KEY_VALID);
      buddy->state |= SESSION_KEY_VALID;

      bc_log("get_handler", BC_LOG_INFO,
	     "Received session key (%s)...", bc_normalize(who));

      /* Now that we've received the session key, see if we still need to
       * send ours. */
      if ((buddy->state & SENT_SESSION_KEY) == 0) {
        ret = send_session_key(mxfer, buddy);
        if (ret != 1) {
          bc_log("get_handler", BC_LOG_ERROR,
		 "Error while sending session key to [%s].\n", who);

	  change_state(buddy, STATE_FLAG_SET, 0);
	  buddy->state = 0;

          remove_buddy_from_pending_list(buddy);
          wipe_buddy(buddy);  free(buddy);  buddy = NULL;
          goto get_handler_error;
        } else {

	  change_state(buddy, STATE_FLAG_ADD, SENT_SESSION_KEY);
          buddy->state |= SENT_SESSION_KEY;
          bc_log("get_handler", BC_LOG_INFO,
		 "Sent session key (%s)...", bc_normalize(who));
        }
      }


      /* At this point, we've gotten & sent the session keys, and so, in
       * theory, we should be ready to jump into hyperspace (ie: go to the
       * CONNECTED state).  But instead of just switching right to it, we'll
       * make one last sanity check:  see that the other proper flags are set
       * now as well. */

      if (((buddy->state & SENT_PUBKEYS) == SENT_PUBKEYS) &&
          ((buddy->state & SENT_SESSION_KEY) == SENT_SESSION_KEY) &&
          ((buddy->state & SESSION_KEY_VALID) == SESSION_KEY_VALID)) {
	char *pending_message = NULL;
	  /*((buddy->state & SIGNKEY_FINGERPRINT_VALID) ==
                                                SIGNKEY_FINGERPRINT_VALID) &&
						((buddy->state & ENCRYPTKEY_VALID) == ENCRYPTKEY_VALID) &&*/

        /* Take this buddy out of the pending list... */
        remove_buddy_from_pending_list(buddy);

        /* Ahh, yes... the coveted CONNECTED state! */
	change_state(buddy, STATE_FLAG_SET, CONNECTED);
        buddy->state = CONNECTED;

	/* Zero the timestamp.  More paranoia... */
	buddy->last_touched = 0;

        bc_log("get_handler", BC_LOG_INFO,
	       "Connected! (%s)", bc_normalize(who));

        /* Put the buddy into the "cool" list. */
        add_to_validated_list(buddy);


        /* Now that we've just switched to CONNECTED, see if there's a
	 * message waiting to be sent.  This happens only when the local user
         * initializes the handshake. */
	pending_message = get_pending_message(bc_normalize(who), 1);
        if (pending_message != NULL) {
	  ret = send_encrypted_im(mxfer, buddy, pending_message);
          if (ret != 1) {
	    bc_log("get_handler", BC_LOG_ERROR,
		   "Error while sending pending message (%s) to [%s]",
		   pending_message, who);
	  }
          free(pending_message);  pending_message = NULL;
        }

      }


    } else {

      /* This would be a fatal error, so its best to exit the entire program
       * to prevent anything worse from happening.  Since our code obviously
       * misbehaved to get here, who knows what else it could do?  Rely on
       * weak authentication?  Leak keys?  Scary thought.*/

      bc_log("get_handler", BC_LOG_ERROR, "ERROR IN GET HANDLER!\n");
      if (buddy != NULL)
	bc_log("get_handler", BC_LOG_INFO, "STATE: %u", buddy->state);
      exit(1);
    }


    /* Notice that this isn't in an 'else' block.  This is a good example of
     * proactive and defensive programming. */
  } else if (buddy->state == CONNECTED) {

    /* Decrypt and check the message's signature. */
    get_encrypted_im(mxfer, buddy, message);

    /* If the IM was successfully parsed, then notify the upper layer that
     * this conversation is sure to be immune from replay attacks (if we
     * didn't already). */
    if ((mxfer->type != MXFER_TYPE_ERROR) && (buddy->notify_noreplay == 0)) {
      change_state(buddy, STATE_FLAG_SET, REPLAY_IMMUNE);
      buddy->notify_noreplay = 1;
    }

  } else {

      /* This would be a fatal error, so its best to exit the entire program
       * to prevent anything worse from happening.  Since our code obviously
       * misbehaved to get here, who knows what else it could do?  Rely on
       * weak authentication?  Leak keys?  Scary thought. */

      bc_log("get_handler", BC_LOG_ERROR,
	     "FATAL ERROR IN GET HANDLER [%s][%s]", who, message);

      if (buddy != NULL)
	bc_log("get_handler", BC_LOG_ERROR, "STATE: %u", buddy->state);
      exit(1);
  }

  bc_mutex_unlock(pending_list_mutex);
  return;

 get_handler_error:
  pending_message = get_pending_message(bc_normalize(who), 1);
  if (pending_message != NULL) {
    free(pending_message);  pending_message = NULL;
  }

  bc_mutex_unlock(pending_list_mutex);

  free(screenname);  screenname = NULL;
  bc_log("get_handler", BC_LOG_ERROR, "GET_HANDLER_ERROR!\n");
  return;
}



/* Sends the public signing key along with the public encryption key, signed by
 * the signing key. */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19th, 2003
 */
int send_public_keys(MXFER *mxfer, char *who) {

  char *message = NULL;
  char *base64_buffer = NULL;
  const char *key_component_buf = NULL;
  size_t key_component_len = 0;
  unsigned int i = 0;
  char *sign_buffer = NULL;
  size_t sign_buffer_len = 0;
  int ret = 0;
  BCString *string = NULL;

  gcry_sexp_t public_encrypt_key;
  gcry_sexp_t public_sign_key;
  gcry_sexp_t key_component;
  gcry_sexp_t to_sign_sexp;
  gcry_sexp_t signed_encrypt_key_sexp;
  gcry_mpi_t encrypt_key_mpi;

  memset(&public_encrypt_key, 0, sizeof(gcry_sexp_t));
  memset(&public_sign_key, 0, sizeof(gcry_sexp_t));
  memset(&key_component, 0, sizeof(gcry_sexp_t));
  memset(&to_sign_sexp, 0, sizeof(gcry_sexp_t));
  memset(&signed_encrypt_key_sexp, 0, sizeof(gcry_sexp_t));
  memset(&encrypt_key_mpi, 0, sizeof(gcry_mpi_t));


  /*printf("\tsend_public_keys\n");
    printf("\twho:          [%s]\n", who);*/
  /*printf("\tusername:     [%s]\n\n", gc->account->username);*/


  message = calloc(MESSAGE_LEN + 8, sizeof(char));
  base64_buffer = calloc(BASE64_BUFFER_LEN + 8, sizeof(char));
  sign_buffer = calloc(SIGN_BUFFER_LEN + 8, sizeof(char));


  if ((message == NULL) || (base64_buffer == NULL) || (sign_buffer == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error in send_public_keys (1): "
				 "Out of memory");
    bc_log("send_public_keys_error", BC_LOG_ERROR, mxfer->error->str);
    goto send_public_keys_error;
  }


  /* We start off by sending the version of the Ultramagnetic protocol we're
   * using.  Both buddies must be using the same exact version in order to
   * communicate.  Backwards compatibility is not allowed due to increased
   * risks of security vulnerabilities. */
  bc_strlcpy(message, UM_PROTOCOL_VERSION, MESSAGE_LEN - 8);
  bc_strlcat(message, "|", MESSAGE_LEN - 8);


  /* Grab a reference to the public key section of our signing and encrypting
   * keypairs. */
  public_sign_key = gcry_sexp_find_token(*my_signing_keypair, "public-key", 0);
  public_encrypt_key = gcry_sexp_find_token(*my_encrypting_keypair,
                                            "public-key", 0);

  /* Loop through all the fields in our public signing key and encode them to
   * base64 and append them to our message. */
  for (i = 0; sign_key_fieldz[ i ] != NULL; i++) {

    encode_key_component(public_sign_key, sign_key_fieldz[ i ],
                         base64_buffer, BASE64_BUFFER_LEN - 8);

    bc_strlcat(message, base64_buffer, MESSAGE_LEN - 8);
    bc_strlcat(message, "|", MESSAGE_LEN - 8);

  }

  /* We're done with our public sign key. */
  UM_SEXP_RELEASE(public_sign_key);


  string = bc_string_new(NULL);
  bc_string_append_printf(string, "%u", strlen(message));

  /* NOTE:  When implementing binary outputs, can't use bc_string_new! */
  mxfer->type = MXFER_TYPE_MESSAGE_ACTIONABLE;
  mxfer->results = bc_slist_append(mxfer->results, strdup(message));
  mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

  bc_string_free(&string, 0);


  /* Clear the message buffer now that we've sent our public signing key. */
  memset(message, 0, MESSAGE_LEN - 8);


  /* Base64 encode all the public encryption key fields, then append them to
   * the message buffer.  Also, we must sign the public encryption key, so the
   * public key encryption fields are put into the sign_buffer. */
  for (i = 0; encrypt_key_fieldz[ i ] != NULL; i++) {
    key_component = gcry_sexp_find_token(public_encrypt_key,
                                         encrypt_key_fieldz[ i ], 0);
    key_component_buf = gcry_sexp_nth_data(key_component, 1,
                                           &key_component_len);

    /* Make sure that there's enough space in the sign_buffer. */
    if (sign_buffer_len + key_component_len < SIGN_BUFFER_LEN - 8) {
      memcpy(sign_buffer + sign_buffer_len,
             key_component_buf, key_component_len);
      sign_buffer_len += key_component_len;
    } else {
      /* ERROR WHILE SIGNING! */
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error in send_public_keys (2):"
				   "signing error");
      bc_log("send_public_keys", BC_LOG_ERROR, mxfer->error->str);
      goto send_public_keys_error;
    }

    base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN,
		  (unsigned char *)key_component_buf, key_component_len);

    bc_strlcat(message, base64_buffer, MESSAGE_LEN);
    bc_strlcat(message, "|", MESSAGE_LEN);
    UM_SEXP_RELEASE(key_component);
  }

  /* Now we're done with the public encryption key. */
  UM_SEXP_RELEASE(public_encrypt_key);

  /* Make the sign_buffer into an MPI so we can sign it... */
  gcry_mpi_scan(&encrypt_key_mpi, GCRYMPI_FMT_STD,
                (unsigned char *)sign_buffer, sign_buffer_len, NULL);

  /* Now put the MPI (sign_buffer) into the appropriate S-expression... */
  gcry_sexp_build(&to_sign_sexp, &sign_buffer_len,
                  "(data\n (flags raw)\n (value %m))\n", encrypt_key_mpi);

  /* Now sign it! */
  ret = gcry_pk_sign(&signed_encrypt_key_sexp, to_sign_sexp,
                     *my_signing_keypair);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error in send_public_keys (3): ");
    mxfer->error = bc_string_append(mxfer->error, (char *)gcry_strerror(ret));
    bc_log("send_public_keys", BC_LOG_ERROR, mxfer->error->str);
    goto send_public_keys_error;
  }


  /* Release the MPI and S-expression we just made. */
  UM_MPI_RELEASE(encrypt_key_mpi);
  UM_SEXP_RELEASE(to_sign_sexp);


  /* Extract and encode the signed hash of our message, then append to the
   * message. */

  /* First part... */
  encode_key_component(signed_encrypt_key_sexp, "r",
                       base64_buffer, BASE64_BUFFER_LEN - 8);

  bc_strlcat(message, base64_buffer, MESSAGE_LEN - 8);
  bc_strlcat(message, "|", MESSAGE_LEN - 8);

  /* Second part... */
  encode_key_component(signed_encrypt_key_sexp, "s",
                       base64_buffer, BASE64_BUFFER_LEN - 8);

  bc_strlcat(message, base64_buffer, BASE64_BUFFER_LEN - 8);


  /* Now we're done with the signed result... */
  UM_SEXP_RELEASE(signed_encrypt_key_sexp);


  /* PARANOIA:  add a random timeout here to prevent anyone from deducing
   * timing information regarding our signing operation. */

  /* Send the signed public encryption key. */
  /*serv_send_im(gc, who, message, 0);*/

  string = bc_string_new(NULL);
  bc_string_append_printf(string, "%u", strlen(message));

  /* NOTE:  When implementing binary outputs, can't use bc_string_new! */
  mxfer->type = MXFER_TYPE_MESSAGE_ACTIONABLE;
  mxfer->results = bc_slist_append(mxfer->results, strdup(message));
  mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

  bc_string_free(&string, 0);


  free(sign_buffer);    sign_buffer = NULL;
  free(base64_buffer);  base64_buffer = NULL;
  free(message);  message = NULL;

  /* Add checks here to make sure that everything is NULL again! */
  /*printf("\n\n\nREAL KEYS:\n\n");
  gcry_sexp_dump(*my_signing_keypair);
  gcry_sexp_dump(*my_encrypting_keypair);*/

  return 1;


 send_public_keys_error:
    free(message);        message = NULL;
    free(base64_buffer);  base64_buffer = NULL;
    free(sign_buffer);    sign_buffer = NULL;
    UM_SEXP_RELEASE(public_encrypt_key);
    UM_SEXP_RELEASE(public_sign_key);
    UM_SEXP_RELEASE(key_component);
    UM_SEXP_RELEASE(to_sign_sexp);
    UM_SEXP_RELEASE(signed_encrypt_key_sexp);
    UM_MPI_RELEASE(encrypt_key_mpi);
    return -1;
}


/* This function looks through the validated list and returns a reference to
 * the UM_BUDDY that matches the screen name argument, or NULL if nothing was
 * found. */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19, 2003
 */
struct UM_BUDDY *find_buddy_in_validated_list(char *who) {
  struct UM_BUDDY *ret = NULL;
  struct UM_BUDDY *current = NULL;

  current = validated_buddies_list;
  while ((current != NULL) && (ret == NULL)) {
    if (strcmp(who, current->screenname) == 0)
      ret = current;

    current = current->next;
  }

  return ret;
}



/* Removes a buddy from the list of validated peers.  Note:  this function does
 * NOT call wipe_buddy() on the buddy that was removed; the caller is
 * responsible for doing this (if appropriate). */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19, 2003
 */
void remove_buddy_from_validated_list(struct UM_BUDDY *buddy) {
  struct UM_BUDDY *previous = NULL;
  struct UM_BUDDY *current = NULL;
  unsigned int deleted = 0;
  char *who = NULL;

  who = buddy->screenname;

  if (validated_buddies_list == NULL)
    return;

  /* If the buddy to remove is at the head of the list, then make the head's
   * 'next' pointer the new head. If the validated list contained only one
   * entry, then validated_buddies_list will equal NULL, which is correct. */
  if (strcmp(validated_buddies_list->screenname, who) == 0) {
    validated_buddies_list = validated_buddies_list->next;
    return;
  }

  previous = validated_buddies_list;
  current = validated_buddies_list->next;
  while((current != NULL) && (deleted == 0)) {

    if (strcmp(current->screenname, who) == 0) {
      previous->next = current->next;
      deleted = 1;
    } else {
      previous = current;
      current = current->next;
    }

  }

  return;
}



/* This function completely scrubs-down a UM_BUDDY structure. */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19, 2003
 */
void wipe_buddy(struct UM_BUDDY *um_buddy) {

  if (um_buddy == NULL)
    return;

  if (um_buddy->screenname != NULL) {
    char *pending_message = NULL;

    bc_log("wipe_buddy", BC_LOG_ERROR,
	   "Wiping buddy: [%s]", um_buddy->screenname);

    pending_message =
      get_pending_message(bc_normalize(um_buddy->screenname), 1);
    if (pending_message != NULL) {
      free(pending_message);  pending_message = NULL;
    }

  } else
    bc_log("wipe_buddy", BC_LOG_ERROR, "Wiping buddy...");


  if (um_buddy->session_key != NULL) {
    memset(um_buddy->session_key, 0, SESSION_KEY_BYTE_SIZE);
    gcry_free(um_buddy->session_key);  um_buddy->session_key = NULL;
  }

  if (um_buddy->session_iv != NULL) {
    memset(um_buddy->session_iv, 0, SESSION_KEY_BYTE_SIZE);
    gcry_free(um_buddy->session_iv);  um_buddy->session_iv = NULL;
  }

  if (um_buddy->session_hmac != NULL) {
    memset(um_buddy->session_hmac, 0, SESSION_KEY_BYTE_SIZE);
    gcry_free(um_buddy->session_hmac);  um_buddy->session_hmac = NULL;
  }

  if (um_buddy->cipher != NULL) {
    gcry_cipher_close(*(um_buddy->cipher));
    gcry_free(um_buddy->cipher);  um_buddy->cipher = NULL;
  }

  if (um_buddy->md != NULL) {
    gcry_md_close(*(um_buddy->md));
    gcry_free(um_buddy->md);  um_buddy->md = NULL;
  }

  if (um_buddy->public_encrypt_key != NULL) {
    UM_SEXP_RELEASE(*(um_buddy->public_encrypt_key));
    free(um_buddy->public_encrypt_key);  um_buddy->public_encrypt_key = NULL;
  }

  if (um_buddy->public_sign_key != NULL) {
    UM_SEXP_RELEASE(*(um_buddy->public_sign_key));
    free(um_buddy->public_sign_key);  um_buddy->public_sign_key = NULL;
  }

  if (um_buddy->screenname != NULL) {
    memset(um_buddy->screenname, 0, strlen(um_buddy->screenname));
    free(um_buddy->screenname);  um_buddy->screenname = NULL;
  }

  if (um_buddy->um_id != NULL) {
    memset(um_buddy->um_id, 0, strlen(um_buddy->um_id));
    free(um_buddy->um_id);  um_buddy->um_id = NULL;
  }

  if (um_buddy->my_sequence_number != NULL) {
    UM_MPI_RELEASE(*(um_buddy->my_sequence_number));
    gcry_free(um_buddy->my_sequence_number);
    um_buddy->my_sequence_number = NULL;
  }

  if (um_buddy->his_sequence_number != NULL) {
    UM_MPI_RELEASE(*(um_buddy->his_sequence_number));
    gcry_free(um_buddy->his_sequence_number);
    um_buddy->his_sequence_number = NULL;
  }

  change_state(um_buddy, STATE_FLAG_SET, 0);
  um_buddy->state = 0;
  um_buddy->next = NULL;

  memset(um_buddy, 0,   sizeof(struct UM_BUDDY));
  memset(um_buddy, 255, sizeof(struct UM_BUDDY));
  memset(um_buddy, 0,   sizeof(struct UM_BUDDY));
}



/* Adds a buddy to the pending list. */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19, 2003
 */
void add_to_pending_list(struct UM_BUDDY *new_buddy) {
  struct UM_BUDDY *current = NULL;

  if (new_buddy->screenname == NULL) {
    bc_log("add_to_pending_list", BC_LOG_ERROR, "Attempt to add a buddy with no screen name!");
    exit(1);
  }



  new_buddy->next = NULL;

  if(pending_buddies_list == NULL)
    pending_buddies_list = new_buddy;
  else {
    current = pending_buddies_list;
    while(current->next != NULL)
      current = current->next;

    current->next = new_buddy;
  }

  /*print_pending_list();*/

  return;
}



/* Returns the UM_BUDDY structure that matches the screen name argument, or
 * NULL if none is found. */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 14th, 2005
 *    - Joe Testa: July 19, 2003
 */
struct UM_BUDDY *find_buddy_in_pending_list(char *who) {
  struct UM_BUDDY *ret = NULL;
  struct UM_BUDDY *current = NULL;

  current = pending_buddies_list;
  while ((current != NULL) && (ret == NULL)) {
    /*if (current->screenname == NULL) {
      printf("screenname in pending list is NULL!\n");
      exit(1);
      }*/
    if (strcmp(current->screenname, who) == 0)
      ret = current;

    current = current->next;
  }

  return ret;
}



/* Retrieves the buddy's public signing key from an incoming message.  The
 * UM_BUDDY argument's public_sign_key field is set. */
/* Security audit trail:
 *       - Joe Testa: January 15th, 2005
 *       - Joe Testa: March 6th, 2004
 *       - Joe Testa: July 19th, 2003
 */
int get_pubkeys_1(MXFER *mxfer, struct UM_BUDDY *buddy, char *message) {

  char *pipe_ptr = NULL;
  char *segment = NULL;
  unsigned int i = 0;
  int decode_len = 0;
  char *base64_buffer = NULL;
  gcry_sexp_t *buddy_signing_keypair = NULL;
  int ret = 0;
  size_t err = 0;

  gcry_mpi_t mpis[ 8 ];


  memset(mpis, 0, sizeof(mpis));


  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));


  base64_buffer = calloc(BASE64_BUFFER_LEN + 8, sizeof(unsigned char));
  if (base64_buffer == NULL) {
    goto get_pubkeys_1_error;
  }

  pipe_ptr = strchr(message, '|');
  if (pipe_ptr == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_1", BC_LOG_ERROR,
	   "Malformed input (1): pipe not found in: [%s]", message);
    goto get_pubkeys_1_error;
  }

  /* Overwrite the first pipe with a NULL; this breaks off a segment to work
   * on. */
  segment = message;
  message = pipe_ptr + 1;
  *pipe_ptr = '\0';


  if (atoi(segment) != atoi(UM_PROTOCOL_VERSION)) {
    /* PROTOCOL MISMATCH! */
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Protocol mismatch.");
    bc_log("get_pubkeys_1", BC_LOG_ERROR,
	   "PROTOCOL MISMATCH! (2): wanted %d, got %d.",
	   atoi(UM_PROTOCOL_VERSION), atoi(segment));
    goto get_pubkeys_1_error;
  }

  /* Extract and decode the four fields in the buddy's public sign key. */
  /* Note: do not increase the 4 below without increasing mpis[]. */
  for (i = 0; i < 4; i++) {

    pipe_ptr = strchr(message, '|');
    if (pipe_ptr == NULL) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_1", BC_LOG_ERROR,
	     "Malformed input (3): Pipe not found in: [%s]", message);
      goto get_pubkeys_1_error;
    }

    segment = message;
    message = pipe_ptr + 1;
    *pipe_ptr = '\0';


    decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)segment);
    if (decode_len < 1) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_1", BC_LOG_ERROR, "Malformed input. (3.5)");
      goto get_pubkeys_1_error;
    }
    ret = gcry_mpi_scan(&mpis[ i ], GCRYMPI_FMT_STD,
                        (unsigned char *)base64_buffer, decode_len, NULL);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_1", BC_LOG_ERROR,
	     "Internal error while scanning MPI. (4)");
      goto get_pubkeys_1_error;
    }
  }

  free(base64_buffer);  base64_buffer = NULL;

  buddy_signing_keypair = calloc(1, sizeof(gcry_sexp_t));
  if (buddy_signing_keypair == NULL)
    goto get_pubkeys_1_error;

  /* Reconstruct the buddy's public sign key. */
  ret = gcry_sexp_build(buddy_signing_keypair, &err,
                  "(key-data(public-key(dsa(p %m)(q %m)(g %m)(y %m))))",
                  mpis[ 0 ], mpis[ 1 ], mpis[ 2 ], mpis[ 3 ] );
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_1", BC_LOG_ERROR, "Error while building s-exp. (5)");
    goto get_pubkeys_1_error;
  }

  /* Now that the key was built, the MPIs can be released. */
  for (i = 0; i < 4; i++) {
    UM_MPI_RELEASE(mpis[ i ]);
  }


  buddy->public_sign_key = buddy_signing_keypair;

  /* There is no actionable return value since all we had to do was parse the
   * signing key. */
  mxfer->type = MXFER_TYPE_MESSAGE_NONACTIONABLE;

  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));

  return 1;

 get_pubkeys_1_error:
  bc_log("get_pubkeys_1", BC_LOG_ERROR, "ERROR IN get_pubkeys_1()!\n");
  free(base64_buffer);  base64_buffer = NULL;
  free(buddy_signing_keypair);  buddy_signing_keypair = NULL;
  for (i = 0; i < 4; i++) {
    UM_MPI_RELEASE(mpis[ i ]);
  }

  return -1;
}



/* Retrieves the buddy's public encryption key from an incoming message and
 * verifies the signature.  The UM_BUDDY argument's public_encrypt_key field
 * is set. */
/* Security audit trail:
 *    - Joe Testa: January 15th, 2005
 *    - Joe Testa: March 6th, 2004
 *    - Joe Testa: July 20th, 2003
 */
#define VERIFY_BUFFER_LEN 4088  /* 4088 = 4096 - 8 */
int get_pubkeys_2(MXFER *mxfer, struct UM_BUDDY *buddy, char *message) {

  char *pipe_ptr = NULL;
  char *segment = NULL;
  unsigned int i = 0;
  int decode_len = 0;
  char *base64_buffer = NULL;
  gcry_sexp_t *buddy_encrypting_keypair = NULL;
  char *verify_buffer = NULL;
  size_t verify_buffer_len = 0;
  int ret = 0;
  size_t err = 0;

  gcry_mpi_t mpis[ 8 ];
  gcry_mpi_t r_mpi;
  gcry_mpi_t s_mpi;
  gcry_mpi_t data_mpi;
  gcry_sexp_t signed_result;
  gcry_sexp_t signed_data;


  memset(mpis, 0, sizeof(mpis));
  memset(&r_mpi, 0, sizeof(gcry_mpi_t));
  memset(&s_mpi, 0, sizeof(gcry_mpi_t));
  memset(&data_mpi, 0, sizeof(gcry_mpi_t));
  memset(&signed_result, 0, sizeof(gcry_sexp_t));
  memset(&signed_data, 0, sizeof(gcry_sexp_t));


  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));


  base64_buffer = calloc(BASE64_BUFFER_LEN + 8, sizeof(unsigned char));
  verify_buffer = calloc(VERIFY_BUFFER_LEN + 8, sizeof(unsigned char));
  if ((verify_buffer == NULL) || (base64_buffer == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Out of memory. (1)");
    goto get_pubkeys_2_error;
  }


  /* Extract, decode, and scan the components to the buddy's public encryption
   * key. */
  for (i = 0; i < 3; i++) {

    pipe_ptr = strchr(message, '|');
    if (pipe_ptr == NULL) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_2", BC_LOG_ERROR,
	     "Malformed input (2): no pipe found in: [%s]", message);
      goto get_pubkeys_2_error;
    }

    segment = message;
    message = pipe_ptr + 1;
    *pipe_ptr = '\0';


    decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)segment);
    if (decode_len < 1) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (3).");
      goto get_pubkeys_2_error;
    }


    /* We need to verify that the encryption key matches the signature, so
     * we save the raw components into the verify_buffer. */
    if ((verify_buffer_len + decode_len) < (VERIFY_BUFFER_LEN - 8)) {
      memcpy(verify_buffer + verify_buffer_len, base64_buffer, decode_len);
      verify_buffer_len += decode_len;
    } else {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (3).");
      goto get_pubkeys_2_error;
    }

    ret = gcry_mpi_scan(&mpis[ i ], GCRYMPI_FMT_STD,
                        (unsigned char *)base64_buffer, decode_len, NULL);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Malformed input.");
      bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (4).");
      goto get_pubkeys_2_error;
    }

  }


  pipe_ptr = strchr(message, '|');
  if (pipe_ptr == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (5).");
    goto get_pubkeys_2_error;
  }

  segment = message;
  message = pipe_ptr + 1;
  *pipe_ptr = '\0';


  /* Scan the first part of the signed hash. */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)segment);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (5.5).");
    goto get_pubkeys_2_error;
  }

  ret = gcry_mpi_scan(&r_mpi, GCRYMPI_FMT_STD, (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (6).");
    goto get_pubkeys_2_error;
  }

  /* Scan the second part of the signed hash. */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)message);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (6.5).");
    goto get_pubkeys_2_error;
  }

  gcry_mpi_scan(&s_mpi, GCRYMPI_FMT_STD, (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (7).");
    goto get_pubkeys_2_error;
  }

  free(base64_buffer);  base64_buffer = NULL;

  /* Build the S-expression that represents the signed hash of the encryption
   * key. */
  ret = gcry_sexp_build(&signed_result, &err,
                  "(sig-val(dsa(r %m)(s %m)))", r_mpi, s_mpi );
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (8).");
    goto get_pubkeys_2_error;
  }


  UM_MPI_RELEASE(r_mpi);
  UM_MPI_RELEASE(s_mpi);


  /* Make the verify_buffer into an MPI. */
  ret = gcry_mpi_scan(&data_mpi, GCRYMPI_FMT_STD,
                      (unsigned char *)verify_buffer, verify_buffer_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (9).");
    goto get_pubkeys_2_error;
  }

  free(verify_buffer);  verify_buffer = NULL;
  verify_buffer_len = 0;


  /* Build the verify_buffer MPI into an S-expression. */
  ret = gcry_sexp_build(&signed_data, &verify_buffer_len,
                  "(data\n (flags raw)\n (value %m))\n", data_mpi);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input (10).");
    goto get_pubkeys_2_error;
  }
  UM_MPI_RELEASE(data_mpi);  


  if (buddy->public_sign_key == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR,
	   "BUDDY'S PUBLIC SIGN KEY IS NULL! (11)");
    goto get_pubkeys_2_error;
  }

  ret = gcry_pk_verify(signed_result, signed_data, *(buddy->public_sign_key));
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.  "
				 "Sig on encryption key is invalid.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR,
	   "Signature on encryption key is INVALID! (12)");
    goto get_pubkeys_2_error;
  } else {
    change_state(buddy, STATE_FLAG_ADD, ENCRYPTKEY_VALID);
    buddy->state |= ENCRYPTKEY_VALID;
  }

  UM_SEXP_RELEASE(signed_result);
  UM_SEXP_RELEASE(signed_data);

  buddy_encrypting_keypair = calloc(1, sizeof(gcry_sexp_t));
  if (buddy_encrypting_keypair == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Out of memory. (13)");
    goto get_pubkeys_2_error;
  }

  ret = gcry_sexp_build(buddy_encrypting_keypair, &err,
                  "(key-data(public-key(elg(p %m)(g %m)(y %m))))",
                  mpis[ 0 ], mpis[ 1 ], mpis[ 2 ] );
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_pubkeys_2", BC_LOG_ERROR, "Malformed input. (14)");
    goto get_pubkeys_2_error;
  }

  for (i = 0; i < 3; i++) {
    UM_MPI_RELEASE(mpis[ i ]);
  }


  buddy->public_encrypt_key = buddy_encrypting_keypair;

  /* There is no actionable return value since all we had to do was parse the
   * encryption key. */
  mxfer->type = MXFER_TYPE_MESSAGE_NONACTIONABLE;

  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));

  return 1;

 get_pubkeys_2_error:
  bc_log("get_pubkeys_2", BC_LOG_ERROR, "ERROR IN get_pubkeys_2()!");
  for (i = 0; i < 3; i++) {
     UM_MPI_RELEASE(mpis[ i ]);
  }
  UM_MPI_RELEASE(r_mpi);
  UM_MPI_RELEASE(s_mpi);
  UM_MPI_RELEASE(data_mpi);
  UM_SEXP_RELEASE(signed_result);
  UM_SEXP_RELEASE(signed_data);

  free(base64_buffer);  base64_buffer = NULL;
  free(verify_buffer);  verify_buffer = NULL;
  free(buddy_encrypting_keypair);  buddy_encrypting_keypair = NULL;
  return -1;
}



/* This function generates a random session key and sends it to the buddy
 * argument. */
/* Security audit trail:
 *    - Joe Testa: January 15th, 2005
 *    - Joe Testa: March 6th, 2004
 *    - Joe Testa: August 13th, 2003
 */
#define BUFFER_TO_ENCRYPT_LEN  1016
#define ENCRYPTED_BUFFER_LEN   1016
int send_session_key(MXFER *mxfer, struct UM_BUDDY *buddy) {

  unsigned int session_junk_len = 0;
  unsigned char *session_junk = NULL;
  unsigned int session_key_len = 0;
  unsigned char *session_key = NULL;
  unsigned int session_iv_len = 0;
  unsigned char *session_iv = NULL;
  unsigned int session_hmac_len = 0;
  unsigned char *session_hmac = NULL;
  unsigned int session_sequence_num_len = 0;
  unsigned char *session_sequence_num = NULL;
  char *buffer_to_encrypt = NULL;
  size_t err = 0;
  char *base64_buffer = NULL;
  unsigned int buffer_to_encrypt_len = 0;
  int ret = 0;
  char *encrypted_buffer = NULL;
  gcry_cipher_hd_t *cipher = NULL;
  gcry_md_hd_t *md = NULL;
  BCString *string = NULL;

  gcry_sexp_t key_component;
  gcry_sexp_t sexp_to_sign;
  gcry_sexp_t signed_info;
  gcry_sexp_t sexp_to_encrypt;
  gcry_sexp_t sexp_encrypted;
  gcry_mpi_t data_mpi;
  gcry_mpi_t to_encrypt_mpi;


  memset(&key_component, 0, sizeof(gcry_sexp_t));
  memset(&sexp_to_sign, 0, sizeof(gcry_sexp_t));
  memset(&signed_info, 0, sizeof(gcry_sexp_t));
  memset(&sexp_to_encrypt, 0, sizeof(gcry_sexp_t));
  memset(&sexp_encrypted, 0, sizeof(gcry_sexp_t));
  memset(&data_mpi, 0, sizeof(gcry_mpi_t));
  memset(&to_encrypt_mpi, 0, sizeof(gcry_mpi_t));


  /* Generate a random 256-bit session key, 256-bit initialization vector,
   * and a 256-bit HMAC key (will be used in the future for file transfers). */
  session_junk_len = SESSION_JUNK_SIZE;
  session_junk = (unsigned char *)gcry_malloc_secure(SESSION_JUNK_SIZE);
                 /*(unsigned char *)gcry_random_bytes_secure(
                                 SESSION_JUNK_SIZE,
                                 GCRY_VERY_STRONG_RANDOM);*/
  session_key_len = SESSION_KEY_BYTE_SIZE;
  session_key = (unsigned char *)gcry_malloc_secure(session_key_len + 8);

  session_iv_len = SESSION_KEY_BYTE_SIZE;
  session_iv = (unsigned char *)gcry_malloc_secure(session_iv_len + 8);

  session_hmac_len = SESSION_KEY_BYTE_SIZE;
  session_hmac = (unsigned char *)gcry_malloc_secure(session_hmac_len + 8);

  session_sequence_num_len = SESSION_KEY_BYTE_SIZE;
  session_sequence_num = (unsigned char *)gcry_malloc_secure(session_sequence_num_len + 8);

  if ((session_junk == NULL) || (session_key == NULL) ||
      (session_iv == NULL) || (session_hmac == NULL) ||
      (session_sequence_num == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR,
	   "Out of memory. (1)");
    goto send_session_key_error;
  }

  /*hex_dump("session_junk", session_junk, session_junk_len);*/

  /* The first part of the random session junk will be the key, the second
   * part will be the IV, the third part will be the HMAC, and the fourth
   * part will be the nonce/sequence number. */
  /*memcpy(session_key, session_junk, SESSION_KEY_BYTE_SIZE);
  memcpy(session_iv, session_junk + SESSION_KEY_BYTE_SIZE,
	 SESSION_KEY_BYTE_SIZE);
  memcpy(session_hmac, session_junk + (SESSION_KEY_BYTE_SIZE * 2),
	 SESSION_KEY_BYTE_SIZE);
  memcpy(session_sequence_num, session_junk + (SESSION_KEY_BYTE_SIZE * 3),
  SESSION_KEY_BYTE_SIZE);*/

  /* The message key and HMAC key should both be very strongly random, while
   * the IV and sequence number can pass being only strongly random.  This
   * saves some precious entropy. */
  gcry_randomize(session_key, session_key_len, GCRY_VERY_STRONG_RANDOM);
  gcry_randomize(session_iv, session_iv_len, GCRY_STRONG_RANDOM);
  gcry_randomize(session_hmac, session_hmac_len, GCRY_VERY_STRONG_RANDOM);
  gcry_randomize(session_sequence_num, session_sequence_num_len, GCRY_STRONG_RANDOM);

  /* Copy the message key, iv, HMAC key, and sequence number, respectively,
   * into the 'session_junk' buffer for transmission. */
  memcpy(session_junk, session_key, session_key_len);
  memcpy(session_junk + (SESSION_KEY_BYTE_SIZE * 1),
	 session_iv, session_iv_len);
  memcpy(session_junk + (SESSION_KEY_BYTE_SIZE * 2),
	 session_hmac, session_hmac_len);
  memcpy(session_junk + (SESSION_KEY_BYTE_SIZE * 3),
	 session_sequence_num, session_sequence_num_len);

  /*hex_dump("session_key", session_key, session_key_len);
  hex_dump("session_iv", session_iv, session_iv_len);
  hex_dump("\t\tX session_hmac", session_hmac, session_hmac_len);*/
  hex_dump("send_session_sequence", (char *)session_sequence_num,
	   session_sequence_num_len);

  buddy->his_sequence_number = gcry_calloc(1, sizeof(gcry_mpi_t));
  *(buddy->his_sequence_number) = gcry_mpi_snew(sizeof(gcry_mpi_t));
  ret = gcry_mpi_scan(buddy->his_sequence_number, GCRYMPI_FMT_USG,
		      session_sequence_num, session_sequence_num_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("send_session_key", BC_LOG_ERROR, "Malformed input. (2)");
    goto send_session_key_error;
  }

  /* Create an MPI out of the session junk. */
  data_mpi = gcry_mpi_snew(session_junk_len * 8);
  ret = gcry_mpi_scan(&data_mpi, GCRYMPI_FMT_USG,
		      session_junk, session_junk_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (3)");
    goto send_session_key_error;
  }

  /* Construct an S-expression to sign using our session key MPI. */
  ret = gcry_sexp_build(&sexp_to_sign, &err,
                  "(data\n (flags raw)\n (value %m))\n", data_mpi);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (4)");
    goto send_session_key_error;
  }

  /* Sign the session key MPI with the local buddy's keypair. */
  ret = gcry_pk_sign(&signed_info, sexp_to_sign, *my_signing_keypair);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR,
	   "Internal error. %s (5)", gcry_strerror(ret));
    goto send_session_key_error;
  }

  /* Sanity check:  make sure that signature is valid... */
  ret = gcry_pk_verify(signed_info, sexp_to_sign, *my_signing_keypair);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR,
	   "Internal error. %s (6)", gcry_strerror(ret));
    goto send_session_key_error;
  }


  UM_MPI_RELEASE(data_mpi);
  UM_SEXP_RELEASE(sexp_to_sign);


  buffer_to_encrypt = (char *)gcry_malloc_secure(BUFFER_TO_ENCRYPT_LEN + 8);
  base64_buffer = (char *)gcry_malloc_secure(BASE64_BUFFER_LEN + 8);
  if ((buffer_to_encrypt == NULL) || (base64_buffer == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (7)");
    goto send_session_key_error;
  } else {
    memset(buffer_to_encrypt, 0, BUFFER_TO_ENCRYPT_LEN);
    memset(base64_buffer, 0, BASE64_BUFFER_LEN);
  }


  /* Encode the session key, IV, and HMAC... */
  base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8,
                (unsigned char *)session_junk, session_junk_len);

  /* Copy the encoded session key into a buffer that we'll encrypt later. */
  bc_strlcpy(buffer_to_encrypt, base64_buffer, BUFFER_TO_ENCRYPT_LEN - 8);
  buffer_to_encrypt_len = strlen(base64_buffer);


  bc_strlcat(buffer_to_encrypt, "|", BUFFER_TO_ENCRYPT_LEN - 8);
  buffer_to_encrypt_len++;


  /* Put the first part of the signed hash into the buffer_to_encrypt. */
  ret = encode_key_component(signed_info, "r",
                             base64_buffer, BASE64_BUFFER_LEN - 8);
  if (ret == -1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (8)");
    goto send_session_key_error;
  }

  bc_strlcat(buffer_to_encrypt, base64_buffer, BUFFER_TO_ENCRYPT_LEN - 8);
  buffer_to_encrypt_len += strlen(base64_buffer);


  bc_strlcat(buffer_to_encrypt, "|", BUFFER_TO_ENCRYPT_LEN - 8);
  buffer_to_encrypt_len++;

  /* Put the second part of the signed hash into the buffer_to_encrypt. */
  ret = encode_key_component(signed_info, "s",
                             base64_buffer, BASE64_BUFFER_LEN - 8);
  if (ret == -1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (9)");
    goto send_session_key_error;
  }

  bc_strlcat(buffer_to_encrypt, base64_buffer, BUFFER_TO_ENCRYPT_LEN - 8);
  buffer_to_encrypt_len += strlen(base64_buffer);


  /* We don't need the signed info anymore since its now in the
   * buffer_to_encrypt. */
  UM_SEXP_RELEASE(signed_info);


  /* Sanity check. */
  if (buffer_to_encrypt_len >= BUFFER_TO_ENCRYPT_LEN - 8) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (10)");
    goto send_session_key_error;
  }


  /* Make the buffer_to_encrypt into an MPI.  32 bits = 4 bytes.  This ensures
   * that it will exist in secure memory. */
  to_encrypt_mpi = gcry_mpi_snew(32);
  ret = gcry_mpi_scan(&to_encrypt_mpi, GCRYMPI_FMT_USG,
                      (unsigned char *)buffer_to_encrypt,
		      buffer_to_encrypt_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (11)");
    goto send_session_key_error;
  }

  /* We don't need the buffer_to_encrypt anymore since its now an MPI. */
  gcry_free(buffer_to_encrypt);  buffer_to_encrypt = NULL;
  buffer_to_encrypt_len = 0;


  /* Construct an S-expression from the buffer_to_encrypt MPI. */
  ret = gcry_sexp_build(&sexp_to_encrypt, &err,
                  "(data\n (flags)\n (value %m))\n", to_encrypt_mpi);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (12)");
    goto send_session_key_error;
  }

  UM_MPI_RELEASE(to_encrypt_mpi);


  /* Encrypt the S-expression which holds the session key and its
   * corresponding signed hash. */
  ret = gcry_pk_encrypt(&sexp_encrypted, sexp_to_encrypt,
                        *(buddy->public_encrypt_key));

  UM_SEXP_RELEASE(sexp_to_encrypt);

  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (13)");
    goto send_session_key_error;
  }


  encrypted_buffer = calloc(ENCRYPTED_BUFFER_LEN + 8, sizeof(unsigned char));
  if (encrypted_buffer == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (14)");
    goto send_session_key_error;
  }

  /* components 'a' and 'b' must be base64'ed and sent. */
  ret = encode_key_component(sexp_encrypted, "a",
                             base64_buffer, BASE64_BUFFER_LEN - 8);
  if (ret == -1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (15)");
    goto send_session_key_error;
  }

  bc_strlcpy(encrypted_buffer, base64_buffer, ENCRYPTED_BUFFER_LEN - 8);
  bc_strlcat(encrypted_buffer, "|", ENCRYPTED_BUFFER_LEN - 8);

  ret = encode_key_component(sexp_encrypted, "b",
                             base64_buffer, BASE64_BUFFER_LEN - 8);
  if (ret == -1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (16)");
    goto send_session_key_error;
  }

  bc_strlcat(encrypted_buffer, base64_buffer, ENCRYPTED_BUFFER_LEN - 8);


  gcry_free(base64_buffer);  base64_buffer = NULL;
  UM_SEXP_RELEASE(sexp_encrypted);


  /* Save a reference to the cleartext session key we're gonna send, then send
   * the signed and encrypted version of it. */

  /* Now we need to XOR the two keys from both buddies to create the session
   * info.  But... this step is done asynchronously with the buddy, so we
   * may or may not have the other half yet.  If we do, then XOR the two
   * pieces and we're done.  If not, just save what we have so far, and we'll
   * XOR it all later. */

  /* If the session stuff is NULL, then we don't have the buddy's info yet, so
   * we'll just cache what we have so far... */
  if ((buddy->session_key == NULL) &&
      (buddy->session_iv == NULL) &&
      (buddy->session_hmac == NULL)) {
    buddy->session_key = session_key;
    buddy->session_iv = session_iv;
    buddy->session_hmac = session_hmac;
  } else {
    int i = 0;

    /* We have both halves of the session info, so lets XOR it all together! */
    for (i = 0; i < SESSION_KEY_BYTE_SIZE; i++) {
      buddy->session_key[ i ] ^= session_key[ i ];
      buddy->session_iv[ i ] ^= session_iv[ i ];
      buddy->session_hmac[ i ] ^= session_hmac[ i ];
    }

    /* These aren't needed anymore now that the two have been XOR'ed and
     * stored back in the UM_BUDDY struct above. */
    gcry_free(session_key);   session_key = NULL;
    gcry_free(session_iv);    session_iv = NULL;
    gcry_free(session_hmac);  session_hmac = NULL;


    /* Now that we have complete session information, we can now construct
     * the cipher object and set the IV, etc correctly... */
    cipher = (gcry_cipher_hd_t *)gcry_malloc_secure(sizeof(gcry_cipher_hd_t));
    md = (gcry_md_hd_t *)gcry_malloc_secure(sizeof(gcry_md_hd_t));
    if ((cipher == NULL) || (md == NULL)) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (17)");
      goto send_session_key_error;
    }

    memset(cipher, 0, sizeof(gcry_cipher_hd_t));
    memset(md, 0, sizeof(gcry_md_hd_t));

    ret = gcry_cipher_open(cipher, GCRY_CIPHER_AES256,
			   GCRY_CIPHER_MODE_ECB, GCRY_CIPHER_SECURE);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR,
	     "Internal error. %s (18)", gcry_strerror(ret));
      goto send_session_key_error;
    }

    /*if (gcry_cipher_setctr(*cipher, buddy->session_iv,
			   SYMMETRIC_BLOCK_SIZE)) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (19)");
      goto send_session_key_error;
      }*/

    if (gcry_cipher_setkey(*cipher, buddy->session_key,
			   SESSION_KEY_BYTE_SIZE)) { 
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (20)");
      goto send_session_key_error;
    }


    buddy->cipher = cipher;


    /* Now set up the message-digest object that we'll use for HMAC'ing... */
    ret = gcry_md_open(md, GCRY_MD_SHA1,
		       GCRY_MD_FLAG_SECURE | GCRY_MD_FLAG_HMAC);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (21)");
      goto send_session_key_error;
    }

    ret = gcry_md_setkey(*md, buddy->session_hmac, 32);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("send_session_key", BC_LOG_ERROR, "Internal error. (22)");
      goto send_session_key_error;
    }


    buddy->md = md;
  }


  string = bc_string_new(NULL);
  bc_string_append_printf(string, "%u", strlen(encrypted_buffer));

  /* NOTE:  When implementing binary outputs, can't use bc_string_new! */
  mxfer->type = MXFER_TYPE_MESSAGE_ACTIONABLE;
  mxfer->results = bc_slist_append(mxfer->results, strdup(encrypted_buffer));
  mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

  bc_string_free(&string, 0);

  /* BLASTOFF! */
  /*serv_send_im(gc, buddy->screenname, encrypted_buffer, 0);*/

  free(encrypted_buffer);  encrypted_buffer = NULL;
  gcry_free(session_junk);  session_junk = NULL;

  return 1;


 send_session_key_error:
  gcry_free(session_junk);  session_junk = NULL;
  gcry_free(session_key);   session_key = NULL;
  gcry_free(session_iv);    session_iv = NULL;
  gcry_free(session_hmac);  session_hmac = NULL;
  gcry_free(cipher);        cipher = NULL;       buddy->cipher = NULL;
  gcry_free(md);            md = NULL;           buddy->md = NULL;

  gcry_free(buffer_to_encrypt);  buffer_to_encrypt = NULL;
  gcry_free(base64_buffer);  base64_buffer = NULL;
  free(encrypted_buffer);  encrypted_buffer = NULL;

  UM_MPI_RELEASE(data_mpi);
  UM_MPI_RELEASE(to_encrypt_mpi);

  UM_SEXP_RELEASE(key_component);
  UM_SEXP_RELEASE(sexp_to_sign);
  UM_SEXP_RELEASE(signed_info);
  UM_SEXP_RELEASE(sexp_to_encrypt);
  UM_SEXP_RELEASE(sexp_encrypted);

  return -1;
}



/* This function extracts the buddy's session key in the message and checks its
 * signature.  The UM_BUDDY's session_key_from_buddy field is set. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 *    - Joe Testa: August 14th, 2003
 */
int get_session_key(MXFER *mxfer, struct UM_BUDDY *buddy, char *message) {

  char *pipe_ptr = NULL;
  char *a_ptr = NULL;
  char *b_ptr = NULL;
  char *key_ptr = NULL;
  char *r_ptr = NULL;
  char *s_ptr = NULL;
  gcry_cipher_hd_t *cipher = NULL;
  gcry_md_hd_t *md = NULL;
  int decode_len = 0;
  char *base64_buffer = NULL;
  int ret = 0;
  char *key_component_buf = NULL;
  size_t key_component_len = 0;
  unsigned int key_len = 0;
  unsigned char *session_key = NULL;
  unsigned char *session_junk = NULL;
  unsigned char *session_iv = NULL;
  unsigned char *session_hmac = NULL;
  unsigned char *session_sequence_num = NULL;
  size_t err = 0;

  gcry_mpi_t a_mpi;
  gcry_mpi_t b_mpi;
  gcry_mpi_t r_mpi;
  gcry_mpi_t s_mpi;
  gcry_mpi_t key_mpi;
  gcry_sexp_t sexp_encrypted_key;
  gcry_sexp_t sexp_decrypted_key;
  gcry_sexp_t key_component;
  gcry_sexp_t signed_result;
  gcry_sexp_t signed_data;

  memset(&a_mpi, 0, sizeof(gcry_mpi_t));
  memset(&b_mpi, 0, sizeof(gcry_mpi_t));
  memset(&r_mpi, 0, sizeof(gcry_mpi_t));
  memset(&s_mpi, 0, sizeof(gcry_mpi_t));
  memset(&key_mpi, 0, sizeof(gcry_mpi_t));
  memset(&sexp_encrypted_key, 0, sizeof(gcry_sexp_t));
  memset(&sexp_decrypted_key, 0, sizeof(gcry_sexp_t));
  memset(&key_component, 0, sizeof(gcry_sexp_t));
  memset(&signed_result, 0, sizeof(gcry_sexp_t));
  memset(&signed_data, 0, sizeof(gcry_sexp_t));


  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));


  pipe_ptr = strchr(message, '|');
  if (pipe_ptr == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_session_key", BC_LOG_ERROR, "Internal Error. (1)");
    goto get_session_key_error;
  }

  /* Split the message into the 'a' and 'b' parts (which directly correspond to
   * the encrypted chunks). */
  a_ptr = message;
  b_ptr = pipe_ptr + 1;
  *pipe_ptr = '\0';


  base64_buffer = gcry_malloc_secure(BASE64_BUFFER_LEN + 8);
  if (base64_buffer == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_session_key", BC_LOG_ERROR, "Internal Error. (2)");
    goto get_session_key_error;
  }

  memset(base64_buffer, 0, BASE64_BUFFER_LEN);


  /* Decode and scan the 'a' component of the encrypted message. */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)a_ptr);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (2.5)");
    goto get_session_key_error;
  }

  ret = gcry_mpi_scan(&a_mpi, GCRYMPI_FMT_USG,
                      (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (3)");
    goto get_session_key_error;
  }

  /* Decode and scan the 'b' component of the encrypted message. */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)b_ptr);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (3.5)");
    goto get_session_key_error;
  }

  ret = gcry_mpi_scan(&b_mpi, GCRYMPI_FMT_USG,
                      (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (4)");
    goto get_session_key_error;
  }

  /* Make an S-expression out of the 'a' and 'b' encrypted message
   * components. */
  ret = gcry_sexp_build(&sexp_encrypted_key, &err,
                        "(enc-val(flags)(elg(a %m)(b %m)))", a_mpi, b_mpi);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (5)");
    goto get_session_key_error;
  }

  UM_MPI_RELEASE(a_mpi);
  UM_MPI_RELEASE(b_mpi);

  /* Initialize some secure memory for the plaintext key to go into. */
  /* BROKEN:  there is no 'secure S-expression' creation function.  =\ */
  /*sexp_decrypted_key = gcry_mpi_snew(8 * 4);*/

  /* Decrypt the session key. */
  ret = gcry_pk_decrypt(&sexp_decrypted_key, sexp_encrypted_key,
                        *my_encrypting_keypair);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (6)");
    goto get_session_key_error;
  }


  key_component_buf = (char *)gcry_sexp_nth_data(sexp_decrypted_key,
                                                          1,
                                                          &key_component_len);
  if (key_component_buf == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (7)");
    goto get_session_key_error;
  }


  /* Save a pointer to where the session key starts.  Note that this pointer is
   * only valid as long as sexp_decrypted_key above isn't modified! */
  key_ptr = key_component_buf;


  /* Here's a quirk in libgcrypt:  gcry_sexp_nth_data() above returns an
   * EOT-terminated string (EOT = ASCII character #4).  We'll just replace
   * that EOT with a NULL to make the code below happy... */
  if ((key_component_len > 2) &&
      (key_component_buf[ key_component_len ] == (char)4)) {
    key_component_buf[ key_component_len ] = '\0';
  }

  pipe_ptr = strchr(key_component_buf, '|');
  if (pipe_ptr == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (8)");
    goto get_session_key_error;
  }

  /* Save a pointer to the 'r' signed hash component. */
  r_ptr = pipe_ptr + 1;
  *pipe_ptr = '\0';

  pipe_ptr = strchr(r_ptr, '|');
  if (pipe_ptr == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (9)");
    goto get_session_key_error;
  }

  /* Save a pointer to the 's' signed hash component. */
  s_ptr = pipe_ptr + 1;
  *pipe_ptr = '\0';


  /* Decode the session key. */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)key_ptr);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (9.5)");
    goto get_session_key_error;
  }

  key_len = decode_len;

  if (decode_len < SESSION_JUNK_SIZE) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (10)");
    goto get_session_key_error;
  }


  session_junk = gcry_malloc_secure(SESSION_JUNK_SIZE + 8);
  session_key = gcry_malloc_secure(SESSION_KEY_BYTE_SIZE + 8);
  session_iv = gcry_malloc_secure(SESSION_KEY_BYTE_SIZE + 8);
  session_hmac = gcry_malloc_secure(SESSION_KEY_BYTE_SIZE + 8);
  session_sequence_num = gcry_malloc_secure(SESSION_KEY_BYTE_SIZE + 8);
  if ((session_junk == NULL) || (session_key == NULL) ||
      (session_iv == NULL) || (session_hmac == NULL) ||
      (session_sequence_num == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (11)");

    goto get_session_key_error;
  }


  /* Copy the session key into its final destination buffer. */
  memcpy(session_junk, base64_buffer, SESSION_JUNK_SIZE);
  memcpy(session_key, session_junk, SESSION_KEY_BYTE_SIZE);
  memcpy(session_iv, session_junk + SESSION_KEY_BYTE_SIZE,
	 SESSION_KEY_BYTE_SIZE);
  memcpy(session_hmac, session_junk + (SESSION_KEY_BYTE_SIZE * 2),
	 SESSION_KEY_BYTE_SIZE);
  memcpy(session_sequence_num, session_junk + (SESSION_KEY_BYTE_SIZE * 3),
	 SESSION_KEY_BYTE_SIZE);


  /* Now that we pulled the session junk into its key, iv, and hmac, we free
   * it.*/
  gcry_free(session_junk);  session_junk = NULL;

  /*hex_dump("get_session_key", session_key, SESSION_KEY_BYTE_SIZE);
  hex_dump("get_session_iv", session_iv, SESSION_KEY_BYTE_SIZE);
  hex_dump("\t\tX get_session_hmac", session_hmac, SESSION_KEY_BYTE_SIZE);
  hex_dump("get_session_sequence", session_sequence_num, SESSION_KEY_BYTE_SIZE);*/
  hex_dump("get_session_sequence", (char *)session_sequence_num,
	   SESSION_KEY_BYTE_SIZE);


  /* Make an MPI out of the session key so that its signature can be checked.*/
  key_mpi = gcry_mpi_new((SESSION_JUNK_SIZE + 8) * 8);
  ret = gcry_mpi_scan(&key_mpi, GCRYMPI_FMT_USG, (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (12)");
    goto get_session_key_error;
  }

  /* Make an MPI out of the 'r' signed hash component  */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)r_ptr);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (12.5)");
    goto get_session_key_error;
  }

  ret = gcry_mpi_scan(&r_mpi, GCRYMPI_FMT_USG, (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (13)");
    goto get_session_key_error;
  }


  /* Make an MPI out of the 's' signed hash component  */
  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)s_ptr);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (13.5)");
    goto get_session_key_error;
  }

  ret = gcry_mpi_scan(&s_mpi, GCRYMPI_FMT_USG, (unsigned char *)base64_buffer, decode_len, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (14)");
    goto get_session_key_error;
  }

  gcry_free(base64_buffer);  base64_buffer = NULL;


  /* Construct an S-expression to hold the signed hash of the session key. */
  ret = gcry_sexp_build(&signed_result, &err,
                  "(sig-val(dsa(r %m)(s %m)))", r_mpi, s_mpi );
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (15)");
    goto get_session_key_error;
  }

  /*printf("signed_result:\n\n");
    gcry_sexp_dump(signed_result);*/


  UM_MPI_RELEASE(r_mpi);
  UM_MPI_RELEASE(s_mpi);


  /* Build the session key S-expression so we can check the signature. */
  ret = gcry_sexp_build(&signed_data, &err,
                        "(data\n (flags raw)\n (value %m))\n", key_mpi);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (16)");
    goto get_session_key_error;
  }

  UM_MPI_RELEASE(key_mpi);  



  if (buddy->public_sign_key == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (17)");
    goto get_session_key_error;
  }

  ret = gcry_pk_verify(signed_result, signed_data, *(buddy->public_sign_key));
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_session_key", BC_LOG_ERROR, "Malformed input. (18)");
    goto get_session_key_error;
  }

  /*printf("SIGNATURE ON SESSION KEY IS GOOD!\n");*/
  UM_SEXP_RELEASE(sexp_encrypted_key);
  UM_SEXP_RELEASE(sexp_decrypted_key);

  UM_SEXP_RELEASE(signed_result);
  UM_SEXP_RELEASE(signed_data);


  /* Now we need to XOR the two keys from both buddies to create the session
   * info.  But... this step is done asynchronously with the buddy, so we
   * may or may not have the other half yet.  If we do, then XOR the two
   * pieces and we're done.  If not, just save what we have so far, and we'll
   * XOR it all later. */

  /* If the session stuff is NULL, then we don't have the buddy's info yet, so
   * we'll just cache what we have so far... */
  if ((buddy->session_key == NULL) &&
      (buddy->session_iv == NULL) &&
      (buddy->session_hmac == NULL)) {
    buddy->session_key = session_key;
    buddy->session_iv = session_iv;
    buddy->session_hmac = session_hmac;
  } else {
    int i = 0;

    /* We have both halves of the session info, so lets XOR it all together! */
    for (i = 0; i < SESSION_KEY_BYTE_SIZE; i++) {
      buddy->session_key[ i ] ^= session_key[ i ];
      buddy->session_iv[ i ] ^= session_iv[ i ];
      buddy->session_hmac[ i ] ^= session_hmac[ i ];
    }

    /*hex_dump("FINAL: session_key", buddy->session_key, SESSION_KEY_BYTE_SIZE);
    hex_dump("FINAL: session_iv", buddy->session_iv, SESSION_KEY_BYTE_SIZE);
    hex_dump("FINAL: session_hmac", buddy->session_hmac,SESSION_KEY_BYTE_SIZE);*/

    /* These aren't needed anymore now that the two have been XOR'ed and
     * stored back in the UM_BUDDY struct above. */
    gcry_free(session_key);   session_key = NULL;
    gcry_free(session_iv);    session_iv = NULL;
    gcry_free(session_hmac);  session_hmac = NULL;


    /* Make decrypting cipher for this buddy, and set its key & IV (ctr). */
    cipher = (gcry_cipher_hd_t *)gcry_malloc_secure(sizeof(gcry_cipher_hd_t));
    md = (gcry_md_hd_t *)gcry_malloc_secure(sizeof(gcry_md_hd_t));
    if ((cipher == NULL) && (md == NULL)) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (19)");
      goto get_session_key_error;
    }

    memset(cipher, 0, sizeof(gcry_cipher_hd_t));
    ret = gcry_cipher_open(cipher, GCRY_CIPHER_AES256,
			   GCRY_CIPHER_MODE_ECB, GCRY_CIPHER_SECURE);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (20)");
      goto get_session_key_error;
    }

    /*if (gcry_cipher_setctr(*cipher, buddy->session_iv,
			   SYMMETRIC_BLOCK_SIZE)) { 
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (21)");
      goto get_session_key_error;
      }*/

    if (gcry_cipher_setkey(*cipher, buddy->session_key,
			   SESSION_KEY_BYTE_SIZE)) { 
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (22)");
      goto get_session_key_error;
    }


    buddy->cipher = cipher;


    /* Now set up the message-digest object that we'll use for HMAC'ing... */
    memset(md, 0, sizeof(gcry_md_hd_t));
    ret = gcry_md_open(md, GCRY_MD_SHA1,
		       GCRY_MD_FLAG_SECURE | GCRY_MD_FLAG_HMAC);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (23)");
      goto get_session_key_error;
    }

    ret = gcry_md_setkey(*md, buddy->session_hmac, 32);
    if (ret != 0) {
      mxfer->type = MXFER_TYPE_ERROR;
      mxfer->error = bc_string_new("Internal error.");
      bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (24)");
      goto get_session_key_error;
    }


    buddy->md = md;

  }


  buddy->my_sequence_number = gcry_calloc(1, sizeof(gcry_mpi_t));
  *(buddy->my_sequence_number) = gcry_mpi_snew(sizeof(gcry_mpi_t));
  ret = gcry_mpi_scan(buddy->my_sequence_number, GCRYMPI_FMT_USG,
		      session_sequence_num, SESSION_KEY_BYTE_SIZE, NULL);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_session_key", BC_LOG_ERROR, "Internal error. (25)");
    goto get_session_key_error;
  }

  /* There is no actionable message since we just parsed the session keys. */
  mxfer->type = MXFER_TYPE_MESSAGE_NONACTIONABLE;

  /* Touch the timeout timestamp. */
  time(&(buddy->last_touched));

  return 1;


 get_session_key_error:
  gcry_free(base64_buffer);  base64_buffer = NULL;
  gcry_free(session_key);    session_key = NULL;
  gcry_free(session_junk);   session_junk = NULL;
  gcry_free(session_iv);     session_iv = NULL;
  gcry_free(session_hmac);   session_hmac = NULL;
  gcry_free(cipher);         cipher = NULL;   buddy->cipher = NULL;
  gcry_free(md);             md = NULL;       buddy->md = NULL;

  UM_MPI_RELEASE(a_mpi);
  UM_MPI_RELEASE(b_mpi);
  UM_MPI_RELEASE(r_mpi);
  UM_MPI_RELEASE(s_mpi);
  UM_MPI_RELEASE(key_mpi);
  UM_SEXP_RELEASE(sexp_encrypted_key);
  UM_SEXP_RELEASE(sexp_decrypted_key);
  UM_SEXP_RELEASE(key_component);
  UM_SEXP_RELEASE(signed_result);
  UM_SEXP_RELEASE(signed_data);

  return -1;
}


/* This function extracts a named component in the specified S-expression,
 * base64 encodes it, then copies it into the output buffer. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int encode_key_component(gcry_sexp_t x, char *component,
                         char *output, unsigned int output_len ) {

  unsigned char *key_component_buf = NULL;
  size_t key_component_len = 0;
  int ret = -1;

  gcry_sexp_t key_component;


  memset(&key_component, 0, sizeof(gcry_sexp_t));


  key_component = gcry_sexp_find_token(x, component, 0);
  if (key_component == NULL) {
    return -1;
  }

  key_component_buf = (unsigned char *)gcry_sexp_nth_data(key_component, 1,
                                                          &key_component_len);
  if (key_component_buf == NULL) {
    UM_SEXP_RELEASE(key_component);
    return -1;
  }

  /* Base64'ing a set of bytes results in an encoding that is 4 / 3rds times
   * the size of the input.  We should check to see if the output buffer is
   * long enough, and return an explicit error, if appropriate, instead of
   * just silently returning without doing anything (which is what
   * base64_encode would normally do...). */
  /* No longer necessary since base64_encode now has a return value. */
  /*if ((key_component_len * 1.34) > output_len) {
    UM_SEXP_RELEASE(key_component);
    return -1;
    }*/

  ret = base64_encode((unsigned char *)output, output_len,
                      (unsigned char *)key_component_buf, key_component_len);
  UM_SEXP_RELEASE(key_component);
  return ret;
}


/* This function is identical to 'encode_key_component', except that the
 * nth data field is encoded instead of always the 1st. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int encode_key_component_nth(gcry_sexp_t x, char *component,
			     unsigned int nth,
			     char *output, unsigned int output_len) {

  unsigned char *key_component_buf = NULL;
  size_t key_component_len = 0;
  int ret = -1;

  gcry_sexp_t key_component;


  memset(&key_component, 0, sizeof(gcry_sexp_t));


  key_component = gcry_sexp_find_token(x, component, 0);
  if (key_component == NULL) {
    return -1;
  }

  key_component_buf = (unsigned char *)gcry_sexp_nth_data(key_component, nth,
                                                          &key_component_len);
  if (key_component_buf == NULL) {
    UM_SEXP_RELEASE(key_component);
    return -1;
  }

  ret = base64_encode((unsigned char *)output, output_len,
                      (unsigned char *)key_component_buf, key_component_len);
  UM_SEXP_RELEASE(key_component);
  return ret;
}



/* Removes a buddy from the pending list. Buddy is still valid afterwards
 * (ie: it is not wiped). */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
void remove_buddy_from_pending_list(struct UM_BUDDY *buddy) {

  char *who = NULL;
  struct UM_BUDDY *previous = NULL;
  struct UM_BUDDY *current = NULL;
  unsigned int deleted = 0;

  who = buddy->screenname;

  if ((pending_buddies_list != NULL) &&
      (strcmp(pending_buddies_list->screenname, who) == 0)) {
    pending_buddies_list = pending_buddies_list->next;
    /*print_pending_list();*/
    return;
  }

  current = pending_buddies_list;
  while((current != NULL) && (deleted == 0)) {

    if (strcmp(current->screenname, who) == 0) {
      previous->next = current->next;
      deleted = 1;
    } else {
      previous = current;
      current = current->next;
    }

  }

  /*print_pending_list();*/
  return;
}


/* Empties the pending list and wipes all the buddies in it.  Useful only
 * when signing off. */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
void wipe_entire_pending_list() {

  struct UM_BUDDY *next = NULL;
  struct UM_BUDDY *current = NULL;

  bc_log("wipe_entire_pending_list", BC_LOG_INFO, "called.");
/*if (pending_buddies_list == NULL)
  return;*/

  current = pending_buddies_list;
  while(current != NULL) {
    next = current->next;
    bc_log("wipe_entire_pending_list", BC_LOG_INFO,
	   "wiping: [%s]", current->screenname);
    wipe_buddy(current);  free(current); current = NULL;
    current = next;
  }

    /*  do {
    bc_log("wipe_entire_pending_list", BC_LOG_ERROR,
	   "wiping: [%s]", current->screenname);
    next = current->next;
    wipe_buddy(current);  free(current);  current = NULL;
  } while(next != NULL);
    */

  pending_buddies_list = NULL;
  return;
}



/* Empties the validated list and wipes all the buddies in it.  Useful only
 * when signing off. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
void wipe_entire_validated_list() {

  struct UM_BUDDY *next = NULL;
  struct UM_BUDDY *current = NULL;

  bc_log("wipe_entire_validated_list", BC_LOG_INFO, "called.");
  /*if (validated_buddies_list == NULL)
    return;

  current = validated_buddies_list;
  do {
    bc_log("wipe_entire_validated_list", BC_LOG_INFO,
	   "wiping: [%s]", current->screenname);
    next = current->next;
    wipe_buddy(current);  free(current);  current = NULL;
    } while(next != NULL);*/

  current = validated_buddies_list;
  while(current != NULL) {
    next = current->next;
    bc_log("wipe_entire_pending_list", BC_LOG_INFO,
	   "wiping: [%s]", current->screenname);
    wipe_buddy(current);  free(current); current = NULL;
    current = next;
  }


  validated_buddies_list = NULL;
  return;

}



/* Sends an encrypted message to a buddy. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 *    - Joe Testa: August 14th, 2003
 */
int send_encrypted_im(MXFER *mxfer, struct UM_BUDDY *buddy, char *message) {

  char *buffer_to_encrypt = NULL;
  char *base64_buffer = NULL;
  unsigned int buffer_to_encrypt_len = 0;
  int ret = 0;
  char *encrypted_buffer = NULL;
  char *padded_message = NULL;
  u_int32_t padded_message_len = 0;
  unsigned int padded_message_size = 0;
  char *hmac_digest = NULL;
  unsigned char *sequence_number = NULL;
  BCString *string = NULL;
  BCString *string2 = NULL;


  char header_pad[ 96 ];

  memset(header_pad, 0, sizeof(header_pad));


  padded_message_size = ((int)strlen(message) * 1.5) + 1 + 40;
  padded_message = calloc(padded_message_size, sizeof(char));

  padded_message_len = (u_int32_t)um_pad_message(padded_message, padded_message_size, message);


  /* Is there any point in putting this into a secure buffer?  GAIM just
   * copied the cleartext message all over memory already, so... */
  buffer_to_encrypt = (char *)gcry_malloc_secure(BUFFER_TO_ENCRYPT_LEN + 8);
  base64_buffer = (char *)gcry_malloc_secure(BASE64_BUFFER_LEN + 8);
  if ((buffer_to_encrypt == NULL) || (base64_buffer == NULL)) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error in send_encrypted_im (1): ");
    mxfer->error = bc_string_append(mxfer->error, (char *)gcry_strerror(ret));
    bc_log("send_encrypted_im", BC_LOG_ERROR, mxfer->error->str);
    goto send_encrypted_im_error;
  } else {
    memset(buffer_to_encrypt, 0, BUFFER_TO_ENCRYPT_LEN);
    memset(base64_buffer, 0, BASE64_BUFFER_LEN);
  }



  sequence_number = get_sequence_number(buddy);


  /*gcry_md_write(*(buddy->md), &padded_message_len, sizeof(u_int32_t));*/
  gcry_md_write(*(buddy->md), sequence_number, SESSION_KEY_BYTE_SIZE);
  gcry_md_write(*(buddy->md), padded_message, padded_message_len);
  hmac_digest = (char *)gcry_md_read(*(buddy->md), GCRY_MD_SHA1);


  /*********/
  /*bc_log("send_encrypted_im", BC_LOG_DEBUG, "Padded message len: %u", padded_message_len);
  hex_dump("send_encrypted_im/Sequence number", sequence_number, SESSION_KEY_BYTE_SIZE);
  hex_dump("send_encrypted_im/Padded message", padded_message, padded_message_len);
  hex_dump("send_encrypted_im/HMAC", hmac_digest, HMAC_DIGEST_BYTE_SIZE);*/
  /*********/



  /* Put the padded message length in first.  This lets the receiver know
   * how long it is despite the presence of any cipher padding at the end of
   * the block. */
  /*if (sizeof(u_int32_t) > BUFFER_TO_ENCRYPT_LEN - 8) {
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (0)");
    goto send_encrypted_im_error;
  } else {
    memcpy(buffer_to_encrypt, &padded_message_len, sizeof(u_int32_t));
    buffer_to_encrypt_len = sizeof(u_int32_t);
    }*/

  um_pad_header(header_pad, (char *)sequence_number, hmac_digest);
  if (HEADER_PAD_SIZE > BUFFER_TO_ENCRYPT_LEN - 8) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (1)");
    goto send_encrypted_im_error;
  } else {
    memcpy(buffer_to_encrypt, header_pad, HEADER_PAD_SIZE);
    buffer_to_encrypt_len = HEADER_PAD_SIZE;
  }

  /* Put the sequence number in the buffer first... */
  /*if (sizeof(u_int32_t) + SESSION_KEY_BYTE_SIZE > BUFFER_TO_ENCRYPT_LEN - 8) {
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (1)");
    goto send_encrypted_im_error;
  } else {
    memcpy(buffer_to_encrypt + buffer_to_encrypt_len,
	   sequence_number, SESSION_KEY_BYTE_SIZE);
    buffer_to_encrypt_len += SESSION_KEY_BYTE_SIZE;
  }
  */

  /* We don't need the sequence number anymore, so free it. */
  gcry_free(sequence_number);  sequence_number = NULL;


  /* Put the HMAC digest in second... */
  /*if (buffer_to_encrypt_len+HMAC_DIGEST_BYTE_SIZE > BUFFER_TO_ENCRYPT_LEN-8) {
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (2)");
    goto send_encrypted_im_error;
  } else {
    memcpy(buffer_to_encrypt + buffer_to_encrypt_len,
	   hmac_digest, HMAC_DIGEST_BYTE_SIZE);
    buffer_to_encrypt_len += HMAC_DIGEST_BYTE_SIZE;
    }*/

  /* Lastly, put in the padded message... */
  if (buffer_to_encrypt_len + padded_message_len > BUFFER_TO_ENCRYPT_LEN - 8) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (3)");
    goto send_encrypted_im_error;
  } else {
    memcpy(buffer_to_encrypt + buffer_to_encrypt_len,
	   padded_message, padded_message_len);
    buffer_to_encrypt_len += padded_message_len;
  }


  /* We need to align the buffer to encrypt to a 16-byte boundary. */
  /*cipher_padding_len = 16 - ((sizeof(u_int32_t) + SESSION_KEY_BYTE_SIZE + HMAC_DIGEST_BYTE_SIZE + padded_message_len) % 16);
  if (buffer_to_encrypt_len + cipher_padding_len > BUFFER_TO_ENCRYPT_LEN - 8) {
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (4)");
    goto send_encrypted_im_error;
  } else if (cipher_padding_len > 0) {
    char pad[ 32 ];

    memset(pad, 0, sizeof(pad));


    * NOTE:  THESE PAD BYTES ARE NOT SIGNED!! *
    gcry_create_nonce(pad, MY_MIN(sizeof(pad), cipher_padding_len));

    memcpy(buffer_to_encrypt + buffer_to_encrypt_len,
	   pad, MY_MIN(sizeof(pad), cipher_padding_len));
    buffer_to_encrypt_len += MY_MIN(sizeof(pad), cipher_padding_len);
  }*/


  /* Since we wrote bytes into the message-digest object, we should reset it
   * so that all the messages are not linked together (this would kill an
   * entire connection if one transmission error occurred). */
  gcry_md_reset(*(buddy->md));


  /* Encrypt the message! */
  ret = gcry_cipher_encrypt(*(buddy->cipher),
                            (unsigned char *)buffer_to_encrypt, buffer_to_encrypt_len,
                            NULL, 0);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("send_encrypted_im", BC_LOG_ERROR, "Malformed input (3)");
    goto send_encrypted_im_error;
  }


  /* Base64 the encrypted message. */
  if (base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8,
		    (unsigned char *)buffer_to_encrypt, buffer_to_encrypt_len) < 0) {
    bc_log("send_encrypted_im", BC_LOG_ERROR,
	   "Error encoding encrypted message: %u %u", BASE64_BUFFER_LEN - 8,
	   buffer_to_encrypt_len);
    goto send_encrypted_im_error;
  }


  string = bc_string_new(NULL);
  string2 = bc_string_new(base64_buffer);

  bc_string_append_printf(string, "%u", strlen(base64_buffer));

  /* NOTE:  When implementing binary outputs, can't use bc_string_new! */
  mxfer->type = MXFER_TYPE_MESSAGE_ACTIONABLE;
  mxfer->results = bc_slist_append(mxfer->results, string2->str);
  mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

  bc_string_free(&string, 0);
  bc_string_free(&string2, 0);


  gcry_free(padded_message);     padded_message = NULL;
  gcry_free(buffer_to_encrypt);  buffer_to_encrypt = NULL;
  gcry_free(base64_buffer);      base64_buffer = NULL;
  return 1;

 send_encrypted_im_error:
  gcry_free(padded_message);    padded_message = NULL;
  gcry_free(encrypted_buffer);  encrypted_buffer = NULL;
  gcry_free(base64_buffer);     base64_buffer = NULL;
  gcry_md_reset(*(buddy->md));
  return -1;
}



/* Gets the encrypted message, decodes it, decrypts it, and checks its
 * signature. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 *    - Joe Testa: March 5th, 2004
 *    - Joe Testa: August 15th, 2003
 */
void get_encrypted_im(MXFER *mxfer, struct UM_BUDDY *buddy, char *message) {
  BCString *string = NULL;
  int decode_len = 0;
  int ret = 0;
  u_int32_t padded_message_len = 0;
  unsigned int decrypted_message_len = 0;
  char *base64_buffer = NULL;
  char *decrypted_message = NULL;
  char *return_buffer = NULL;
  unsigned char *hmac_digest_mine = NULL;

  char hmac_digest_from_message[ HMAC_DIGEST_BYTE_SIZE + 8 ];
  unsigned char sequence_number[ SESSION_KEY_BYTE_SIZE + 8 ];
  char header_pad[ 96 ];

  memset(hmac_digest_from_message, 0, sizeof(hmac_digest_from_message));
  memset(sequence_number, 0, sizeof(sequence_number));
  memset(header_pad, 0, sizeof(header_pad));


  base64_buffer = calloc(BASE64_BUFFER_LEN + 8, sizeof(unsigned char));
  if (base64_buffer == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Internal error. (1)");
    goto get_encrypted_im_error;
  }

  decode_len = base64_decode((unsigned char *)base64_buffer, BASE64_BUFFER_LEN - 8, (unsigned char *)message);
  if (decode_len < 1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (2)");
    goto get_encrypted_im_error;
  }


  decrypted_message_len = decode_len * 2;
  decrypted_message = calloc(decrypted_message_len + 8, sizeof(unsigned char));
  if (decrypted_message == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Internal error. (3)");
    goto get_encrypted_im_error;
  }

  ret = gcry_cipher_decrypt(*(buddy->cipher),
                            (unsigned char *)decrypted_message, decrypted_message_len - 8,
                            (unsigned char *)base64_buffer, decode_len);
  if (ret != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (4)");
    goto get_encrypted_im_error;
  }


  /*memcpy(&padded_message_len, decrypted_message, sizeof(u_int32_t));*/
  /*bc_log("get_encrypted_im", BC_LOG_DEBUG, "Padded message len: %u", padded_message_len);*/
  memcpy(header_pad, decrypted_message, HEADER_PAD_SIZE);
  um_unpad_header((char *)sequence_number, hmac_digest_from_message, header_pad);



  /* The padded message length must be less than the size of the decrypted
   * block minus the length itself, the sequence number, and the digest.  After
   * subtracting all those parts, all that is left is the padded message and
   * the block padding (if any). */
  /*if (padded_message_len > (decode_len - (sizeof(u_int32_t) + SESSION_KEY_BYTE_SIZE + HMAC_DIGEST_BYTE_SIZE))) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (5)");
    goto get_encrypted_im_error;
    }*/


  /* Copy out the raw sequence number. */
  /*if (sizeof(sequence_number) - 8 < SESSION_KEY_BYTE_SIZE) {
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (1)");
    goto get_encrypted_im_error;
  } else {
    memcpy(sequence_number, decrypted_message + sizeof(u_int32_t), SESSION_KEY_BYTE_SIZE);
    *hex_dump("get_encrypted_im/Sequence number", sequence_number, SESSION_KEY_BYTE_SIZE);*
  }*/


  /* Ensure that the sequence number sent is good. */
  ret = check_sequence_number(buddy, sequence_number, SESSION_KEY_BYTE_SIZE);
  if (ret == -1) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (6)");
    goto get_encrypted_im_error;
  }


  /*if (sizeof(hmac_digest_from_message) - 8 < HMAC_DIGEST_BYTE_SIZE) {
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Buffer overflow prevented (2)");
    goto get_encrypted_im_error;
  } else {
    memcpy(hmac_digest_from_message, decrypted_message + sizeof(u_int32_t) + SESSION_KEY_BYTE_SIZE, HMAC_DIGEST_BYTE_SIZE);
    }*/


  /* Make a new buffer to return. */
  padded_message_len = decode_len - HEADER_PAD_SIZE;
  return_buffer = calloc(padded_message_len + 1 + 16, sizeof(unsigned char));
  if (return_buffer == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Internal error. (9)");
    goto get_encrypted_im_error;
  } else {
    /* Here, we want to copy the padded message in the buffer, which starts
     * AFTER the padded sequence number and HMAC digest.  We will skip them and
     * copy the rest of that buffer into 'return_buffer'. */
    memcpy(return_buffer, decrypted_message + HEADER_PAD_SIZE,
	   padded_message_len);
  }


  /* Calculate the HMAC *before* unpadding... */
  /*gcry_md_write(*(buddy->md), &padded_message_len, sizeof(u_int32_t));*/
  gcry_md_write(*(buddy->md), sequence_number, SESSION_KEY_BYTE_SIZE);
  gcry_md_write(*(buddy->md), return_buffer, padded_message_len);
  hmac_digest_mine = gcry_md_read(*(buddy->md), GCRY_MD_SHA1);


  if (memcmp(hmac_digest_mine, hmac_digest_from_message,
	     HMAC_DIGEST_BYTE_SIZE) != 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (10)");
    goto get_encrypted_im_error;
  }


  /* Now that the message has been authenticated, unpad the message. */
  if (um_unpad_message(return_buffer, padded_message_len,
		       return_buffer, padded_message_len) < 0) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Malformed input.");
    bc_log("get_encrypted_im", BC_LOG_ERROR, "Malformed input. (11)");
    goto get_encrypted_im_error;
  }

  free(decrypted_message);  decrypted_message = NULL;
  free(base64_buffer);  base64_buffer = NULL;
  gcry_md_reset(*(buddy->md));


  string = bc_string_new(NULL);
  bc_string_append_printf(string, "%u", strlen(return_buffer));

  mxfer->type = MXFER_TYPE_MESSAGE_CRYPTED;
  mxfer->results = bc_slist_append(mxfer->results, strdup(return_buffer));
  mxfer->result_sizes = bc_slist_append(mxfer->result_sizes, string->str);

  bc_string_free(&string, 0);
  free(return_buffer);  return_buffer = NULL;
  return;

 get_encrypted_im_error:
  free(base64_buffer);      base64_buffer = NULL;
  free(decrypted_message);  decrypted_message = NULL;
  free(return_buffer);      return_buffer = NULL;
  gcry_md_reset(*(buddy->md));
  return;
}



/* Adds a CONNECTED buddy to the validated list. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 *    - Joe Testa: March 5th, 2004
 */
void add_to_validated_list(struct UM_BUDDY *new_buddy) {
  struct UM_BUDDY *current = NULL;

  /* Double-check that this buddy is in the CONNECTED state... */
  if (new_buddy->state != CONNECTED) {
    bc_log("add_to_validated_list", BC_LOG_ERROR,
	   "NEW BUDDY'S STATE IS NOT CONNECTED!: %u\n", new_buddy->state);
    return;
  }

  /* Make sure the new buddy's next pointer is NULL, otherwise future list
   * operations will fail. */
  new_buddy->next = NULL;

  /* If the head of the list is NULL, then make the new buddy the head. */
  if(validated_buddies_list == NULL)
    validated_buddies_list = new_buddy;
  else {

    /* Cycle to the end of the list and add the new buddy there. */
    current = validated_buddies_list;
    while(current->next != NULL)
      current = current->next;

    current->next = new_buddy;

  }

  return;
}



/* Returns the hex fingerprint of the specified key.  Note that the buffer's
 * size must be 128 bytes or larger, otherwise the result is truncated. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 *    - Joe Testa: March 5th, 2004
 */
void get_key_fingerprint_from_sexp(gcry_sexp_t *key, char *buffer,
			           unsigned int buffer_len) {
  unsigned int i = 0;

  unsigned char temp[ 64 ];
  char hex[ 16 ];

  memset(temp, 0, sizeof(temp));
  memset(hex, 0, sizeof(hex));

  if (buffer == NULL)
    return;

  /* libgcrypt docs say that the passed buffer needs to be at least 20 bytes.*/
  gcry_pk_get_keygrip(*key, temp);

  /* temp now has the fingerprint, but we need to convert to hex ASCII... */
  *buffer = '\0';
  for (i = 0; i < 20; i++) {
    snprintf(hex, sizeof(hex), "%02X ", temp[ i ]);
    bc_strlcat(buffer, hex, buffer_len - 8);
  }

  /* Yeah, I'm paranoid... */
  if (buffer_len > 2) {
    /* Truncate the trailing space... */
    buffer[ strlen(buffer) - 1 ] = '\0';
  } else {
    buffer[ 0 ] = '\0';
  }

  return;
}



/* Loads the encrypting and signing key from a file, given the password to
 * decrypt that file.  Optionally, it can also load the signing key's
 * fingerprint. */
/* Returns 1 on success, 0 if the password was invalid, and -1 on error. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int load_keys_and_fingerprint(char *um_file,
			      char *pw,
			      gcry_sexp_t *signing_keypair,
			      gcry_sexp_t *encrypting_keypair,
			      char *buffer, size_t buffer_len) {
/*unsigned int file_len = 0;*/
  char *file_buffer = NULL;
/*  FILE *file_handle = NULL;*/
  size_t err = 0;
  unsigned int i = 0;
  int decode_len = 0;
  char *equal_ptr = NULL;
  char *key = NULL;
  char *value = NULL;
  gcry_mpi_t *mpi_ptr = NULL;
  int ret = 0;
  unsigned int key_args_are_null = 0;
  unsigned int buffer_arg_is_null = 0;
  gcry_sexp_t *local_signing_keypair = NULL;
  gcry_sexp_t *local_encrypting_keypair = NULL;
  char *newline_pos = NULL;
  char *data_ptr = NULL;
  char *entropy_filename = NULL;
  size_t entropy_filename_len = 0;

  /*#ifdef _WIN32
  HANDLE hFile = 0;
  DWORD file_len_high = 0;
  #endif*/

  struct stat file_status;
  gcry_mpi_t sign_public_p;
  gcry_mpi_t sign_public_q;
  gcry_mpi_t sign_public_g;
  gcry_mpi_t sign_public_y;
  gcry_mpi_t misc_sign_info_1;
  gcry_mpi_t misc_sign_info_2;
  gcry_mpi_t misc_sign_info_3;
  gcry_mpi_t misc_sign_info_4;
  gcry_mpi_t encrypt_public_p;
  gcry_mpi_t encrypt_public_g;
  gcry_mpi_t encrypt_public_y;
  gcry_mpi_t sign_private_x;
  gcry_mpi_t encrypt_private_x;
  unsigned char base64_buffer[ 1024 ];
  char databuf[ 4096 ];
  char fingerprint_buffer[ 128 ];

  memset(&file_status, 0, sizeof(struct stat));
  memset(&sign_public_p, 0, sizeof(gcry_mpi_t));
  memset(&sign_public_q, 0, sizeof(gcry_mpi_t));
  memset(&sign_public_g, 0, sizeof(gcry_mpi_t));
  memset(&sign_public_y, 0, sizeof(gcry_mpi_t));
  memset(&misc_sign_info_1, 0, sizeof(gcry_mpi_t));
  memset(&misc_sign_info_2, 0, sizeof(gcry_mpi_t));
  memset(&misc_sign_info_3, 0, sizeof(gcry_mpi_t));
  memset(&misc_sign_info_4, 0, sizeof(gcry_mpi_t));
  memset(&encrypt_public_p, 0, sizeof(gcry_mpi_t));
  memset(&encrypt_public_g, 0, sizeof(gcry_mpi_t));
  memset(&encrypt_public_y, 0, sizeof(gcry_mpi_t));
  memset(&sign_private_x, 0, sizeof(gcry_mpi_t));
  memset(&encrypt_private_x, 0, sizeof(gcry_mpi_t));
  memset(base64_buffer, 0, sizeof(base64_buffer));
  memset(databuf, 0, sizeof(databuf));
  memset(fingerprint_buffer, 0, sizeof(fingerprint_buffer));

  entropy_filename = strdup(um_file);
  entropy_filename_len = strlen(entropy_filename);
  entropy_filename[ entropy_filename_len - 2 ] = 'e';
  entropy_filename[ entropy_filename_len - 1 ] = 'n';

  set_entropy_file(entropy_filename);


  data_ptr = databuf;
  if (!decrypt_keyfile(um_file, pw,
		       fingerprint_buffer, sizeof(fingerprint_buffer) - 8,
		       databuf, sizeof(databuf) - 8)) {
    bc_log("load_keys_and_fingerprint",BC_LOG_ERROR,"decrypt_keyfile failed.");
    return 0;
  }

  newline_pos = strchr(data_ptr, '\n');
  if (newline_pos == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }

  file_buffer = data_ptr;
  *newline_pos = '\0';
  data_ptr = newline_pos + 1;
  /*printf("read: [%s]\n", file_buffer);*/

  if ((file_buffer[ 0 ] != 'F') ||
      (file_buffer[ 1 ] != 'V') ||
      (file_buffer[ 2 ] != ' ')) {
    bc_log("load_keys_and_fingerprint",BC_LOG_ERROR,
	   "Decrypted content is invalid.");
    return 0;
  }


  local_signing_keypair = signing_keypair;
  local_encrypting_keypair = encrypting_keypair;


  /* If the caller passes NULL for the keypairs, then they simply want the
   *     fingerprint.  We still need an S-exp, so we'll make them here and
   *     free them later.
   */
  if ((signing_keypair == NULL) || (encrypting_keypair == NULL)) {
    key_args_are_null = 1;
    local_signing_keypair = (gcry_sexp_t *)gcry_malloc_secure(sizeof(gcry_sexp_t));
    local_encrypting_keypair = (gcry_sexp_t *)gcry_malloc_secure(sizeof(gcry_sexp_t));

    if ((local_signing_keypair == NULL) || (local_encrypting_keypair == NULL)){
      bc_log("load_keys_and_fingerprint", BC_LOG_ERROR, "Out of memory!");
      goto load_keys_and_fingerprint_error;
    }
  }

  /* If the caller passes a null buffer, then we simply won't calculate the
   *     fingerprint. */
  if (buffer == NULL)
    buffer_arg_is_null = 1;



  sign_private_x = gcry_mpi_snew(8 * 4);
  encrypt_private_x = gcry_mpi_snew(8 * 4);

  if ((sign_private_x == NULL) || (encrypt_private_x == NULL)) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Couldn't allocate space for signing & encrypting MPIs!");
    goto load_keys_and_fingerprint_error;
  }

  /*
#ifdef _WIN32
  hFile = CreateFile(um_file, GENERIC_READ, FILE_SHARE_READ, NULL,
		     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "File %s is invalid.", um_file);
    goto load_keys_and_fingerprint_error;
  }

  file_len = GetFileSize(hFile, &file_len_high);
  if ((file_len <= 0) || (file_len > (1024 * 10)) || (file_len_high != 0)) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "File %s is invalid.", um_file);
    goto load_keys_and_fingerprint_error;
  }
#else
  stat(um_file, &file_status);
  file_len = file_status.st_size;

  if ((file_len <= 0) || (file_len > (1024 * 10)) ||
      (file_status.st_mode & S_IFMT) != S_IFREG) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "File %s is invalid.", um_file);
    goto load_keys_and_fingerprint_error;
  }
#endif
  */

  /*file_buffer = gcry_malloc_secure(file_len + 8);
  if (file_buffer == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Couldn't allocate space for file buffer!");
    goto load_keys_and_fingerprint_error;
    }*/



  /*file_handle = fopen(um_file, "r");
  if (file_handle == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Couldn't open %s for reading!", um_file);
    goto load_keys_and_fingerprint_error;
  }
*/

  for (i = 0; i < 11; i++) {

    newline_pos = strchr(data_ptr, '\n');
    if (newline_pos == NULL) {
      bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	     "%s is corrupt!", um_file);
      goto load_keys_and_fingerprint_error;
    }

    file_buffer = data_ptr;
    *newline_pos = '\0';
    data_ptr = newline_pos + 1;
    /*printf("read: [%s]\n", file_buffer);*/


    /*fgets(file_buffer, file_len, file_handle);*/

    /* Remove the ending newline. */
    /*file_buffer[ strlen(file_buffer) - 1 ] = '\0';*/


    equal_ptr = strstr(file_buffer, "=");
    if (equal_ptr == NULL) {
      bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	     "%s is corrupt!", um_file);
      goto load_keys_and_fingerprint_error;
    }

    /* The key is the text to the left of the equal sign, or the beginning of
     * the line that was just read. */
    key = file_buffer;

    /* The value is everything after the equal sign. */
    value = equal_ptr + 1;

    /* Change the equal sign to a NULL, so that the key & value are
     * separated. */
    *equal_ptr = '\0';

    /* See which key we're dealing with, then set a pointer to point to the
     * correct MPI (this cuts down on ugly, repetitive code).*/
    if (strcmp(key, "sign_public_p") == 0)
      mpi_ptr = &sign_public_p;
    else if (strcmp(key, "sign_public_q") == 0)
      mpi_ptr = &sign_public_q;
    else if (strcmp(key, "sign_public_g") == 0)
      mpi_ptr = &sign_public_g;
    else if (strcmp(key, "sign_public_y") == 0)
      mpi_ptr = &sign_public_y;
    else if (strcmp(key, "misc_info_1") == 0)
      mpi_ptr = &misc_sign_info_1;
    else if (strcmp(key, "misc_info_2") == 0)
      mpi_ptr = &misc_sign_info_2;
    else if (strcmp(key, "misc_info_3") == 0)
      mpi_ptr = &misc_sign_info_3;
    else if (strcmp(key, "misc_info_4") == 0)
      mpi_ptr = &misc_sign_info_4;
    else if (strcmp(key, "encrypt_public_p") == 0)
      mpi_ptr = &encrypt_public_p;
    else if (strcmp(key, "encrypt_public_g") == 0)
      mpi_ptr = &encrypt_public_g;
    else if (strcmp(key, "encrypt_public_y") == 0)
      mpi_ptr = &encrypt_public_y;
    else
      mpi_ptr = NULL;

    decode_len = base64_decode((unsigned char *)base64_buffer, sizeof(base64_buffer) - 8, (unsigned char *)value);
    if (decode_len < 1) {
      goto load_keys_and_fingerprint_error;
    }

    gcry_mpi_scan(mpi_ptr, GCRYMPI_FMT_STD, base64_buffer, decode_len, NULL);
    mpi_ptr = NULL;

  }


  /* Now we gotta read in the secret components to the signing & encrypting
   * key. */

  /* Read in the extra newline separator. */
  /*fgets(file_buffer, file_len, file_handle);*/


  newline_pos = strchr(data_ptr, '\n');
  if (newline_pos == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }
  
  file_buffer = data_ptr;
  *newline_pos = '\0';
  data_ptr = newline_pos + 1;
  /*printf("read: [%s]\n", file_buffer);*/


  /*fgets(file_buffer, file_len, file_handle);
    file_buffer[ strlen(file_buffer) - 1 ] = '\0';*/

  newline_pos = strchr(data_ptr, '\n');
  if (newline_pos == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }
  
  file_buffer = data_ptr;
  *newline_pos = '\0';
  data_ptr = newline_pos + 1;
  /*printf("read: [%s]\n", file_buffer);*/


  equal_ptr = strstr(file_buffer, "=");
  if (equal_ptr == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }


  decode_len = base64_decode((unsigned char *)base64_buffer, sizeof(base64_buffer) - 8, (unsigned char *)equal_ptr + 1);
  if (decode_len < 1) {
    goto load_keys_and_fingerprint_error;
  }
  gcry_mpi_scan(&sign_private_x, GCRYMPI_FMT_STD, base64_buffer, decode_len, NULL);




  /*fgets(file_buffer, file_len, file_handle);
    file_buffer[ strlen(file_buffer) - 1 ] = '\0';*/

  newline_pos = strchr(data_ptr, '\n');
  if (newline_pos == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }
  
  file_buffer = data_ptr;
  *newline_pos = '\0';
  data_ptr = newline_pos + 1;
  /*printf("read: [%s]\n", file_buffer);*/


  equal_ptr = strstr(file_buffer, "=");
  if (equal_ptr == NULL) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "%s is corrupt!", um_file);
    goto load_keys_and_fingerprint_error;
  }

  decode_len = base64_decode((unsigned char *)base64_buffer, sizeof(base64_buffer) - 8, (unsigned char *)equal_ptr + 1);
  if (decode_len < 1) {
    goto load_keys_and_fingerprint_error;
  }
  gcry_mpi_scan(&encrypt_private_x, GCRYMPI_FMT_STD, base64_buffer,
                decode_len, NULL);


  gcry_sexp_build(local_signing_keypair, &err,
                  "(key-data(public-key(dsa(p %m)(q %m)(g %m)(y %m)))"
                            "(private-key(dsa(p %m)(q %m)(g %m)(y %m)(x %m)))"
                            "(misc-key-info(pm1-factors %m %m %m %m)))",
                   sign_public_p, sign_public_q, sign_public_g, sign_public_y,
                   sign_public_p, sign_public_q, sign_public_g, sign_public_y,
                   sign_private_x, misc_sign_info_1, misc_sign_info_2,
		   misc_sign_info_3, misc_sign_info_4);


  gcry_sexp_build(local_encrypting_keypair, &err,
		  "(key-data(public-key(elg(p %m)(g %m)(y %m)))"
                           "(private-key(elg(p %m)(g %m)(y %m)(x %m))))",
                  encrypt_public_p, encrypt_public_g, encrypt_public_y,
                  encrypt_public_p, encrypt_public_g, encrypt_public_y,
                  encrypt_private_x);




  /*fclose(file_handle);*/


  /*gcry_free(file_buffer);  file_buffer = NULL;*/


  UM_MPI_RELEASE(sign_public_p);
  UM_MPI_RELEASE(sign_public_q);
  UM_MPI_RELEASE(sign_public_g);
  UM_MPI_RELEASE(sign_public_y);
  UM_MPI_RELEASE(misc_sign_info_1);
  UM_MPI_RELEASE(misc_sign_info_2);
  UM_MPI_RELEASE(misc_sign_info_3);
  UM_MPI_RELEASE(misc_sign_info_4);
  UM_MPI_RELEASE(encrypt_public_p);
  UM_MPI_RELEASE(encrypt_public_g);
  UM_MPI_RELEASE(encrypt_public_y);

  UM_MPI_RELEASE(sign_private_x);
  UM_MPI_RELEASE(encrypt_private_x);


  /*gcry_sexp_dump( *my_signing_keypair );
    gcry_sexp_dump( *my_encrypting_keypair );*/
  ret = gcry_pk_testkey(*local_signing_keypair);
  if (ret != 0) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Signing keypair is not valid: %d: %s", ret, gcry_strerror(ret));
    goto load_keys_and_fingerprint_error;
  }

  ret = gcry_pk_testkey(*local_encrypting_keypair);
  if (ret != 0) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Encrypting keypair is not valid: %d: %s", ret, gcry_strerror(ret));
    goto load_keys_and_fingerprint_error;
  }


  if (buffer_arg_is_null == 0)
    get_key_fingerprint_from_sexp(local_signing_keypair, buffer, buffer_len);


  if (key_args_are_null == 1) {
    UM_SEXP_RELEASE(*local_signing_keypair);
    UM_SEXP_RELEASE(*local_encrypting_keypair);

    gcry_free(local_signing_keypair);    local_signing_keypair = NULL;
    gcry_free(local_encrypting_keypair); local_encrypting_keypair = NULL;
  }


  /* Ensure that the key fingerprint stored as plaintext inside the file is
   * correct by recalculating the fingerprint and comparing. */
  get_key_fingerprint_from_sexp(local_signing_keypair,
				databuf, sizeof(databuf) - 8);

  /*printf("loaded keyfp: [%s]\nstored keyfp: [%s]\n",
    databuf, fingerprint_buffer);*/
  if (strcmp(databuf, fingerprint_buffer) != 0) {
    bc_log("load_keys_and_fingerprint", BC_LOG_ERROR,
	   "Loaded and stored key fingerprints do not match!: [%s][%s]",
	   databuf, fingerprint_buffer);
    goto load_keys_and_fingerprint_error;
  }


  return 1;

 load_keys_and_fingerprint_error:
    UM_MPI_RELEASE(sign_private_x);
    UM_MPI_RELEASE(encrypt_private_x);

    UM_MPI_RELEASE(sign_public_p);
    UM_MPI_RELEASE(sign_public_q);
    UM_MPI_RELEASE(sign_public_g);
    UM_MPI_RELEASE(sign_public_y);
    UM_MPI_RELEASE(misc_sign_info_1);
    UM_MPI_RELEASE(misc_sign_info_2);
    UM_MPI_RELEASE(misc_sign_info_3);
    UM_MPI_RELEASE(misc_sign_info_4);
    UM_MPI_RELEASE(encrypt_public_p);
    UM_MPI_RELEASE(encrypt_public_g);
    UM_MPI_RELEASE(encrypt_public_y);

    if (local_signing_keypair != NULL) {
      UM_SEXP_RELEASE(*local_signing_keypair);
    }

    if (local_encrypting_keypair != NULL ) {
      UM_SEXP_RELEASE(*local_encrypting_keypair);
    }

    UM_GCRY_FREE(local_signing_keypair);
    UM_GCRY_FREE(local_encrypting_keypair);
    return -1;
}



/* Returns 1 if an encrypting/signing key was loaded, and -1 if not. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int is_encryption_initialized() {
  if ((my_signing_keypair == NULL) || (my_encrypting_keypair == NULL))    
    return -1;
  else
    return 1;
}



/* Loads an ID from the specified directory.  Also sets the load/save directory
 * and database path.  Returns 1 on success, 0 if the password was invalid, 
 * and -1 on error. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int load_database_and_keys(char *directory_path, char *id, char *pw) {
#ifdef _WIN32
  HANDLE hFile = 0;
/*#else
  unsigned int file_len = 0;*/
#endif
  int ret = 0;

  char um_file[ 256 ];
  struct stat file_status;


  memset(um_file, 0, sizeof(um_file));
  memset(&file_status, 0, sizeof(file_status));


  my_signing_keypair = gcry_malloc_secure(sizeof(gcry_sexp_t));
  my_encrypting_keypair = gcry_malloc_secure(sizeof(gcry_sexp_t));

  if ((my_signing_keypair == NULL) || (my_encrypting_keypair == NULL)) {
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "Couldn't allocate space for signing & encrypting keypairs!");
    goto load_database_and_keys_error;
  }
  bc_strlcpy(um_file, directory_path, sizeof(um_file) - 8);
  if (ends_with_slash(directory_path) == -1)
    bc_strlcat(um_file, "/", sizeof(um_file) - 8);

  /* Quick!  While no one's looking!  Copy the Load/Save directory into
   *     this buffer!  Quick! */
  bc_strlcpy(um_load_save_dir, um_file, sizeof(um_load_save_dir) - 8);
  bc_strlcpy(um_database_path, um_file, sizeof(um_database_path) - 8);
  bc_strlcat(um_database_path, UM_DATABASE_FILENAME, sizeof(um_database_path) - 8);


  bc_strlcat(um_file, id, sizeof(um_file) - 8);
  bc_strlcat(um_file, ".bc", sizeof(um_file) - 8);

#ifdef _WIN32

  hFile = CreateFile(um_file, GENERIC_READ, FILE_SHARE_READ, NULL,
		     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "File %s is invalid!", um_file);
    goto load_database_and_keys_error;
  }
  CloseHandle(hFile);

  hFile = CreateFile(um_database_path, GENERIC_READ, FILE_SHARE_READ, NULL,
		     OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "Error while loading or initializing database: %s",
	   um_database_path );
    goto load_database_and_keys_error;
  }
  CloseHandle(hFile);

#else

  /* Make sure that the um_file is a regular file. */
  if (is_regular_file(um_file) != 1) {
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "File %s is does not exist/is a symlink!", um_file);
    goto load_database_and_keys_error;
  }


  /* Lets check the database too. */
  ret = is_regular_file(um_database_path);
  if (ret == 0) {  /* Database does not exist... */
    FILE *tmp = NULL;
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "%s has not been initialized...", um_database_path);
    tmp = fopen(um_database_path, "a");
    fclose(tmp);
  } else if (ret == 1) {  /* db is already initialized. */
    /* Bleh. */
  } else { /* We got a symlink on our hands. :( */
    bc_log("load_database_and_keys", BC_LOG_ERROR,
	   "ERROR:  %s is a symlink!  Load/Save directory rejected.",
	   um_database_path);
    goto load_database_and_keys_error;
  }

#endif

  if (my_signing_keypair != NULL) {
    UM_SEXP_RELEASE(*my_signing_keypair);
  }
  if (my_encrypting_keypair != NULL) {
    UM_SEXP_RELEASE(*my_encrypting_keypair);
  }

  ret = load_keys_and_fingerprint(um_file, pw, my_signing_keypair,
				  my_encrypting_keypair, NULL, 0);
  if (ret <= 0) {
    gcry_free(my_signing_keypair);     my_signing_keypair = NULL;
    gcry_free(my_encrypting_keypair);  my_encrypting_keypair = NULL;
  }
  return ret;


load_database_and_keys_error:
    gcry_free(my_signing_keypair);     my_signing_keypair = NULL;
    gcry_free(my_encrypting_keypair);  my_encrypting_keypair = NULL;
    return -1;
}



/* Generates a new encryption key & signing key, then writes it disk inside the
 * specified directory. */
/* Security audit trail:
 *    - Joe Testa: January 16th, 2005
 */
int generate_key(char *directory_entry_text,
		 char *id_combo_entry_text,
		 char *pw,
		 char *fingerprint_buffer,
		 unsigned int fingerprint_buffer_len) {
    gcry_sexp_t *local_signing_keypair = NULL;
    gcry_sexp_t *local_encrypting_keypair = NULL;
    char *filename = NULL;
    BCString *entropy_filename = NULL;
    unsigned int filename_len = 0;
    char *base64_buffer = NULL;
    const char *key_component_buf = NULL;
    size_t key_component_len = 0;
    /*FILE *output_file = NULL;*/
    unsigned int i = 0;
    char *output_data = NULL;
    size_t output_data_bufsize = 0;

    gcry_sexp_t keygen_params;
    gcry_sexp_t public_sign_key, private_sign_key;
    gcry_sexp_t public_encrypt_key, private_encrypt_key;
    gcry_sexp_t misc_sign_info;
    gcry_sexp_t key_component;
    gcry_md_hd_t message_digest;
    char hex[ 8 ];
    char digest_hex_bytes[ 128 ];


    memset(&keygen_params, 0, sizeof(gcry_sexp_t));
    memset(&public_sign_key, 0, sizeof(gcry_sexp_t));
    memset(&private_sign_key, 0, sizeof(gcry_sexp_t));
    memset(&public_encrypt_key, 0, sizeof(gcry_sexp_t));
    memset(&private_encrypt_key, 0, sizeof(gcry_sexp_t));
    memset(&misc_sign_info, 0, sizeof(gcry_sexp_t));
    memset(&key_component, 0, sizeof(gcry_sexp_t));
    memset(&message_digest, 0, sizeof(gcry_md_hd_t));
    memset(hex, 0, sizeof(hex));
    memset(digest_hex_bytes, 0, sizeof(digest_hex_bytes));




    local_signing_keypair = gcry_malloc_secure(sizeof(gcry_sexp_t));
    local_encrypting_keypair = gcry_malloc_secure(sizeof(gcry_sexp_t));

    if ((local_signing_keypair == NULL) || (local_encrypting_keypair == NULL)) {
      gcry_free(local_signing_keypair);  local_signing_keypair = NULL;
      gcry_free(local_encrypting_keypair);  local_encrypting_keypair = NULL;
      bc_log("generate_key", BC_LOG_ERROR,
	     "Couldn't allocate space for keypair!");
      goto generate_key_error;
    }


    gcry_sexp_build(&keygen_params, NULL,
		    "(genkey(dsa(nbits %d)))", 1024);
    if (gcry_pk_genkey(local_signing_keypair, keygen_params)) {
      bc_log("generate_key", BC_LOG_ERROR, "Couldn't generate keypair!");
      goto generate_key_error;
    }
    UM_SEXP_RELEASE(keygen_params);

    gcry_sexp_build(&keygen_params, NULL,
                     "(genkey(elg(nbits %d)))", 2048);
    if (gcry_pk_genkey(local_encrypting_keypair, keygen_params)) {
      bc_log("generate_key", BC_LOG_ERROR, "Couldn't generate keypair!");
      goto generate_key_error;
    }
    UM_SEXP_RELEASE(keygen_params);

    /* +3 for ".bc", +1 for terminating null. */
    filename_len = strlen(directory_entry_text) +
                   strlen(id_combo_entry_text) + 3 + 1;
    filename = calloc(filename_len + 16, sizeof(char)); /* 16 = paranoid factor. */
    if (filename == NULL) {
      gcry_free(local_signing_keypair);     local_signing_keypair = NULL;
      gcry_free(local_encrypting_keypair);  local_encrypting_keypair = NULL;
      bc_log("generate_key", BC_LOG_ERROR,
	     "Couldn't allocate space for filenames!");
      goto generate_key_error;
    }


    bc_strlcpy(filename, directory_entry_text, filename_len);
    bc_strlcat(filename, id_combo_entry_text, filename_len);
    bc_strlcat(filename, ".bc", filename_len);

    entropy_filename = bc_string_new(directory_entry_text);
    entropy_filename = bc_string_append(entropy_filename, id_combo_entry_text);
    entropy_filename = bc_string_append(entropy_filename, ".en");

    set_entropy_file(entropy_filename->str);

    bc_string_free(&entropy_filename, 1);  entropy_filename = NULL;

    /*
  if( 1 ) {
    unsigned char plaintext[] = "1234567890123456789012345678901234567890";
    gcry_mpi_t mpi_plaintext;
    gcry_sexp_t sexp_data;
    gcry_sexp_t *sign_result;
    int ret = 0;

    sign_result = ( gcry_sexp_t * )malloc( sizeof( gcry_sexp_t ) );
    memset(sign_result, 0, sizeof(gcry_sexp_t));


    printf( "version: %s\n", gcry_check_version(NULL) );

    ret = strlen(plaintext);
    gcry_mpi_scan(&mpi_plaintext, GCRYMPI_FMT_STD, plaintext, ret, NULL);
    gcry_sexp_build(&sexp_data, &ret,
                    "(data\n (flags raw)\n (value %m))\n",
                    mpi_plaintext);
    

    gcry_sexp_dump(sexp_data);
    ret = gcry_pk_sign(sign_result, sexp_data, *signing_keypair);

    gcry_sexp_dump(*sign_result);

    if ( ret != 0 )
      printf( "error:  %d: %s\n", ret, gcry_strerror(ret));
    else
      printf( "no error.\n" );

    exit( 0 );

    }*/



    /*gcry_sexp_dump( *signing_keypair );
      gcry_sexp_dump( *encrypting_keypair );*/


    /*output_file = fopen(filename, "w");

    if (output_file == NULL) {
      gcry_free(local_signing_keypair);  local_signing_keypair = NULL;
      gcry_free(local_encrypting_keypair);  local_encrypting_keypair = NULL;
      bc_log("generate_key", BC_LOG_ERROR,
	     "Couldn't open %s for writing!", filename);
      exit(1);
      }*/

    output_data_bufsize = 4096;
    output_data = (char *)gcry_malloc_secure(output_data_bufsize);
    memset(output_data, 0, output_data_bufsize);

    /* The first three bytes of out file format is "FV " so that we can
     * check if a decryption operation is correct.  The number that comes
     * after refers to the file version, currently at 1. */
    bc_strlcat(output_data, "FV 1\n", output_data_bufsize - 8);

    public_sign_key = gcry_sexp_find_token(*local_signing_keypair,
                                           "public-key", 0);
    misc_sign_info = gcry_sexp_find_token(*local_signing_keypair,
                                          "misc-key-info", 0);
    public_encrypt_key = gcry_sexp_find_token(*local_encrypting_keypair,
                                              "public-key", 0);

    /*gcry_sexp_dump( *encrypting_keypair );*/

    base64_buffer = gcry_malloc_secure(sizeof(unsigned char *) *
				       BASE64_BUFFER_SIZE);
    if (base64_buffer == NULL) {
      gcry_free(local_signing_keypair);  local_signing_keypair = NULL;
      gcry_free(local_encrypting_keypair);  local_encrypting_keypair = NULL;
      bc_log("generate_key", BC_LOG_ERROR,
	     "Couldn't allocate mem for base64_buffer!");
      goto generate_key_error;
    }


    for (i = 0; sign_key_fieldz[ i ] != 0; i++) {
      key_component = gcry_sexp_find_token(public_sign_key,
					   sign_key_fieldz[ i ], 0);
      key_component_buf = gcry_sexp_nth_data(key_component, 1,
                                            &key_component_len);

      base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_SIZE - 8,
		    (unsigned char *)key_component_buf, key_component_len);


      bc_strlcat(output_data, "sign_public_", output_data_bufsize - 8);
      bc_strlcat(output_data, sign_key_fieldz[ i ], output_data_bufsize - 8);
      bc_strlcat(output_data, "=", output_data_bufsize - 8);
      bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
      bc_strlcat(output_data, "\n", output_data_bufsize - 8);
      /*fprintf(output_file, "sign_public_%s=%s\n",
	sign_key_fieldz[ i ], base64_buffer);*/
      UM_SEXP_RELEASE(key_component);
    }
    encode_key_component(misc_sign_info, "pm1-factors",
                         base64_buffer, BASE64_BUFFER_LEN - 8);

    /*fprintf(output_file, "misc_info_1=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "misc_info_1=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);


    encode_key_component_nth(misc_sign_info, "pm1-factors", 2,
                         base64_buffer, BASE64_BUFFER_LEN - 8);
    /*fprintf(output_file, "misc_info_2=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "misc_info_2=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);



    encode_key_component_nth(misc_sign_info, "pm1-factors", 3,
                         base64_buffer, BASE64_BUFFER_LEN - 8);
    /*fprintf(output_file, "misc_info_3=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "misc_info_3=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);


    encode_key_component_nth(misc_sign_info, "pm1-factors", 4,
                         base64_buffer, BASE64_BUFFER_LEN - 8);
    /*fprintf(output_file, "misc_info_4=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "misc_info_4=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);

    UM_SEXP_RELEASE(misc_sign_info);
    UM_SEXP_RELEASE(public_sign_key);


    for (i = 0; encrypt_key_fieldz[ i ] != 0; i++) {
      key_component = gcry_sexp_find_token(public_encrypt_key,
                                           encrypt_key_fieldz[ i ], 0);
      key_component_buf = gcry_sexp_nth_data(key_component, 1,
                                             &key_component_len);

      base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_SIZE - 8,
		    (unsigned char * )key_component_buf, key_component_len);
      /*      fprintf(output_file, "encrypt_public_%s=%s\n",
	      encrypt_key_fieldz[ i ], base64_buffer);*/
      bc_strlcat(output_data, "encrypt_public_", output_data_bufsize - 8);
      bc_strlcat(output_data, encrypt_key_fieldz[ i ],output_data_bufsize - 8);
      bc_strlcat(output_data, "=", output_data_bufsize - 8);
      bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
      bc_strlcat(output_data, "\n", output_data_bufsize - 8);


      UM_SEXP_RELEASE(key_component);
    }
    UM_SEXP_RELEASE(public_encrypt_key);


    private_sign_key = gcry_sexp_find_token(*local_signing_keypair,
                                            "private-key", 0);
    private_encrypt_key = gcry_sexp_find_token( *local_encrypting_keypair,
                                                "private-key", 0 );

    key_component = gcry_sexp_find_token(private_sign_key, "x", 0);
    key_component_buf = gcry_sexp_nth_data(key_component, 1,
                                           &key_component_len);
    base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_SIZE - 8,
                  (unsigned char *)key_component_buf, key_component_len);

    /*fprintf(output_file, "\nsign_private_x=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "\nsign_private_x=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);


    UM_SEXP_RELEASE(key_component);
    UM_SEXP_RELEASE(private_sign_key);


    key_component = gcry_sexp_find_token(private_encrypt_key, "x", 0);
    key_component_buf = gcry_sexp_nth_data(key_component, 1,
                                           &key_component_len);
    base64_encode((unsigned char *)base64_buffer, BASE64_BUFFER_SIZE - 8,
		  (unsigned char *)key_component_buf, key_component_len);

    /*fprintf(output_file, "encrypt_private_x=%s\n", base64_buffer);*/
    bc_strlcat(output_data, "encrypt_private_x=", output_data_bufsize - 8);
    bc_strlcat(output_data, base64_buffer, output_data_bufsize - 8);
    bc_strlcat(output_data, "\n", output_data_bufsize - 8);


    UM_SEXP_RELEASE(key_component);
    UM_SEXP_RELEASE(private_encrypt_key);



    memset(&key_component, 0, sizeof(gcry_sexp_t));
    memset(base64_buffer, 0, BASE64_BUFFER_SIZE);

    gcry_free(base64_buffer);  base64_buffer = NULL;
    UM_SEXP_RELEASE(private_sign_key);
    UM_SEXP_RELEASE(private_encrypt_key);

    memset(&private_sign_key, 0, sizeof(gcry_sexp_t));
    memset(&private_encrypt_key, 0, sizeof(gcry_sexp_t));


    /*fclose(output_file);*/

    get_key_fingerprint_from_sexp(local_signing_keypair, fingerprint_buffer,
                                  fingerprint_buffer_len);

    encrypt_keyfile(local_signing_keypair, filename, output_data, pw);
    free(filename);  filename = NULL;

    UM_GCRY_FREE(output_data);

    /* Update the entropy file now.  Key generation isn't done very often, and
     * when it is, the user isn't doing anything else critical.  So this
     * ensures that our precious entropy won't be lost in case something goes
     * wrong later. */
    update_entropy_file();
    return 1;


 generate_key_error:
  return -1;
}



/* Adds a pending message to the pending message queue.  Note!:  The 'who'
 * must be dynamic memory, and becomes the property of the queue; it will be
 * free()'ed when not needed automatically.  The 'what' arg should also be
 * dynamicly allocated, but is not deleted automatically unless
 * clear_all_pending_messages() is called. */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
void add_pending_message(char *who, char *what) {
  struct PENDING_MESSAGE *new_pending_message = NULL;
  struct PENDING_MESSAGE *current = NULL;

  new_pending_message = calloc(1, sizeof(struct PENDING_MESSAGE));
  if (new_pending_message == NULL) {
    bc_log("add_pending_message", BC_LOG_ERROR, "Error:  out of memory!\n");
    return;
  }
  new_pending_message->who = who;
  new_pending_message->what = what;

  bc_log("add_pending_message", BC_LOG_INFO,
	 "add_pending_message(%s, %s);", who, what);

  if (pending_messages == NULL) {
    pending_messages = new_pending_message;
    return;
  }

  current = pending_messages;
  while(current->next != NULL)
    current = current->next;

  current->next = new_pending_message;
  new_pending_message->next = NULL;

}



/* Retrieves a queued pending message.  If flag is set to 1, the message
 * will be returned and removed from the pending message list (note that the
 * 'who' field, as provided in the call to add_pending_message,
 * shall be free()'ed in this case too!) */
/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
char *get_pending_message(char *who, unsigned int flag) {
  struct PENDING_MESSAGE *current = NULL;
  struct PENDING_MESSAGE *previous = NULL;
  char *retval = NULL;

  unsigned int found = 0;


  if (pending_messages == NULL)
    return NULL;

  /*printf("pending_messages:  [%s], [%s]\n", pending_messages->who,
    pending_messages->what);*/

  previous = NULL;
  current = pending_messages;
  while((current != NULL) && (found == 0)) {
    /*printf("current->who = [%s]\n", current->who);*/
    if (strcmp(current->who, who) == 0) {
      found = 1;
      retval = current->what;
    } else {
      previous = current;
      current = current->next;
    }
  }

  if ((found == 1) && (flag == 1)) {
    if (previous == NULL)
      pending_messages = current->next;
    else
      previous->next = current->next;

    free(current->who);  current->who = NULL;
    free(current); current = NULL;
  }

  return retval;
}



/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
void clear_all_pending_messages(void) {
  struct PENDING_MESSAGE *current = NULL;
  struct PENDING_MESSAGE *next = NULL;

  if (pending_messages == NULL)
    return;

  current = pending_messages;
  while (current != NULL) {
    next = current->next;

    free(current->who);  current->who = NULL;
    free(current->what);  current->what = NULL;

    free(current);  current = NULL;
    current = next;
  }

  pending_messages = NULL;
}



/* IMPORTANT:  In order to call this function safely, you must be holding
 * the 'pending_list_mutex' lock! (this condition is satisfied when inside
 * the 'send_handler' or 'get_handler') */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
void print_pending_list(void) {
  struct UM_BUDDY *current = NULL;

  bc_log("print_pending_list", BC_LOG_INFO, "xxxxxxxxxxxxxxx");
  current = pending_buddies_list;
  while(current != NULL) {
    bc_log("print_pending_list", BC_LOG_INFO, "[%s]: %u", current->screenname, current->state);
    current = current->next;
  }
  bc_log("print_pending_list", BC_LOG_INFO, "xxxxxxxxxxxxxxx");
}



/* This function runs in a seperate thread.  It monitors the
 * pending_buddies_list and ensures that no entry is older than 32 seconds.
 * If an old entry is found, it is purged from the list (and thus its
 * connection state is reset). */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
#ifdef _WIN32
DWORD WINAPI crypto_state_timeout_thread(void *nothing) {
#else
void *crypto_state_timeout_thread(void *nothing) {
#endif

  struct UM_BUDDY *previous = NULL;
  struct UM_BUDDY *current = NULL;
  char *ptr = NULL;

  char timedout_name[ 128 ];

  /*return NULL;
    printf("XXX!\n");*/

  memset(timedout_name, 0, sizeof(timedout_name));

  thread_continue_flag = 1;
  while(thread_continue_flag == 1) {
#ifdef _WIN32
    Sleep(2000);
#else
    sleep(2);
#endif

    bc_mutex_lock(pending_list_mutex);


    /* Debugging... */
    /*printf("----------------\n");
    current = pending_buddies_list;
    while(current != NULL) {
      	printf("[%s]: %u\n", current->screenname, current->state);
	current = current->next;
    }
    printf("----------------\n");*/


    previous = NULL;
    current = pending_buddies_list;

    while(current != NULL) {
      if ((time(NULL) - current->last_touched) >= 45) {
	bc_log("crypto_state_timeout_thread", BC_LOG_INFO, 
	       "\t[%s] has timed out; state: %u.",
	       current->screenname, current->state);

	/* Patch the linked list.  If there is no 'previous' link (ie: we're at
	 * the head of the list), then set the head pointer to NULL. */
	if (previous != NULL)
	  previous->next = current->next;
	else
	  pending_buddies_list = current->next;

	/* Delete any pending message that may be waiting... */
        ptr = get_pending_message(current->screenname, 1);
        if (ptr != NULL) {
          free(ptr);  ptr = NULL;
        }


        /* Copy the screen name so that we can write to the UI below.  We
	 * don't write to the UI now because we're in a critical section,
	 * and its bad to hold multiple locks concurrently (to modify
	 * the UI from a thread necessitates aquiring the GDK lock). */
	bc_strlcpy(timedout_name, current->screenname,
		   sizeof(timedout_name) - 8);


	/* Notify the calling prog that this buddy timed out, but we simply
	 * reset the buddy. */
	change_state(current, STATE_FLAG_SET, TIMEDOUT);
	current->state = BUDDY_RESET;


	wipe_buddy(current);
	free(current);  current = NULL;

	/* Since we just removed a timed-out buddy, don't bother looking for
	 * more until after we sleep again.  This keeps the logic here simple.
	 * Not a problem unless the pending list is large, which does not
	 * normally occur. */
	current = NULL;

      } else {
	/*printf("\t[%s] is good for now: %d!\n", current->screenname,
	  (time(NULL) - current->last_touched));*/
	previous = current;
	current = current->next;
      }
    }
    bc_mutex_unlock(pending_list_mutex);

    /* Check if the 'timedout_name' array is empty or not.  Checking the
     * first byte is sufficient since we're using memset below to clear the
     * entire array. */
    if (*timedout_name != '\0') {
      bc_log("crypto_state_timeout_thread", BC_LOG_INFO, 
	     "Timeout expired; connection setup has failed.");
      memset(timedout_name, 0, sizeof(timedout_name));
    }

  }

#ifdef _WIN32
  ExitThread(0);
#else
  pthread_exit(0);
#endif


}

void kill_state_timeout_thread(BC_THREAD **thread) {
  BC_THREAD *t = NULL;
#ifdef _WIN32
  unsigned int continue_flag = 0;
  unsigned int i = 0;
  DWORD ecode = 0;
  BOOL ret = FALSE;
#endif

  thread_continue_flag = 0;

  t = *thread;
  if (t == NULL)
    return;

#ifdef _WIN32
  /* Wait for the thread to exit for at most 5 seconds before reporting an
   * error and moving on. */
  continue_flag = 1;
  for (i = 0; (i < 5) && (continue_flag == 1); i++) {
    ret = GetExitCodeThread(*t, &ecode);
    if ((ret == FALSE) || (ecode == STILL_ACTIVE))
      Sleep(1000);
    else
      continue_flag = 0;
  }

  if (continue_flag == 1) {
    bc_log("kill_state_timeout_thread", BC_LOG_ERROR,
	   "Thread did not exit!");
  }
#else
  pthread_join(*t, NULL);
#endif

  free(t);  t = NULL;
  *thread = NULL;

}

/* Retrieves a raw listing of the specified database. */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
void get_database_raw_list(char *database_path,
			   BCSList **umid_list, BCSList **fingerprint_list) {
  FILE *handle = NULL;
  char *pipe_ptr = NULL;
  char *temp_umid = NULL;
  char *temp_fingerprint = NULL;

  char file_line[ 128 ];

  memset(file_line, 0, sizeof(file_line));


  handle = fopen(database_path, "r");
  if (handle == NULL) {
    bc_log("get_database_raw_list", BC_LOG_ERROR,
	   "ERROR WHILE OPENING DATABASE!");
    return;
  }

  while(fgets(file_line, sizeof(file_line) - 8, handle) != NULL) {
    /* Remove the ending newline. */
    file_line[ strlen(file_line) - 1 ] = '\0';


    pipe_ptr = strchr(file_line, '|');
    if (pipe_ptr == NULL) {
      bc_log("get_database_raw_list", BC_LOG_ERROR, 
	     "ERROR: Pipe not found in database line:  [%s]", file_line);
      fclose(handle);
      *umid_list = NULL;
      *fingerprint_list = NULL;
      return;
    }

    temp_umid = file_line;
    temp_fingerprint = pipe_ptr + 1;
    *pipe_ptr = '\0';

    *umid_list = bc_slist_append(*umid_list, strdup(temp_umid));
    *fingerprint_list = bc_slist_append(*fingerprint_list,
				      strdup(temp_fingerprint));
  }

  fclose(handle);

}


/* Frees the raw database lists generated by 'get_database_raw_list()' */
void free_database_raw_list(BCSList **umid_list, BCSList **fingerprint_list) {
  BCSList *ids = NULL;
  BCSList *fps = NULL;
  unsigned int len = 0;
  unsigned int i = 0;
  char *ptr = NULL;

  if ((umid_list == NULL) || (fingerprint_list == NULL))
    return;

  ids = *umid_list;
  fps = *fingerprint_list;

  if ((ids == NULL) || (fps == NULL)) {
    *umid_list = NULL;
    *fingerprint_list = NULL;
    return;
  }

  len = bc_slist_length(ids);
  for(i = 0; i < len; i++) {
    ptr = bc_slist_nth_data(ids, i);
    free(ptr);  ptr = NULL;
  }
  bc_slist_free(&ids);

  
  len = bc_slist_length(fps);
  for(i = 0; i < len; i++) {
    ptr = bc_slist_nth_data(fps, i);
    free(ptr);  ptr = NULL;
  }
  bc_slist_free(&fps);

  
  *umid_list = NULL;
  *fingerprint_list = NULL;
}




/* Retrieves the sequence number needed to send a single message to the
 * specified buddy.  The sequence number is automatically incremented by 1. */
/* NOTE!:  The return value of this function must be deallocated eventually
 * using gcry_free(). */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
unsigned char *get_sequence_number(struct UM_BUDDY *buddy) {
  unsigned char *ptr = NULL;

  gcry_mpi_add(*(buddy->my_sequence_number),
	       *(buddy->my_sequence_number), mpi_one);
  gcry_mpi_aprint(GCRYMPI_FMT_USG, &ptr, NULL, *(buddy->my_sequence_number));

  return ptr;
}



/* Checks that a buddy's sequence number is equal to its expected value, and
 * returns 1 if it is (-1 if otherwise).  If the sequence number is not legit,
 * the connection should be aborted immediately! (Update:  this doesn't
 * actually happen, however: the message that had the bogus sequence number
 * just gets dropped silently... hmphf.  Good enough for now...) */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 */
int check_sequence_number(struct UM_BUDDY *buddy,
			  unsigned char *sequence_number,
			  size_t sequence_number_len) {
  int ret = 0;
  size_t buf_len = 0;

  gcry_mpi_t mpi_to_check;
  unsigned char buf[ 256 ];


  memset(&mpi_to_check, 0, sizeof(gcry_mpi_t));
  memset(buf, 0, sizeof(buf));


  mpi_to_check = gcry_mpi_snew(SESSION_KEY_BYTE_SIZE * 8);
  ret = gcry_mpi_scan(&mpi_to_check, GCRYMPI_FMT_USG,
		      (unsigned char *)sequence_number, sequence_number_len, NULL);
  if (ret != 0) {
    bc_log("check_sequence_number", BC_LOG_ERROR, "check: can't scan!");
    goto check_sequence_number_error;
  }

  /*gcry_mpi_print(GCRYMPI_FMT_USG, buf, sizeof(buf) - 8, &buf_len,
		 mpi_to_check);
  hex_dump("Seq num", buf, buf_len);*/

  gcry_mpi_add(*(buddy->his_sequence_number),
	       *(buddy->his_sequence_number), mpi_one);

  if (gcry_mpi_cmp(*(buddy->his_sequence_number), mpi_to_check) != 0) {
    size_t buf_len2 = 0;
    unsigned char buf2[ 256 ];

    memset(buf2, 0, sizeof(buf2));

    gcry_mpi_print(GCRYMPI_FMT_USG, buf, sizeof(buf) - 8, &buf_len,
		   mpi_to_check);

    gcry_mpi_print(GCRYMPI_FMT_USG, buf2, sizeof(buf2) - 8, &buf_len2,
		   *(buddy->his_sequence_number));

    bc_log("check_sequence_number", BC_LOG_ERROR,
	   "check_sequence_number: failed.");
    hex_dump("Expected", (char *)buf2, buf_len2);
    hex_dump("Got", (char *)buf, buf_len);

    goto check_sequence_number_error;   
  }

  UM_MPI_RELEASE(mpi_to_check);
  return 1;

 check_sequence_number_error:
  UM_MPI_RELEASE(mpi_to_check);
  return -1;
}



/* This function is called when a user right-clicks on a name in the buddy
 * list and chooses the 'Reset Encrypted Connection' option. */
/*void reset_crypto_cb(GtkWidget *w, GaimBuddy *b) {
  struct UM_BUDDY *buddy = NULL;
  gchar *sn = (gchar *)gaim_normalize(NULL, b->name);

  * Check if the buddy is in the pending list, and remove if so. *
  bc_mutex_lock(pending_list_mutex);
  buddy = find_buddy_in_pending_list(sn);
  bc_mutex_unlock(pending_list_mutex);

  if (buddy != NULL) {
    unsigned char *pending_message = NULL;

    bc_mutex_lock(pending_list_mutex);
    remove_buddy_from_pending_list(buddy);
    * If the buddy signed off while there was still a pending message
     * for them, remove it now. *
    pending_message = get_pending_message(sn, 1);
    if (pending_message != NULL) {
      free(pending_message);  pending_message = NULL;
    }
    bc_mutex_unlock(pending_list_mutex);


    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
    gaim_debug(GAIM_DEBUG_INFO, "gtkblist-crypto",
	       "Removed [%s] buddy from pending list.", sn);
  }

  * Check if the buddy is in the validated list, and remove if so. *
  buddy = find_buddy_in_validated_list(sn);
  if (buddy != NULL) {
    remove_buddy_from_validated_list(buddy);
    wipe_buddy(buddy);  free(buddy);  buddy = NULL;
    gaim_debug(GAIM_DEBUG_INFO, "gtkblist-crypto",
	       "Removed [%s] buddy from validated list.", sn);
  }

  um_system_error_message(sn, "Encrypted connection reset.");
}*/

/*int show_crypto_reset(GaimBuddy *b) {
  gchar *sn = (gchar *)gaim_normalize(NULL, b->name);
  if ((find_buddy_in_validated_list(sn) != NULL) ||
      (find_buddy_in_pending_list(sn) != NULL))
    return 1;
  else
    return -1;
    }*/




/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 */
int buddy_fingerprint_is_good(MXFER *mxfer, char *who) {
  struct UM_BUDDY *buddy = NULL;
  int ret = 0;

  buddy = find_buddy_in_pending_list(bc_normalize(who));
  if (buddy == NULL) {
    mxfer->type = MXFER_TYPE_ERROR;
    mxfer->error = bc_string_new("Internal error.");
    bc_log("buddy_fingerprint_is_good", BC_LOG_ERROR,
	   "Internal error: invalid state.");
    return -1;
  }

  /* Now that the buddy's keys have been read in successfully, see if we
   * have sent our public keys already.  We have already if and only if
   * we initialized the handshake, but we've been silent (until now), if
   * the buddy initialized it.  */
  if ((buddy->state & SENT_PUBKEYS) == 0) {

    ret = send_public_keys(mxfer, bc_normalize(who));
    if (ret != 1) {
      change_state(buddy, STATE_FLAG_SET, 0);
      buddy->state = 0;
      remove_buddy_from_pending_list(buddy);
      wipe_buddy(buddy);  free(buddy);  buddy = NULL;
      return -1;
    } else {
      change_state(buddy, STATE_FLAG_ADD, SENT_PUBKEYS);
      buddy->state |= SENT_PUBKEYS;
      bc_log("buddy_fingerprint_is_good", BC_LOG_INFO,
	     "Sent public keys(%s)...", bc_normalize(who));
    }
    
  }
  

  if ((buddy->state & SENT_SESSION_KEY) == 0) {
    ret = send_session_key(mxfer, buddy);
    if (ret != 1) {
      bc_log("buddy_fingerprint_is_good", BC_LOG_ERROR,
	     "ERROR WHILE SENDING SESSION KEY, YO!\n");
      change_state(buddy, STATE_FLAG_SET, 0);
      buddy->state = 0;
      remove_buddy_from_pending_list(buddy);
      wipe_buddy(buddy);  free(buddy);  buddy = NULL;
      return -1;
    } else {
      change_state(buddy, STATE_FLAG_ADD, SENT_SESSION_KEY);
      buddy->state |= SENT_SESSION_KEY;
      bc_log("buddy_fingerprint_is_good", BC_LOG_INFO,
	     "Sent session key (%s)...", bc_normalize(who));
    }
  }

  return 1;
}



/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 */
int decrypt_keyfile(char *path, char *pw,
		    char *storedkeyfpbuf, size_t storedkeyfpbufsize,
		    char *databuf, size_t databufsize) {
  gcry_error_t ret = 0;

  gcry_cipher_hd_t cipher;
  char hash[ 48 ];

#ifdef _WIN32
  HANDLE hFile = NULL;
  DWORD file_len_high = 0;
  DWORD file_len = 0;
  DWORD read_len = 0;
#else
  FILE *hFile = NULL;
  size_t read_len = 0;

  struct stat filestat;

  memset(&filestat, 0, sizeof(struct stat));
#endif

  memset(&cipher, 0, sizeof(gcry_cipher_hd_t));
  memset(hash, 0, sizeof(hash));

#ifdef _WIN32
  hFile = CreateFile(path, GENERIC_READ, FILE_SHARE_READ, NULL,
                     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "Error opening %s.", path);
    goto decrypt_keyfile_error;
  }

  file_len = GetFileSize(hFile, &file_len_high);
  if ((file_len <= 0) || (file_len > (1024 * 4)) || (file_len_high != 0)) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR,
           "File %s is invalid.", path);
     goto decrypt_keyfile_error;
  }

  if (!ReadFile(hFile, storedkeyfpbuf, MY_MIN(59, storedkeyfpbufsize),
                &read_len, NULL)) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "ReadFile failed.");
    goto decrypt_keyfile_error;
  }

/*bc_log("decrypt_keyfile", BC_LOG_ERROR, "storedkey: [%s]", storedkeyfpbuf);*/

  if (!ReadFile(hFile, databuf,
                MY_MIN(file_len - 59, databufsize), &read_len, NULL)) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "ReadFile failed.");
    goto decrypt_keyfile_error;
  }

/*bc_log("decrypt_keyfile", BC_LOG_ERROR, "more read: %d", read_len);*/

  CloseHandle(hFile);  hFile = 0;

#else

  stat(path, &filestat);
  if (filestat.st_size > 4096) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "%s is too large!", path);
    goto decrypt_keyfile_error;
  }

  if (!(hFile = fopen(path, "r"))) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "fopen failed.");
    goto decrypt_keyfile_error;
  }


  read_len = fread(storedkeyfpbuf, sizeof(char),
                   MY_MIN(59, storedkeyfpbufsize), hFile);
  if (read_len != 59) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "fread failed: %d|%d|%d", read_len,
	   MY_MIN(59, storedkeyfpbufsize), storedkeyfpbufsize);
    goto decrypt_keyfile_error;
  }

  read_len = fread(databuf, sizeof(char),
                   MY_MIN(filestat.st_size - 59, databufsize), hFile);

  fclose(hFile);  hFile = NULL;
#endif

  gcry_md_hash_buffer(GCRY_MD_SHA256, hash, pw, strlen(pw));

  ret = gcry_cipher_open(&cipher, GCRY_CIPHER_AES256,
                         GCRY_CIPHER_MODE_ECB, GCRY_CIPHER_SECURE);
  if (ret != 0) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_open failed.");
    goto decrypt_keyfile_error;
  }

  ret = gcry_cipher_setkey(cipher, hash, MY_MIN(32, sizeof(hash)));
  if (ret != 0) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_setkey failed.");
    goto decrypt_keyfile_error;
  }

  ret = gcry_cipher_decrypt(cipher, (unsigned char *)databuf, read_len, NULL, 0);
  if (ret != 0) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_encrypt failed.");
    goto decrypt_keyfile_error;
  }

  gcry_cipher_close(cipher);


  if (um_unpad_message(databuf, read_len, databuf, read_len) < 0) {
    bc_log("decrypt_keyfile", BC_LOG_ERROR, "um_unpad_message failed.");
    goto decrypt_keyfile_error;
  }

  /*bc_log("decrypt_keyfile", BC_LOG_ERROR, "returning: [%s]", databuf);*/
  /*printf("Returning: [%s]\n", databuf);*/
  return 1;

 decrypt_keyfile_error:
  return 0;
}



/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 */
int encrypt_keyfile(gcry_sexp_t *sign_key, char *path, char *data, char *pw) {
  gcry_error_t ret = 0;
  char *padded_keyfile = NULL;
#ifdef _WIN32
  HANDLE hFile = NULL;
  DWORD bytesWritten = 0;
#else
  FILE *hFile = NULL;
#endif
  int padded_keyfile_len = 0;
  int padded_keyfile_bufsize = 0;

  gcry_cipher_hd_t cipher;
  char hash[ 48 ];
  char fingerprint_buffer[ 128 ];

  memset(&cipher, 0, sizeof(gcry_cipher_hd_t));
  memset(hash, 0, sizeof(hash));
  memset(fingerprint_buffer, 0, sizeof(fingerprint_buffer));


  get_key_fingerprint_from_sexp(sign_key, fingerprint_buffer,
				sizeof(fingerprint_buffer) - 8);


  padded_keyfile_bufsize = (int)((strlen(data) * 3) / 2) + 8;
  padded_keyfile = (char *)gcry_malloc_secure(padded_keyfile_bufsize);

  padded_keyfile_len = um_pad_message(padded_keyfile,
				      padded_keyfile_bufsize, data);
  if (padded_keyfile_len == 0) {
    bc_log("encrypt_keyfile", BC_LOG_ERROR, "um_padded_message failed.");
    goto encrypt_keyfile_error;
  }

  gcry_md_hash_buffer(GCRY_MD_SHA256, hash, pw, strlen(pw));

/*hex_dump("HASH OF PASSWORD", hash, 32);*/

  ret = gcry_cipher_open(&cipher, GCRY_CIPHER_AES256,
			 GCRY_CIPHER_MODE_ECB, GCRY_CIPHER_SECURE);
  if (ret != 0) {
    bc_log("encrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_open failed.");
    goto encrypt_keyfile_error;
  }

  ret = gcry_cipher_setkey(cipher, hash, 32);
  if (ret != 0) {
    bc_log("encrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_setkey failed.");
    goto encrypt_keyfile_error;
  }

  ret = gcry_cipher_encrypt(cipher, (unsigned char *)padded_keyfile, padded_keyfile_len,
			    NULL, 0);
  if (ret != 0) {
    bc_log("encrypt_keyfile", BC_LOG_ERROR, "gcry_cipher_encrypt failed: %s",
           gcry_strerror(ret));
    goto encrypt_keyfile_error;
  }

  gcry_cipher_close(cipher);

#ifdef _WIN32
  hFile = CreateFile(path, GENERIC_WRITE, FILE_SHARE_WRITE, NULL,
                     CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if (!WriteFile(hFile, fingerprint_buffer, 59, &bytesWritten, NULL))
     goto encrypt_keyfile_error;

  if (!WriteFile(hFile, padded_keyfile, padded_keyfile_len,&bytesWritten,NULL))
     goto encrypt_keyfile_error;

  CloseHandle(hFile);
#else
  hFile = fopen(path, "w+");
  fwrite(fingerprint_buffer, 59, sizeof(char), hFile);
  fwrite(padded_keyfile, padded_keyfile_len, sizeof(char), hFile);
  fclose(hFile);  hFile = NULL;
#endif

  UM_GCRY_FREE(padded_keyfile);
  return 1;

 encrypt_keyfile_error:
  UM_GCRY_FREE(padded_keyfile);
  return 0;
}



/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 */
unsigned int get_fingerprint_from_keyfile(char *filepath,
                                          char *buf, size_t bufsize) {
  FILE *hFile = NULL;
  size_t bytesRead = 0;


  if (is_regular_file(filepath) != 1)
    return 0;

  if (!(hFile = fopen(filepath, "r")))
    return 0;

  bytesRead = fread(buf, sizeof(char), MY_MIN(59, bufsize), hFile);
  fclose(hFile);

  if (bytesRead != 59)
    return 0;

  if (bufsize > 60) {
    buf[ 59 ] = '\0';
    return 1;
  } else
    return 0;

}


/* Encrypts a file given a path to an input file and an output target (with
 * respect to the key belonging to a particular buddy. */
/* Security audit trail:
 */
#define ENCRYPT_BUFFER_SIZE (64 * 1024)
#define DECRYPT_BUFFER_SIZE ENCRYPT_BUFFER_SIZE
int encrypt_file(char *who, char *infile, char *outfile) {
  gcry_error_t ret = 0;
  struct UM_BUDDY *buddy = NULL;
  char *buffer = NULL;
  size_t buffer_size = 0;
  unsigned int continue_flag = 0;
  unsigned int i = 0;
  unsigned char *hmac_digest = NULL;
#ifdef _WIN32
  HANDLE hFileIn = NULL;
  HANDLE hFileOut = NULL;
  DWORD infile_len = 0;
  DWORD file_len_high = 0;
  DWORD n = 0;
#else
  u_int32_t infile_len = 0;
  FILE *hFileIn = NULL;
  /*FILE *hFileOut = NULL;*/
  int hFileOut = -1;
  int n = 0;
#endif

  /*char temp_key[] = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";
  char temp_iv[] = "AAAAAAAAAAAAAAAAXAAAAAAAAAAAAAAA";
  char temp_hmac[] = "BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB";*/

  gcry_cipher_hd_t cipher;
  gcry_md_hd_t md;
  struct stat thestat;


  memset(&cipher, 0, sizeof(gcry_cipher_hd_t ));
  memset(&md, 0, sizeof(gcry_md_hd_t));
  memset(&thestat, 0, sizeof(struct stat));


  /* Make sure that the file to encrypt is a regular file. */
  if (is_regular_file(infile) != 1) {
    bc_log("encrypt_file", BC_LOG_ERROR, "%s is not a regular file!", infile);
    goto encrypt_file_error;
  }


  /* Make sure that the output file does not yet exist. */
  if (is_regular_file(outfile) != 0) {
    bc_log("encrypt_file", BC_LOG_ERROR, "%s already exists!", outfile);
    goto encrypt_file_error;
  }


  buddy = find_buddy_in_validated_list(who);
  if (buddy == NULL)
    goto encrypt_file_error;

  if (gcry_cipher_open(&cipher, GCRY_CIPHER_AES256,
		       GCRY_CIPHER_MODE_CTR, GCRY_CIPHER_SECURE)) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not open cipher.");
    goto encrypt_file_error;
  }

  if (gcry_cipher_setctr(cipher, /*temp_iv*/ buddy->session_iv, SYMMETRIC_BLOCK_SIZE)) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not set counter.");
    goto encrypt_file_error;
  }

  if (gcry_cipher_setkey(cipher, /*temp_key*/ buddy->session_key, SESSION_KEY_BYTE_SIZE)) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not set cipher key.");
    goto encrypt_file_error;
  }

  ret = gcry_md_open(&md, GCRY_MD_SHA1, GCRY_MD_FLAG_SECURE|GCRY_MD_FLAG_HMAC);
  if (ret != 0) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not open digest.");
    goto encrypt_file_error;
  }

  ret = gcry_md_setkey(md, /*temp_hmac*/ buddy->session_hmac, 32);
  if (ret != 0) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not set md key.");
    goto encrypt_file_error;
  }

#ifdef _WIN32
  hFileIn = CreateFile(infile, GENERIC_READ, FILE_SHARE_READ, NULL,
                       OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  hFileOut = CreateFile(outfile, GENERIC_WRITE, 0, NULL,
                        CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);

  infile_len = GetFileSize(hFileIn, &file_len_high);
  if ((hFileIn == INVALID_HANDLE_VALUE) || (hFileOut == INVALID_HANDLE_VALUE)) {
#else
  stat(infile, &thestat);
  infile_len = thestat.st_size;

  hFileIn = fopen(infile, "r");
  /*hFileOut = fopen(outfile, "w");*/
  hFileOut = open(outfile, O_WRONLY | O_CREAT | O_EXCL, 0600); 
  if ((hFileIn == NULL) || (hFileOut < 0)) {
#endif
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not open files.");
    goto encrypt_file_error;
  }

  /* Convert to network order so that systems with differing endian-ness can
   * xfer files appropriately. */
  infile_len = htonl(infile_len);

  /* Write the original file's size first, so we know how big the decrypted
   * file should be (since it will be padded to 32 bytes below). */
#ifdef _WIN32
  WriteFile(hFileOut, &infile_len, 4, &n, NULL);
#else
  /*fwrite(&infile_len, sizeof(u_int32_t), 1, hFileOut);*/
  write(hFileOut, &infile_len, sizeof(u_int32_t));
#endif

  /* Default the buffer size to 65K.  If the file we're encrypting is smaller
   * than that, use the smallest 32-byte-aligned buffer possible.  This can
   * be done in a more efficient way... */
  buffer_size = ENCRYPT_BUFFER_SIZE;
  while (buffer_size > (infile_len + 32))
    buffer_size = buffer_size - 32;



  buffer = (char *)gcry_malloc_secure(buffer_size + 8);
  if (buffer == NULL) {
    bc_log("encrypt_file", BC_LOG_ERROR, "Could not allocate.");
    goto encrypt_file_error;
  }

  continue_flag = 1;
  while(continue_flag == 1) {
#ifdef _WIN32
    if (!ReadFile(hFileIn, buffer, buffer_size, &n, NULL))
      n = -1;
#else
    n = fread(buffer, 1, buffer_size, hFileIn);
#endif
    if (n <= 0)
      continue_flag = 0;
    else {
      gcry_md_write(md, buffer, n);

      /* If the number of bytes read does not equal the size of the buffer,
       * then we blank out the end of the buffer. */
      if (n != buffer_size) {
	/*printf("n != buffersize: %d\n", n);*/
	for (i = n; i < buffer_size; i++)
	  buffer[ i ] = '\0';
      }

      ret = gcry_cipher_encrypt(cipher, (unsigned char *)buffer, buffer_size, NULL, 0);
      if (ret != 0) {
	bc_log("encrypt_file", BC_LOG_ERROR, "Error while encrypting...");
	goto encrypt_file_error;
      }

#ifdef _WIN32
      WriteFile(hFileOut, buffer, buffer_size, &n, NULL);
#else
      /*fwrite(buffer, 1, buffer_size, hFileOut);*/
      write(hFileOut, buffer, buffer_size);
#endif
    }
  }

  /* Write the file's HMAC at the end of the output file. */
  hmac_digest = gcry_md_read(md, GCRY_MD_SHA1);

#ifdef _WIN32
  WriteFile(hFileOut, hmac_digest, 20, &n, NULL);

  CloseHandle(hFileOut);  hFileOut = NULL;
  CloseHandle(hFileIn);   hFileIn = NULL;
#else
  /*fwrite(hmac_digest, 1, 20, hFileOut);*/
  write(hFileOut, hmac_digest, 20);
  close(hFileOut);  hFileOut = -1;
  fclose(hFileIn);  hFileIn = NULL;
#endif

  gcry_cipher_close(cipher);  cipher = NULL;
  gcry_md_close(md);          md = NULL;
  gcry_free(buffer);          buffer = NULL;
  return 1;

 encrypt_file_error:
#ifdef _WIN32
  if (hFileOut != NULL) {
    CloseHandle(hFileOut);  hFileOut = NULL;
  }
  if (hFileIn != NULL) {
    CloseHandle(hFileIn);   hFileIn = NULL;
  }
#else
  if (hFileOut > -1) {
    close(hFileOut);  hFileOut = -1;
  }
  if (hFileIn != NULL) {
    fclose(hFileIn);   hFileIn = NULL;
  }
#endif
  gcry_cipher_close(cipher);  cipher = NULL;
  gcry_md_close(md);          md = NULL;
  gcry_free(buffer);          buffer = NULL;
  return -1;
}


/* Decrypts a file given a path to an input file and an output target (with
 * respect to the key belonging to a particular buddy. */
/* Security audit trail:
 */
int decrypt_file(char *who, char *infile, char *outfile) {
  gcry_error_t ret = 0;
  struct UM_BUDDY *buddy = NULL;
  char *buffer = NULL;
  size_t buffer_size = 0;
#ifdef _WIN32
  HANDLE hFileIn = NULL;
  HANDLE hFileOut = NULL;
  DWORD infile_len = 0;
  DWORD outfile_len = 0;
  DWORD file_len_high = 0;
  DWORD n = 0;
  DWORD x = 0;
#else
  u_int32_t infile_len = 0;
  u_int32_t outfile_len = 0;
  FILE *hFileIn = NULL;
  /*FILE *hFileOut = NULL;*/
  int hFileOut = -1;
  int n = 0;
#endif
  unsigned int i = 0;
  unsigned char *hmac_digest = NULL;
  size_t body_len = 0;
  size_t plaintext_left_to_read = 0;

  /*char temp_key[] = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";
  char temp_iv[] = "AAAAAAAAAAAAAAAAXAAAAAAAAAAAAAAA";
  char temp_hmac[] = "BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB";*/

  gcry_cipher_hd_t cipher;
  gcry_md_hd_t md;
  struct stat thestat;


  memset(&cipher, 0, sizeof(gcry_cipher_hd_t ));
  memset(&md, 0, sizeof(gcry_md_hd_t));
  memset(&thestat, 0, sizeof(struct stat));


  /* Make sure that the file to encrypt is a regular file. */
  if (is_regular_file(infile) != 1) {
    bc_log("decrypt_file", BC_LOG_ERROR, "%s is not a regular file!", infile);
    goto decrypt_file_error;
  }


  /* Make sure that the output file does not yet exist. */
  if (is_regular_file(outfile) != 0) {
    bc_log("decrypt_file", BC_LOG_ERROR, "%s already exists!", outfile);
    goto decrypt_file_error;
  }


  /* We need the buddy so that we can decrypt with the corresponding session
   * key. */
  buddy = find_buddy_in_validated_list(who);
  if (buddy == NULL) {
    bc_log("decrypt_file", BC_LOG_ERROR, "No connection to %s exists; no key to decrypt with.", who);
    goto decrypt_file_error;
  }


  /* Create a cipher to decrypt with. */
  if (gcry_cipher_open(&cipher, GCRY_CIPHER_AES256,
		       GCRY_CIPHER_MODE_CTR, GCRY_CIPHER_SECURE)) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not open cipher.");
    goto decrypt_file_error;
  }


  /* We're going to use CTR mode. */
  if (gcry_cipher_setctr(cipher, /*temp_iv*/ buddy->session_iv, SYMMETRIC_BLOCK_SIZE)) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not set counter.");
    goto decrypt_file_error;
  }


  /* Give the cipher our session key. */
  if (gcry_cipher_setkey(cipher, /*temp_key*/ buddy->session_key, SESSION_KEY_BYTE_SIZE)) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not set cipher key.");
    goto decrypt_file_error;
  }


  /* Create a message digest to calculate the HMAC with. */
  ret = gcry_md_open(&md, GCRY_MD_SHA1, GCRY_MD_FLAG_SECURE|GCRY_MD_FLAG_HMAC);
  if (ret != 0) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not open digest.");
    goto decrypt_file_error;
  }


  /* Give the message digest our HMAC key. */
  ret = gcry_md_setkey(md, /*temp_hmac*/ buddy->session_hmac, 32);
  if (ret != 0) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not set md key.");
    goto decrypt_file_error;
  }


#ifdef _WIN32
  hFileIn = CreateFile(infile, GENERIC_READ, FILE_SHARE_READ, NULL,
                       OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  hFileOut = CreateFile(outfile, GENERIC_WRITE, 0, NULL,
                        CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);

  infile_len = GetFileSize(hFileIn, &file_len_high);

  if ((hFileIn == INVALID_HANDLE_VALUE) || (hFileOut == INVALID_HANDLE_VALUE)) {
#else
  /* Get the size of the file to decrypt. */
  stat(infile, &thestat);
  infile_len = thestat.st_size;

  hFileIn = fopen(infile, "r");
  /*hFileOut = fopen(outfile, "w");*/
  hFileOut = open(outfile, O_WRONLY | O_CREAT | O_EXCL, 0600);
  if ((hFileIn == NULL) || (hFileOut < 0)) {
#endif
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not open files.");
    goto decrypt_file_error;
  }



  /* Read in the original length of the file. */
#ifdef _WIN32
  if (!ReadFile(hFileIn, &outfile_len, 4, &n, NULL))
    n = -1;
#else
  n = fread(&outfile_len, sizeof(u_int32_t), 1, hFileIn);
#endif
  outfile_len = ntohl(outfile_len);

  /* If the original file length is too big, then this is corrupt... */
  /* 20 = HMAC trailer; 4 = the length we just read. */
  /*if (outfile_len > infile_len) {*/
  if ((outfile_len + 20 + 4) > infile_len) {
    bc_log("decrypt_file", BC_LOG_ERROR, "File length is corrupt: %u %u",
	   outfile_len, infile_len);
    goto decrypt_file_error;
  }


  /* Calculate the size of the encrypted bytes we need to read.  Don't count
   * the file length header or the HMAC footer. */
#ifdef _WIN32
  body_len = infile_len - 4 - 20;
#else
  body_len = infile_len - sizeof(u_int32_t) - 20;
#endif
  plaintext_left_to_read = outfile_len;


  /* If the file is less than 65K, we'll read the file all at once, otherwise,
   * we will read it in blocks of 65K. */
  buffer_size = MY_MIN(DECRYPT_BUFFER_SIZE, body_len);
  buffer = (char *)gcry_malloc_secure(MY_MAX(buffer_size + 8, 64));
  if (buffer == NULL) {
    bc_log("decrypt_file", BC_LOG_ERROR, "Could not allocate.");
    goto decrypt_file_error;
  }

  /*printf("body_len: %u; buffer_size: %u; loops: %u; plaintext_left_to_read: %u\n", body_len, buffer_size, (body_len / buffer_size), plaintext_left_to_read);*/
  /* We will loop as many times as needed based on our buffer size. */
  for (i = 0; i < (body_len / buffer_size); i++) {
#ifdef _WIN32
    if (!ReadFile(hFileIn, buffer, buffer_size, &n, NULL))
      n = -1;
#else
    n = fread(buffer, 1, buffer_size, hFileIn);
#endif
    if (n <= 0) {
      bc_log("decrypt_file", BC_LOG_ERROR, "Miscalculation of loops! %u %u", body_len, buffer_size);
      goto decrypt_file_error;
    }

    /* Decrypt the buffer. */
    ret = gcry_cipher_decrypt(cipher, (unsigned char *)buffer, n, NULL, 0);
    if (ret != 0) {
      bc_log("decrypt_file", BC_LOG_ERROR, "Error while decrypting.");
      goto decrypt_file_error;
    }

    /* If we're reading multiple blocks, then the last block might be bigger
     * than the last plaintext block (since the encrypt_file must pad the input
     * to align to 32 bytes).  So, we must shorted the last block here
     * ourselves so the HMAC is properly calculated and the output file is not
     * padded. */
    if (plaintext_left_to_read < buffer_size) {
      n = plaintext_left_to_read;
    }

    gcry_md_write(md, buffer, n);
#ifdef _WIN32
    WriteFile(hFileOut, buffer, n, &x, NULL);
#else
    /*fwrite(buffer, 1, n, hFileOut);*/
    write(hFileOut, buffer, n);
#endif

    /* We just handled some bytes, so update the number of bytes left to
     * read. */
    if (plaintext_left_to_read >= n)
      plaintext_left_to_read = plaintext_left_to_read - n;

  }

  /* Read in the 160-bit SHA1 HMAC at the very end of the file. */
#ifdef _WIN32
  if (!ReadFile(hFileIn, buffer, 20, &n, NULL))
    n = -1;
#else
  fread(buffer, 1, 20, hFileIn);
#endif

  /* Calculate the HMAC based on what we decrypted. */
  hmac_digest = gcry_md_read(md, GCRY_MD_SHA1);

#ifdef _WIN32
  CloseHandle(hFileOut);  hFileOut = NULL;
  CloseHandle(hFileIn);   hFileIn = NULL;
#else
  close(hFileOut);  hFileOut = -1;
  fclose(hFileIn);  hFileIn = NULL;
#endif

  /* See if the stored HMAC and the calculated HMAC match.  If so, the
   * signature is good.  Otherwise its bad. */
  if (memcmp(buffer, hmac_digest, 20) != 0) {
    bc_log("decrypt_file", BC_LOG_ERROR, "File signature is bad!");

    /* If the signature is bad, we will delete the output file. */
    if (bc_delete_file(outfile) < 0)
      bc_log("decrypt_file", BC_LOG_ERROR, "Error while unlinking!");
    goto decrypt_file_error;
  }


  gcry_cipher_close(cipher);  cipher = NULL;
  gcry_md_close(md);          md = NULL;
  gcry_free(buffer);          buffer = NULL;
  return 1;

 decrypt_file_error:
#ifdef _WIN32
  if (hFileOut != NULL) {
    CloseHandle(hFileOut);  hFileOut = NULL;
  }
  if (hFileIn != NULL) {
    CloseHandle(hFileIn);   hFileIn = NULL;
  }
#else
  if (hFileOut > -1) {
    close(hFileOut);  hFileOut = -1;
  }
  if (hFileIn != NULL) {
    fclose(hFileIn);  hFileIn = NULL;
  }
#endif
  gcry_cipher_close(cipher);  cipher = NULL;
  gcry_md_close(md);  md = NULL;
  gcry_free(buffer);  buffer = NULL;
  return -1;
}
