/**
 * module.h
 *
 * Copyright (C) 2003-2006, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef MODULE_H
#define MODULE_H

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#ifdef _WIN32
#include <winsock2.h>
#else
#include <arpa/inet.h>
#endif
#include <dirent.h>
#include <fcntl.h>
#include <gcrypt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include "utils.h"
#include "key_database.h"


#define MAX_MESSAGE_SIZE (256 * 1024)

#define STATE_FLAG_REMOVE 0
#define STATE_FLAG_ADD 1
#define STATE_FLAG_SET 2


#define REQUIRED_LIBGCRYPT_VERSION "1.2.0"

#define MESSAGE_LEN        1016
#define BASE64_BUFFER_LEN  1016
#define SIGN_BUFFER_LEN    1016

#define BASE64_BUFFER_SIZE 1024

#define UMID_MAX_LEN 32

/* AES-256 uses a 16-byte block size. */
#define SYMMETRIC_BLOCK_SIZE 16

/* Also defined in um_encryption_ui.h! */
#define UM_DATABASE_FILENAME "um_keycache"

/* 32 bytes x 8 bits per byte = 256 bits */
/* NOTE: IF THESE TWO CONSTANTS ARE CHANGED, BE VERY CAREFUL BECAUSE THEY'RE
 * USED IN MEMCPY's! */
#define SESSION_KEY_BYTE_SIZE 32
#define SESSION_JUNK_SIZE SESSION_KEY_BYTE_SIZE * 4

#define HMAC_DIGEST_BYTE_SIZE 20

#define UM_PROTOCOL_VERSION "0"

#define BUDDY_RESET                 0
#define SENT_PUBKEYS                1
#define SENT_SESSION_KEY            2
#define AWAITING_PUBKEYS_1          4
#define AWAITING_PUBKEYS_2          8
#define AWAITING_SESSION_KEY       16
#define SIGNKEY_FINGERPRINT_VALID  32
#define ENCRYPTKEY_VALID           64
#define SESSION_KEY_VALID         128
#define CONNECTED                 256
#define TIMEDOUT                  512
#define REPLAY_IMMUNE            1024

#define MAX_PATH_LEN 256

#define UM_SEXP_RELEASE(x)  { gcry_sexp_release((x));  memset(&(x), 0, sizeof(gcry_sexp_t)); }

#define UM_MPI_RELEASE(x)  { gcry_mpi_release((x));  memset(&(x), 0, sizeof(gcry_mpi_t)); }

#define UM_CIPHER_RELEASE(x)  { gcry_cipher_close((x));  memset(&(x), 0, sizeof(gcry_cipher_hd_t)); }

#define UM_GCRY_FREE(x)  { if ((x) != NULL) { gcry_free((x)); (x) = NULL; } }


#define MY_MIN(a,b) (((a)<(b))?(a):(b))
#define MY_MAX(a,b) (((a)<(b))?(b):(a))



/* MAKE SURE TO ZERO OUT ALL FIELDS BEFORE USING!!! */
struct UM_BUDDY {

  /* 32-byte (256-bit) AES session key. */
  unsigned char *session_key;

  /* 32-byte (256-bit) Initialization Vector. */
  unsigned char *session_iv;

  /* 32-byte (256-bit) AES HMAC key. */
  unsigned char *session_hmac;

  /* The cipher object we use to send messages with. */
  gcry_cipher_hd_t *cipher;

  /* The nonce sent by the buddy for use as a sequence number. */
  gcry_mpi_t *my_sequence_number;

  /* The nonce sent to the buddy for use as a sequence number. */
  gcry_mpi_t *his_sequence_number;
  
  /* The message-digest object we use to perform HMAC integrity functions
   * with. */
  gcry_md_hd_t *md;

  /* The buddy's public encryption key. */
  gcry_sexp_t *public_encrypt_key;

  /* The buddy's public signing key. */
  gcry_sexp_t *public_sign_key;

  /* The buddy's protocol-specific screen name. */
  char *screenname;

  /* The buddy's protocol-independent Ultramagnetic ID. */
  char *um_id;

  /* The communication state that this buddy is in. */
  unsigned int state;

  /* The last time that this buddy structed has been accessed/modified.  This
   * is relevant only when this buddy is stored in the pending list.  If not
   * touched for more than 30 seconds, then the 'crypto_state_timeout_thread'
   * function will immediately free this buddy and thus the pending connection
   * will be declared timed-out. */
  time_t last_touched;

  /* Flag which holds whether or not we notified the upper layer that the
   * conversation with this buddy is immune to replay attacks.
   * (0 = no notification made, 1 = already notified). */
  unsigned int notify_noreplay;

  /* Pointer to the next UM_BUDDY in the linked list. */
  struct UM_BUDDY *next;
};


struct PENDING_MESSAGE {
  char *who;
  char *what;
  struct PENDING_MESSAGE *next;
};



void add_pending_message(char *who, char *what);

void add_to_pending_list(struct UM_BUDDY *new_buddy);

void add_to_validated_list(struct UM_BUDDY *new_buddy);

int buddy_fingerprint_is_good(MXFER *mxfer, char *who);

int check_sequence_number(struct UM_BUDDY *buddy,
			  unsigned char *sequence_number,
			  size_t sequence_number_len);

void clear_all_pending_messages(void);

#ifdef _WIN32
DWORD WINAPI crypto_state_timeout_thread(void *nothing);
#else
void *crypto_state_timeout_thread(void *);
#endif

int decrypt_file(char *who, char *infile, char *outfile);

int decrypt_keyfile(char *path, char *pw,
		    char *storedkeyfpbuf, size_t storedkeyfpbufsize,
		    char *databuf, size_t databufsize);

int encode_key_component(gcry_sexp_t x, char *component,
                         char *output, unsigned int output_len);

int encode_key_component_nth(gcry_sexp_t x, char *component,
			     unsigned int nth,
			     char *output, unsigned int output_len);

int encrypt_file(char *who, char *infile, char *outfile);

int encrypt_keyfile(gcry_sexp_t *sign_key, char *path, char *data, char *pw);

struct UM_BUDDY *find_buddy_in_pending_list(char *who);

struct UM_BUDDY *find_buddy_in_validated_list(char *who);

void free_database_raw_list(BCSList **umid_list, BCSList **fingerprint_list);

int generate_key(char *directory_entry_text,
		 char *id_combo_entry_text,
		 char *pw,
		 char *fingerprint_buffer,
		 unsigned int fingerprint_buffer_len);

void get_database_raw_list(char *database_path,
			   BCSList **umid_list, BCSList **fingerprint_list);

void get_encrypted_im(MXFER *mxfer, struct UM_BUDDY *buddy, char *message);

unsigned int get_fingerprint_from_keyfile(char *filepath,
                                          char *buf, size_t bufsize);

void get_handler(MXFER *mxfer, char *who, char *message);

void get_key_fingerprint_from_sexp(gcry_sexp_t *key, char *buffer,
				   unsigned int buffer_len);

char *get_pending_message(char *who, unsigned int flag);

int get_pubkeys_1(MXFER *mxfer, struct UM_BUDDY *buddy, char *message);

int get_pubkeys_2(MXFER *mxfer, struct UM_BUDDY *buddy, char *message);

unsigned char *get_sequence_number(struct UM_BUDDY *buddy);

int get_session_key(MXFER *mxfer, struct UM_BUDDY *buddy, char *message);

void init_gcrypt(void);

void init_threads(void);

int is_encryption_initialized(void);

void kill_state_timeout_thread(BC_THREAD **thread);

int load_database_and_keys(char *directory_path, char *id, char *pw);

void load_id_file(unsigned char *directory, unsigned char *file);

BCSList *load_ids_from_directory(char *dir_name);

int load_keys_and_fingerprint(char *um_file,
			      char *pw,
			      gcry_sexp_t *signing_keypair,
			      gcry_sexp_t *encrypting_keypair,
			      char *buffer, size_t buffer_len);

void parse_directory_string(char *directory_path,
                            char *output_buffer,
                            unsigned int output_buffer_len);

void print_pending_list(void);

void remove_buddy_from_pending_list(struct UM_BUDDY *buddy);

void remove_buddy_from_validated_list(struct UM_BUDDY *buddy);

int send_encrypted_im(MXFER *mxfer, struct UM_BUDDY *buddy,
		      char *message);

void send_handler(MXFER *mxfer, char *who, char *message);

int send_session_key(MXFER *mxfer, struct UM_BUDDY *buddy);

int send_public_keys(MXFER *mxfer, char *who);

int um_encryption_init(void);

void wipe_buddy(struct UM_BUDDY *um_buddy);

void wipe_entire_pending_list();

void wipe_entire_validated_list();

#endif
