/**
 * utils.c
 *
 * Copyright (C) 2004-2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif


#include "utils.h"
#ifdef DMALLOC
  #include "dmalloc.h"
#endif


char base64_table[ 256 ] = { 
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
  'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
  'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
  'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
  'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
  'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
  'w', 'x', 'y', 'z', '0', '1', '2', '3',
  '4', '5', '6', '7', '8', '9', '+', '/',
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 
  ' ', ' ', ' ', ' ', ' ', ' ', ' ', '\0' };


int base64_table_inverted[ 256 ] = {
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
  52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,
  -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
  15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
  -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
  41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};


char logfile_path[ LOGFILE_PATH_SIZE ];
int logfile_fd = -1;

unsigned int entropy_file_set = 0;

/* Base64's a block of data. Returns 1 on success, and -1 if the output buffer
 * was too small or some other error occured. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
int base64_encode(unsigned char *output, size_t output_len,
		  unsigned char *input, size_t input_len) {
  unsigned int j = 0;

  while (input_len >= 3) {
    if (output_len <= j + 5)
      goto encode_error;

    output[ j++ ] = base64_table[ input[ 0 ] >> 2 ];
    output[ j++ ] = base64_table[ (input[ 1 ] >> 4) |
                                  ((input[ 0 ] << 4) & 0x30) ];
    output[ j++ ] = base64_table[ ((input[ 1 ] << 2) & 0x3c) |
                                  (input[ 2 ] >> 6) ];
    output[ j++ ] = base64_table[ input[ 2 ] & 0x3f ];

    input += 3;
    input_len -= 3;
  }


  if ((input_len == 1) && (output_len > j + 4)) {
    output[ j++ ] = base64_table[ input[ 0 ] >> 2 ];
    output[ j++ ] = base64_table[ (input[ 0 ] << 4) & 0x30 ];
    output[ j++ ] = '=';
    output[ j++ ] = '=';
  } else if ((input_len == 2) && (output_len > j + 4)) {
    output[ j++ ] = base64_table[ input[ 0 ] >> 2 ];
    output[ j++ ] = base64_table[ (input[ 1 ] >> 4) |
                                  ((input[ 0 ] << 4) & 0x30) ];
    output[ j++ ] = base64_table[ (input[ 1 ] << 2) & 0x3c ];
    output[ j++ ] = '=';
  } else if (input_len != 0)
    goto encode_error;


  if (j < output_len)
    output[ j ] = '\0';
  else 
    goto encode_error;


  return 1;

encode_error:
  output[ 0 ] = '\0';
  return -1;
}


/* Decodes a base64 string.  Returns the number of bytes decoded into the
 * output buffer, or -1 on error. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
int base64_decode(unsigned char *output, size_t output_len, unsigned char *input) {
  int input_len = strlen((char *)input);
  int one = 0;
  int two = 0;
  int three = 0;
  int four = 0;
  int j = 0;

  while (input_len >= 4) {

    one = base64_table_inverted[ (unsigned int)input[ 0 ] ];
    if (one == -1)
      goto decode_error;

    two = base64_table_inverted[ (unsigned int)input[ 1 ] ];
    if (two == -1)
      goto decode_error;

    three = base64_table_inverted[ (unsigned int)input[ 2 ] ];
    if (input[ 2 ] == '=')
      three = 0;
    else if (three == -1)
      goto decode_error;

    four = base64_table_inverted[ (unsigned int)input[ 3 ] ];
    if (input[ 3 ] == '=')
      four = 0;
    else if (four == -1)
      goto decode_error;


    if (output_len < j + 4)
      goto decode_error;
    else {
      output[ j++ ] = (one << 2) | (two >> 4);
      if (input[ 2 ] != '=') {
	output[ j++ ] = ((two << 4) & 0xF0) | (three >> 2);
	if (input[ 3 ] != '=')
	  output[ j++ ] = ((three << 6) & 0xC0) | four;
      }
    }

    input += 4;
    input_len -= 4;
  }


  if (input_len != 0)
    goto decode_error;

  return j;

 decode_error:
  output[ 0 ] = '\0';
  bc_log("my_base64_decode", BC_LOG_ERROR, "Error!");
  return -1;
}


/* Normalizes a string.  This consists of lowercasing all characters and
 * stripping out all spaces.  The character array argument is modified
 * in-place. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
char *bc_normalize(char *txt) {
  unsigned int i = 0;
  unsigned int j = 0;

  while(i < strlen(txt) && (j <= i)) {
    if (txt[ i ] != ' ') {
      txt[ j ] = tolower(txt[ i ]);
      j++;
    }
    i++;
  }

  txt[ j ] = '\0';
  return txt;
}


/* Retrieves the data from the nth element in the BCSList. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
void *bc_slist_nth_data(BCSList *list, unsigned int n) {
  BCSList *current = list;
  unsigned int j = 0;

  if (list == NULL)
    return NULL;

  while ((current->next != NULL) && (j < n)) {
    current = current->next;
    j++;
  }

  return current->data;
}


/* Appends data to the end of a BCSList.  Returns a pointer to the new head
 * of the list (this changes if the list was previously NULL). */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
BCSList *bc_slist_append(BCSList *list, void *data) {
  BCSList *retVal = NULL;

  BCSList *new_node = (BCSList *)calloc(1, sizeof(BCSList));
  new_node->data = data;
  new_node->next = NULL; /* Not necessary since we just calloc'ed, but hey.. */

  if (list == NULL)
    retVal = new_node;
  else {
    retVal = list;

    while(list->next != NULL)
      list = list->next;

    /* Note that new_node->next is set to NULL above... */
    list->next = new_node;
  }

  return retVal;
}


/* Frees a BCSList.  The data pointers held inside the list are not
 * modified. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
void bc_slist_free(BCSList **list) {
  BCSList *l = NULL;
  BCSList *current = NULL;
  BCSList *next = NULL;

  if (list == NULL)
    return;

  l = *list;
  if (l == NULL) {
    *list = NULL;
    return;
  }

  current = l;
  while(current != NULL) {
    next = current->next;
    free(current); /* current = NULL;  Not needed in this case. */
    current = next;
  }

  *list = NULL;
}


/* Returns the length of a BCSList. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
unsigned int bc_slist_length(BCSList *list) {
  unsigned int retVal = 0;
  BCSList *current = list;

  while (current != NULL) {
    retVal++;
    current = current->next;
  }

  return retVal;
}

/* Creates a new BCString set to a default value (may be NULL).   Note that the
 * default value must be a null-terminated string. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
BCString *bc_string_new(char *init_data) {
  BCString *ret = NULL;

  ret = (BCString *)calloc(1, sizeof(BCString));
  if (ret) {
    unsigned int init_alloc_size = 0;
    unsigned int init_data_size = 0;


    if (init_data == NULL)
      init_alloc_size = DEFAULT_ALLOC_SIZE;
    else {
      init_data_size = strlen(init_data) + 1;
      init_alloc_size = 1;

      /* Double the size of the initial allocation block until it is larger
       * than the initial data size. */
      while(init_data_size > init_alloc_size)
	init_alloc_size = init_alloc_size << 1;

    }

    ret->str = (char *)calloc(init_alloc_size, sizeof(char));
    if (ret->str) {
      if (init_data) {
	bc_strlcpy(ret->str, init_data, init_alloc_size);
	ret->str_len = strlen(init_data);
      } else {
	ret->str_len = 0;
      }

      ret->allocated_len = init_alloc_size;
    } else {
      free(ret);  ret = NULL;
    }

  }
  return ret;
}


/* Appends a null-terminated string to the end of this BCString.  The string
 * will be reallocated as necessary. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
BCString *bc_string_append(BCString *string, char *new_data) {
  BCString *ret = NULL;

  if (string != NULL) {
    ret = bc_string_append_len(string, new_data, strlen(new_data) + 1);
    string->str_len--;
  }

  return ret;
}


/* Appends binary data to the end of this BCString.  The string will be
 * reallocated as necessary. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
BCString *bc_string_append_len(BCString *string,
			       char *new_data,
			       size_t new_data_len) {

  /* string can be not NULL while string->str is if the user free()'ed the
   * string but is using it again accidentally... */
  if ((string == NULL) || (new_data == NULL) ||
      (new_data_len == 0) || (string->str == NULL))
    return string;


  /*printf("string: %d %d\n", string->str_len, string->allocated_len);*/

  if ((string->str_len + new_data_len) > string->allocated_len) {
    unsigned int new_buffer_size = string->allocated_len;
    char *new_buffer = NULL;

    /* Double the current size of the allocation buffer until it is big enough
     * to hold the new data. */
    while((string->str_len + new_data_len) > new_buffer_size)
      new_buffer_size = new_buffer_size << 1;

    /*printf("resizing buffer.  new size is %d\n", new_buffer_size);*/

#if EXTRA_SAFE_BCSTRINGS == 1
    new_buffer = (char *)calloc(new_buffer_size, sizeof(char));
    memcpy(new_buffer, string->str, string->str_len);
    free(string->str);  string->str = NULL;
    string->str = new_buffer;
#elif EXTRA_SAFE_BCSTRINGS == 0
    new_buffer = (unsigned char *)realloc(string->str, new_buffer_size);
#else
    #error <utils.h>:EXTRA_SAFE_BCSTRINGS must be 0 or 1!
#endif

    string->allocated_len = new_buffer_size;
  }

  memcpy(string->str + string->str_len, new_data, new_data_len);
  string->str_len += new_data_len;
  return string;
}


/* This frees a string.  If the caller is still using the character string
 * but wants to kill this BCString, then 'free_everything' can be set to 0.
 * In this case, the caller assumes the responsibility of free()'ing the
 * character string later.  A pointer to the internal character array is
 * returned if the caller chose not to free it.  In all cases the string
 * argument be set to NULL to prevent subsequent (and accidental) use. */
/* Security audit trail:
 *    - Joe Testa: January 3rd, 2005
 */
char *bc_string_free(BCString **string, unsigned int free_everything){
  char *ret = NULL;
  BCString *lstring = *string;

  if ((string == NULL) || (*string == NULL))
    return NULL;

  if (free_everything == 0)
    ret = lstring->str;
  else {
    free(lstring->str);  lstring->str = NULL;
  }

  memset(lstring, 0, sizeof(BCString));
  /*lstring->str = NULL;
  lstring->str_len = 0;
  lstring->allocated_len = 0;*/
  free(lstring);  lstring = NULL;

  *string = NULL;
  return ret;
}


/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
unsigned int bc_string_startswith(BCString *string, BCString *tofind) {
  return bc_startswith(string->str, tofind->str);
}


/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
unsigned int bc_string_startswith_char(BCString *string, char *tofind) {
  return bc_startswith(string->str, tofind);
}


/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
unsigned int bc_startswith(char *line, char *tofind) {
  unsigned int i = 0;
  unsigned int line_len = 0;
  unsigned int tofind_len = 0;

  if ((line == NULL) || (tofind == NULL))
    return 0;

  line_len = strlen(line);
  tofind_len = strlen(tofind);

  /* If the tofind string is longer than the line to search, then we know that
   * tofind isn't a prefix. */
  if (tofind_len > line_len)
    return 0;

  /* Compare each character in tofind with line.  If a difference is found,
   * then tofind isn't a prefix. */
  for(i = 0; i < tofind_len; i++) {
    if (line[ i ] != tofind[ i ])
      return 0;
  }

  /* If we reached here, then tofind must be a prefix. */
  return 1;
}


/* Concatenates a string with another string.  This function guarantees that
 * dest will be null terminated. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
unsigned int bc_strlcat(char *dest, char *src, size_t dest_len) {
  unsigned int i = 0, j = 0;

  if ((src == NULL) || (dest == NULL) || (dest_len == 0))
    return 0;

  /* Get the index of the null terminator in dest. */
  /*while((dest[ i ] != '\0') && (i < dest_len))
    i++;*/
  i = strlen(dest);

  while((src[ j ] != '\0') && (i < dest_len)) {
    dest[ i ] = src[ j ];
    i++;
    j++;
  }

  if (i == dest_len) {
    dest[ dest_len - 1 ] = '\0';
    return dest_len - 1;
  } else if (i < dest_len) {
    dest[ i ] = '\0';
    return i;
  } else {
    dest[ 0 ] = '\0';
    return 0;
  }
}


/* Copies src into dest and ensures that dest will be null-terminated
 * afterwards. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
unsigned int bc_strlcpy(char *dest, char *src, size_t dest_len) {
  unsigned int i = 0;

  if ((src == NULL) || (dest == NULL) || (dest_len == 0))
    return 0;

  while((src[ i ] != '\0') && (i < dest_len)) {
    dest[ i ] = src[ i ];
    i++;
  }

  if (i == dest_len) {
    dest[ dest_len - 1 ] = '\0';
    return dest_len - 1;
  } else if (i < dest_len) {
    dest[ i ] = '\0';
    return i;
  } else {
    dest[ 0 ] = '\0';
    return 0;
  }
}


/* Appends a character array to the end of a BCString using the printf
 * formatting features. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_string_append_printf(BCString *string, const char *format, ...) {
  va_list args;
  unsigned int buffer_size = 128;
  char *buffer = NULL;
  unsigned int done = 0;
  int n = 0;


  buffer = (char *)calloc(buffer_size, sizeof(char));
  if (buffer == NULL)
    return;

  while(done == 0) {

    va_start(args, format);
    n = vsnprintf(buffer, buffer_size, format, args);
    va_end(args);

    if ((n > -1) && (n < buffer_size)) {
      string = bc_string_append(string, buffer);
      done = 1;
    } else {
      buffer_size = buffer_size << 1;
      if ((buffer = (char *)realloc(buffer, buffer_size)) == NULL)
	return;
    }

  }

  free(buffer);  buffer = NULL;
}


/* Clears a BCString. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_string_clear(BCString *string) {
  if ((string == NULL) || (string->str == NULL))
    return;

  memset(string->str, 0, string->allocated_len);
  string->str_len = 0;
}


/* Logs text to the log file, if one was specified at startup. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_log(char *function, unsigned int type, char *format, ...) {
  va_list args;
  unsigned int i = 0;
  /*unsigned buffer_len = 0;*/

  if (logfile_fd != -1) {
    unsigned int continueFlag = 1;
    char *buffer = NULL;
    size_t buffer_len = 128;
    int actually_printed = 0;
    /*char buffer[ 1024 ];
    memset(buffer, 0, sizeof(buffer));*/

    buffer = (char *)calloc(buffer_len, sizeof(char));
    if (buffer == NULL)
      return;

    while (continueFlag == 1) {
      va_start(args, format);
      actually_printed = vsnprintf(buffer, buffer_len, format, args);
      va_end(args);
      if (((actually_printed > -1) && (actually_printed < buffer_len)) ||
	  (actually_printed < 0)) {
	continueFlag = 0;
      } else {
	/*buffer_len = actually_printed + 1;*/
	buffer_len = buffer_len << 1;
	buffer = (char *)realloc(buffer, buffer_len);
	if (buffer == NULL)
	  return;
      }

    }


    /* Make sure that newlines are not injected as part of the message. */
    for(i = 0; i < strlen(buffer); i++) {
      if ((buffer[ i ] == '\n') || (buffer[ i ] == '\r'))
	buffer[ i ] = ' ';
    }

    write(logfile_fd, function, strlen(function));
    write(logfile_fd, ": ", 2);
    write(logfile_fd, buffer, strlen(buffer));
    write(logfile_fd, "\n", strlen("\n"));

    free(buffer);  buffer = NULL;

  }

}


/* Trims a newline off the end of a string, if it exists. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void trim_line(char *line) {
  size_t line_size = strlen(line);
  unsigned int i = 0;

  for(i = line_size - 1; i >= 0; i--) {
    if (line[ i ] == '\n') {
      line[ i ] = '\0';
      break;
    }
  }
}


/* Creates a new mutex. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
BC_MUTEX *bc_mutex_new(void) {
  BC_MUTEX *retVal = (BC_MUTEX *)calloc(1, sizeof(BC_MUTEX));

#ifdef _WIN32
  InitializeCriticalSection(retVal);
#else
  if (pthread_mutex_init(retVal, NULL)) {
    perror("pthread_mutex_init");
  }
#endif

  return retVal;
}


/* Locks a mutex. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_mutex_lock(BC_MUTEX *mutex) {
#ifdef _WIN32
   EnterCriticalSection(mutex);
#else
   pthread_mutex_lock(mutex);
#endif
}


/* Unlocks a mutex. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_mutex_unlock(BC_MUTEX *mutex) {
#ifdef _WIN32
   LeaveCriticalSection(mutex);
#else
   pthread_mutex_unlock(mutex);
#endif
}


/* Destroys a mutex. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
void bc_mutex_destroy(BC_MUTEX **mutex) {
#ifdef _WIN32
   DeleteCriticalSection(*mutex);
#else
   pthread_mutex_destroy(*mutex);
#endif

   free(*mutex);  *mutex = NULL;
}


/* Creates a thread. */
/* Security audit trail:
 *    - Joe Testa: January 4th, 2005
 */
#ifdef _WIN32
BC_THREAD *bc_thread_create(DWORD (WINAPI *func) (void *arg)) {
#else
BC_THREAD *bc_thread_create(void *(*func)(void *)) {
#endif
  BC_THREAD *retVal = calloc(1, sizeof(BC_THREAD));

#ifdef _WIN32
  DWORD thread_id;
  *retVal = CreateThread(NULL, 512, func, NULL, 0, &thread_id);
#else
  pthread_attr_t thread_attr;

  memset(&thread_attr, 0, sizeof(pthread_attr_t));
  
  pthread_attr_init(&thread_attr);
  pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_JOINABLE);

  if (pthread_create(retVal, &thread_attr, func, NULL))
    perror("pthread_create");

  pthread_attr_destroy(&thread_attr);
#endif

  return retVal;
}



/*void bc_thread_join(pthread_t *thread_handle) {}*/
/* This function returns 1 if the specified path points to a regular file,
 * returns 0 if the path does not exist, and returns -1 if otherwise (for
 * symlinks, directories, etc). */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
int is_regular_file(char *path) {
#ifndef _WIN32
  struct stat thestat;

  memset(&thestat, 0, sizeof(thestat));

  if (lstat(path, &thestat) != -1) {
    if (S_ISREG(thestat.st_mode))
      return 1;
    else
      return -1;
  } else
    return 0;
#else
  WIN32_FILE_ATTRIBUTE_DATA stat;

  memset(&stat, 0, sizeof(stat));

  if (GetFileAttributesEx(path, GetFileExInfoStandard, &stat)) {
    if (stat.dwFileAttributes & (FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_ARCHIVE))
      return 1;
    else
      return -1;
  } else
    return 0;
#endif
}



/* Returns 1 if the path is a symbolic link, 0 if the path does not exist, and
 * -1 if otherwise (path is a file, socket, directory, etc). */
/* Security audit trail:
 *    - Joe Testa: January 11th, 2005
 */
int is_symlink(char *path) {
#ifdef _WIN32
  WIN32_FILE_ATTRIBUTE_DATA stat;

  memset(&stat, 0, sizeof(stat));

  if (GetFileAttributesEx(path, GetFileExInfoStandard, &stat))
    return -1;
  else
    return 0;
#else
  struct stat thestat;

  memset(&thestat, 0, sizeof(thestat));

  if (lstat(path, &thestat) != -1) {
    if (S_ISLNK(thestat.st_mode))
      return 1;
    else
      return -1;
  } else
    return 0;
#endif
}


void enable_logging(char *path) {
  int ret = 0;
  int flags = 0;

  time_t c;


  memset(&c, 0, sizeof(c));


  /* If logging is already on, then do nothing. */
  if (logfile_fd != -1)
    return;

  bc_strlcpy(logfile_path, path, sizeof(logfile_path) - 8);

  ret = is_regular_file(logfile_path);
  if (ret < 0) {
    fprintf(stderr, "open() failed, logging will not occur.\n");
    memset(logfile_path, 0, sizeof(logfile_path));
    exit(666);
  } else if (ret == 0)
    flags = O_WRONLY | O_CREAT | O_EXCL;
  else
    flags = O_WRONLY | O_APPEND;


  logfile_fd = open(logfile_path, flags, (S_IREAD | S_IWRITE));
  if (logfile_fd == -1) {
    fprintf(stderr, "open() failed, logging will not occur.\n");
    memset(logfile_path, 0, sizeof(logfile_path));
  }

  time(&c);
  bc_log("enable_logging", BC_LOG_INFO, "Logging enabled at %s", ctime((const time_t *)&c));
  bc_log("enable_logging", BC_LOG_DEBUG, "Version: %s; API Version: %s; " \
	 "Libgcrypt: %s", VERSION, API_VERSION, gcry_check_version(NULL));
}


void check_for_debug_file(char *load_save_dir) {
  BCString *debug_file_path = NULL;

  if (load_save_dir == NULL)
    return;

  debug_file_path = bc_string_new(load_save_dir);
  debug_file_path = bc_string_append(debug_file_path, "enable_debugging");

  if (is_regular_file(debug_file_path->str) == 1) {
    bc_string_free(&debug_file_path, 1);  debug_file_path = NULL;

    debug_file_path = bc_string_new(load_save_dir);
    debug_file_path = bc_string_append(debug_file_path, "log.txt");
    enable_logging(debug_file_path->str);
  }

  bc_string_free(&debug_file_path, 1);  debug_file_path = NULL;
}


int bc_delete_file(char *file) {
  if (is_regular_file(file) != 1)
    return -1;

#ifdef _WIN32
  if (DeleteFile(file) != 0)
    return -1;
#else
  if (unlink(file) < 0)
    return -1;
#endif
  return 1;
}



/* Pads a message before it is to be encrypted.  This function will sprinkle
 * random bytes into the message in every 4th position, then align the
 * result to 32 bytes to frustrate traffic analysis.  Returns the number of
 * bytes written to the output buffer (don't use strlen(output) anymore!) */
/* Security audit trail:
 *    - Joe Testa: January 18th, 2005
 *    - Joe Testa: March 4th, 2004
 */
unsigned int um_pad_message(char *output, size_t output_len, char *message) {
  unsigned int retval = 0;
  unsigned char *random_bytes = NULL;
  unsigned int message_len = 0;
  unsigned int random_bytes_len = 0;
  unsigned int newlen = 0;
  unsigned int i = 0;
  unsigned int j = 0;

  message_len = strlen(message);

  /* Make sure that the output buffer is large enough to hold the result.  This
   * size calculation uses the worst-case scenario since its easier than
   * calculating the true size... */
  /* 40 = 32 (worse case padding for phase 2, plus the standard 8 for
   * safety) */
  if (((message_len * 1.34) + 40) > output_len) {
    return 0;
  }


  /* We need one random byte for every three bytes in the message for the
   * first phase of padding... */
  random_bytes_len = (int)(message_len / 3);


  /* For messages aligned to 3 bytes, we need an extra one that isn't
   * calculated above.  This is just a quirk of the algorithm below... */
  if ((message_len % 3) == 0)
     random_bytes_len++;
  else {
     /* Because of typcasting truncation above, we might need to add 1 or 2. */
     while ((random_bytes_len * 3) < message_len)
       random_bytes_len++;
  }


  /* BEGIN PHASE 1 PADDING */

  /* Allocate a secure buffer and fill it with random bytes. */
  random_bytes = (unsigned char *)gcry_malloc_secure(random_bytes_len + 8);
  gcry_randomize(random_bytes, random_bytes_len, GCRY_STRONG_RANDOM);

  /* During this phase of padding, we can't allow a zero byte to occur,
   * otherwise the message will be truncated when the message is unpadded
   * later.  So, we'll just replace all the zeros with a one. */
  for(i = 0; i < random_bytes_len; i++)
    if (random_bytes[ i ] == 0)
      random_bytes[ i ] = 1;

  /* Put a random byte at the first position in the output buffer, and every
   * fourth position thereafter. */
  for(i = 0, j = 0; (i < output_len) && (j < random_bytes_len); i += 4, j++)
    output[ i ] = random_bytes[ j ];

  /* Now we're done with this array, so free it. */
  gcry_free(random_bytes);  random_bytes = NULL;

  /* Now overlay the message into the output buffer three bytes at a time,
   * around the random bytes we placed in earlier. We loop until
   * j < message_len + 1 so that the message's NULL value is also copied. */
  for (i = 1, j = 0; (i < output_len) && (j < message_len + 1); i++, j++) {
    output[ i ] = message[ j ];
    /* Skip over the random byte we put in. */
    if (((i + 1) % 4) == 0) { i++; }
  }


  /* BEGIN PHASE 2 PADDING */

  /* We must now align the message to 32 bytes.  If it is already aligned,
   * then we don't need to do anything. Notice that we re-calculate the
   * output's len because if the message was aligned to 3 bytes above, the
   * counter 'i' would be incremented to far... */
  newlen = strlen(output) + 1;
  if ((newlen % 32) != 0) {
    unsigned int needed_bytes = 0;

    unsigned char more_random_bytes[ 32 + 8 ];

    memset(more_random_bytes, 0, sizeof(more_random_bytes));

    /* Get whatever random bytes we need to align to 32 bytes... */
    needed_bytes = 32 - (newlen % 32);
    gcry_randomize(more_random_bytes, sizeof(more_random_bytes) - 8,
		   GCRY_STRONG_RANDOM);

    /* Put these random bytes all at the end of the output buffer. */
    for (i = newlen, j = 0; (i < output_len) &&
                            ((i - newlen) < needed_bytes) &&
                            (j < (sizeof(more_random_bytes) - 8));
         i++, j++) {
      output[ i ] = more_random_bytes[ j ];
    }

    retval = i;
  } else {
    /* If the message is aligned to 32 bytes, then it is also aligned to 4
     * bytes, which means that i is one more than the true length (due to the
     * mod 4 wrapping above). */
    retval = i - 1;
  }

  /* Return the size of the output buffer. */
  return retval;
}



/* This function reverses the padding introduced by the 'um_pad_message'
 * function.  Returns 1 on success, -1 on error. */
/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 *    - Joe Testa: March 4th, 2004
 */
int um_unpad_message(char *output, size_t output_len,
		     char *input, size_t input_len) {

  unsigned int output_ptr = 0;
  unsigned int i = 0;
  unsigned int null_char_found = 0;

  if ((input == NULL) || (output == NULL))
    return -1;

  /* If the input is not aligned to 32 bytes, then there is a serious error,
   * and we should not do anything here. */
  if ((input_len % 32) != 0) {
    *output = '\0';
    return -1;
  }

  /* Of every quartet of bytes, extract the last three into the output
   * buffer. Stop when a null byte is found. */
  for (i = 1; (i < input_len) && (null_char_found == 0); i++) {
    if (output_ptr < output_len) {
      output[ output_ptr ] = input[ i ];  output_ptr++;
    }

    if ((input[ i ] == '\0') || (input[ i + 1 ] == '\0'))
      null_char_found = 1;

    if (((i + 1) % 4) == 0) {  i++;  }
  }

  /* Put a null terminator at the end of the string since we didn't do it
   * above.  If we do not have room to put in the null terminator, then return
   * nothing at all. */
  if (output_ptr < output_len) {
    output[ output_ptr ] = '\0';
    return 1;
  } else {
    *output = '\0';
    return -1;
  }
}



/* Simple method to convert raw bytes to hex, then dump to the console. */
/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 *    - Joe Testa: March 4th, 2004
 */
void hex_dump(char *title, char *buffer, unsigned int buffer_len) {
  BCString *string_title = NULL;
  BCString *string_bytes = NULL;
  unsigned int i = 0;

/*char temp[ 16 ];

memset(temp, 0, sizeof(temp));*/

  string_title = bc_string_new("hex_dump/");
  string_title = bc_string_append(string_title, title);

  string_bytes = bc_string_new(NULL);
  for (i = 0; i < buffer_len; i++) {
/*snprintf(temp, sizeof(temp) - 8, "%0X", buffer[ i ]);
  bc_string_append_printf(string_bytes, "%s ", temp);*/
    bc_string_append_printf(string_bytes, "%0X", (unsigned char)buffer[ i ]);
  }

  /*printf("%s: %s\n", string_title->str, string_bytes->str);*/
  bc_log(string_title->str, BC_LOG_DEBUG, string_bytes->str);
  bc_string_free(&string_title, 1);
  bc_string_free(&string_bytes, 1);
}



/* Returns 1 if the directory path argument ends in a slash character, and
 * returns -1 if it does not. */
/* Security audit trail:
 *    - Joe Testa: January 22nd, 2005
 */
int ends_with_slash(char *directory) {
  unsigned int directory_len = strlen(directory);
  unsigned char lastChar = '\0';

  if (directory_len == 0)
    return -1;

  lastChar = directory[ directory_len - 1 ];
  if (lastChar == SLASH_CHAR)
    return 1;
  else
    return -1;
}


/* Pads a message header given a sequence number and message digest.  A random
 * byte is appended to every third byte of the sequence, then the digest is
 * tacked on at the end.  The sequence number MUST be 32 bytes and the digest
 * MUST be 20 bytes.  The result is always a 64-byte header. */
void um_pad_header(char *result, char *sequence_num, char *digest) {
  unsigned int i = 0;
  unsigned char pad[ 16 ];

  memset(pad, 0, sizeof(pad));


  gcry_create_nonce(pad, 12);


  for (i = 0; i < 10; i++) {
    result[ i * 4 ]       = sequence_num[ i * 3 ];
    result[ (i * 4) + 1 ] = sequence_num[ (i * 3) + 1 ];
    result[ (i * 4) + 2 ] = sequence_num[ (i * 3) + 2 ];
    result[ (i * 4) + 3 ] = pad[ i ];
  }
  result[ 40 ] = sequence_num[ 30 ];
  result[ 41 ] = sequence_num[ 31 ];
  result[ 42 ] = pad[ 10 ];
  result[ 43 ] = pad[ 11 ];


  for (i = 0; i < 20; i++)
    result[ 44 + i ] = digest[ i ];

  /*result[ 0 ] = sequence_num[ 0 ];
  result[ 1 ] = sequence_num[ 1 ];
  result[ 2 ] = sequence_num[ 2 ];
  result[ 3 ] = pad[ 0 ];

  result[ 4 ] = sequence_num[ 3 ];
  result[ 5 ] = sequence_num[ 4 ];
  result[ 6 ] = sequence_num[ 5 ];
  result[ 7 ] = pad[ 1 ];

  result[ 8 ] = sequence_num[ 6 ];
  result[ 9 ] = sequence_num[ 7 ];
  result[ 10 ] = sequence_num[ 8 ];
  result[ 11 ] = pad[ 2 ];

  result[ 12 ] = sequence_num[ 9 ];
  result[ 13 ] = sequence_num[ 10 ];
  result[ 14 ] = sequence_num[ 11 ];
  result[ 15 ] = pad[ 3 ];

  result[ 16 ] = sequence_num[ 12 ];
  result[ 17 ] = sequence_num[ 13 ];
  result[ 18 ] = sequence_num[ 14 ];
  result[ 19 ] = pad[ 4 ];

  result[ 20 ] = sequence_num[ 15 ];
  result[ 21 ] = sequence_num[ 16 ];
  result[ 22 ] = sequence_num[ 17 ];
  result[ 23 ] = pad[ 5 ];

  result[ 24 ] = sequence_num[ 18 ];
  result[ 25 ] = sequence_num[ 19 ];
  result[ 26 ] = sequence_num[ 20 ];
  result[ 27 ] = pad[ 6 ];

  result[ 28 ] = sequence_num[ 21 ];
  result[ 29 ] = sequence_num[ 22 ];
  result[ 30 ] = sequence_num[ 23 ];
  result[ 31 ] = pad[ 7 ];

  result[ 32 ] = sequence_num[ 24 ];
  result[ 33 ] = sequence_num[ 25 ];
  result[ 34 ] = sequence_num[ 26 ];
  result[ 35 ] = pad[ 8 ];

  result[ 36 ] = sequence_num[ 27 ];
  result[ 37 ] = sequence_num[ 28 ];
  result[ 38 ] = sequence_num[ 29 ];
  result[ 39 ] = pad[ 9 ];

  result[ 40 ] = sequence_num[ 30 ];
  result[ 41 ] = sequence_num[ 31 ];
  result[ 42 ] = pad[ 10 ];
  result[ 43 ] = pad[ 11 ];*/

}

void um_unpad_header(char *sequence_num, char *digest, char *padded_header) {
  unsigned int i = 0;

  for (i = 0; i < 10; i++) {
    sequence_num[ (i * 3) ]     = padded_header[ (i * 4) ];
    sequence_num[ (i * 3) + 1 ] = padded_header[ (i * 4) + 1 ];
    sequence_num[ (i * 3) + 2 ] = padded_header[ (i * 4) + 2 ];
  }
  sequence_num[ 30 ] = padded_header[ 40 ];
  sequence_num[ 31 ] = padded_header[ 41 ];

  for (i = 0; i < 20; i++)
    digest[ i ] = padded_header[ 44 + i ];
}


int set_entropy_file(char *file_path) {
  if (entropy_file_set == 0) {
    /*char buffer[ 32 ];

    memset(buffer, 0, sizeof(buffer));*/

    bc_log("utils.c", BC_LOG_INFO, "set_entropy_file: [%s]", file_path);
    gcry_control(GCRYCTL_SET_RANDOM_SEED_FILE, file_path);

    if (is_regular_file(file_path) != 1) {
      bc_log("utils.c", BC_LOG_INFO, "GETTING RANDOM BYTES, Y0");
      gcry_control(GCRYCTL_FAST_POLL);
      gcry_free(gcry_random_bytes(8, GCRY_VERY_STRONG_RANDOM));
    }

    /*gcry_randomize(buffer, 16, GCRY_VERY_STRONG_RANDOM);*/

    entropy_file_set = 1;
    return 1;
  } else
    return -1;
}

int update_entropy_file(void) {
  if (entropy_file_set == 1) {
    bc_log("utils.c", BC_LOG_INFO, "update_entropy_file: called: %d",
	   gcry_control(GCRYCTL_UPDATE_RANDOM_SEED_FILE));
    return 1;
  } else
    return -1;
}
