/**
 * utils.h
 *
 * Copyright (C) 2004-2005, J. Salvatore Testa II
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef UTILS_H
#define UTILS_H

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#include <ctype.h>
#include <fcntl.h>
#include <gcrypt.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#ifdef _WIN32
#include <windows.h>
#else
#include <pthread.h>
#endif

#define BC_LOG_INFO 0
#define BC_LOG_DEBUG 1
#define BC_LOG_WARNING 2
#define BC_LOG_ERROR 3

#define LOGFILE_PATH_SIZE 256

#define MXFER_TYPE_ERROR -1
#define MXFER_TYPE_UNUSED 0
#define MXFER_TYPE_MESSAGE_CRYPTED 1
#define MXFER_TYPE_MESSAGE_CLEAR 2
#define MXFER_TYPE_MESSAGE_ACTIONABLE 3
#define MXFER_TYPE_MESSAGE_NONACTIONABLE 4
#define MXFER_TYPE_MESSAGE_KEYFP 5

/* Default size for new BCStrings with no default text. */
#define DEFAULT_ALLOC_SIZE 16

/* 1 here means that strings will be calloc()'ed during reallocation to ensure
 * that the buffer is zeroed out first.  0 will cause the buffer to be
 * realloc()'ed instead, which is faster and more efficient, but does not give
 * the extra security of zeroed space.  Standard uses for this software won't
 * notice the difference of a 1-setting.  Embedded or intensive applications
 * probably want to set this to 0. */
#define EXTRA_SAFE_BCSTRINGS 1

typedef struct _BCSList BCSList;
struct _BCSList {
  void *data;
  struct _BCSList *next;
};

typedef struct _BCString BCString;
struct _BCString {
  char *str;
  size_t str_len;
  size_t allocated_len;
};

typedef struct _MXFER MXFER;
struct _MXFER {
  int type;
  BCSList *result_sizes;
  BCSList *results;
  BCString *error;
};

#ifdef _WIN32
  #define SLASH_CHAR '\\'
  typedef CRITICAL_SECTION BC_MUTEX;
  typedef HANDLE BC_THREAD;
#else
  #define SLASH_CHAR '/'
  typedef pthread_mutex_t BC_MUTEX;
  typedef pthread_t BC_THREAD;
#endif

/* Base64's a block of data. Returns 1 on success, and -1 if the output buffer
 * was too small or some other error occured. */
int base64_decode(unsigned char *output, size_t output_len, unsigned char *input);

/* Decodes a base64 string.  Returns the number of bytes decoded into the
 * output buffer, or -1 on error. */
int base64_encode(unsigned char *output, size_t output_len,
                  unsigned char *input, size_t input_len);

/* Logs text to the log file, if one was specified at startup. */
void bc_log(char *function, unsigned int type, char *format, ...);

/* Normalizes a string.  This consists of lowercasing all characters and
 * stripping out all spaces.  The character array argument is modified
 * in-place. */
char *bc_normalize(char *txt);

/* Creates a new BCString set to a default value (may be NULL).   Note that the
 * default value must be a null-terminated string. */
BCString *bc_string_new(char *init_data);

/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
unsigned int bc_startswith(char *line, char *tofind);

/* Concatenates a string with another string.  This function guarantees that
 * dest will be null terminated. */
unsigned int bc_strlcat(char *dest, char *src, size_t dest_len);

/* Copies src into dest and ensures that dest will be null-terminated
 * afterwards. */
unsigned int bc_strlcpy(char *dest, char *src, size_t dest_len);

/* Appends a null-terminated string to the end of this BCString.  The string
 * will be reallocated as necessary. */
BCString *bc_string_append(BCString *string, char *new_data);

/* Appends binary data to the end of this BCString.  The string will be
 * reallocated as necessary. */
BCString *bc_string_append_len(BCString *string,
			       char *new_data,
			       size_t new_data_len);

/* Appends a character array to the end of a BCString using the printf
 * formatting features. */
void bc_string_append_printf(BCString *string, const char *format, ...);

/* Clears a BCString. */
void bc_string_clear(BCString *string);

/* This frees a string.  If the caller is still using the character string
 * but wants to kill this BCString, then 'free_everything' can be set to 0.
 * In this case, the caller assumes the responsibility of free()'ing the
 * character string later.  A pointer to the internal character array is
 * returned if the caller chose not to free it.  In all cases the string
 * argument be set to NULL to prevent subsequent (and accidental) use. */
char *bc_string_free(BCString **string, unsigned int free_everything);

/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
unsigned int bc_string_startswith(BCString *string, BCString *tofind);

/* Returns 1 if the second argument is a prefix is the first argument, and 0 if
 * otherwise. */
unsigned int bc_string_startswith_char(BCString *string, char *tofind);

/* Trims a newline off the end of a string, if it exists. */
void trim_line(char *line);

/* Retrieves the data from the nth element in the BCSList. */
void *bc_slist_nth_data(BCSList *list, unsigned int i);

/* Appends data to the end of a BCSList.  Returns a pointer to the new head
 * of the list (this changes if the list was previously NULL). */
BCSList *bc_slist_append(BCSList *list, void *data);

/* Frees a BCSList.  The data pointers held inside the list are not
 * modified. */
void bc_slist_free(BCSList **list);

/* Returns the length of a BCSList. */
unsigned int bc_slist_length(BCSList *list);

/* Creates a new mutex. */
BC_MUTEX *bc_mutex_new(void);

/* Locks a mutex. */
void bc_mutex_lock(BC_MUTEX *mutex);

/* Unlocks a mutex. */
void bc_mutex_unlock(BC_MUTEX *mutex);

/* Destroys a mutex. */
void bc_mutex_destroy(BC_MUTEX **mutex);

/* Creates a thread. */
#ifdef _WIN32
BC_THREAD *bc_thread_create(DWORD (WINAPI *func) (void *arg));
#else
BC_THREAD *bc_thread_create(void *(*func)(void *));
#endif


int is_symlink(char *path);
int is_regular_file(char *path);

/*void bc_thread_join(pthread_t *thread_handle);*/
void check_for_debug_file(char *load_save_dir);
void enable_logging(char *path);

int bc_delete_file(char *file);
int um_unpad_message(char *output, size_t output_len,
		     char *input, size_t input_len);
unsigned int um_pad_message(char *output, size_t output_len,
			    char *message);
int ends_with_slash(char *directory);
void hex_dump(char *title, char *buffer,
	      unsigned int buffer_len);

int set_entropy_file(char *file_path);
int update_entropy_file(void);

#define HEADER_PAD_SIZE 64

/* Pads a message header given a sequence number and message digest.  A random
 * byte is appended to every third byte of the sequence, then the digest is
 * tacked on at the end.  The sequence number MUST be 32 bytes and the digest
 * MUST be 20 bytes.  The result is always a 64-byte header. */
void um_pad_header(char *result, char *sequence_num, char *digest);

void um_unpad_header(char *sequence_num, char *digest, char *padded_header);

#endif
