/**
 * shared_crypto_executable.c
 *
 * Copyright (C) 2004-2005, Joe Testa
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#ifdef BUILD_EXECUTABLE

#include "shared_crypto.h"

#ifdef DMALLOC
  #include "dmalloc.h"
#endif

#ifdef _WIN32
   HANDLE hOutputRead = NULL;
   HANDLE hInputWrite = NULL;
   HANDLE hErrorRead = NULL;
   HANDLE hChildProcess = NULL;
#else
   int pipe_to_mod[ 2 ];
   int pipe_from_mod[ 2 ];
   int pipe_from_mod2[ 2 ];
#endif

void mod_kill() {
#ifdef _WIN32
   TerminateProcess(hChildProcess, 0);
#endif
}

void run_module() {
#ifdef _WIN32
   int ret = 0;
   char bc_path[ MAX_PATH ];
   TCHAR current_dir[ MAX_PATH ];

   char tmp[ 128 ];
   

   HANDLE hOutputReadTmp, hOutputWrite;
   HANDLE hInputWriteTmp, hInputRead;
   HANDLE hErrorReadTmp, hErrorWrite;
   /*HANDLE hThread;*/

    PROCESS_INFORMATION processInformation;
    STARTUPINFO startupInfo;

    SECURITY_ATTRIBUTES securityAttributes;

    memset(current_dir, 0, MAX_PATH);

    securityAttributes.nLength = sizeof(SECURITY_ATTRIBUTES);
    securityAttributes.lpSecurityDescriptor = NULL;
    securityAttributes.bInheritHandle = TRUE;

    CreatePipe(&hOutputReadTmp, &hOutputWrite, &securityAttributes, 0);
    CreatePipe(&hInputRead, &hInputWriteTmp, &securityAttributes, 0);
    CreatePipe( &hErrorReadTmp, &hErrorWrite, &securityAttributes, 0 );


    DuplicateHandle(GetCurrentProcess(), hOutputReadTmp, GetCurrentProcess(),
                    &hOutputRead, 0, TRUE, DUPLICATE_SAME_ACCESS);
    DuplicateHandle(GetCurrentProcess(), hInputWriteTmp, GetCurrentProcess(),
                    &hInputWrite, 0, TRUE, DUPLICATE_SAME_ACCESS);
    DuplicateHandle(GetCurrentProcess(), hErrorReadTmp, GetCurrentProcess(),
                    &hErrorRead, 0, TRUE, DUPLICATE_SAME_ACCESS);

    CloseHandle(hOutputReadTmp);
    CloseHandle(hInputWriteTmp);
    CloseHandle(hErrorReadTmp);

    ZeroMemory(&startupInfo, sizeof(STARTUPINFO));
    startupInfo.cb = sizeof(STARTUPINFO);
    startupInfo.dwFlags = STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW;
    startupInfo.hStdOutput = hOutputWrite;
    startupInfo.hStdInput  = hInputRead;
    startupInfo.hStdError  = hErrorWrite;


    GetCurrentDirectory(sizeof(current_dir) - 1, current_dir);
    bc_strlcpy(bc_path, current_dir, sizeof(bc_path) - 1);
    bc_strlcat(bc_path, "\\..\\src\\scatterchatmod.exe", sizeof(bc_path) - 1);

    bc_strlcpy(tmp, "scatterchatmod.exe --log blehXX.txt", sizeof(tmp));

    tmp[ 27 ] = 'a' + (rand() % 26);
    tmp[ 28 ] = 'a' + (rand() % 26);

    ret = CreateProcess(bc_path, tmp, NULL, NULL, TRUE, 0, NULL, NULL,
			&startupInfo, &processInformation);
    if (ret == 0) {
       printf("CreateProcess failed.\n");  fflush(stdout);
       return;
    }


    hChildProcess = processInformation.hProcess;
    CloseHandle(processInformation.hThread);

    CloseHandle(hOutputWrite);
    CloseHandle(hInputRead);
    CloseHandle(hErrorWrite);
    return;
#else
  pid_t pid = 0;
  char buffer[ 256 ];

  memset(buffer, 0, sizeof(buffer));

  if ((pipe(pipe_to_mod) < 0) || (pipe(pipe_from_mod) < 0) ||
      (pipe(pipe_from_mod2) < 0)) {
    fprintf(stderr, "Pipe error!\n");
    exit(1);
  }

  /* read from pipes[ 0 ], write to pipes[ 1 ]. */
  pid = fork();
  if (pid < 0) {
    fprintf(stderr, "Error: could not fork()!\n");
    exit(1);
  } else if (pid == 0) {
    /* Child. */
    close(pipe_from_mod[ 0 ]);
    close(pipe_from_mod2[ 0 ]);
    close(pipe_to_mod[ 1 ]);


    dup2(pipe_from_mod[ 1 ], fileno(stdout));
    dup2(pipe_from_mod2[ 1 ], fileno(stderr));
    dup2(pipe_to_mod[ 0 ], fileno(stdin));



    /*printf("child execing...\n");*/
    execlp("../src/scatterchatmod", "../src/scatterchatmod", NULL);
    printf("Error: execv returned!\n");
    exit(1);
  } else if (pid > 0) {
    char buffer[ 256 ];

    memset(buffer, 0, sizeof(buffer));

    /* Parent. */
    close(pipe_from_mod[ 1 ]);
    close(pipe_from_mod2[ 1 ]);
    close(pipe_to_mod[ 0 ]);

  }
#endif
}


void mod_write(char *string) {
#ifdef _WIN32
  DWORD bytesWritten = 0;
  WriteFile(hInputWrite, string, strlen(string), &bytesWritten, NULL);
#else
  write(pipe_to_mod[ 1 ], string, strlen(string));
#endif
}


void mod_writeln(char *string) {
#ifdef _WIN32
  DWORD bytesWritten = 0;
  WriteFile(hInputWrite, string, strlen(string), &bytesWritten, NULL);
  WriteFile(hInputWrite, "\n", strlen("\n"), &bytesWritten, NULL);
#else
  write(pipe_to_mod[ 1 ], string, strlen(string));
  write(pipe_to_mod[ 1 ], "\n", strlen("\n"));
#endif
}


/*void mod_writebin(unsigned char *binary_data, ssize_t length) {
  write(pipe_to_mod[ 1 ], binary_data, length);
  }*/


int mod_readline(char *buffer, size_t buffer_len) {
#ifdef _WIN32
  DWORD bytesRead = 0;
  char one_char_buf[ 2 ] = { '\0', '\0' };
  unsigned int i = 0;
  unsigned int crlf_found = 0;

  while(!crlf_found && (i < buffer_len - 1)) {
     if (!ReadFile(hOutputRead, one_char_buf, 1, &bytesRead, NULL))
        return -1;

    buffer[ i ] = one_char_buf[ 0 ];
    i++;

    if ((i >= 2) && (buffer[ i - 2 ] == '\r') && (buffer[ i - 1 ] == '\n'))
      crlf_found = 1;

  }

  buffer[ i ] = '\0';

  /* If a CRLF was read in, truncate it. */
  if (crlf_found == 1) {
    trim_crlf(buffer);
    return i - 3;
  } else
    return -1;
#else
  char one_char_buf[ 1 ] = { '\0' };
  unsigned int i = 0;
  unsigned int crlf_found = 0;


  while(!crlf_found && (i < buffer_len - 1)) {
    read(pipe_from_mod[ 0 ], one_char_buf, 1);

    buffer[ i ] = one_char_buf[ 0 ];
    i++;

    if ((i >= 2) && (buffer[ i - 2 ] == '\r') && (buffer[ i - 1 ] == '\n'))
      crlf_found = 1;
  }

  buffer[ i ] = '\0';

  /* If a CRLF was read in, truncate it. */
  if (crlf_found == 1)
    trim_crlf(buffer);

  return i - 3;
#endif
}


int mod_readdata(char *buffer, size_t buffer_len, size_t bytes_to_read) {
#ifdef _WIN32
   DWORD bytesRead = 0;

   if (bytes_to_read > buffer_len) {
      buffer[ 0 ] = '\0';
      return -1;
   }

   if (!ReadFile(hOutputRead, buffer, bytes_to_read, &bytesRead, NULL))
      return -1;
   else
      return (int)bytesRead;
#else
  int ret = 0;

  if (bytes_to_read > buffer_len) {
    buffer[ 0 ] = '\0';
    return -1;
  }

  if ((ret = read(pipe_from_mod[ 0 ], buffer, bytes_to_read)) > 0)
    return ret;
  else
    return 0;

#endif
}


void trim_crlf(char *buffer) {
  size_t buffer_len = strlen(buffer);

  if (buffer_len < 2)
    return;

  if ((buffer[ buffer_len - 2 ] == '\r') &&
      (buffer[ buffer_len - 1 ] == '\n')) {
    buffer[ buffer_len - 2 ] = '\0';
  }
}


void init_module() {
  unsigned int num_ids = 0;
  unsigned int num_fps = 0;
  unsigned int i = 0;
  unsigned int j = 0;
  char *pipe_pos = NULL;

  char buffer[ 256 ];
  char id_name[ 256 ];
  char id_pw[ 32 ];


  memset(buffer, 0, sizeof(buffer));
  memset(id_name, 0, sizeof(id_name));
  memset(id_pw, 0, sizeof(id_pw));

  mod_readline(buffer, sizeof(buffer) - 1);
  printf("Using scatterchat module version %s (API version ", buffer);fflush(stdout);

  mod_readline(buffer, sizeof(buffer) - 1);
  printf("%s).\n", buffer);fflush(stdout);

  /*mod_writeln("NOOP");
  mod_readline(buffer, sizeof(buffer) - 1);
  printf("returned from noop: [%s]\n", buffer);*/


  printf("Enter the load/save directory for the encryption keys: ");
  fflush(stdout);
  console_read_line(load_save_dir, sizeof(load_save_dir));

  mod_write("INIT ");
  mod_writeln(load_save_dir);

  mod_readline(buffer, sizeof(buffer) - 1);
  /*printf("init result: %s\n", buffer);
    fflush(stdout);*/
  if (strstr(buffer, "ERROR") != NULL) {
    printf("Error while initializing directory: %s\n", buffer);
    exit(1);
  }


  mod_readline(buffer, sizeof(buffer) - 1);
  num_ids = (unsigned int)atoi(buffer);


  if (num_ids == 0) {
    printf("No IDS were found in %s.\n", load_save_dir);
    fflush(stdout);
  } else {

    printf("+-------------------------------------------------------------+\n");
    printf("| LOADED IDS:                                                 |\n");
    printf("+-------------------------------------------------------------+\n");
    printf("|                                                             |\n");
    for (i = 0; i < num_ids; i++) {
      mod_readline(buffer, sizeof(buffer) - 1);
      pipe_pos = strchr(buffer, '|');
      if (pipe_pos == NULL) {
	printf("Dang.  The module returned an invalid ID line: [%s]\n",buffer);
      } else {
	char *id = buffer;
	char *fp = pipe_pos + 1;

	*pipe_pos = '\0';

	printf("| ID: %s", id);
	for(j = 0; j < (56 - strlen(id)); j++)
	  printf(" ");
	printf("|\n");


        printf("| %s |\n", fp);
	printf("|                                                             |\n");
      }

      /*printf("line: %s\n", buffer);*/
    }
    printf("+-------------------------------------------------------------+\n");
    fflush(stdout);
  }

  mod_readline(buffer, sizeof(buffer) - 1);
  num_fps = (unsigned int)atoi(buffer);

  if (num_fps == 0) {
    printf("No peer fingerprints are in the database.\n");
  } else {

    printf("+-------------------------------------------------------------+\n");
    printf("| LOADED DATABASE                                             |\n");
    printf("+-------------------------------------------------------------+\n");
    printf("|                                                             |\n");

    for (i = 0; i < num_fps; i++) {
      mod_readline(buffer, sizeof(buffer) - 1);
      pipe_pos = strchr(buffer, '|');
      /*printf("line: %s\n", buffer);*/
      pipe_pos = strchr(buffer, '|');
      if (pipe_pos == NULL) {
	printf("Dang.  The module returned an invalid ID line: [%s]\n",buffer);
      } else {
	char *id = buffer;
	char *fp = pipe_pos + 1;

	*pipe_pos = '\0';

	printf("| ID: %s", id);
	for(j = 0; j < (56 - strlen(id)); j++)
	  printf(" ");
	printf("|\n");


        printf("| %s |\n", fp);
	printf("|                                                             |\n");
      }
    }
    printf("+-------------------------------------------------------------+\n");
    fflush(stdout);

  }


  if (num_ids == 0) {

    printf("Do you want me to generate an ID now? (y/N) ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));
    if (strcmp(buffer, "y") != 0) {
      fprintf(stderr, "Can't continue without at least one available ID.  "
	              "Exiting...\n");
      exit(1);
    }


    printf("What name would you like to give this new ID? ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));

    bc_strlcpy(id_name, buffer, sizeof(id_name));


    printf("Enter a password to protect the new key with (NOTE: the password will be printed on the screen!): ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));

    bc_strlcpy(id_pw, buffer, sizeof(id_pw));


    printf("Generating key.  Please wait...\n");
    fflush(stdout);
    mod_write("GENERATE_KEY ");
    mod_write(id_name);
    mod_write("|");
    mod_writeln(id_pw);


    mod_readline(buffer, sizeof(buffer) - 1);
    if (strstr(buffer, "ERROR") != NULL) {
      fprintf(stderr, "Error: %s\n", buffer);
      exit(1);
    }

    mod_readline(buffer, sizeof(buffer) - 1);
    printf("Key generated successfully.  Its fingerprint is:\n%s\n", buffer);
    printf("This ID will be used for the current session.\n");
    fflush(stdout);
  } else {
    printf("Which ID would you like to use? ");
    fflush(stdout);
    console_read_line(id_name, sizeof(id_name));

    printf("Enter a password to unlock this ID (NOTE: the password will be printed on the screen!): ");
    fflush(stdout);
    console_read_line(id_pw, sizeof(id_pw));
  }


  mod_write("SELECT_USER ");
  mod_write(id_name);
  mod_write("|");
  mod_writeln(id_pw);

  mod_readline(buffer, sizeof(buffer) - 1);
  if (strstr(buffer, "ERROR") != NULL) {
    fprintf(stderr, "Error: %s\n", buffer);
    exit(1);
  }

  /*noop("init_module");*/

  printf("Module has been initialized successfully.\n");
  fflush(stdout);
}


void console_read_line(char *buffer, size_t buffer_len) {

  fgets(buffer, buffer_len - 1, stdin);
  if (strlen(buffer) > 0)
    buffer[ strlen(buffer) - 1 ] = '\0';

}


#ifdef _WIN32
void send_message(SOCKET socket, char *who, char *message) {
#else
void send_message(int socket, char *who, char *message) {
#endif
  unsigned int i = 0;
  char *data_ptr = NULL;
  size_t total_size = 0;
  unsigned int offset = 0;
  unsigned int type = 0;
  unsigned int num_results = 0;

  char line[ 2048 ];
  char text_len[ 16 ];
  size_t size_array[ 4 ];

  /*noop("send_message");*/

  memset(text_len, 0, sizeof(text_len));
  memset(line, 0, sizeof(line));
  memset(size_array, 0, sizeof(size_array));

  sprintf(text_len, "%u", (unsigned int)strlen(message));

  mod_write("SEND ");
  mod_write(who);
  mod_write("|");
  mod_writeln(text_len);
  mod_writeln(message);

  mod_readline(line, sizeof(line) - 8);

  if (bc_startswith(line, "ERROR")) {
    printf("Error: %s\n", line);
    fflush(stdout);
    exit(1);
  }

  parseOK(line, &type, &num_results);

  /*printf("send_message: type: %d num_results: %d\n", type, num_results);
  fflush(stdout);*/

  total_size = get_sizes(size_array, sizeof(size_array));

  /*printf("total_size: %u\n", (unsigned int)total_size);
    fflush(stdout);*/

  data_ptr = (char *)calloc(total_size + 8, sizeof(char));
  if (data_ptr == NULL) {
    fprintf(stderr, "Can't calloc!\n");
    exit(1);
  }

  mod_readdata(data_ptr, total_size, total_size);
  mod_readline(line, sizeof(line) - 1);

  /*printf("mod_readdata: [%s]\n", data_ptr);
  fflush(stdout);*/

  for(i = 0; i < num_results; i++) {
    /*int k = 0;
    printf("\n\t\tblock: ");
    for (k = 0; k < size_array[ i ]; k++)
      printf("%c", *(data_ptr + offset + k));
    printf("\n\n");
    fflush(stdout);*/

    send(socket, (u_int32_t *)&size_array[ i ], sizeof(u_int32_t), 0);
    send(socket, data_ptr + offset, size_array[ i ], 0);
    /*my_sleep(1000);*/

    offset += size_array[ i ];
  }

  free(data_ptr);  data_ptr = NULL;
}


#ifdef _WIN32
int get_message(SOCKET socket, char *who, char *raw_message,
		size_t raw_message_len) {
#else
int get_message(int socket, char *who, char *raw_message,
		size_t raw_message_len) {
#endif
  unsigned int type = 0;
  unsigned int num_results = 0;
  size_t total_size = 0;
  unsigned int offset = 0;
  unsigned int i = 0;
  char *data_ptr = NULL;

  char line[ 256 ];
  char text_len[ 16 ];
  size_t size_array[ 4 ];

  memset(line, 0, sizeof(line));
  memset(text_len, 0, sizeof(text_len));
  memset(size_array, 0, sizeof(size_array));

  /*noop("get_message");*/

  /*printf("\t\tGET_MESSAGE: [%u][%s]\n\n", strlen(raw_message), raw_message);  fflush(stdout);*/

  sprintf(text_len, "%u", (unsigned int)raw_message_len);

  mod_write("PROCESS ");
  mod_write(who);
  mod_write("|");
  mod_writeln(text_len);
  mod_writeln(raw_message);

  /*printf("Sent the process stuff...\n");  fflush(stdout);*/
  mod_readline(line, sizeof(line) - 8);
  /*printf("bleh: [%s]\n", line);  fflush(stdout);*/

  if (bc_startswith(line, "ERROR")) {
    printf("Error: %s\n", line);  fflush(stdout);
    exit(1);
  }

  /*printf("x[%s]\t\t\n", line);  fflush(stdout);*/
  parseOK(line, &type, &num_results);
  /*printf("get_message(): type: %d, num_results: %d\n", type, num_results);  fflush(stdout);*/

  if (type == 4) {
    /*printf("Got type 4.\n");  fflush(stdout);*/
    /*my_sleep(100);*/
    return 1;
  }

  total_size = get_sizes(size_array, sizeof(size_array));
  /*printf("total_size: %u\n", (unsigned int)total_size);  fflush(stdout);*/

  data_ptr = (char *)calloc(total_size + 8, sizeof(char));
  if (data_ptr == NULL) {
    fprintf(stderr, "Can't calloc!\n");
    exit(1);
  }

  mod_readdata(data_ptr, total_size, total_size);
  mod_readline(line, sizeof(line) - 1);

  if (type == 5) {
    /*printf("fp: %s\n", data_ptr);  fflush(stdout);*/
    handle_fingerprint(socket, who, data_ptr);
    /*my_sleep(100);*/
  } else if (type == 3) {
    /*send(socket, data_ptr, size_array[ i ], 0);*/
    for(i = 0; i < num_results; i++) {
      /*int k = 0;
      printf("\t\tblock: ");
	for (k = 0; k < size_array[ i ]; k++)
	printf("%c", *(data_ptr + offset + k));
	printf("\n\n");*/
      /*my_sleep(500);*/

      send(socket, (u_int32_t *)&size_array[ i ], sizeof(u_int32_t), 0);
      send(socket, data_ptr + offset, size_array[ i ], 0);
      /*usleep(333333);*/  /* Sleep 1/3rd of a second. */

      offset += size_array[ i ];
    }
  } else if (type == 1) {
    printf("Them:\t%s\n", data_ptr);
    fflush(stdout);
  } else if (type == 2) {
    printf("Plaintext message received! (Uh-oh!): [%s]\n", data_ptr);
    fflush(stdout);
  }

  return 0;
}

#ifdef _WIN32
int handle_chat(SOCKET fd) {
  int continue_flag = 1;
  DWORD thread_id = 0;

  char line[ 128 ];

  memset(line, 0, sizeof(line));

  /*noop("handle_chat");*/

  CreateThread(NULL, 512, read_thread, &fd, 0, &thread_id);
  while(continue_flag == 1) {
    console_read_line(line, sizeof(line) - 1);
    if((strcmp(line, "exit") == 0) || (strcmp(line, "quit") == 0)) {
      continue_flag = 0;
      mod_kill();
    } else {
      send_message(fd, "Them", line);
    }
  }

  if (continue_flag == 0)
    return -1;
  else
    return 1;
}

DWORD WINAPI read_thread(void *ptr) {
  SOCKET *fd = (SOCKET *)ptr;
  u_int32_t the_size = 0;
  int len = 0;
  unsigned int continue_flag = 1;

  char buffer[ 2048 ];

  memset(buffer, 0, sizeof(buffer));

  while(continue_flag == 1) {
    /*if (get_message(*fd, buffer, sizeof(buffer)) < 0)
      break;*/
    len = recv(*fd, &the_size, sizeof(u_int32_t), 0);
    if (len < 0) {
      continue_flag = 0;
    } else if (the_size > sizeof(buffer) - 1) {
      printf("The size of the message (%u) is larger than the size of the buffer (%u)!\n", the_size, sizeof(buffer));  fflush(stdout);
      continue_flag = 0;
    } else {
      size_t read_so_far = 0;

      do {
	len = recv(*fd, buffer, the_size, 0);
	if (len > 0) {
	  read_so_far += len;
	  buffer[ len ] = '\0';
	  get_message(*fd, "Them", buffer, strlen(buffer));
	} else
	  continue_flag = 0;
      } while ((continue_flag == 1) && (read_so_far < the_size));
    }
    /*printf("Them:\t%s\n", buffer);
      fflush(stdout);*/
  }
  exit(0);
}
#else
int handle_chat(int fd) {
  int retval = 0;
  int len = 0;
  unsigned int continue_flag = 0;
  u_int32_t the_size = 0;

  char line[ 2048 ];
  fd_set rfds;
  struct timeval tv;

  memset(line, 0, sizeof(line));
  memset(&rfds, 0, sizeof(fd_set));
  memset(&tv, 0, sizeof(struct timeval));

  FD_SET(0, &rfds);
  FD_SET(fd, &rfds);


  continue_flag = 1;
  while(continue_flag == 1) {
    tv.tv_sec = 1;
    tv.tv_usec = 0;
    /*    retval = select(fd + 1, &rfds, NULL, NULL, &tv);*/
    retval = select(fd + 1, &rfds, NULL, NULL, NULL);
    if (retval) {
      if (FD_ISSET(0, &rfds)) {
	console_read_line(line, sizeof(line) - 1);
	if ((strcmp(line, "exit") == 0) || strcmp(line, "quit") == 0)
	  continue_flag = 0;
	else if (strlen(line) > 0) {
	  printf("You:\t%s\n", line);
	  /*send(fd, line, strlen(line), 0);*/
	  send_message(fd, "Them", line);
	  FD_SET(fd, &rfds);
	}
      } else if (FD_ISSET(fd, &rfds)) {
	len = recv(fd, &the_size, sizeof(u_int32_t), 0);
	if (len < 0) {
	  continue_flag = 0;
	} else if (the_size > sizeof(line) - 1) {
	  printf("The size of the message (%u) is larger than the size of the buffer (%u)!\n", (unsigned int)the_size, (unsigned int)sizeof(line));  fflush(stdout);
	  continue_flag = 0;
	} else {
	  size_t read_so_far = 0;

	  do {
	    len = recv(fd, line, the_size, 0);
	    if (len > 0) {
	      read_so_far += len;
	      line[ len ] = '\0';
	      get_message(fd, "Them", line, strlen(line));
	      FD_SET(0, &rfds);
	    } else
	      continue_flag = 0;
	  } while ((continue_flag == 1) && (read_so_far < the_size));
	}
      } else {
	FD_SET(0, &rfds);
	FD_SET(fd, &rfds);
      }

    }

  }


  if (continue_flag == 0)
    return -1;
  else
    return 1;
}
#endif

int parseOK(char *line, unsigned int *type, unsigned int *num_results) {
  char *space_pos = NULL;

  if ((strlen(line) < 4) || (bc_startswith(line, "OK ") == 0))
    return -1;

  space_pos = strchr(line + 3, ' ');
  if (space_pos == NULL)
    return -1;

  *space_pos = '\0';
  *type = atoi(line + 3);
  *num_results = atoi(space_pos + 1);

  return 1;
}

size_t get_sizes(size_t *size_array, size_t size_array_len) {
  unsigned int i = 0;
  unsigned int j = 0;
  unsigned int k = 0;
  int ret = 0;

  char line[ 256 ];
  char text_size[ 16 ];

  memset(line, 0, sizeof(line));
  memset(text_size, 0, sizeof(text_size));


  mod_readline(line, sizeof(line) - 1);
  for(i = 0; i < strlen(line); i++) {

    if ((line[ i ] != ' ') && (j < sizeof(text_size) - 1)) {
      text_size[ j ] = line[ i ];
      j++;
    } else if (k < size_array_len - 1) {
      size_array[ k ] = (size_t)atoi(text_size);
      k++;

      /*printf("text_size: [%s]\n", text_size);*/
      memset(text_size, 0, sizeof(text_size));
      j = 0;
    }

  }

  for (i = 0; i < k; i++) {
    ret += size_array[ i ];
  }

  return ret;
}


#ifdef _WIN32
void handle_fingerprint(SOCKET socket, char *who, char *fingerprint) {
#else
void handle_fingerprint(int socket, char *who, char *fingerprint) {
#endif
  unsigned int type = 0;
  unsigned int num_results = 0;
  size_t total_size = 0;
  unsigned int i = 0;
  unsigned int offset = 0;
  int selection = 0;
  char *data_ptr = NULL;

  size_t size_array[ 4 ];
  char line[ 512 ];

  memset(size_array, 0, sizeof(size_array));
  memset(line, 0, sizeof(line));


#ifdef _WIN32
  /* This is really lame.  Because of the ugly Win32 select() hack, we can't
   * read from the console in this thread, so we have to use a message box
   * to get input from the user... */

  bc_strlcpy(line, "New fingerprint: ", sizeof(line));
  bc_strlcat(line, fingerprint, sizeof(line));
  bc_strlcat(line, "\n\nClick 'Yes' to accept this key permanently\n\nClick 'No to accept this key temporarily\n\nClick 'Cancel' to reject this key.\n\n(P.S.:  this dialog box is UGLY!)", sizeof(line));

  selection = MessageBox(NULL, line, "New Fingerprint Found!", MB_YESNOCANCEL | MB_ICONQUESTION | MB_TOPMOST);
  if (selection == IDYES)
    selection = 2;
  else if (selection == IDNO)
    selection = 1;
  else
    selection = 3;

#else
  printf("KEY FINGERPRINT: [%s]\n\n", fingerprint);
  printf("\t(1) Accept key temporarily\n");
  printf("\t(2) Accept key permanently\n");
  printf("\t(3) Reject key\n\n");
  printf("> ");  fflush(stdout);
  console_read_line(line, sizeof(line) - 1);

  /*printf("selection: [%s]\n", line);*/
  selection = atoi(line);
  if ((selection != 1) && (selection != 2))
    selection = 3;
#endif
  
  if (selection == 3) {
    mod_write("KEY_REJECT ");
    mod_writeln(who);
    mod_readline(line, sizeof(line) - 1);
    if (strcmp(line, "OK") == 0)
      printf("Key rejected.\n");
    else
      printf("Error: %s\n", line);
    return;
  } else if (selection == 1) {
    mod_write("KEY_ACCEPT_TEMPORARY ");
    mod_write(who);
    mod_write("|");
    mod_writeln(who);
  } else if (selection == 2) {
    mod_write("KEY_ACCEPT_PERMANENTLY ");
    mod_write(who);
    mod_write("|");
    mod_writeln(who);
  }

  mod_readline(line, sizeof(line) - 8);
  if (bc_startswith(line, "ERROR")) {
    printf("Error: %s\n", line);
    return;
  }
  
  parseOK(line, &type, &num_results);

  total_size = get_sizes(size_array, sizeof(size_array));
  /*printf("total_size: %u\n", (unsigned int)total_size);*/
  
  data_ptr = (char *)calloc(total_size + 8, sizeof(char));
  if (data_ptr == NULL) {
    fprintf(stderr, "Can't calloc!\n");
    exit(1);
  }

  mod_readdata(data_ptr, total_size, total_size);
  mod_readline(line, sizeof(line) - 1);
    
  for(i = 0; i < num_results; i++) {

    /*printf("block: ");
      for (k = 0; k < size_array[ i ]; k++)
      printf("%c", *(data_ptr + offset + k));
      printf("\n");*/

    send(socket, (u_int32_t *)&size_array[ i ], sizeof(u_int32_t), 0);
    send(socket, data_ptr + offset, size_array[ i ], 0);
    /*my_sleep(333);*/  /* Sleep 1/3rd of a second. */

    offset += size_array[ i ];
  }
  
}

/*void noop(char *title) {
  char buffer[ 32 ];

  memset(buffer, 0, sizeof(buffer));

  printf("NOOP (%s)...", title);  fflush(stdout);
  mod_writeln("NOOP");
  mod_readline(buffer, sizeof(buffer) - 1);
  printf("done. [%s]\n", buffer);  fflush(stdout);
  }*/

void reset_module(void) {
  char buffer[ 32 ];

  memset(buffer, 0, sizeof(buffer));

  mod_writeln("RESET");
  mod_readline(buffer, sizeof(buffer) - 1);
  if (strstr(buffer, "OK") != buffer) {
    printf("Error while resetting module: %s\n", buffer);
    exit(1);
  }
}

void shutdown_module(void) {
    char buffer[ 32 ];

  memset(buffer, 0, sizeof(buffer));

  mod_writeln("SHUTDOWN");
  mod_readline(buffer, sizeof(buffer) - 1);
  if (strstr(buffer, "OK") != buffer) {
    printf("Error while shutting down module: %s\n", buffer);
    exit(1);
  }
}

#endif
