/**
 * shared_crypto_library.c
 *
 * Copyright (C) 2004-2005, Joe Testa
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "shared_crypto.h"
#include "../src/library.h"

#ifdef BUILD_SHARED
void state_change_callback(char *sn, char *umid, unsigned int operation,
			   unsigned int old_state, unsigned int new_state);
char *parse_state_into_text(unsigned int state);


char them[ 5 ] = { 'T', 'h', 'e', 'm', '\0'};

void mod_kill(void) {
}

void console_read_line(char *buffer, size_t buffer_len) {
  fgets(buffer, buffer_len - 1, stdin);
  if (strlen(buffer) > 0)
    buffer[ strlen(buffer) - 1 ] = '\0';
}


void init_module(void) {
  BC_IDS_AND_FINGERPRINTS *ids_and_fps = NULL;
  int ret = 0;
  unsigned int num_ids = 0;
  unsigned int num_db = 0;
  unsigned int i = 0;
  unsigned int j = 0;

  char bc_version[ 32 ];
  char bc_api_version[ 32 ];
  char buffer[ 256 ];
  char id_name[ 64 ];
  char id_pw[ 64 ];


  memset(bc_version, 0, sizeof(bc_version));
  memset(bc_api_version, 0, sizeof(bc_api_version));
  memset(buffer, 0, sizeof(buffer));
  memset(id_name, 0, sizeof(id_name));
  memset(id_pw, 0, sizeof(id_pw));

  if (bc_startup(bc_version, sizeof(bc_version),
		 bc_api_version, sizeof(bc_api_version),
		 state_change_callback) < 0) {
    printf("bc_startup failed!\n"); fflush(stdout);
  } else {
    printf("Using shared scatterchat module version %s (API version %s).\n", bc_version, bc_api_version); fflush(stdout);
  }

  printf("Enter the load/save directory for the encryption keys: ");
  fflush(stdout);
  console_read_line(load_save_dir, sizeof(load_save_dir));

  if ((ret = bc_init(load_save_dir, &ids_and_fps)) < 0) {
    printf("Error while loading directory: %d\n", ret);
    exit(1);
  }

  num_ids = ids_and_fps->num_mine;
  if (num_ids == 0) {
    printf("No IDS were found in %s.\n", load_save_dir);
    fflush(stdout);
  } else {

    printf("+-------------------------------------------------------------+\n");
    printf("| LOADED IDS:                                                 |\n");
    printf("+-------------------------------------------------------------+\n");
    printf("|                                                             |\n");
    for (i = 0; i < num_ids; i++) {
      printf("| ID: %s", ids_and_fps->my_ids[ i ]);
      for(j = 0; j < (56 - strlen(ids_and_fps->my_ids[ i ])); j++)
	printf(" ");
      printf("|\n");


      printf("| %s |\n", ids_and_fps->my_fps[ i ]);
      printf("|                                                             |\n");
    }

  }
  printf("+-------------------------------------------------------------+\n");
  fflush(stdout);


  num_db = ids_and_fps->num_db;
  if (num_db == 0) {
    printf("No peer fingerprints are in the database.\n");
  } else {

    printf("+-------------------------------------------------------------+\n");
    printf("| LOADED DATABASE                                             |\n");
    printf("+-------------------------------------------------------------+\n");
    printf("|                                                             |\n");
    for (i = 0; i < num_db; i++) {
      printf("| ID: %s", ids_and_fps->db_ids[ i ]);
      for(j = 0; j < (56 - strlen(ids_and_fps->db_ids[ i ])); j++)
	printf(" ");
      printf("|\n");


      printf("| %s |\n", ids_and_fps->db_fps[ i ]);
      printf("|                                                             |\n");
    }
  }
  printf("+-------------------------------------------------------------+\n");
  fflush(stdout);


  if (num_ids == 0) {

    printf("Do you want me to generate an ID now? (y/N) ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));
    if (strcmp(buffer, "y") != 0) {
      fprintf(stderr, "Can't continue without at least one available ID.  "
	              "Exiting...\n");
      exit(1);
    }


    printf("What name would you like to give this new ID? ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));

    bc_strlcpy(id_name, buffer, sizeof(id_name));


    printf("Enter a password to protect the new key with (NOTE: the password will be printed on the screen!): ");
    fflush(stdout);
    console_read_line(buffer, sizeof(buffer));

    bc_strlcpy(id_pw, buffer, sizeof(id_pw));


    printf("Generating key.  Please wait...\n");
    fflush(stdout);


    if ((ret = bc_generate_key(id_name, id_pw, buffer, sizeof(buffer))) < 0) {
      printf("Error while generating key: %d\n", ret);
      exit(1);
    }

    printf("Key generated successfully.  Its fingerprint is:\n%s\n", buffer);
    printf("This ID will be used for the current session.\n");
    fflush(stdout);
  } else {
    printf("Which ID would you like to use? ");
    fflush(stdout);
    console_read_line(id_name, sizeof(id_name));

    printf("Enter a password to unlock this ID (NOTE: the password will be printed on the screen!): ");
    fflush(stdout);
    console_read_line(id_pw, sizeof(id_pw));
  }


  if ((ret = bc_select_user(id_name, id_pw)) < 0) {
    printf("Error while selecting user: %u\n", ret);
    exit(1);
  }



  printf("Module has been initialized successfully.\n");
  fflush(stdout);


  bc_free_ids_and_fingerprints(&ids_and_fps);
}


void run_module(void) {
  
}

void show_usage_and_die(char *prog) {

}

void noop(char *title) {

}



#ifdef _WIN32
int handle_chat(SOCKET fd) {
  int continue_flag = 1;
  DWORD thread_id = 0;

  char line[ 128 ];

  memset(line, 0, sizeof(line));


  CreateThread(NULL, 512, read_thread, &fd, 0, &thread_id);
  while(continue_flag == 1) {
    console_read_line(line, sizeof(line) - 1);
    if((strcmp(line, "exit") == 0) || (strcmp(line, "quit") == 0)) {
      continue_flag = 0;
      mod_kill();
    } else {
      send_message(fd, them, line);
    }
  }
}

DWORD WINAPI read_thread(void *ptr) {
  SOCKET *fd = (SOCKET *)ptr;
  int len = 0;
  unsigned int continue_flag = 1;
  u_int32_t the_size = 0;

  char buffer[ 2048 ];

  memset(buffer, 0, sizeof(buffer));

  while(continue_flag == 1) {
    len = recv(*fd, &the_size, sizeof(u_int32_t), 0);
    if (len < 0)
      continue_flag = 0;
    } else if (the_size > sizeof(buffer) - 1) {
      printf("The size of the message (%u) is larger than the size of the buffer (%u)!\n", the_size, sizeof(buffer));  fflush(stdout);
      continue_flag = 0;
    } else {
      size_t read_so_far = 0;

      do {
	len = recv(*fd, buffer, the_size, 0);
	if (len > 0) {
	  read_so_far += len;
	  buffer[ len ] = '\0';
	  get_message(*fd, them, buffer, strlen(buffer));
	} else
	  continue_flag = 0;
      } while ((continue_flag == 1) && (read_so_far < the_size));
    }

  }
  exit(0);
}

#else
int handle_chat(int fd) {
  int retval = 0;
  int len = 0;
  unsigned int continue_flag = 0;
  u_int32_t the_size = 0;

  char line[ 2048 ];
  fd_set rfds;

  memset(line, 0, sizeof(line));
  memset(&rfds, 0, sizeof(fd_set));

  FD_SET(0, &rfds);
  FD_SET(fd, &rfds);


  continue_flag = 1;
  while(continue_flag == 1) {
    retval = select(fd + 1, &rfds, NULL, NULL, NULL);
    if (retval) {
      if (FD_ISSET(0, &rfds)) {
	console_read_line(line, sizeof(line) - 1);
	if ((strcmp(line, "exit") == 0) || strcmp(line, "quit") == 0)
	  continue_flag = 0;
	else if (strlen(line) > 0) {
	  printf("You:\t%s\n", line);
	  send_message(fd, them, line);
	  FD_SET(fd, &rfds);
	}
      } else if (FD_ISSET(fd, &rfds)) {


	len = recv(fd, &the_size, sizeof(u_int32_t), 0);
	if (len < 0) {
	  continue_flag = 0;
	} else if (the_size > sizeof(line) - 1) {
	  printf("The size of the message (%u) is larger than the size of the buffer (%u)!\n", (unsigned int)the_size, (unsigned int)sizeof(line));  fflush(stdout);
	  continue_flag = 0;
	} else {
	  size_t read_so_far = 0;

	  do {
	    len = recv(fd, line, the_size, 0);
	    if (len > 0) {
	      read_so_far += len;
	      line[ len ] = '\0';
	      get_message(fd, them, line, strlen(line));
	      FD_SET(0, &rfds);
	    } else
	      continue_flag = 0;
	  } while ((continue_flag == 1) && (read_so_far < the_size));
	}
      }
    } else {
      FD_SET(0, &rfds);
      FD_SET(fd, &rfds);
    }
  }
  return 1;
}
#endif

#ifdef _WIN32
void handle_fingerprint(SOCKET socket, char *who, char *fingerprint) {
#else
void handle_fingerprint(int socket, char *who, char *fingerprint) {
#endif
  unsigned int type = 0;
  unsigned int num = 0;
  unsigned int i = 0;
  int selection = 0;
  u_int32_t the_size = 0;
  BC_PROCESSED_MESSAGES *bpm = NULL;
  BC_ERROR *error = NULL;

  char line[ 256 ];

  memset(line, 0, sizeof(line));

#ifdef _WIN32
  /* This is really lame.  Because of the ugly Win32 select() hack, we can't
   * read from the console in this thread, so we have to use a message box
   * to get input from the user... */

  bc_strlcpy(line, "New fingerprint: ", sizeof(line));
  bc_strlcat(line, fingerprint, sizeof(line));
  bc_strlcat(line, "\n\nClick 'Yes' to accept this key permanently\n\nClick 'No to accept this key temporarily\n\nClick 'Cancel' to reject this key.\n\n(P.S.:  this dialog box is UGLY!)", sizeof(line));

  selection = MessageBox(NULL, line, "New Fingerprint Found!", MB_YESNOCANCEL | MB_ICONQUESTION | MB_TOPMOST);
  if (selection == IDYES)
    selection = 2;
  else if (selection == IDNO)
    selection = 1;
  else
    selection = 3;

#else
  printf("KEY FINGERPRINT: [%s]\n\n", fingerprint);
  printf("\t(1) Accept key temporarily\n");
  printf("\t(2) Accept key permanently\n");
  printf("\t(3) Reject key\n\n");
  printf("> ");  fflush(stdout);
  console_read_line(line, sizeof(line) - 1);

  /*printf("selection: [%s]\n", line);*/
  selection = atoi(line);
  if ((selection != 1) && (selection != 2))
    selection = 3;
#endif
  
  if (selection == 3) {
    if (bc_key_reject(who) < 0) {
      printf("Error while rejecting key from %s.\n", who);
    }
    return;
  } else if (selection == 1) {
    if (bc_key_accept_temporary(who, who, &bpm, &error) < 0) {
      printf("Error while accepting key temporarily: %s\n", error->message);
      bc_error_free(&error);
      return;
    }
  } else if (selection == 2) {
    if (bc_key_accept_permanently(who, who, &bpm, &error) < 0) {
      printf("Error while accepting key permanently: %s\n", error->message);
      bc_error_free(&error);
      return;
    }
  }

  type = bpm->type;
  num = bpm->num;
  
  for(i = 0; i < num; i++) {
    the_size = strlen(bpm->result[ i ]);
    send(socket, &the_size, sizeof(u_int32_t), 0);
    send(socket, bpm->result[ i ], the_size, 0);
  }

  bc_processed_messages_free(&bpm);
}

#ifdef _WIN32
int get_message(SOCKET socket, char *who, char *raw_message,
		size_t raw_message_len) {
#else
int get_message(int socket, char *who, char *raw_message,
		size_t raw_message_len) {
#endif
  BC_PROCESSED_MESSAGES *bpm = NULL;
  BC_ERROR *error = NULL;
  unsigned int type = 0;
  unsigned int num = 0;
  unsigned int i = 0;
  u_int32_t the_size = 0;

  if (bc_process(who, raw_message, &bpm, &error) < 0) {
    printf("bc_process: %s\n", error->message); fflush(stdout);
    bc_error_free(&error);
    return -1;
  }

  type = bpm->type;
  num = bpm->num;
  if (type == BC_MESSAGE_TYPE_NONACTIONABLE_HANDSHAKE) {
    /* Do nothing. */
  } else if (type == BC_MESSAGE_TYPE_FINGERPRINT) {
    handle_fingerprint(socket, who, bpm->result[ 0 ]);
  } else if (type == BC_MESSAGE_TYPE_ACTIONABLE_HANDSHAKE) {
    for (i = 0; i < num; i++) {
      the_size = strlen(bpm->result[ i ]);
      send(socket, &the_size, sizeof(u_int32_t), 0);
      send(socket, bpm->result[ i ], the_size, 0);
    }
  } else if (type == BC_MESSAGE_TYPE_WAS_ENCRYPTED) {
    printf("Them:\t\%s\n", bpm->result[ 0 ]); fflush(stdout);
  } else if (type == BC_MESSAGE_TYPE_PLAINTEXT) {
    printf("Plaintext message received! (Uh-oh!): [%u]\n", num); fflush(stdout);
  }


  bc_processed_messages_free(&bpm);
  return 1;
}

#ifdef _WIN32
void send_message(SOCKET socket, char *who, char *message) {
#else
void send_message(int socket, char *who, char *message) {
#endif
  BC_PROCESSED_MESSAGES *bpm = NULL;
  BC_ERROR *error = NULL;
  unsigned int type = 0;
  unsigned int i = 0;
  unsigned int num = 0;
  u_int32_t the_size = 0;

  if (bc_send(who, message, &bpm, &error) < 0) {
    printf("bc_send: %s\n", error->message); fflush(stdout);
    bc_error_free(&error);
    return;
  }

  type = bpm->type;
  num = bpm->num;

  for(i = 0; i < num; i++) {
    the_size = strlen(bpm->result[ i ]);
    send(socket, &the_size, sizeof(u_int32_t), 0);
    send(socket, bpm->result[ i ], the_size, 0);
  }

  bc_processed_messages_free(&bpm);
  return;
}

/* TODO: implement error code checking! */
void reset_module(void) {
  bc_reset();
}

void shutdown_module(void) {
  bc_shutdown();
}


void state_change_callback(char *sn, char *umid, unsigned int operation,
    unsigned int old_state, unsigned int new_state) {

   if ((operation == BC_STATE_OPERATION_ADD) ||
      ((operation == BC_STATE_OPERATION_SET) &&
         (new_state == BC_STATE_CONNECTED)) ||
      ((operation == BC_STATE_OPERATION_SET) &&
         (new_state == BC_STATE_TIMEDOUT))) {
     printf("--> %s\n", parse_state_into_text(new_state));
   }

/*printf("state_change_callback(%s, %s, %u, %u, %u)\n", sn, umid, operation, old_state, new_state);*/
}

char *parse_state_into_text(unsigned int state) {

  switch(state) {

  case BC_STATE_BUDDY_RESET:
    return "Encrypted conversation has been reset and closed.";

  case BC_STATE_SENT_PUBKEYS:
    return "Sent public keys to buddy.";

  case BC_STATE_SENT_SESSION_KEY:
    return "Sent session key to buddy.";

  case BC_STATE_AWAITING_PUBKEYS_1:
    return "Awaiting signature key from buddy.";

  case BC_STATE_AWAITING_PUBKEYS_2:
    return "Awaiting encryption key from buddy.";

  case BC_STATE_AWAITING_SESSION_KEY:
    return "Awaiting session key from buddy.";

  case BC_STATE_SIGNKEY_FINGERPRINT_VALID:
    return "Buddy's signature key fingerprint is valid.";

  case BC_STATE_ENCRYPTKEY_VALID:
    return "Buddy's encryption key is valid.";

  case BC_STATE_SESSION_KEY_VALID:
    return "Session key is valid.";

  case BC_STATE_CONNECTED:
    return "Encrypted conversation successfully started.";

  case BC_STATE_TIMEDOUT:
    return "Encrypted conversation failed to begin due to timeout.";

  default:
    return "Unknown state.";
  }
}


#endif
