/**
 * shared_normal.c
 * 
 * Copyright (C) 2004-2005, Joe Testa
 *
 * This software, including this piece of code, is distributed according
 * to the Hacktivismo Software License, as stated in LICENSE.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "shared_normal.h"


void console_read_line(char *buffer, size_t buffer_len) {
  fgets(buffer, buffer_len - 1, stdin);
  if (strlen(buffer) > 0)
    buffer[ strlen(buffer) - 1 ] = '\0';
}


#ifdef _WIN32
void send_message(SOCKET socket, char *message) {
#else
void send_message(int socket, char *message) {
#endif
  u_int32_t the_size = (u_int32_t)strlen(message);
  send(socket, &the_size, sizeof(u_int32_t), 0);
  send(socket, message, strlen(message), 0);
}


#ifdef _WIN32
int get_message(SOCKET socket, char *buffer, size_t buf_size) {
#else
int get_message(int socket, char *buffer, size_t buf_size) {
#endif
  u_int32_t the_size = 0;

  ssize_t len = recv(socket, &the_size, sizeof(u_int32_t), 0);
  if (len < 0) {
    perror("recv");
    exit(1);
  } else if (len == 0) {
    printf("Peer has terminated the connection.\n"); fflush(stdout);
    exit(1);
  }

  if (the_size > buf_size - 1) {
    fprintf(stderr, "Size of message (%u) is bigger than size of buffer (%u)!\n", the_size, (unsigned int)buf_size);  fflush(stderr);
  } else {
    size_t read_so_far = 0;

    do {
      len = recv(socket, buffer, buf_size - 1, 0);
      if (len < 0) {
	perror("recv");
	exit(1);
      } else if (len == 0) {
	printf("Peer has terminated the connection.\n"); fflush(stdout);
	exit(1);
      }
      read_so_far += len;
    } while (read_so_far < the_size);

    buffer[ read_so_far ] = '\0';
  }

  return len;
}


#ifdef _WIN32
/* Ok, the select() call doesn't work on Win32 as it does on UNIX.  I'm sure
 * there's an elegant way to get the desired effect, but I don't feel like
 * spending much more time on this.  Sooo... I'm just gonna make a new thread
 * which reads from the socket and have this thread just send messages read
 * from the console.  Quick n' dirty.  I'm definitely not a Windows guy...
 */
void handle_chat(SOCKET fd) {
  int continue_flag = 1;
  DWORD thread_id = 0;

  char line[ 128 ];

  memset(line, 0, sizeof(line));


  CreateThread(NULL, 512, read_thread, &fd, 0, &thread_id);
  while(continue_flag == 1) {
    console_read_line(line, sizeof(line) - 1);
    if((strcmp(line, "exit") == 0) || (strcmp(line, "quit") == 0))
      continue_flag = 0;
    else {
      send_message(fd, line);
    }
  }
}

DWORD WINAPI read_thread(void *ptr) {
  char buffer[ 2048 ];
  SOCKET *fd = (SOCKET *)ptr;

  memset(buffer, 0, sizeof(buffer));

  while(1) {
    if (get_message(*fd, buffer, sizeof(buffer)) < 0)
      break;
    printf("Them:\t%s\n", buffer);
    fflush(stdout);
  }
  exit(0);
}

#else
void handle_chat(int fd) {
  int retval = 0;
  unsigned int continue_flag = 0;

  char line[ 2048 ];
  char buffer[ 2048 ];
  fd_set rfds;
  struct timeval tv;

  memset(line, 0, sizeof(line));
  memset(buffer, 0, sizeof(buffer));
  memset(&rfds, 0, sizeof(fd_set));
  memset(&tv, 0, sizeof(struct timeval));

  FD_SET(0, &rfds);
  FD_SET(fd, &rfds);

  continue_flag = 1;
  while(continue_flag == 1) {
    tv.tv_sec = 1;
    tv.tv_usec = 0;
    retval = select(fd + 1, &rfds, NULL, NULL, &tv);
    if (retval) {
      if (FD_ISSET(0, &rfds)) {
	console_read_line(line, sizeof(line) - 1);
	if ((strcmp(line, "exit") == 0) || strcmp(line, "quit") == 0)
	  continue_flag = 0;
	else if (strlen(line) > 0) {
	  printf("You:\t%s\n", line);
	  fflush(stdout);
	  send_message(fd, line);
	  FD_SET(fd, &rfds);
	}
      } else if (FD_ISSET(fd, &rfds)) {
	if (get_message(fd, buffer, sizeof(buffer)) > 0) {
	  printf("Them:\t%s\n", buffer);
	  fflush(stdout);
	  FD_SET(0, &rfds);
	} else
	  continue_flag = 0;
      }
    } else {
      FD_SET(0, &rfds);
      FD_SET(fd, &rfds);
    }

  }
}
#endif
