// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//


#include "ezFCPlib.h"


//
// IMPORTED DECLARATIONS
//

extern time_t	_mkgmtime(struct tm *p_tm);	// thank God for this gem of a function!

extern int		meta_parse(METADATA *meta);

extern char		*_fcpHost;
extern int		_fcpPort;
extern int		_fcpHtl;
extern char		*_fcpProgPath;
extern int		_fcpFileNum;	// temporary file count
extern char     _fcpID[];


//
// PRIVATE DECLARATIONS
//

static int		fcpOpenKeyRead(HFCP *hfcp, char *key, int maxRegress);
static int		fcpOpenKeyWrite(HFCP *hfcp, char *key);
static int		calc_new_date(char *newdate, char *baseline, int increment, int regress);
static time_t	date_to_secs(char *datestr);

#ifdef YYDEBUG
extern int meta_debug;
#endif


//
// Function:	fcpOpenKey()
//
// Arguments:	hfcp - hfcp handle, previously opened by call to fcpCreateHandle()
//
//				key	- freenet URI of the key to be opened
//
//				mode -  _FCP_O_READ - open an existing Freenet key for reading
//						_FCP_O_WRITE - create a new Freenet key for writing
//						_FCP_O_RAW - disable automatic handling of metadata
//									 (enabled by default)
//						read and write access are mutually exclusive
//
// Description:	This is really two functions in one, because opening keys for reading and
//				opening keys for writing are two completely different operations.
//
// Returns:		0 if successful
//				-1 if failed
//

fcpOpenKey(HFCP *hfcp, char *key, int mode)
{
	// Validate flags
	if ((mode & _FCP_O_READ) && (mode & _FCP_O_WRITE))
		return -1;		// read/write access is impossible
	if ((mode & (_FCP_O_READ | _FCP_O_WRITE)) == 0)
		return -1;		// neither selected - illegal
	if (mode & _FCP_O_RAW)
		hfcp->raw = 1;

	if (mode & _FCP_O_READ)
		return fcpOpenKeyRead(hfcp, key, hfcp->regress);
	else
		return fcpOpenKeyWrite(hfcp, key);

}		// 'fcpOpenKey()'




static int fcpOpenKeyRead(HFCP *hfcp, char *key, int maxRegress)
{
	char	buf[1024];
	int		n;
	int		len;
	int		bytesread = 0;
	int		bytesneeded = 0;
	FCP_URI	uri;
	char	was_mapfile = 0;
	int		parse_error;
	int		retcode;

	METADATA *metadata  = NULL;
	int redirecting;
	int currRegress = 0;

	uri.path[0] = uri.subpath[0] = uri.keyid[0] = '\0';

	// analyse the key URI
	_fcpParseUri(key, &uri);
	if (uri.is_msk)
		was_mapfile = 1;
	else if (was_mapfile)
		uri.is_msk = 1;

	// connect to Freenet FCP
	if (_fcpSockConnect(hfcp) != 0)
	{

//_endthread();

		return -1;
	}

	// create and send key request
	sprintf(buf, "ClientGet\nURI=%s\nHopsToLive=%x\nEndMessage\n", uri.uri_str, hfcp->htl);
	len = strlen(buf);
#ifdef FCP_ID_REQUIRED
   	_fcpSockSend(hfcp, _fcpID, 4);
#endif
	n = _fcpSockSend(hfcp, buf, len);
	if (n < len)
	{
		_fcpSockDisconnect(hfcp);

//_endthread();

		return -1;
	}

	// expecting a datafound response
	if (_fcpRecvResponse(hfcp) != FCPRESP_TYPE_DATAFOUND)
	{
		_fcpSockDisconnect(hfcp);

//_endthread();

		return -1;
	}

	// record the key's data size
	hfcp->keysize = hfcp->conn.response.body.datafound.dataLength;

	// suck in the metadata, if any
	bytesneeded = hfcp->conn.response.body.datafound.metaLength;
	if (bytesneeded > 0)
	{
		char *ptr = malloc(bytesneeded + 1);	// extra byte for '\0'
		int count;

		// get all the metadata
		hfcp->conn.response.body.datafound.metaData = ptr;
		count = _fcpReadBlk(hfcp, ptr, bytesneeded);
		ptr[count] = '\0';
		metadata = malloc(sizeof(METADATA));
		metadata->raw = metadata->rawptr = ptr;
		metadata->data = NULL;
		metadata->len = count;
		metadata->is_sol = 1;
#ifdef YYDEBUG
		meta_debug = 1;
#endif

#ifdef _DEBUG
//	printf("Metadata:\n--------\n%s\n--------\n", metadata->raw);
#endif
		parse_error = meta_parse(metadata);
		if (parse_error != 0)
		{
			free(ptr);
			free(metadata);
			return -1;
		}
	}

	// do we need to handle metadata?
	if (parse_error == 0 && metadata != NULL && !hfcp->raw)
	{
		char newdate[15];
		MAPFILE *map;
		MAPFILE_FILE *map_file;

//		printf("playing with metadata\n");
//			printf("type = '%d', REDIRECT = '%d'\n", hfcp->meta->data->type, META_TYPE_REDIRECT);
		switch (metadata->data->type)
		{
		case META_TYPE_REDIRECT:
			//sprintf(hfcp->meta->data->body.redirect->uri->uri_str);
			// TO DO - clean up old keys - memory leak!!
			if (uri.is_msk)
			{
				sprintf(buf, "MSK@%s//", metadata->data->body.redirect->uri->uri_str);
				if (uri.subpath[0])
					strcat(buf, uri.subpath);
			}
			else
				strcpy(buf, metadata->data->body.redirect->uri->uri_str);
			
			_fcpSockDisconnect(hfcp);
			if ((retcode = fcpOpenKeyRead(hfcp, buf, maxRegress)) != 0)
			{

//_endthread();

				hfcp->meta = NULL;
			//free(key);
			}
			return retcode;

		case META_TYPE_DBR:
			for (currRegress = 0; currRegress <= maxRegress; currRegress++)
			{
				calc_new_date(newdate,
								metadata->data->body.dbr->baseline,
								metadata->data->body.dbr->increment,
								currRegress);
				sprintf(buf, "freenet:SSK@%s/%s-%s",
							metadata->data->body.dbr->uri->keyid,
							newdate,
							metadata->data->body.dbr->uri->path);
				if (uri.is_msk)
					strcat(buf, "//");
				if (uri.subpath[0])
					strcat(buf, uri.subpath);
				_fcpSockDisconnect(hfcp);
				key = strdup(buf);
				if ((retcode = fcpOpenKeyRead(hfcp, key, maxRegress)) == 0)
					break;

				// no key - perhaps spin around again
				free(key);
				hfcp->meta = NULL;
			}
			return retcode;

		case META_TYPE_MAPFILE:
			// do nothing unless requested key is an MSK
			if (!uri.is_msk)
			{
				redirecting = 0;
				break;
			}

			// hmmm, we've gotta find the needed file in the site map
			map = metadata->data->body.mapfile;
			if (!uri.subpath[0])
				// no subpath - fall back onto default
				strcpy(uri.subpath, map->default_file);

			// need to find the required file
			for (map_file = map->first_file; map_file != NULL; map_file = map_file->next)
				if (!strcmp(uri.subpath, map_file->file))
					break;		// found it
			if (map_file == NULL)
			{
				// we're stuffed - no way to find key
				_fcpSockDisconnect(hfcp);

//_endthread();

				hfcp->meta = NULL;
				return -1;
			}

			// cool - found file - construct new URI
			key = strdup(map_file->uri->uri_str);
			_fcpSockDisconnect(hfcp);
			retcode = fcpOpenKeyRead(hfcp, key, maxRegress);
			free(key);
			return retcode;

		case META_TYPE_CONTENT:
			//hfcp->meta->data->body.contenttype = NULL;
			hfcp->meta = metadata;
			return 0;
		}
	}

	hfcp->openmode = _FCP_O_READ;
	hfcp->meta = metadata;
	return 0;

}		// 'fcpOpenKeyRead()'


static int fcpOpenKeyWrite(HFCP *hfcp, char *key)
{
	// connect to Freenet FCP
	if (_fcpSockConnect(hfcp) != 0)
	{
//_endthread();

		return -1;
	}

	// save the key
	if (_fcpParseUri(key, &hfcp->wr_info.uri) != 0)
		return -1;

	// generate unique filenames
#ifdef WINDOWS
	sprintf(hfcp->wr_info.data_temp_file, "%s\\fcp-%d.tmp", _fcpProgPath, _fcpFileNum++);
	sprintf(hfcp->wr_info.meta_temp_file, "%s\\fcp-%d.tmp", _fcpProgPath, _fcpFileNum++);
#else
	sprintf(hfcp->wr_info.data_temp_file, "%s/fcp-%d.tmp", _fcpProgPath, _fcpFileNum++);
	sprintf(hfcp->wr_info.meta_temp_file, "%s/fcp-%d.tmp", _fcpProgPath, _fcpFileNum++);
#endif

	// open the files
#ifdef WINDOWS
	if ((hfcp->wr_info.fd_data = _open(hfcp->wr_info.data_temp_file,
										_O_CREAT | _O_RDWR | _O_BINARY,
										_S_IREAD | _S_IWRITE)) < 0)
			return -1;
	if ((hfcp->wr_info.fd_meta = _open(hfcp->wr_info.meta_temp_file,
										_O_CREAT | _O_RDWR | _O_BINARY,
										_S_IREAD | _S_IWRITE)) < 0)
#else
	if ((hfcp->wr_info.fd_data = open(hfcp->wr_info.data_temp_file, O_CREAT, S_IREAD | S_IWRITE)) < 0)
			return -1;
	if ((hfcp->wr_info.fd_meta = open(hfcp->wr_info.meta_temp_file, O_CREAT, S_IREAD | S_IWRITE)) < 0)
#endif
	{
#ifdef WINDOWS
		_close(hfcp->wr_info.fd_data);
#else
		close(hfcp->wr_info.fd_data);
#endif
		return -1;
	}

	hfcp->wr_info.num_data_wr = hfcp->wr_info.num_meta_wr = 0;
	hfcp->openmode = _FCP_O_WRITE;

	return 0;
}


int _fcpParseUri(char *key, FCP_URI *uri)
{
	char skip_keytype = 1;

	char *dupkey = strdup(key);
	char *oldkey = dupkey;
	char *subpath;

	// set defaults
	uri->path[0] = uri->subpath[0] = '\0';
	uri->is_msk = 0;

	// skip 'freenet:'
	if (!strncmp(dupkey, "freenet:", 8))
		dupkey += 8;

	// ignore redundant 'MSK@' prefix
	if (!strncmp(dupkey, "MSK@", 4))
		dupkey += 4;

	// check for tell-tale '//' MSK specifier
	if ((subpath = strstr(dupkey, "//")) != NULL)
	{
		// found '//' - it's an MSK
		*subpath = '\0';
		subpath += 2;
		strcpy(uri->subpath, subpath);
		uri->is_msk = 1;
	}

	// classify key header
	if (!strncmp(dupkey, "SSK@", 4))
	{
		char *path = strchr(dupkey, '/');

		dupkey += 4;
		uri->type = KEY_TYPE_SSK;
		*path++ = '\0';
		strcpy(uri->keyid, dupkey);
		strcpy(uri->path, path);
		sprintf(uri->uri_str, "SSK@%s/%s", uri->keyid, uri->path);
	}
	else if (!strncmp(dupkey, "CHK@", 4))
	{
		uri->type = KEY_TYPE_KSK;
		dupkey += 4;
		strcpy(uri->keyid, dupkey);
		sprintf(uri->uri_str, "CHK@%s", uri->keyid);
	}
	else if (!strncmp(dupkey, "KSK@", 4))
	{
		uri->type = KEY_TYPE_CHK;
		dupkey += 4;
		strcpy(uri->keyid, dupkey);
		sprintf(uri->uri_str, "KSK@%s", uri->keyid);
	}
	else
	{
		// just a KSK
		strcpy(uri->keyid, dupkey);
		uri->type = KEY_TYPE_KSK;
		sprintf(uri->uri_str, "KSK@%s", uri->keyid);
	}

	free(oldkey);
	return 0;

}


static int calc_new_date(char *newdate, char *baseline, int increment, int daysRegress)
{
	struct tm tm_lastupdate;

	time_t secs_now;
	time_t secs_baseline;
	time_t secs_last_update;
	time_t secs_since_baseline;

	// get current time in seconds since epoch
	time(&secs_now);                 /* Get time in seconds */

	// convert baseline to tm format
//	sscanf(baseline, "%04d%02d%02d%02d%02d%02d",
//					&tmb.tm_year, &tmb.tm_mon, &tmb.tm_mday, &tmb.tm_hour, &tmb.tm_min, &tmb.tm_sec);
//	tmb.tm_mon--;
//	tmb.tm_year -= 1900;

// convert baseline AS GMT to seconds since epoch
//	secs_baseline = mktime(&tmb);
//	secs_baseline = _mkgmtime(&tmb);	// thank God for this gem of a function!

	secs_baseline = date_to_secs(baseline);

	// calculate time of last update as seconds since epoch
	secs_since_baseline = secs_now - secs_baseline;
	secs_last_update = (secs_since_baseline / increment) * increment + secs_baseline;

	// go back zero or more days according to daysRegress
	secs_last_update -= daysRegress * 24 * 60 * 60;

	// now convert to a tm structure as GMT
	memcpy(&tm_lastupdate, gmtime(&secs_last_update), sizeof(struct tm));

	// Finally, convert to freenet format date-time string yyyymmddhhmmss
	strftime(newdate, 16, "%Y%m%d%H%M%S", &tm_lastupdate);

	return 0;
}


//
// warning - revolting function follows
//
// this is made necessary because unix lacks a GMT equivalent of mktime(),
// so we have to manually convert a date string to seconds since epoch
//

static time_t date_to_secs(char *datestr)
{
	static int mon_days[12] = {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};
//	struct tm acttm;
	time_t basesecs;
	int basedays;
	int year, mon, day, hour, min, sec;

	// break up into individual fields
	sscanf(datestr, "%04d%02d%02d%02d%02d%02d",	&year, &mon, &day, &hour, &min, &sec);

	// calculate days, not including leap years
	basedays = (year - 1970) * 365 + mon_days[mon - 1] + day - 1;

	// add the leap years
	basedays += (year + 2 - 1970) / 4;

	// exclude if this year is a leap year and prior to feb 29
	if (year % 4 && mon < 3 && year != 1970)
	{
//		printf("docking a day for this years leap year\n");
		basedays--;
	}

	basesecs = basedays * 86400 + hour * 3600 + min * 60 + sec;

//	memcpy(&acttm, gmtime(&basesecs), sizeof(struct tm));
//	printf("datestr = '%s', basedays = %d, basesecs = %ld\n", datestr, basedays, basesecs);
//	printf("%04d-%02d-%02d %02d:%02d:%02d\n", acttm.tm_year+1900, acttm.tm_mon+1, acttm.tm_mday, acttm.tm_hour, acttm.tm_min, acttm.tm_sec);

	return basesecs;
}
