// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//

/* Yacc/Bison Grammar File for parsing Freenet metadata */

%{

/* C Declarations */

#define YYSTYPE META_TOK *

#include <math.h>
#include <stdio.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#ifdef WINDOWS
#include <io.h>
#endif
#include <time.h>


#include "ezFCPlib.h"

extern int meta_error();
extern meta_lex(int *lvalp, METADATA *meta);

#define YYPARSE_PARAM meta
#define YYLEX_PARAM meta


%}

%pure_parser

%token TOK_NEWLINE TOK_SLASH TOK_PLUS TOK_COLON TOK_AT TOK_EQUALS

%token TOK_REDIRECT TOK_END TOK_URI TOK_MAPFILE TOK_DEFAULT
%token TOK_FREENET TOK_MSK TOK_SSK TOK_CHK TOK_KSK
%token TOK_CONTENTTYPE TOK_INCREMENT TOK_BASELINE

%token TOK_STRING

%token TOK_EOF

%token NUM
%token TOK_VERSION
%token ENDPART


%% /* Grammar rules and actions follow */

metadata
	: metadata_parse
		{
			((METADATA *)meta)->data = $1; 
			//printf("metadata_parse->metadata: data = 0x%x\n", $1);
		}
	;

metadata_parse
	: metadata_old
		{
			$$ = $1;

			//printf("metadata_old->metadata_parse: data = 0x%x\n", $1);
		}
	;


metadata_old
	: simple_redirect
		{
			//printf("Got redirect\n");
			$$ = $1;
		}
	| date_redirect
		{
			//printf("Got date redirect\n");
			$$ = $1;
		}
	| content_type
		{
			//printf("Got content-type = 0x%x\n", $1);
			$$ = $1;
		}
	| mapfile
		{
			//printf("Got mapfile\n");
			$$ = $1;
		}
	;


simple_redirect
	: TOK_REDIRECT TOK_NEWLINE TOK_END TOK_NEWLINE freenet_uri TOK_NEWLINE
		{
			META_TOK *redir = malloc(sizeof(META_TOK));
			META_TOK *uri = $5;

			//printf("Got redirect\n");
			redir->type = META_TYPE_REDIRECT;
			redir->body.redirect = malloc(sizeof(REDIRECT));
			redir->body.redirect->uri = uri->body.uri;
			free(uri);	// could be deadly

			//printf("Redirect: key='%s', path='%s'\n",
			//		redir->body.redirect->uri->keyid,
			//		redir->body.redirect->uri->path);
			$$ = redir;
		}
	;


date_redirect
	: TOK_REDIRECT TOK_NEWLINE
	  increment TOK_NEWLINE
	  baseline TOK_NEWLINE
	  TOK_END TOK_NEWLINE
	  freenet_uri TOK_NEWLINE
		{
			META_TOK *dbr = malloc(sizeof(META_TOK));
			META_TOK *inc = $3;
			META_TOK *baseline = $5;
			META_TOK *uri = $9;

			//printf("Got redirect\n");
			dbr->type = META_TYPE_DBR;
			dbr->body.dbr = malloc(sizeof(DBR));
			dbr->body.dbr->increment = atoi(inc->body.str);
			dbr->body.dbr->baseline = baseline->body.str;
			dbr->body.dbr->uri = uri->body.uri;
			free(uri);	// could be deadly
			free(inc);
			free(baseline);

			//printf("DBR: key='%s', path='%s', increment='%d', baseline='%s'\n",
			//		dbr->body.dbr->uri->keyid,
			//		dbr->body.dbr->uri->path,
			//		dbr->body.dbr->increment,
			//		dbr->body.dbr->baseline
			//		);
			$$ = dbr;
		}
	| TOK_REDIRECT TOK_NEWLINE
	  baseline TOK_NEWLINE
	  increment TOK_NEWLINE
	  TOK_END TOK_NEWLINE
	  freenet_uri TOK_NEWLINE
		{
			META_TOK *dbr = malloc(sizeof(META_TOK));
			META_TOK *baseline = $3;
			META_TOK *inc = $5;
			META_TOK *uri = $9;

			//printf("Got redirect\n");
			dbr->type = META_TYPE_DBR;
			dbr->body.dbr = malloc(sizeof(DBR));
			dbr->body.dbr->increment = atoi(inc->body.str);
			dbr->body.dbr->baseline = baseline->body.str;
			dbr->body.dbr->uri = uri->body.uri;
			free(uri);	// could be deadly
			free(inc);
			free(baseline);

			//printf("DBR: key='%s', path='%s', increment='%d', baseline='%s'\n",
			//		dbr->body.dbr->uri->keyid,
			//		dbr->body.dbr->uri->path,
			//		dbr->body.dbr->increment,
			//		dbr->body.dbr->baseline
			//		);
			$$ = dbr;
		}
	;


mapfile
	: TOK_MAPFILE TOK_NEWLINE
	  mapfile_default
	  TOK_END TOK_NEWLINE
	  mapfile_list
		{
			META_TOK *map = malloc(sizeof(META_TOK));
			META_TOK *def = $3;
			META_TOK *filelist = $6;

			map->type = META_TYPE_MAPFILE;
			map->body.mapfile = malloc(sizeof(MAPFILE));
			map->body.mapfile->default_file = def->body.str;
			map->body.mapfile->first_file = filelist->body.mapfile_file;
			$$ = map;
		}
	;


mapfile_default
	: TOK_DEFAULT TOK_EQUALS TOK_STRING TOK_NEWLINE
		{ $$ = $3; }
	| /* empty */
		{
			META_TOK *def = malloc(sizeof(META_TOK));
			def->body.mapfile->default_file = strdup("index.html");
			$$ = def;
		}
	;


mapfile_list
	: mapfile_list_entry
		{ $$ = $1; }
	| mapfile_list_entry mapfile_list
		{
			META_TOK *listent = $1;
			META_TOK *list = $2;

			/* extract file chain out of list, chain it in */
			((META_TOK *)$1)->body.mapfile_file->next = ((META_TOK *)$2)->body.mapfile_file;
			free($2);
			$$ = $1;
		}
	;


mapfile_list_entry
	: mapfile_list_file TOK_EQUALS freenet_uri TOK_NEWLINE
		{
			META_TOK *tok = malloc(sizeof(META_TOK));
			FCP_URI *uri = ((META_TOK *)$3)->body.uri;

			tok->body.mapfile_file = malloc(sizeof(MAPFILE_FILE));
			tok->body.mapfile_file->file = ((META_TOK *)$1)->body.str;
			tok->body.mapfile_file->uri = uri;
			tok->body.mapfile_file->next = NULL;

			//printf("mapfile entry: '%s' = '%c' '%s' '%s'\n", tok->body.mapfile_file->file,
			//										tok->body.mapfile_file->uri->type,
			//										tok->body.mapfile_file->uri->keyid,
			//										tok->body.mapfile_file->uri->path
			//											? tok->body.mapfile_file->uri->path
			//											: ""
			//										);
			free($1);
			free($3);
			$$ = tok;
		}
	;


mapfile_list_file
	: TOK_STRING
		{ $$ = $1; }
	| TOK_STRING TOK_SLASH mapfile_list_file
		{
			META_TOK *tok = malloc(sizeof(META_TOK));
			char buf[256];

			sprintf(buf, "%s/%s", ((META_TOK *)$1)->body.str, ((META_TOK *)$3)->body.str);
			free(((META_TOK *)$1)->body.str);
			((META_TOK *)$1)->body.str = strdup(buf);
			free(((META_TOK *)$3)->body.str);
			free($3);
		}
	;


increment
	: TOK_INCREMENT TOK_EQUALS TOK_STRING
		{ $$ = $3; }
	;


baseline
	: TOK_BASELINE TOK_EQUALS TOK_STRING
		{ $$ = $3; }
	;


freenet_uri
	: freenet_uri_ksk
		{
			//printf("parse: got KSK\n");
			$$ = $1;
		}
	| freenet_uri_ssk
		{
			//printf("parse: got SSK\n");
			$$ = $1;
		}
	| freenet_uri_chk
		{
			//printf("parse: got CHK\n");
			$$ = $1;
		}
	;

freenet_uri_ssk
	: freenet_prefix TOK_SSK TOK_AT TOK_STRING TOK_SLASH uri_path
		{
			META_TOK *ptok = malloc(sizeof(META_TOK));

			//printf("Got SSK\n");
			ptok->body.uri = malloc(sizeof(FCP_URI));
			ptok->body.uri->type = KEY_TYPE_SSK;
			strcpy(ptok->body.uri->keyid, ((META_TOK *)$4)->body.str);
			strcpy(ptok->body.uri->path, ((META_TOK *)$6)->body.str);
			ptok->body.uri->subpath[0] = '\0';
			ptok->type = TOK_URI;
			sprintf(ptok->body.uri->uri_str,
					"SSK@%s/%s",
					ptok->body.uri->keyid,
					ptok->body.uri->path);
			$$ = ptok;
			free($4);
			free($6);
		}
	;

freenet_uri_ksk
	: freenet_prefix TOK_KSK TOK_AT uri_path
		{
			META_TOK *ptok = malloc(sizeof(META_TOK));

			//printf("Got formal ksk\n");
			ptok->body.uri = malloc(sizeof(FCP_URI));
			ptok->body.uri->type = KEY_TYPE_KSK;
			strcpy(ptok->body.uri->keyid, ((META_TOK *)$4)->body.str);
			ptok->body.uri->path[0] = '\0';
			ptok->body.uri->subpath[0] = '\0';
			ptok->type = TOK_URI;
			sprintf(ptok->body.uri->uri_str,
					"KSK@%s",
					ptok->body.uri->keyid);
			$$ = ptok;
			//free($1);
		}
	| uri_path
		{
			META_TOK *ptok = malloc(sizeof(META_TOK));

			//printf("Got KSK\n");
			ptok->body.uri = malloc(sizeof(FCP_URI));
			ptok->body.uri->type = KEY_TYPE_KSK;
			strcpy(ptok->body.uri->keyid, ((META_TOK *)$1)->body.str);
			ptok->body.uri->path[0] = '\0';
			ptok->body.uri->subpath[0] = '\0';
			ptok->type = TOK_URI;
			sprintf(ptok->body.uri->uri_str,
					"KSK@%s",
					ptok->body.uri->keyid);
			$$ = ptok;
			free($1);
		}
	;

freenet_uri_chk
	: freenet_prefix TOK_CHK TOK_AT TOK_STRING
		{
			META_TOK *ptok = malloc(sizeof(META_TOK));

			//printf("Got CHK\n");
			ptok->type = TOK_URI;
			ptok->body.uri = malloc(sizeof(FCP_URI));
			ptok->body.uri->type = KEY_TYPE_CHK;
			strcpy(ptok->body.uri->keyid, ((META_TOK *)$4)->body.str);
			ptok->body.uri->path[0] = ptok->body.uri->subpath[0] = '\0';
			sprintf(ptok->body.uri->uri_str,
					"CHK@%s",
					ptok->body.uri->keyid);
			$$ = ptok;
			free($4);
		}
	;

uri_path
	: TOK_STRING
		{ $$ = $1; }
	| uri_path TOK_SLASH TOK_STRING
		{
			char buf[256];
			META_TOK *ptok = malloc(sizeof(META_TOK));

			sprintf(buf, "%s/%s", ((META_TOK *)$1)->body.str, ((META_TOK *)$3)->body.str);
			free(((META_TOK *)$1)->body.str);
			free(((META_TOK *)$3)->body.str);
			free($3);
			((META_TOK *)$1)->body.str = strdup(buf);
			$$ = $1;
		}
	| TOK_STRING TOK_PLUS TOK_STRING
		{
			char buf[256];
			META_TOK *ptok = malloc(sizeof(META_TOK));

			sprintf(buf, "%s %s", ((META_TOK *)$1)->body.str, ((META_TOK *)$3)->body.str);
			free(((META_TOK *)$1)->body.str);
			free(((META_TOK *)$3)->body.str);
			free($3);
			((META_TOK *)$1)->body.str = strdup(buf);
			$$ = $1;
		}
	;


content_type
	: TOK_CONTENTTYPE content_type_delim TOK_STRING TOK_SLASH TOK_STRING TOK_NEWLINE optional_end
		{
			META_TOK *ptok = malloc(sizeof(META_TOK));
			char buf[128];

			//printf("Got Content-type\n");
			ptok->type = META_TYPE_CONTENT;
			ptok->body.contenttype = malloc(sizeof(CONTENTTYPE));
			sprintf(buf, "%s/%s", ((META_TOK *)$3)->body.str, ((META_TOK *)$5)->body.str);
			ptok->body.contenttype->type = strdup(buf);
			$$ = ptok;
			free($3);
			free($5);
		}
	;

content_type_delim
	: TOK_EQUALS
	| TOK_COLON
	;

optional_end
	: TOK_END TOK_NEWLINE
	| /* empty */
	;

freenet_prefix
	: TOK_FREENET TOK_COLON
	| /* empty */
	;


/*
 * NEW METADATA - NOT IMPLEMENTED
 */


metadata_new
	: version cmds
	;

version
	: /* empty */
	| version_dec
	;

version_dec
	: TOK_VERSION fld_version ENDPART
	;

/*****************************
 * commands
 *****************************/
cmds
	: cmd cmds
	;

cmd
	: cmd_redirect
	| cmd_dateredirect
	| cmd_splitfile
	| cmd_info
	| cmd_trailinginfo
	;

cmd_redirect
	:
	;

cmd_dateredirect
	:
	;

cmd_splitfile
	:
	;

cmd_info
	:
	;

cmd_trailinginfo
	:
	;

/****************************
 * fields
 ****************************/

fld_version
	:
	;


%%

