/*
 * FreeWeb Agent
 *
 * config.c
 *
 * Routines to read and write basic configs to disk file
 */

#include <stdio.h>
#include <string.h>
#include <windows.h>

#include "freeweb.h"


//#include "fwagent.h"


#define AGENT_CONFIG_FILE	"fwagent.ini"


/*
 * IMPORTED DECLARATIONS
 */


/*
 * EXPORTED DECLARATIONS
 */

FW_CONFIGS fwconfigs;


/*
 * PRIVATE DECLARATIONS
 */

static char *skipequals(char *s1);

static char proxy_host[128] = "127.0.0.1";
static char my_haddr[128] = "127.0.0.1";




/*
 * load_configs
 *
 * Load configuration info from file
 *
 * Arguments:
 *
 *		none
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */


int load_configs()
{
	FILE	*fp;
	char	buf[1024];

	char *progpath = strdup(_pgmptr); // 'pgmptr' is a global windows-specific string
	char *exename = strrchr(progpath, '\\'); // point to slash between path and filename
	*exename++ = '\0'; // split the string and point to filename part

	fwconfigs.debug = 0;
	fwconfigs.multi_threaded = 0;
	fwconfigs.hideConsole = 0;
	fwconfigs.use_remote_proxy = 0;
	fwconfigs.rem_proxy_host = proxy_host;
	fwconfigs.rem_proxy_port = 8080;
	fwconfigs.haddr = my_haddr;
	fwconfigs.hport = 8888;
	fwconfigs.fproxy_port = 8081;
	fwconfigs.webblockdefault = 1;		// on by default
	fwconfigs.allowlocalhost = 0;

	sprintf(buf, "%s\\%s", progpath, AGENT_CONFIG_FILE);
	if ((fp = fopen(buf, "rt")) == NULL)
	{
		char buf[256];

		sprintf(buf, "FreeWeb Agent configuration file '%s' is missing or corrupted\n\n"
					 "Press OK to quit",
						AGENT_CONFIG_FILE);
		MessageBox(0, buf, "FreeWeb Agent - Error", 0);
		return 1;
	}

	while (fgets(buf, 1024, fp) != NULL)
	{
		char c;
		char *s;

		/* skip if first char is ';'. space or tab */
		if ((c = buf[0]) == ';' || c == ' ' || c == '\t')
			continue;

		/* split line around '=' */
		if ((s = skipequals(buf)) == NULL)
			/* no '=' in line */
			continue;

		if (!strcmp(buf, "debug"))
			;
		else if (!strcmp(buf, "multithread"))
		{
			if (strstr("yesYEStrueTRUEonON", s))
				fwconfigs.multi_threaded = 1;
		}
		else if (!strcmp(buf, "useremproxy"))
		{
			if (strstr("yesYEStrueTRUEonON", s))
				fwconfigs.use_remote_proxy = 1;
		}
		else if (!strcmp(buf, "remproxyaddr"))
			strcpy(proxy_host, s);
		else if (!strcmp(buf, "remproxyport"))
			fwconfigs.rem_proxy_port = atoi(s);
		else if (!strcmp(buf, "listenaddr"))
			fwconfigs.hport = atoi(s);
		else if (!strcmp(buf, "fproxyport"))
			fwconfigs.fproxy_port = atoi(s);
		else if (!strcmp(buf, "webblock"))
		{
			if (!strcmp(s, "off")
				|| !strcmp(s, "no")
				|| !strcmp(s, "OFF")
				|| !strcmp(s, "NO")
				|| !strcmp(s, "false")
				|| !strcmp(s, "FALSE")
				|| !strcmp(s, "0")
			)
				fwconfigs.webblockdefault = 0;
		}
		else if (!strcmp(buf, "allowlocalhost"))
		{
			if (strstr("yesYEStrueTRUEonON", s))
				fwconfigs.allowlocalhost = 1;
		}
	}

	// set current web blocking status according to default
	fwconfigs.webblock = fwconfigs.webblockdefault;

	fclose(fp);
	return 0;

}		/* 'load_configs()' */


/*
 * save_configs
 *
 * Save configuration info to file
 *
 * Arguments:
 *
 *		none
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int save_configs()
{
	char buf[1024];
	FILE *fp;

	char *progpath = strdup(_pgmptr); // 'pgmptr' is a global windows-specific string
	char *exename = strrchr(progpath, '\\'); // point to slash between path and filename
	*exename++ = '\0'; // split the string and point to filename part

	sprintf(buf, "%s\\%s", progpath, AGENT_CONFIG_FILE);
	if ((fp = fopen(buf, "wt")) == NULL)
	{
		MessageBox(0, "Error saving FreeWeb agent configuration",
					"FreeWeb Agent - Problem!",
					0);
		return 1;
	}

	/* Write header comments */
	fprintf(fp,
				"; ================================================\n"
				"; FreeWeb agent configuration file\n"
				"; contains settings for listening port,\n"
				"; and remote proxy\n"
				";\n"
				"; Generally, you won't have to edit this file\n"
				"; because you can use the 'settings' command\n"
				"; on the tray icon menu\n"
				";\n"
				"; warning - spaces are significant - don't use them\n"
				"; ================================================\n\n\n"
			);

	fprintf(fp, "; address for fwagent to listen on\n");
	fprintf(fp, "; must match browser proxy setting\n\n");
	fprintf(fp, "listenaddr=%d\n\n\n", fwconfigs.hport);

	fprintf(fp, "; port number of Freenet FProxy\n\n");
	fprintf(fp, "fproxyport=%d\n\n\n", fwconfigs.fproxy_port);

	fprintf(fp, "; flag to disable access to the mainstream web at startup\n\n");
	fprintf(fp, "webblock=%s\n\n\n", fwconfigs.webblockdefault ? "yes" : "no");

	fprintf(fp, "; flag to enable access to localhost while web blocking is on\n\n");
	fprintf(fp, "allowlocalhost=%s\n\n\n", fwconfigs.allowlocalhost ? "yes" : "no");

	fprintf(fp, "; set if fwagent has to talk to another proxy\n\n");
	fprintf(fp, "useremproxy=%s\n\n\n", fwconfigs.use_remote_proxy ? "yes" : "no");
	
	fprintf(fp, "; address of remote proxy, excluding port number\n\n");
	fprintf(fp, "remproxyaddr=%s\n\n\n", fwconfigs.rem_proxy_host);
	
	fprintf(fp, "; port number of remote proxy\n\n");
	fprintf(fp, "remproxyport=%d\n\n\n", fwconfigs.rem_proxy_port);

	fprintf(fp, "\n; use only for debugging - enables logging etc\n\n");
	fprintf(fp, "debug=%s\n\n\n", fwconfigs.debug ? "yes" : "no");

	fprintf(fp, "; makes browser run at full speed - normally yes\n\n");
	fprintf(fp, "multithread=%s\n\n\n", fwconfigs.multi_threaded ? "yes" : "no");

	fclose(fp);

	return 0;

}		/* 'save_configs()' */



static char *skipequals(char *s1)
{
	int len = strlen(s1);
	char *s2 = strchr(s1, '=');

	if (len == 0)
		return NULL;

	while (--len > 0)
		if (s1[len] == '\r' || s1[len] == '\n')
			s1[len] = '\0';

	if (s2 == NULL)
		return NULL;

	if (s2[1] == '\0')
		return NULL;

	s2[0] = '\0';
	return &(s2[1]);
}