// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//

// config.cpp: implementation of the CConfig class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <wctype.h>
#include <time.h>

#include "stdafx.h"
#include "fwpubgui.h"
#include "constants.h"
#include "config.h"


//#include "fwpublish.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CConfig::CConfig()
{

}

CConfig::~CConfig()
{

}


/*
 * IMPORTED DECLARATIONS
 */

extern time_t	time(time_t *tod); 
extern struct	tm *gmtime(const time_t *tod);



/* state definitions for config file parsing */

#define PARSE_AWAIT_MAIN		0
#define PARSE_IN_MAIN			1
#define	PARSE_IN_SITE			2
#define PARSE_IN_FILE			3
#define PARSE_IN_ROOTDOMAINS	4



/*
 * load()
 *
 * Loads in the node's master freeweb publishing file
 * parses this file into an all sites map structure
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CConfig::load()
{
	FILE *fp;
	char buf[256];
	int tok_cnt;
	char tok1_val[128];
	char tok2_val[128];
	//char *s;
	int i;
	int parse_state;
	Site *thissite;
	SiteFile *thisfile, *thisfile1;

	char *exename;

	progpath = strdup(_pgmptr); // 'pgmptr' is a global windows-specific string
	exename = strrchr(progpath, '\\'); // point to slash between path and filename
	*exename++ = '\0'; // split the string and point to filename part

	strcpy(exepath, progpath);

	// initialise defaults
	strcpy(serverAddress, "127.0.0.1");	// standard Freenet default
	htl = 35;
	htlReq = 60;

	fcpPort = DEFAULT_FCP_PORT;
	listenPort = DEFAULT_LISTEN_PORT;
	daysRegress = 3;
	extProxyEnabled = 0;
	strcpy(extProxyAddr, "127.0.0.1");
	extProxyPort = 8080;
	blockWebAccess = 1;
	blockWebAtStartup = 1;
	trustKsks = 0;
	maxInsertThreads = MAXCONCURRENTINSERTS;

	firstsite = NULL;

	/* open config file */
	sprintf(buf, "%s\\%s", progpath, CONFIGFILENAME);
	if ((fp = fopen(buf, "rt")) == NULL)
	{
		save();
		return 0;
	}

	/* Parse config file */
	parse_state = PARSE_AWAIT_MAIN;
	while (fgets(buf, 256, fp) != NULL)
	{
		/* Skip comments, empty lines, lines beginning with whitespace */
		if (buf[0] == ';' || buf[0] == '#' || iswspace(buf[0]))
			continue;

		/* Hack off whitespace at end */
		for (i = strlen(buf); (i > 0) && iswspace(buf[i-1]);)
			buf[--i] = '\0';

		/* Tokenise line */
		tok_cnt = tokenise_line(buf, tok1_val, tok2_val);

		/* Process according to state */
		switch (parse_state)
		{
		case PARSE_AWAIT_MAIN:
			if (!strcmp(tok1_val, "[main]"))
				parse_state = PARSE_IN_MAIN;
			continue;
		case PARSE_IN_MAIN:
			if (!strcmp(tok1_val, "serverAddress"))
				strcpy(serverAddress, tok2_val);
			else if (!strcmp(tok1_val, "htl"))
				htl = atoi(tok2_val);
			else if (!strcmp(tok1_val, "htlReq"))
				htlReq = atoi(tok2_val);
			else if (!strcmp(tok1_val, "fcpPort"))
				fcpPort = atoi(tok2_val);
			else if (!strcmp(tok1_val, "listenPort"))
				listenPort = atoi(tok2_val);
			else if (!strcmp(tok1_val, "extProxyEnabled"))
				extProxyEnabled = atoi(tok2_val);
			else if (!strcmp(tok1_val, "extProxyAddr"))
				strcpy(extProxyAddr, tok2_val);
			else if (!strcmp(tok1_val, "daysRegress"))
				daysRegress = atoi(tok2_val);
			else if (!strcmp(tok1_val, "extProxyPort"))
				extProxyPort = atoi(tok2_val);
			else if (!strcmp(tok1_val, "blockWebAtStartup"))
				blockWebAtStartup = blockWebAccess = atoi(tok2_val);
			else if (!strcmp(tok1_val, "trustKSKs"))
				trustKsks = atoi(tok2_val);
			else if (!strcmp(tok1_val, "maxInsertThreads"))
				maxInsertThreads = atoi(tok2_val);
			else if (!strcmp(tok1_val, "[site]"))
			{
				/* Create, insert and initialise a new site block */
				thissite = (Site *)malloc(sizeof(Site));
				thissite->next = firstsite;
				firstsite = thissite;
				thissite->firstfile = NULL;
				thissite->domainname[0] = '\0';
				thissite->ssk_pub[0] = '\0';
				thissite->ssk_priv[0] = '\0';
				thissite->ksk_ptr[0] = '\0';
				thissite->datecreated[0] = '\0';
				thissite->dateupdated[0] = '\0';
				thissite->rootdir[0] = '\0';
				thissite->description[0] = '\0';
				thissite->generatedIndex = 0;
				thissite->updating = 0;

				parse_state = PARSE_IN_SITE;
			}
			continue;
		case PARSE_IN_SITE:
			if (!strcmp(tok1_val, "domainname"))
				strcpy(thissite->domainname, tok2_val);
			else if (!strcmp(tok1_val, "datecreated"))
				strcpy(thissite->datecreated, tok2_val);
			else if (!strcmp(tok1_val, "dateupdated"))
				strcpy(thissite->dateupdated, tok2_val);
			else if (!strcmp(tok1_val, "ssk_pub"))
				strcpy(thissite->ssk_pub, tok2_val);
			else if (!strcmp(tok1_val, "ssk_priv"))
				strcpy(thissite->ssk_priv, tok2_val);
			else if (!strcmp(tok1_val, "ksk_ptr"))
				strcpy(thissite->ksk_ptr, tok2_val);
			else if (!strcmp(tok1_val, "generatedIndex"))
				thissite->generatedIndex = 1;
			else if (!strcmp(tok1_val, "rootdir"))
				strcpy(thissite->rootdir, tok2_val);
			else if (!strcmp(tok1_val, "description"))
			{
				strncpy(thissite->description, tok2_val, 127);
				if (strlen(tok2_val) > 127)
					thissite->description[127] = '\0';
			}
			else if (!strcmp(tok1_val, "[site]"))
			{
				/* Create, insert and initialise a new site block */
				thissite = (Site *)malloc(sizeof(Site));
				thissite->next = firstsite;
				thissite->firstfile = NULL;
				thissite->domainname[0] = '\0';
				thissite->ssk_pub[0] = '\0';
				thissite->ssk_priv[0] = '\0';
				thissite->ksk_ptr[0] = '\0';
				thissite->datecreated[0] = '\0';
				thissite->dateupdated[0] = '\0';
				thissite->rootdir[0] = '\0';
				thissite->description[0] = '\0';
				thissite->generatedIndex = 0;
				thissite->updating = 0;
				firstsite = thissite;
				parse_state = PARSE_IN_SITE;
			}
			else if (!strcmp(tok1_val, "[file]"))
			{
				/* adding a file */
				thisfile = (SiteFile *)malloc(sizeof(SiteFile));
				thisfile->next = (SiteFile *)thissite->firstfile;
				thissite->firstfile = thisfile;
				parse_state = PARSE_IN_FILE;
			}

			continue;

		case PARSE_IN_FILE:
			if (!strcmp(tok1_val, "name"))
				strcpy(thisfile->filename, tok2_val);
			else if (!strcmp(tok1_val, "chk"))
				strcpy(thisfile->chk, tok2_val);
			else if (!strcmp(tok1_val, "timeupdated"))
				strcpy(thisfile->time_published, tok2_val);
			else if (!strcmp(tok1_val, "[file]"))
			{
				/* Finish this file def and start another one */
				thisfile1 = thisfile;
				thisfile = (SiteFile *)malloc(sizeof(SiteFile));
				thisfile->next = thissite->firstfile;
				thissite->firstfile = thisfile;
				thisfile->filename[0] = '\0';
				thisfile->chk[0] = '\0';
				thisfile->time_published[0] = '\0';
			}
			else if (!strcmp(tok1_val, "[site]"))
			{
				/* Create, insert and initialise a new site block */
				thissite = (Site *)malloc(sizeof(Site));
				thissite->next = firstsite;
				firstsite = thissite;
				thissite->firstfile = NULL;

				thissite->domainname[0] = '\0';
				thissite->ssk_pub[0] = '\0';
				thissite->ssk_priv[0] = '\0';
				thissite->ksk_ptr[0] = '\0';
				thissite->datecreated[0] = '\0';
				thissite->dateupdated[0] = '\0';
				thissite->rootdir[0] = '\0';
				thissite->description[0] = '\0';
				thissite->generatedIndex = 0;
				thissite->updating = 0;

				parse_state = PARSE_IN_SITE;
			}
			break;
		}	/* 'switch (parse_state)' */
	}		/* 'while (reading config file)' */

	fclose(fp);
	return 0;

}			/* 'load' */



/*
 * save()
 *
 * Writes sites map structure to master publishing file
 *
 * Returns:
 *
 *		1 if successful
 *		0 if failed
 */

int CConfig::save()
{
	FILE		*fp;
	time_t		timenow;
	struct tm	*gmtime_now;
	////int			i;
	Site		*thissite;
	SiteFile	*thisfile;
	char		buf[256];
	char		buf2[256];

	sprintf(buf, "%s\\%s", progpath, CONFIGFILENAME);
	sprintf(buf2, "%s\\%s.bak", progpath, CONFIGFILENAME);

	// back up old config file
	unlink(buf2);
	rename(buf, buf2);

	/* open config file */
	if ((fp = fopen(buf, "wt")) == NULL)
		return 1;

	/* Get today's date */
	time(&timenow);
	gmtime_now = gmtime(&timenow);

	/* write out header, including date saved */
	fprintf(
		fp,
		"; %s - Freenet freeweb configuration file\n",
		CONFIGFILENAME
		);
	fprintf(
		fp,
		"; machine-generated and saved: %d-%02d-%02d %02d:%02d:%02d GMT\n;\n",
		gmtime_now->tm_year+1900,
		gmtime_now->tm_mon+1,
		gmtime_now->tm_mday,
		gmtime_now->tm_hour,
		gmtime_now->tm_min,
		gmtime_now->tm_sec
		);

	/* write out [main] */
	fprintf(fp, "[main]\n");
	fprintf(fp, "serverAddress=%s\n", serverAddress);
	fprintf(fp, "fcpPort=%d\n", fcpPort);
	fprintf(fp, "htl=%d\n", htl);
	fprintf(fp, "htlReq=%d\n", htlReq);
	fprintf(fp, "fcpPort=%d\n", fcpPort);
	fprintf(fp, "listenPort=%d\n", listenPort);
	fprintf(fp, "daysRegress=%d\n", daysRegress);
	fprintf(fp, "extProxyEnabled=%d\n", extProxyEnabled);
	fprintf(fp, "extProxyAddr=%s\n", extProxyAddr);
	fprintf(fp, "extProxyPort=%d\n", extProxyPort);
	fprintf(fp, "blockWebAtStartup=%d\n", blockWebAtStartup);
	fprintf(fp, "trustKSKs=%d\n", trustKsks != 0 ? 0 : 1);
	fprintf(fp, "maxInsertThreads=%d\n", maxInsertThreads);

	/* write out root domains */
	fprintf(fp, "[rootdomains]\n");

	/* write out all the sites */
	for (thissite = firstsite;
		thissite != NULL;
		thissite = thissite->next)
	{
		fprintf(fp, "[site]\n");
		fprintf(fp, "domainname=%s\n",	thissite->domainname);
		fprintf(fp, "datecreated=%s\n", thissite->datecreated);
		fprintf(fp, "dateupdated=%s\n", thissite->dateupdated);
		fprintf(fp, "ssk_pub=%s\n",		thissite->ssk_pub);
		fprintf(fp, "ssk_priv=%s\n",	thissite->ssk_priv);
		fprintf(fp, "rootdir=%s\n",		thissite->rootdir);
		fprintf(fp, "description=%s\n",	thissite->description);
		if (thissite->generatedIndex)
			fprintf(fp, "generatedIndex\n");

		/* Write out each file in this site */
		for (thisfile = thissite->firstfile;
				thisfile != NULL;
				thisfile = thisfile->next)
		{
			fprintf(fp, "[file]\n");
			fprintf(fp, "name=%s\n", thisfile->filename);
			fprintf(fp, "chk=%s\n", thisfile->chk);
			fprintf(fp, "timeupdated=%s\n", thisfile->time_published);
		}
	}

	/* All done */
	fclose(fp);
	return 0;

}			/* 'save()' */



/*
 * tokenise_line()
 *
 * cuts up a 'tag=value' line
 *
 * Arguments:
 *
 *		buf		line to tokenise
 *		val1	string into which first token is written
 *		val2	string into which second token is written
 *
 *	Returns:
 *
 *		1	if only one token on line
 *		2	if 2 tokens found
 */

int CConfig::tokenise_line(char *buf, char *val1, char *val2)
{
	char *token2;

	/* look for '=' */
	if ((token2 = strchr(buf, '=')) == NULL)
	{
		/* just one */
		strcpy(val1, buf);
		return 1;
	}		/* 'tokenise_line()' */

	*token2++ = '\0';
	strcpy(val1, buf);
	strcpy(val2, token2);
	return 2;

}		/* 'tokenise_line()' */



void CConfig::deleteSite(char *domain)
{
	Site *site_to_nuke;
	Site *thissite;

	char fulldomain[128];
	int len = strlen(domain);

	// remove '.free' from domain
	strcpy(fulldomain, domain);
	if (len > 5 && strstr(fulldomain, ".free") == &fulldomain[len - 5])
		fulldomain[len-5] = '\0';

	// Spit if we don't have any sites
	if (firstsite == NULL)
	{
		MessageBox(0, "You don't have any domains defined",
					PROGNAME,
					MB_OK | MB_SYSTEMMODAL);
		return;
	}

	// Check first domain in chain
	if (!strcmp(fulldomain, firstsite->domainname))
	{
		site_to_nuke = firstsite;
		firstsite = firstsite->next;
	}
	else
	{
		// search remaining domains in chain
		for (thissite = firstsite; thissite->next != NULL; thissite = thissite->next)
			if (!strcmp(fulldomain, thissite->next->domainname))
				break;

		if (thissite->next == NULL)
		{
			MessageBox(0, "Site not known", PROGNAME, MB_SYSTEMMODAL);
			return;
		}

		site_to_nuke = thissite->next;
		thissite->next = site_to_nuke->next;
	}

#ifdef MIGHT_NEED_THIS
	// If we get here, we've found a matching site to nuke
	SiteFile *temp;
	while (site_to_nuke->firstfile != NULL)
	{
		temp = site_to_nuke->firstfile;
		site_to_nuke->firstfile = site_to_nuke->firstfile->next;
		free(temp);
	}
#endif
	// Now we can delete the site block itself
	save();
}
