// freenet.cpp: implementation of the CFreenet class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <winbase.h>
#include <process.h>

#include "stdafx.h"
#include "constants.h"

#include "fwpubgui.h"
#include "util.h"
#include "freenet.h"
#include "log.h"
#include "fwpubgui.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CFreenet::CFreenet()
{
	FreenetDir = FREENET_DEF_DIR;

}

CFreenet::~CFreenet()
{

}

//////////////////////////////
// Imports
//////////////////////////////

extern "C" long getpid();

extern CLog *pLog;


//////////////////////////////////////////////////////////////////////
// Public Methods
//////////////////////////////////////////////////////////////////////



/*
 * SetFreenetDir()
 *
 * Used for caller to declare where Freenet is installed
 * on the system
 *
 * If this routine is not called, Freenet is assumed to be
 * installed at the location FREENET_DEF_DIR, in constants.h
 *
 * Arguments:
 *
 *		dir		Directory where Freenet is installed
 *
 * Returns:
 *
 *		void
 */

void CFreenet::SetFreenetDir(char *path)
{
	if (path == NULL || strlen(path) == 0)
		FreenetDir = FREENET_DEF_DIR;
	else
		FreenetDir = strdup(path);
}


/*
 * RequestKey
 *
 * Requests any type of key into a file
 *
 * Arguments:
 *
 *		key		string containing key to request.
 *				may be prepended with 'freenet:'
 *				valid forms are:
 *					KSK@keyname
 *					keyname
 *					CHK@keyname
 *					SVK@keyname
 *					SSK@publickey/subkey
 *					MSK@anykey//
 *					MSK@anykey//subpath
 *					anykey//
 *					anykey//subpath
 *
 *		file	full pathname of file to write key's contents into
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CFreenet::RequestKey(char *key, char *file)
{
	MessageBox(0, "Freenet::RequestKey() - not implemented yet", PROGNAME, MB_SYSTEMMODAL);
	return 0;

}		// RequestKey()



/*
 * InsertKsk
 *
 * Inserts a file as a KSK
 *
 * Arguments:
 *
 *		file	full pathname of file to insert
 *				can be NULL if inserting a map (see 'map' below)
 *
 *		kskkey	string containing key to insert under
 *			    may be prepended with 'freenet:' and/or 'KSK@'
 *
 *		chkkey	address of a buffer into which to
 *				write the CHK key generated
 *				NOTE: is not prepended with 'CHK@'
 *
 *				use NULL if you're not interested in the CHK
 *
 *		map		normally, you'll set this to NULL
 *				But if you want to use the key as an MSK - a 'sitemap',
 *				set this to the full pathname of a map file to
 *				insert as metadata
 *
 *		mime	mime type to set for this key, or NULL for none
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CFreenet::InsertKsk(char *file, char *kskkey, char *chkkey, char *map, char *mime)
{
	char key2[128];
	char key3[128];

	InsertKey(file, kskkey, map, chkkey, key2, key3, mime);

	return 0;

}		// InsertKsk()



/*
 * InsertChk
 *
 * Inserts a file as a CHK
 *
 * Arguments:
 *		file	full pathname of file to insert
 *
 *		chkkey	buffer into which generated CHK is written.
 *				NOTE: is not prepended with 'CHK@'
 *
 *		map		normally, you'll set this to NULL
 *				But if you want to use the key as an MSK - a 'sitemap',
 *				set this to the full pathname of a map file to
 *				insert as metadata
 *
 *		mime	mime type to set for this key, or NULL for none
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CFreenet::InsertChk(char *file, char *chkkey, char *map, char *mime)
{
	char key2[128];
	char key3[128];

	InsertKey(file, "CHK@", map, chkkey, key2, key3, mime);
	return 0;

}		// 'InsertChk()'



/*
 * InsertSvk
 *
 * Inserts a file as an SVK
 *
 * Arguments:
 *
 *		file	full pathname of file to insert
 *
 *		svkroot	buffer into which the SVK's 'root' will be written
 *
 *		svkpub	buffer into which the public key will be written
 *
 *		svkpriv	buffer into which the private key will be written
 *
 *		map		normally, you'll set this to NULL
 *				But if you want to use the key as an MSK - a 'sitemap',
 *				set this to the full pathname of a map file to
 *				insert as metadata
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CFreenet::InsertSvk(char *file,
						char *svkroot,
						char *svkpub,
						char *svkpriv,
						char *map,
						char *mime)
{
	char	key1[128];
	char	key2[128];
	char	key3[128];
	int		i;

	InsertKey(file, "SVK@", map, key1, key2, key3, mime);

	/*
	 * carve the second key into root and public keys
	 * format is 'SVK@publickey,root'
	 */

	/* extract the svk public key part, skipping the 'SVK@' at the front */
	for (i = 0; key2[i+4] != ','; i++)
		svkpub[i] = key2[i+4];
	svkpub[i] = '\0';

	/* extract the svk root key if required - the bit after the comma */
	if (svkroot != NULL)
		strcpy(svkroot, key2 + i + 5);

	/* extract the private key */
	strcpy(svkpriv, key3);

	return 0;

}		// 'InsertSvk()'



/*
 * InsertSsk
 *
 * Inserts a file as an SSK, ie 'under' an SVK
 *
 * Arguments:
 *
 *		file	full pathname of file to insert
 *
 *		svkpriv	buffer containing an SVK's private key
 *
 *		subpath	buffer containing name of subkey
 *
 *		map		normally, you'll set this to NULL
 *				But if you want to use the key as an MSK - a 'sitemap',
 *				set this to the full pathname of a map file to
 *				insert as metadata
 *
 *		mime	mime type to set for this key, or NULL for none
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CFreenet::InsertSsk(char *file, char *svkpriv, char *subpath, char *map, char *mime)
{
	char	key1[128];
	char	key2[128];
	char	key3[128];
	char	keyname[256];

	sprintf(keyname, "%s/%s", svkpriv, subpath);
	InsertKey(file, keyname, map, key1, key2, key3, mime);

	return 0;

}		// 'InsertSsk()'




/*
 * InsertKey
 *
 * Generalised key insertion routine
 *
 * This is really the 'routine from hell', and desperately needs to be
 * replaced with something that calls Freenet much more directly.
 *
 * Arguments:
 *
 *		file		full pathname of file to insert
 *
 *		key			full key name to insert under
 *
 *		metadata	full text of metadata, or NULL.
 *					note - this is not a filename - it is a string containing
 *					the actual metadata
 *
 *		key1		Address of a buffer into which to write the first keyname
 *                  extracted from finsert's output.
 *                  This will be the generated CHK.
 *                  For SVK creation, this will be the SVK private key part
 *
 *      key2		Address of a buffer into which to write the second keyname
 *                  extracted from finsert's output.
 *                  with SVK creation, this will be the 'SVK@<pubkey>,<root>'
 *
 *      key3		Address of a buffer into which to write the third keyname
 *                  extraced from finsert's output
 *                  This will be the SVK private key.
 *
 *		mime		mime type for this key, or NULL if none
 *
 * Returns:
 *
 *		full malloc()'ed text of output from command line insert client,
 *		or NULL if insert failed
 *
 * Remarks:
 *		This will make many programmers cringe. This routine generates a BAT file which
 *		issues the insert command, and collects the output of the command line client.
 *		It is up to callers of this routine to parse the output and extract whatever info
 *		is needed (eg public and private keys from SVK inserts etc).
 */

int CFreenet::InsertKey(char *file,
						char *key,
						char *metadata,
						char *key1,
						char *key2,
						char *key3,
						char *mime)
{
	char *fndir = GetFreenetDir();
	char batfilepath[256];
	char batfilepathq[256];
	char outfilepath[256];
	char outfilepath1[256];
	char metafilepath[256];
	char nullfilepath[256];
	char *client_output;
//	long pid = getpid();
	FILE *fp;
	char buf[256];
	char *keys_returned[3] = { key1, key2, key3 };
	int num_keys_found;
	int  curr_key = 0;
//	DWORD tid = GetCurrentThreadId();
	DWORD tid = clock();

	/* Generate a null file if none given */
	if (file == NULL || file[0] == '\0')
	{
		sprintf(nullfilepath, "%s\\clib-%ld.null", fndir, tid);
		fp = fopen(nullfilepath, "wb");
		fclose(fp);
		file = nullfilepath;
	}

	/* Create temp and output files in Freenet install directory */
	sprintf(batfilepath, "%s\\clib-%ld.bat", fndir, tid);
	sprintf(batfilepathq, "\"%s\\clib-%ld.bat\"", fndir, tid);
	sprintf(outfilepath, "%s\\clib-%ld.tmp", fndir, tid);
	sprintf(outfilepath1, "%s\\clib-%ld-del.tmp", fndir, tid);

	/* If there's metadata, write it to a temp file */
	if (metadata != NULL)
	{
		sprintf(metafilepath, "%s\\clib-%ld.meta", fndir, tid);
		if ((fp = fopen(metafilepath, "wb")) == NULL)
		{
			pLog->Write(-1, LOG_CRITICAL,
						"CFreenet::InsertKey(%s,%s): can't generate metadata file",
						(int)file, (int)key);
			return 0;
		}
		fputs(metadata, fp);
		fflush(fp);
		fclose(fp);
	}


	/* Generate BAT file */
	if ((fp = fopen(batfilepath, "wt")) == NULL)
	{
		pLog->Write(-1, LOG_CRITICAL,
					"CFreenet::InsertKey(%s, %s): can't generate BAT file",
					(int)file, (int)key);
		return 0;
	}

	fprintf(fp,
			"@echo off\n"
				"rem Generated by Freenet C library insert library\n"
				"%c:\n"
				"cd %s\n",
			fndir[0], fndir + 2);

//	fprintf(fp, "finsert -htl %d ", HOPS_TO_LIVE);
	fprintf(fp, "finsert ");
	if (metadata != NULL)
		fprintf(fp, "-metadata clib-%ld.meta ", tid);
	if (mime != NULL)
		fprintf(fp, "-mimeType %s ", mime);

	fprintf(fp, "%s ", key);

	if (file == nullfilepath)
		/* point to the null file */
		fprintf(fp, "clib-%ld.null ", tid);
	else
		/* point to the actual file */
		fprintf(fp, "%s ", file);

	fprintf(fp, "> \"%s\" 2> \"%s\"\n",	outfilepath1, outfilepath);

	if (metadata != NULL)
		/* write command to delete metadata file */
		fprintf(fp, "DEL \"%s\"\n", metafilepath);

	/* add command to delete standard output file */
	fprintf(fp, "DEL clib-%ld-del.tmp\n", tid);

	/* add command to delete null file, if any */
	if (file == nullfilepath)
		fprintf(fp, "DEL clib-%ld.null\n", tid);
	fclose(fp);

	/* submit, then delete, bat file */
//	system(batfilepathq);
//	unlink(batfilepath);

	// Here's some really bad code - basically, we launch the bat file, then wait till it
	// deletes the .tmp file

	// make sure it doesn't already exist
	unlink(outfilepath1);

	// launch the bat file
	ShellExecute(0, "open", batfilepath, NULL, FreenetDir, SW_HIDE);

	// wait for outfilepath1 to come into existence
	while (GetFileAttributes(outfilepath1) == -1)
		Sleep(1000L);

	// wait till outfilepath1 stops existing
	while (GetFileAttributes(outfilepath1) != -1)
		Sleep(1000L);

	// nuke the bat file
	unlink(batfilepath);

	/* Suck up results */
	if ((fp = fopen(outfilepath, "rt")) == NULL)
	{
		pLog->Write(-1, LOG_CRITICAL,
					"CFreenet::InsertKey(%s,%s): can't read output file '%s'\n",
					(int)file, (int)key, (int)outfilepath);
		return 0;
	}

	client_output = NULL;
	while (fgets(buf, 256, fp) != NULL)
	{
		int i;

		/* chomp terminators from end of line */
		for (i = strlen(buf) - 1; i >= 0 && (buf[i] == '\r' || buf[i] == '\n');)
			buf[i--] = '\0';

		/* Extract sent keys */
		if (!strncmp(buf, "Inserted Key   : freenet:", 25))
		{
			if (keys_returned[curr_key] != NULL)
				/* Suck out 'inserted key' */
				strcpy(keys_returned[curr_key], buf+25);
			curr_key++;
		}
		else if (!strncmp(buf, "SVK private key: ", 17))
		{
			if (keys_returned[curr_key] != NULL)
				/* suck out SVK private key */
				strcpy(keys_returned[curr_key], buf+17);
			curr_key++;
		}
		else if (!strncmp(buf, "'freenet:", 9))
		{
			if (keys_returned[curr_key] != NULL)
			{
				if (keys_returned[curr_key] != NULL)
				{
					/* suck out key collision message */
					for (i = 0; buf[i+9] && buf[i+9] != '\''; i++)
						keys_returned[curr_key][i] = buf[i+9];
					keys_returned[curr_key][i] = '\0';
				}
			}
			curr_key++;
		}
	}

	/* nuke any keys not found */
	num_keys_found = curr_key;
	while (curr_key < 3)
		keys_returned[curr_key++][0] = '\0';

	fclose(fp);
	unlink(outfilepath);

	return num_keys_found;

}


/*
 * GetFreenetDir()
 *
 * Returns the directory where Freenet is installed
 *
 * Arguments:
 *
 *		none
 *
 * Returns:
 *
 *		full ASCII pathname of Freenet directory
 */

char *CFreenet::GetFreenetDir()
{
	return FreenetDir;
}
