// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//

// syncsite.cpp: implementation of the CSyncSite class.
//
//////////////////////////////////////////////////////////////////////

/*
 * Class to implement auto-updates of all sites
 */

#include <stdio.h>
#include <string.h>

#include "stdafx.h"

#include "ezFCPlib.h"

#include "freeweb.h"

#include "fwpubgui.h"
#include "fwpubguidlg.h"
#include "syncsite.h"
#include "constants.h"
#include "config.h"
#include "constants.h"
#include "sitedir.h"
#include "sitelist.h"
#include "util.h"
#include "log.h"
#include "gethttp.h"

#include "syncthreads.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

extern CLog			*pLog;
extern CConfig		*pConfig;
extern CLog			*pLog;
extern CSiteDir		*pSiteDir;
extern CUtil		*pUtil;
extern CSiteList	*pSiteList;
extern CFwpubguiDlg	*pDlg;
extern CGetHttp		*pHttp;


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSyncSite::CSyncSite()
{
}

CSyncSite::~CSyncSite()
{

}


/////////////////////////////////////
// Tells whether any site uploads are
// currently in progress
//

BOOL CSyncSite::InitInstance()
{
//	pLog->Write(LOG_DEBUG, LOG_DEBUG, "Site sync thread started");

	// set running threads counter
	CSyncThreads::NumRunningThreads = 0;

	// we stay in this loop waiting for signal from main thread
	while (1)
	{
		Sleep(1000L);
		if (pDlg->UploadState == SYNCTHREAD_IDLE)
			continue;

		// do we just need to send new parameters to Freenet?
		if (pDlg && pDlg->UploadState == SYNCTHREAD_SETPARMS)
		{
			// send new parms to Freenet
			pLog->Write(LOG_DEBUG, LOG_DEBUG,
						"htl='%d'", (int)pConfig->htl);
			pDlg->UploadState = SYNCTHREAD_IDLE;
			continue;
		}

		// otherwise, we've been told to synchronise sites
		pLog->Write(LOG_DEBUG, LOG_DEBUG, "Site sync: starting syncs");

		// we've got the go-ahead from the main thread
		UploadSite(pDlg->guiDisabled ? "all" : pDlg->selectedSite);
//		pSiteList->Refresh();

		// tell the main thread we've finished
		pDlg->UploadState = SYNCTHREAD_IDLE;
		strcpy(pDlg->selectedSite, "all");

	}		// 'while (1)'
}



int CSyncSite::UploadAll()
{
	return UploadSite("all");
}



/*
 * UploadSite()
 *
 * Automatically updates all sites in a given map file
 *
 * Arguments:
 *
 *		domain		individual domain to update, or 'all'
 *		days		number of days ahead to update.
 *
 * Returns:
 *
 *		0 if successful
 *		nonzero if failed
 */

int CSyncSite::UploadSite(char *domain)
{
	char		registered_new_domains = 0;
	int			days = 0;
	Site		*thissite;
	SiteFile	*filelist;
	SiteFile	*filelist_old;
	SiteFile	*thisfile;
	SiteFile	*thisfile_old;

	char		new_msk_needed = 0;
	char		needs_svkroot;
	char		needs_kskptr;
	char		needs_sskmap;

	bool		someInsertsFailed = false;

	pLog->Write(LOG_DEBUG, LOG_DEBUG, "-----------------------------------------");
	pLog->Write(LOG_DEBUG, LOG_DEBUG, "Freshening Site '%s'", (int)domain);

	/* go through all sites, update as needed */
	for (thissite = pConfig->firstsite;
		thissite != NULL;
		thissite = thissite->next)
	{
		char *thisdom = thissite->domainname;

		/* Do we need to update this site? */
		if (strcmp(domain, "all") && strcmp(domain, thisdom))
			continue;	/* no - skip this one */

		/* Is this site up to date? */
		if (strcmp(thissite->dateupdated, pUtil->getgmtdate()) >= 0)
		{
			pLog->Write(LOG_DEBUG, LOG_DEBUG, "domain '%s' is up to date", (int)thisdom);
			continue;
		}

		//
		// Found a site needing updating
		//

		if (!pDlg->guiDisabled)
		{
			thissite->updating = 1;
			//strcpy(thissite->dateupdated, "U");	// mark update date as 'U' to indicate updating
			pSiteList->Refresh();
		}

		needs_svkroot = (thissite->ssk_pub[0] == '\0') ? 1 : 0;
//		needs_kskptr = (thissite->ksk_ptr[0] == '\0') ? 1 : 0;
		needs_kskptr = needs_svkroot;
		needs_sskmap = 1;

		// Go no further with this site unless we can snarf its domain
			pLog->Write(LOG_USER, LOG_USER,
						"Checking if domain '%s' is registered by someone else",
						(int)thisdom);
			pLog->Write(LOG_USER, LOG_USER,
						"This can take 3 minutes (or more if your HTL setting is high); please be patient..");

			// new - check if domain is registered
			if (domainIsRegistered(thisdom))
			{
				pLog->Write(LOG_CRITICAL, LOG_CRITICAL,
							"Sorry - it seems that someone else has registered '%s'",
							(int)thisdom);
				thissite->updating = 0;
				pSiteList->Refresh();
				continue;
			}
			pLog->Write(LOG_USER, LOG_USER,
						"Claiming domain '%s.free' and inserting files...",
						(int)thisdom);

		/* back up this site's file list */
		filelist_old = thissite->firstfile;

		// if needed, generate an index.html file
		// just generate a dummy now so it will show up in the listing
		// it's not nice, i know, but it solves a chicken-and-egg problem
		if (thissite->generatedIndex)
		{
			char buf[256];
			FILE *fp;

			sprintf(buf, "%s\\index.html", thissite->rootdir);
			fp = fopen(buf, "a");
			fclose(fp);
		}
			
		/* Scan this site's directory and chain it in to site */
		filelist = pSiteDir->scan_dir(thissite->rootdir);
		thissite->firstfile = filelist;

		/* mark each file that has a chk and hasn't changed */
		for (thisfile = filelist; thisfile != NULL; thisfile = thisfile->next)
		{
			/* Try to find the file in existing configs for this site */
			for (thisfile_old = filelist_old;
				thisfile_old != NULL;
				thisfile_old = thisfile_old->next)
			{
				if (!strcmp(thisfile->filename, thisfile_old->filename))
				{
					/* file is previously known */
					/* see if it has a CHK and that dates match */
					if ((thisfile_old->chk[0] != '\0')
						&&
						(strcmp(thisfile->time_published,
								thisfile_old->time_published) <= 0)
						)
					{
						/* file hasn't changed - reset update flag */
						thisfile->needs_update = 0;

						/* And grab its CHK since we're re-using it */
						strcpy(thisfile->chk, thisfile_old->chk);
					}
					else
						needs_sskmap = 1;
					break;
				}
			}

			/* Was the file previously known? */
			if (thisfile_old == NULL)
				/* no - we need a new map */
				needs_sskmap = 1;

		}		/* 'for (each file in site's directory)' */

		//
		// if needed, generate an index.html listing all the files
		//
		if (thissite->generatedIndex)
			pSiteDir->generate_index_html(thisdom, thissite->rootdir, filelist);


		////////////////////////////////////////////////////////////////////
		//
		// Perform the insert of this site
		//
		////////////////////////////////////////////////////////////////////

		// Create keypair if needed
		if (needs_svkroot)
		{
			HFCP *hfcp;

			// create a keypair for this domain
			pLog->Write(LOG_VERBOSE, LOG_VERBOSE,
						"domain '%s': creating keypair", (int)thisdom);
			hfcp = fcpCreateHandle();
			fcpMakeSvkKeypair(hfcp, thissite->ssk_pub, thissite->ssk_priv);
			fcpDestroyHandle(hfcp);

			pLog->Write(LOG_VERBOSE, LOG_VERBOSE,
						"site '%s': pubkey='%s'", (int)thisdom, (int)thissite->ssk_pub);
			pLog->Write(LOG_VERBOSE, LOG_VERBOSE,
						"site '%s': privkeykey='%s'", (int)thisdom, (int)thissite->ssk_priv);

		}		// 'if (needs_svkroot)'

		// Now insert the individual files
		pLog->Write(LOG_USER, LOG_USER,
					"Site '%s' - Inserting individual files",
					(int)thissite->domainname);
		for (thisfile = filelist; thisfile != NULL; thisfile = thisfile->next)
		{
			if (thisfile->needs_update)
			{
				Sleep(100L);	// space out the insert threads a bit

				// Wait till running inserts count drops to below maximum
				while (CSyncThreads::NumRunningThreads >= pConfig->maxInsertThreads)
					Sleep(500);

				// We're blow max number of threads - fire up a CHK insert
				CSyncThreads::CSyncThreadChk(thisfile);

			}
		}

		// Check that all insert jobs succeeded
		pLog->Write(LOG_USER, LOG_USER,
					"Now inserting files for site '%s'",
					(int)thissite->domainname);
		for (thisfile = filelist; thisfile != NULL; thisfile = thisfile->next)
		{
			if (strncmp(thisfile->chk, "CHK@", 4) != 0)
			{
				Sleep(100L);	// space out the insert threads a bit

				// Wait till running inserts count drops to below maximum
				while (CSyncThreads::NumRunningThreads >= pConfig->maxInsertThreads)
					Sleep(500);

				// We're blow max number of threads - fire up a thread to re-insert
				CSyncThreads::CSyncThreadChk(thisfile);
			}
		}

		// Wait for all these jobs to complete
		while (CSyncThreads::NumRunningThreads > 0)
			Sleep(500);
		pLog->Write(LOG_USER, LOG_USER,
					"Site '%s' - all files inserted",
					(int)thissite->domainname);

		// launch insert of DBR Root if needed
		if (needs_svkroot)
		{
			//while (CSyncThreads::NumRunningThreads > 0) Sleep(500);
			CSyncThreads::CSyncThreadDbr(thissite->domainname,
										thissite->ssk_pub,
										thissite->ssk_priv);
			pLog->Write(LOG_USER, LOG_USER,
						"Site '%s' - creating DBR root key",
						(int)thissite->domainname);
		}

		// launch insert of KSK pointer if needed
		if (needs_kskptr)
		{
			//while (CSyncThreads::NumRunningThreads > 0) Sleep(500);
			CSyncThreads::CSyncThreadKsk(thissite->domainname,
										thissite->ssk_pub);
			pLog->Write(LOG_USER, LOG_USER,
						"Site '%s' - Creating KSK pointer key",
						(int)thissite->domainname);
		}

		// launch insert of map if needed
		if (needs_sskmap)
		{
			//while (CSyncThreads::NumRunningThreads > 0) Sleep(500);
			CSyncThreads::CSyncThreadMsk(thissite->domainname,
										thissite->firstfile,
										thissite->rootdir,
										thissite->ssk_priv);
			pLog->Write(LOG_USER, LOG_USER,
						"Site '%s' - inserting MSK sitemap",
						(int)thissite->domainname);
		}

		// Wait for all these jobs to complete
		while (CSyncThreads::NumRunningThreads > 0)
			Sleep(500);

		/* Update the update field */
		strcpy(thissite->dateupdated, pUtil->getgmtdate());

		thissite->updating = 0;
		pSiteList->Refresh();
		pLog->Write(LOG_USER, LOG_USER,
					"Site '%s' - INSERTION COMPLETED",
					(int)thissite->domainname);

	}			/* 'for (each site)' */

	////PerformInserts();
	pConfig->save();
	return 0;
}		/* 'fpublish_update()' */


int CSyncSite::domainIsRegistered(char *domain)
{
	HFCP *hfcp = fcpCreateHandle();
	char buf[256];

	sprintf(buf, "freenet:KSK@freeweb/%s", domain);
	fcpSetHtl(hfcp, pConfig->htlReq);
	fcpRawMode(hfcp, 1);
	if (fcpGetKeyToMem(hfcp, buf, NULL, NULL) == 0)
		return 1;	// domain is registered
	else
		return 0;	// key not found - domain not previously registered

}			// 'domainIsRegistered()'

