// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//

// syncthreads.cpp: implementation of the CSyncThreads class.
//
//////////////////////////////////////////////////////////////////////


/*
 * Go through the job queue, and manage freenet inserts concurrently
 */

#include "stdafx.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <windows.h>
#include <winbase.h>

#include "ezFCPlib.h"

#include "freeweb.h"
#include "constants.h"
#include "fwpubgui.h"
#include "config.h"

#include "syncthreads.h"
#include "syncsite.h"

#include "util.h"
#include "log.h"
#include "fwpubguidlg.h"

//#include "freenet.h"
//#include "syncqueue.h"
//#include "fwpublish.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//
// IMPORTED DECLARATIONS
//

//extern CSyncQueue	*pSyncQueue;
//extern CFreenet		*pFreenet;
extern CUtil		*pUtil;
extern CLog			*pLog;
extern CConfig	*pConfig;



// Despicable global var

int CSyncThreads::NumRunningThreads = 0;	// keeps count of number of insert threads currently running


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSyncThreads::CSyncThreads()
{
	// not using plain constructor
}

CSyncThreads::~CSyncThreads()
{

}


//
// Constructor overloads specific to job types
//

	SiteFile		*m_File;			// these fields provide all required info
	char			*m_domain;
	char			*m_pubkey;
	char			*m_privkey;


// create a CHK insertion job

void CSyncThreads::CSyncThreadChk(SiteFile *file)
{
	NumRunningThreads++;
	CSyncThreads *thrd = new CSyncThreads();
	thrd->JobType = INSERTQ_TYPE_CHKFILE;
	thrd->m_File = file;
	thrd->m_RootDir = NULL;
	thrd->m_domain = NULL;
	thrd->m_privkey = NULL;
	thrd->m_pubkey = NULL;
	thrd->CreateThread(0,0);
}

// for DBR root inserts
void CSyncThreads::CSyncThreadDbr(char *domain, char *pub, char *priv)
{
	NumRunningThreads++;
	CSyncThreads *thrd = new CSyncThreads();
	thrd->JobType = INSERTQ_TYPE_SVKROOT;
	thrd->m_File = NULL;
	thrd->m_RootDir = NULL;
	thrd->m_domain = domain;
	thrd->m_privkey = priv;
	thrd->m_pubkey = pub;
	thrd->CreateThread(0,0);
}

// for KSK ptr inserts
void CSyncThreads::CSyncThreadKsk(char *domain, char *pubkey)
{
	NumRunningThreads++;
	CSyncThreads *thrd = new CSyncThreads();
	thrd->JobType = INSERTQ_TYPE_KSKPTR;
	thrd->m_File = NULL;
	thrd->m_RootDir = NULL;
	thrd->m_domain = domain;
	thrd->m_privkey = NULL;
	thrd->m_pubkey = pubkey;
	thrd->CreateThread(0,0);
}

// for MSK map inserts
void CSyncThreads::CSyncThreadMsk(char *domain, SiteFile *filelist, char *rootdir, char *privkey)
{
	NumRunningThreads++;
	CSyncThreads *thrd = new CSyncThreads();
	thrd->JobType = INSERTQ_TYPE_SSKMAP;
	thrd->m_File = filelist;
	thrd->m_RootDir = rootdir;
	thrd->m_domain = domain;
	thrd->m_privkey = privkey;
	thrd->m_pubkey = NULL;
	thrd->CreateThread(0,0);
}


//////////////////////////////////////////////////////////////////////
// Methods
//////////////////////////////////////////////////////////////////////


BOOL CSyncThreads::InitInstance()
{
	switch (JobType)
	{
	case INSERTQ_TYPE_CHKFILE:
		JobThreadChkfile(m_File->filename, m_File->chk);
		break;

	case INSERTQ_TYPE_SVKROOT:
		JobThreadSvkroot(m_domain, m_pubkey, m_privkey);
		break;

	case INSERTQ_TYPE_KSKPTR:
		JobThreadKskptr(m_domain, m_pubkey);
		break;

	case INSERTQ_TYPE_SSKMAP:
		JobThreadSskmap(m_domain, m_RootDir, m_File, m_privkey);
		break;


	}		// 'switch (JobType)'

#ifdef SHIT
	INSERTQ *this_job = pSyncQueue->JobsInProgress[slot_num];

	this->m_bAutoDelete = TRUE;

//	MessageBox(0, "Thread finished", "debug", MB_SYSTEMMODAL);
	JobThread(slot_num);
//	Sleep(2000L);
	this_job->status = INSERTQ_STATUS_DONE;
	this->ExitInstance();
#endif
	return FALSE;
}


//
// JobThreadSvkRoot()
//
// Inserts a Date-Based Redirect update point in SSK@<pubkey>/domainname
//

int CSyncThreads::JobThreadSvkroot(char *domainname, char *svk_pub, char *svk_priv)
{
	char metadata[128];
	char uri[128];
	HFCP *hfcp;

	hfcp = fcpCreateHandle();
	fcpRawMode(hfcp, 1);
	fcpSetHtl(hfcp, pConfig->htl);

	// create a URI
	sprintf(uri, "freenet:SSK@%s/%s", svk_priv, domainname);

	// create metadata for DBR update point
	sprintf(metadata,
			"Redirect\nincrement=86400\nbaseline=20000101000000\nEnd\nfreenet:SSK@%s/%s\n",
			svk_pub,
			domainname);

	// Now insert it into Freenet
	fcpPutKeyFromMem(hfcp, uri, NULL, metadata, 0);

	// finished with this handle
	fcpDestroyHandle(hfcp);

	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadSvkroot: svk_pub='%s'",
				(int)svk_pub);
	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadSvkroot: svk_priv='%s'",
				(int)svk_priv);
	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadSvkroot: metadata:\r\n%s",
				(int)metadata);
	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadSvkroot: uri=%s",
				(int)uri);

	NumRunningThreads--;
	return 0;

}		// 'JobThreadSvkroot()'


int CSyncThreads::JobThreadKskptr(char *domainname, char *svk_pub)
{
	char metadata[128];
	char uri[128];
	HFCP *hfcp;
	char *httpPage = NULL;

	hfcp = fcpCreateHandle();
	fcpRawMode(hfcp, 1);
	fcpSetHtl(hfcp, pConfig->htl);

	// create a URI
	sprintf(uri, "freenet:KSK@freeweb/%s", domainname);

	// create metadata for DBR update point
	sprintf(metadata,
			"Redirect\nincrement=86400\nbaseline=20000101000000\nEnd\nfreenet:SSK@%s/%s\n",
			svk_pub,
			domainname);

	// Create a page which offers a link to secure URI
	//httpPage = pUtil->strsav(NULL, "");

	// Now insert it into Freenet
	fcpPutKeyFromMem(hfcp, uri, NULL, metadata, 0);

	// finished with this handle
	fcpDestroyHandle(hfcp);

	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadKskptr:inserted '%s', uri='%s'",
				(int)svk_pub, (int)uri);

	NumRunningThreads--;
	return 0;

}		/* 'job_thread_kskptr()' */


int CSyncThreads::JobThreadChkfile(char *pathname, char *chk_key)
{
//	CFreenet *pFreenetObj = new CFreenet;
	char *mimetype;
	char metadata[128];
	HFCP *hfcp;

	// Create metadata with mime type
	mimetype = GetMimeType(pathname);
	sprintf(metadata, "Content-Type: %s\n", mimetype);

	// now insert the key
	hfcp = fcpCreateHandle();
	fcpSetHtl(hfcp, pConfig->htl);
	fcpPutKeyFromFile(hfcp, "CHK@", pathname, metadata);

	// uplift the created key URI
	strcpy(chk_key, hfcp->created_uri);

//	pFreenetObj->InsertChk(pathname, chk_key, NULL, NULL);
	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadChkfile:inserted file '%s', key='%s'",
				(int)pathname, (int)chk_key);

	NumRunningThreads--;
	return 0;

}		/* 'job_thread_chkfile()' */


int CSyncThreads::JobThreadSskmap(char *domain, char *rootdir, SiteFile *files, char *svk_priv)
{
	SiteFile	*thisfile;
	char		*ssk_map = NULL;
	////char		*ssk_key;
	////char		*ssk_subpath;

	////char metadata[128];
	char uri[128];
	HFCP *hfcp;

	int rootlen = strlen(rootdir) + 1;		// enable relative pathnames

	// create and set up an FCP handle
	hfcp = fcpCreateHandle();
	fcpRawMode(hfcp, 1);
	fcpSetHtl(hfcp, pConfig->htl);

	// create a URI
	sprintf(uri, "freenet:SSK@%s/%s-%s", svk_priv, pUtil->getgmtmidnight(), domain);

	/* construct mapfile */
	ssk_map = pUtil->strsav(NULL, "Mapfile\ndefault=index.html\nEnd\n");
	for (thisfile = files; thisfile != NULL; thisfile = thisfile->next)
	{
		char filepath[256];
		char *s;

		// convert file path to correct relative path with forward slashes
		strcpy(filepath, thisfile->filename + rootlen);
		for (s = filepath; *s; s++)
			if (*s == '\\')
				*s = '/';

		ssk_map = pUtil->strsav(ssk_map, filepath);
		ssk_map = pUtil->strsav(ssk_map, "=");

		if (thisfile->chk == NULL || strlen(thisfile->chk) == 0)
		{
			char buf[512];

			sprintf(buf, "AAARRRGGGHH!!!\r\n\r\n"
							"The CHK for file '%s' is missing from the manifest!\r\n\r\n"
							"This is a major critical bug - please email me your freeweb.log file so i can fix it",
							filepath);
			MessageBox(0, buf, "FreeWeb is having a hard time :(",
								MB_SYSTEMMODAL | MB_ICONSTOP);
			ssk_map = pUtil->strsav(ssk_map, "KSK@freeweb-404.html");
		}
		else
			ssk_map = pUtil->strsav(ssk_map, thisfile->chk);
		ssk_map = pUtil->strsav(ssk_map, "\n");
	}

	// Now insert it into Freenet
	fcpPutKeyFromMem(hfcp, uri, NULL, ssk_map, 0);

	pLog->Write(LOG_DEBUG, LOG_VERBOSE,
				"JobThreadSskmap: inserted key: '%s' with the map data:",
				(int)uri);

	pLog->Write(LOG_DEBUG, LOG_VERBOSE, ssk_map);
	pLog->Write(LOG_DEBUG, LOG_VERBOSE, "----- END OF MAP -----");

	// finished with this handle, and metadata buffer
	fcpDestroyHandle(hfcp);
	free(ssk_map);

	NumRunningThreads--;
	return 0;

}		// 'JobThreadSskmap()'



struct mimetab_ent CSyncThreads::MimeTab[] = {
	{ "csm", "application/cu-seeme" },
	{ "csm", "application/cu-seeme" },
	{ "cu", "application/cu-seeme" },
	{ "tsp", "application/dsptype" },
	{ "xls", "application/excel" },
	{ "spl", "application/futuresplash" },
	{ "hqx", "application/mac-binhex40" },
	{ "doc", "application/msword" },
	{ "dot", "application/msword" },
	{ "bin", "application/octet-stream" },
	{ "oda", "application/oda" },
	{ "pdf", "application/pdf" },
	{ "pgp", "application/pgp-signature" },
	{ "ps", "application/postscript" },
	{ "ai", "application/postscript" },
	{ "eps", "application/postscript" },
	{ "ppt", "application/powerpoint" },
	{ "rtf", "application/rtf" },
	{ "wp5", "application/wordperfect5.1" },
	{ "zip", "application/zip" },
	{ "wk", "application/x-123" },
	{ "bcpio", "application/x-bcpio" },
	{ "pgn", "application/x-chess-pgn" },
	{ "cpio", "application/x-cpio" },
	{ "deb", "application/x-debian-package" },
	{ "dcr", "application/x-director" },
	{ "dir", "application/x-director" },
	{ "dxr", "application/x-director" },
	{ "dvi", "application/x-dvi" },
	{ "pfa", "application/x-font" },
	{ "pfb", "application/x-font" },
	{ "gsf", "application/x-font" },
	{ "pcf", "application/x-font" },
	{ "pcf.Z", "application/x-font" },
	{ "gtar", "application/x-gtar" },
	{ "tgz", "application/x-gtar" },
	{ "hdf", "application/x-hdf" },
	{ "phtml", "application/x-httpd-php" },
	{ "pht", "application/x-httpd-php" },
	{ "php", "application/x-httpd-php" },
	{ "php3", "application/x-httpd-php3" },
	{ "phps", "application/x-httpd-php3-source" },
	{ "php3p", "application/x-httpd-php3-preprocessed" },
	{ "class", "application/x-java" },
	{ "latex", "application/x-latex" },
	{ "frm", "application/x-maker" },
	{ "maker", "application/x-maker" },
	{ "frame", "application/x-maker" },
	{ "fm", "application/x-maker" },
	{ "fb", "application/x-maker" },
	{ "book", "application/x-maker" },
	{ "fbdoc", "application/x-maker" },
	{ "mif", "application/x-mif" },
	{ "com", "application/x-msdos-program" },
	{ "exe", "application/x-msdos-program" },
	{ "bat", "application/x-msdos-program" },
	{ "dll", "application/x-msdos-program" },
	{ "nc", "application/x-netcdf" },
	{ "cdf", "application/x-netcdf" },
	{ "pac", "application/x-ns-proxy-autoconfig" },
	{ "o", "application/x-object" },
	{ "pl", "application/x-perl" },
	{ "pm", "application/x-perl" },
	{ "shar", "application/x-shar" },
	{ "swf", "application/x-shockwave-flash" },
	{ "swfl", "application/x-shockwave-flash" },
	{ "sit", "application/x-stuffit" },
	{ "sv4cpio", "application/x-sv4cpio" },
	{ "sv4crc", "application/x-sv4crc" },
	{ "tar", "application/x-tar" },
	{ "gf", "application/x-tex-gf" },
	{ "pk", "application/x-tex-pk" },
	{ "PK", "application/x-tex-pk" },
	{ "texinfo", "application/x-texinfo" },
	{ "texi", "application/x-texinfo" },
	{ "~", "application/x-trash" },
	{ "%", "application/x-trash" },
	{ "bak", "application/x-trash" },
	{ "old", "application/x-trash" },
	{ "sik", "application/x-trash" },
	{ "t", "application/x-troff" },
	{ "tr", "application/x-troff" },
	{ "roff", "application/x-troff" },
	{ "man", "application/x-troff-man" },
	{ "me", "application/x-troff-me" },
	{ "ms", "application/x-troff-ms" },
	{ "ustar", "application/x-ustar" },
	{ "src", "application/x-wais-source" },
	{ "wz", "application/x-wingz" },
	{ "au", "audio/basic" },
	{ "snd", "audio/basic" },
	{ "mid", "audio/midi" },
	{ "midi", "audio/midi" },
	{ "mpga", "audio/mpeg" },
	{ "mpega", "audio/mpeg" },
	{ "mp2", "audio/mpeg" },
	{ "mp3", "audio/mpeg" },
	{ "m3u", "audio/mpegurl" },
	{ "aif", "audio/x-aiff" },
	{ "aiff", "audio/x-aiff" },
	{ "aifc", "audio/x-aiff" },
	{ "gsm", "audio/x-gsm" },
	{ "ra", "audio/x-pn-realaudio" },
	{ "rm", "audio/x-pn-realaudio" },
	{ "ram", "audio/x-pn-realaudio" },
	{ "rpm", "audio/x-pn-realaudio-plugin" },
	{ "wav", "audio/x-wav" },
	{ "gif", "image/gif" },
	{ "ief", "image/ief" },
	{ "jpeg", "image/jpeg" },
	{ "jpg", "image/jpeg" },
	{ "jpe", "image/jpeg" },
	{ "png", "image/png" },
	{ "tiff", "image/tiff" },
	{ "tif", "image/tiff" },
	{ "ras", "image/x-cmu-raster" },
	{ "bmp", "image/x-ms-bmp" },
	{ "pnm", "image/x-portable-anymap" },
	{ "pbm", "image/x-portable-bitmap" },
	{ "pgm", "image/x-portable-graymap" },
	{ "ppm", "image/x-portable-pixmap" },
	{ "rgb", "image/x-rgb" },
	{ "xbm", "image/x-xbitmap" },
	{ "xpm", "image/x-xpixmap" },
	{ "xwd", "image/x-xwindowdump" },
	{ "csv", "text/comma-separated-values" },
	{ "html", "text/html" },
	{ "htm", "text/html" },
	{ "mml", "text/mathml" },
	{ "txt", "text/plain" },
	{ "rtx", "text/richtext" },
	{ "tsv", "text/tab-separated-values" },
	{ "h++", "text/x-c++hdr" },
	{ "hpp", "text/x-c++hdr" },
	{ "hxx", "text/x-c++hdr" },
	{ "hh", "text/x-c++hdr" },
	{ "c++", "text/x-c++src" },
	{ "cpp", "text/x-c++src" },
	{ "cxx", "text/x-c++src" },
	{ "cc", "text/x-c++src" },
	{ "h", "text/x-chdr" },
	{ "csh", "text/x-csh" },
	{ "c", "text/x-csrc" },
	{ "java", "text/x-java" },
	{ "moc", "text/x-moc" },
	{ "p", "text/x-pascal" },
	{ "pas", "text/x-pascal" },
	{ "etx", "text/x-setext" },
	{ "sh", "text/x-sh" },
	{ "tcl", "text/x-tcl" },
	{ "tk", "text/x-tcl" },
	{ "tex", "text/x-tex" },
	{ "ltx", "text/x-tex" },
	{ "sty", "text/x-tex" },
	{ "cls", "text/x-tex" },
	{ "vcs", "text/x-vCalendar" },
	{ "vcf", "text/x-vCard" },
	{ "dl", "video/dl" },
	{ "fli", "video/fli" },
	{ "gl", "video/gl" },
	{ "mpeg", "video/mpeg" },
	{ "mpg", "video/mpeg" },
	{ "mpe", "video/mpeg" },
	{ "qt", "video/quicktime" },
	{ "mov", "video/quicktime" },
	{ "asf", "video/x-ms-asf" },
	{ "asx", "video/x-ms-asf" },
	{ "avi", "video/x-msvideo" },
	{ "movie", "video/x-sgi-movie" },
	{ "vrm", "x-world/x-vrml" },
	{ "vrml", "x-world/x-vrml" },
	{ "wrl", "x-world/x-vrml" },
	{ "ogg", "application/x-ogg" },
	{ NULL, NULL }
};


char *CSyncThreads::GetMimeType(char *pathname)
{
	int i;
	char buf[128];
	char *s;
	char delim = DIR_DELIM_CHAR;

	strcpy(buf, pathname);
	s = strrchr(buf, delim);	// find final slash
	if ((s = strrchr(s, '.')) == NULL)
		return "text/plain";		// no file extension - return default mimetype
	s++;	// skip the '.'

	for (i = 0; MimeTab[i].ext != NULL; i++)
		if (!strcmp(s, MimeTab[i].ext))
			return MimeTab[i].mimetype;	// found mimetype

	// fall back on default mime type
	return "text/plain";

}		// 'GetMimeType()'

