// 
//  This code is part of FreeWeb - an FCP-based client for Freenet
//
//  Designed and implemented by David McNab, david@rebirthing.co.nz
//  CopyLeft (c) 2001 by David McNab
//
//  The FreeWeb website is at http://freeweb.sourceforge.net
//  The website for Freenet is at http://freenet.sourceforge.net
//
//  This code is distributed under the GNU Public Licence (GPL) version 2.
//  See http://www.gnu.org/ for further details of the GPL.
//

//
// fwproxy.c
//
// FreeWeb http proxy server
//
// Relays normal web requests out to the mainstream web (if not blocking)
// and redirects requests on .free domain to Freenet
//
// Translation: http://www.mysite.free/file.html -> MSK@KSK@freeweb/mysite//file.html
//


#include "ezFCPlib.h"

//
// EXPORTED DECLARATIONS
//

void fwproxy(int listenport,
			 int extproxyenabled, char *extproxyaddr, int extproxyport,
			 int comeback, int singlethread);
void fwproxy_newport(int newport);
void fwproxy_newextproxy(int enabled, char *addr, int port);
void fwproxy_webblock(int onoff);
void fwproxy_sethtl(int htl);
void fwproxy_setdir(char *dir);


//
// IMPORTED DECLARATIONS
//

extern void proxy_start(int listenPortparm, char *extproxyaddr, int extproxyport,
						int (*callback_fn)(), int comeback, int singlethread);
extern void proxy_newport(int newport);
extern void proxy_newextproxy(int enabled, char *addr, int port);


//
// PRIVATE DECLARATIONS
//

static int fwproxy_callback(char *host, char *headers, int sock);

static int fwproxy_internal_req(char *headers, int sock);
static int fwproxy_freeweb_req(char *host, char *headers, int sock);
static void fwproxy_getpath(char *headers, char *path);
static int fwproxy_reply_webblock(int sock);
static int fwproxy_fproxy_req(char *host, char *headers, int sock);

static int hexdig_to_int(char dig);

static int fwproxy_webblock_mode = 1;
static int fwproxy_htl = 100;

static char fwproxy_dir[256] = "";

static int numhits = 0;


////////////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION DEFINITIONS - EXPORTS
//
////////////////////////////////////////////////////////////////////////////////////////


void fwproxy(int listenport,
			 int extproxyenabled, char *extproxyaddr, int extproxyport,
			 int comeback, int singlethread)
{
	// bail if directory isn't set
	if (fwproxy_dir[0] == '\0')
		return; //// -1;

	printf("fwproxy listening on port %d\n", listenport);

	// fire up proxy server
	proxy_start(listenport, extproxyaddr, extproxyport, fwproxy_callback, comeback, singlethread);
}

void fwproxy_newport(int newport)
{
	proxy_newport(newport);
}

void fwproxy_newextproxy(int enabled, char *addr, int port)
{
	proxy_newextproxy(enabled, addr, port);
}

void fwproxy_webblock(int onoff)
{
	fwproxy_webblock_mode = onoff;
}

void fwproxy_setdir(char *dir)
{
	strcpy(fwproxy_dir, dir);
}


////////////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION DEFINITIONS - PRIVATE
//
////////////////////////////////////////////////////////////////////////////////////////

//
// fwproxy_callback
//
// Here's the guts of FreeWeb Proxy
//
// It gets called every time the browser sends an http request
// It's up to this function to route the request to the mainstream
// web, or divert it to FreeWeb, or serve a hardwired page, according
// to the target host.
//

static int fwproxy_callback(char *host, char *headers, int sock)
{
	int host_len;
	char newhost[128];

	strcpy(newhost, host);
	host_len = strlen(newhost);

	// strip off '.new.net' from end if present - an IE 5.5 quirk
	if (host_len > 8 && !strcmp(newhost + host_len - 8, ".new.net"))
	{
		newhost[host_len - 8] = '\0';
		host_len -= 8;
	}

	// test if we're requesting on .free domain
	if (host_len == 4 && !strcmp(newhost, "free"))
	{
		// request on 'http://free'
		//return fwproxy_internal_req(headers, sock);
		return fwproxy_fproxy_req(newhost, headers, sock);
	}
	else if (host_len > 4 && !strcmp(newhost + host_len - 5, ".free"))
	{
		// host ends in '.free' - here's a freeweb request
		return fwproxy_freeweb_req(newhost, headers, sock);
	}
//#ifdef SSS
	else if (!strcmp(newhost, "127.0.0.1") || !strcmp(newhost, "localhost"))
	{
		// fproxy emulation
		return fwproxy_fproxy_req(newhost, headers, sock);
	}
//#endif
	else if (fwproxy_webblock_mode)
		// mainstream web request, but web blocking is on
		return fwproxy_reply_webblock(sock);
	else
		// mainstream web request - permit this
	{
		numhits++;
		return 0;
	}
}


//
// request sent to freeweb proxy itself
//

static int fwproxy_internal_req(char *headers, int sock)
{
	char *http_reply =
		"HTTP/1.0 200 OK\n"
		"Connection: close\n"
		"Content-Type: text/html\n\n"
		"<html>"
		"<head>"
		"<title>Welcome to FreeWeb</title>\n"
		"</head>"
		"<body bgcolor=\"#000000\" text=#ffff80 link=\"#66FF66\">"
		"<center><h1>Welcome to FreeWeb</h1><br>\n"
		"Your FreeWeb Proxy is working ok\n"
		"</body></html>\n"
		;
	int reply_len = strlen(http_reply);
	send(sock, http_reply, reply_len, 0);
	return 1;
}

strcmp666(char *s1, char *s2)
{
	int result = strcmp(s1, s2);
	return result;
}


//
// fwproxy_freeweb_req
//
// Translate FreeWeb URL to a Freenet key, and retrieve the key from Freenet,
// and send it (or a 404) back to browser
//

static int fwproxy_freeweb_req(char *host, char *headers, int sock)
{
	HFCP *hfcp;
	char *host1 = strdup(host);
	char *host2 = host1;
	char freenet_key[256];
	char path[1024];
	char *docpath = path;

#ifdef XXXX
	// grab first line of header
	strncpy(path, headers, 1023);
	path[1023] = '\0';

	// point to path in header
	*strchr(strchr(path, ' ') + 1, ' ') = '\0';	// chop after pathname
	docpath = strchr(path, ' ') + 1;
	if (!strncmp(docpath, "http://", 7))
	{
		docpath += 7;
		docpath = strchr(docpath, '/');
	}

	// skip leading slash in path
	docpath++;
#endif
	fwproxy_getpath(headers, docpath);

	// strip '.free' from end of host
	host1[strlen(host1) - 5] = '\0';

	// strip 'www' if present
	if (!strncmp(host1, "www.", 4))
		host1 += 4;

	// Create a handle to Freenet FCP and set htl
	hfcp = fcpCreateHandle();
	fcpSetHtl(hfcp, fwproxy_htl);

	// Convert hostname to Freenet key
	sprintf(freenet_key, "freenet:MSK@KSK@freeweb/%s//%s", host1, docpath);

	// Open Freenet key
	if (fcpOpenKey(hfcp, freenet_key, _FCP_O_READ) < 0)
	{
		// Send back a 404
		char *http_404 =
				"HTTP/1.0 404 Page Not Found\n"
				"Connection: close\n"
				"Content-Type: text/html\n\n"
				"<html>"
				"<head>"
				"<title>FreeWeb - Page Not Found</title>\n"
				"</head><body>\n"
				"<h1>Page Not Found</h1>\n"
				"<h3>Sorry, but I was unable to find the Freenet page you requested</h3>\n"
				"</body></html>\n"
				;
		send(sock, http_404, strlen(http_404), 0);
		printf("404: %s\n", freenet_key);
	}
	else
	{
		// Found the key
		char buf[1024];
		char *cont;
		int len;
		char *http_200 =
				"HTTP/1.0 200 OK\n"
				"Connection: close\n"
				;

		// Send back initial 200 response
		send(sock, http_200, strlen(http_200), 0);
		printf("200: %s\n", freenet_key);

		// Notify the content type
		if (hfcp->meta->data != NULL)
		{
			cont = hfcp->meta->data->body.contenttype->type;
			sprintf(buf, "Content-Type: %s\n\n", cont);
			send(sock, buf, strlen(buf), 0);
		}

		// Pass data from Freenet to browser
		while ((len = fcpReadKey(hfcp, buf, 1024)) > 0)
			send(sock, buf, len, 0);
	}

	// Clean up and get out
	free(host2);
	fcpDestroyHandle(hfcp);
	return 1;
}


//
// fwproxy_fproxy_req()
//
// http request handler that emulates Freenet FProxy :)
//

static int fwproxy_fproxy_req(char *host, char *headers, int sock)
{
	char docpath[1024];
	char freenet_key[512];
	HFCP *hfcp;
	int fd;
	char gateway_pathname[256];
	char *s;

	fwproxy_getpath(headers, docpath);

	// delete '?key='
	if (*docpath && !strncmp(docpath, "?key=", 5))
		strcpy(docpath, docpath+5);

	// delete '&submit
	if ((s = strchr(docpath, '&')) != NULL)
		*s = '\0';

	//
	// send 404 on requests for 'robots.txt' so as to allow spiders
	//
	if (!strcmp(docpath, "robots.txt"))
	{
		// Send back a 404
		char *http_404 =
				"HTTP/1.0 404 Page Not Found\n"
				"Connection: close\n"
				"Content-Type: text/html\n\n"
				"<html>"
				"<head>"
				"<title>FreeWeb - Page Not Found</title>\n"
				"</head><body>\n"
				"<h1>Page Not Found</h1>\n"
				"</body></html>\n"
				;
		send(sock, http_404, strlen(http_404), 0);
		return 1;
	}

	// if request is empty, send back the gateway page
	if (*docpath == '\0')
	{
		char buf[1024];
		int len;
		char *http_200 =
				"HTTP/1.0 200 OK\n"
				"Connection: close\n"
				"Content-Type: text/html\n\n"
				;

		// create a path of the gateway page
#ifdef WINDOWS
		sprintf(gateway_pathname, "%s\\gateway.html", fwproxy_dir);
#else
		sprintf(gateway_pathname, "%s/gateway.html", fwproxy_dir);

		printf("opening gateway page '%s'\n", gateway_pathname);
#endif
		if ((fd = open(gateway_pathname, 0)) < 0)
		{
			// Send back a 404
			char *http_404 =
					"HTTP/1.0 404 Page Not Found\n"
					"Connection: close\n"
					"Content-Type: text/html\n\n"
					"<html>"
					"<head>"
					"<title>FreeWeb - Page Not Found</title>\n"
					"</head><body>\n"
					"<h1>Page Not Found</h1>\n"
					"<h3>Sorry, but I was unable to find the Freenet Gateway page</h3>\n"
#ifdef WINDOWS
					"There should be a file called <b>gateway.html</b> in the program directory<br>\n"
#else
					"Please ensure that there's a file called <b>gateway.html</b> in <b>/usr/share/fwproxy</b>\n"
#endif
					"</body></html>\n"
					;
			send(sock, http_404, strlen(http_404), 0);
			return 1;
		}

		// Send gateway page back to browser
		send(sock, http_200, strlen(http_200), 0);
		while ((len = read(fd, buf, 1024)) > 0)
			send(sock, buf, len, 0);
		close(fd);
		return 1;
	}

	// Create a handle to Freenet FCP and set htl
	hfcp = fcpCreateHandle();
	fcpSetHtl(hfcp, fwproxy_htl);

	// Convert hostname to Freenet key
	sprintf(freenet_key, "freenet:%s", docpath);

	//
	// WARNING - FILTHY HACK !!!
	//
	// A certain web spider program converts the double slash in MSK keys to a single slash
	// so we have to detect this and convert it back :(
	//

	if ((s = strstr(freenet_key, "MSK@SSK@")) != NULL)
	{
		if ((s = strchr(s, '/')) != NULL)
		{
			if ((s = strchr(s+2, '/')) != NULL && s[1] != '/')
			{
				// found second slash after 'MSK@SSK@' - convert to '//'
				char temp[256];

				if (s[1])
				{
					// URL doesn't end with double-slash
					s++;
					strcpy(temp, s);
					*s++ = '/';
					strcpy(s, temp);
				}
				else
				{
					// URL ends with double-slash
					*++s = '/';
					*++s = '\0';
				}
			}
		}
	}
	// END OF FILTHY HACK

	// Open Freenet key
	if (fcpOpenKey(hfcp, freenet_key, _FCP_O_READ) < 0)
	{
		// Send back a 404
		char *http_404 =
				"HTTP/1.0 404 Page Not Found\n"
				"Connection: close\n"
				"Content-Type: text/html\n\n"
				"<html>"
				"<head>"
				"<title>FreeWeb - Page Not Found</title>\n"
				"</head><body>\n"
				"<h1>Page Not Found</h1>\n"
				"<h3>Sorry, but I was unable to find the Freenet page you requested</h3>\n"
				"</body></html>\n"
				;
		send(sock, http_404, strlen(http_404), 0);
		printf("404: %s\n", freenet_key);
	}
	else
	{
		// Found the key
		char buf[1024];
		char *cont;
		int len;
		char *http_200 =
				"HTTP/1.0 200 OK\n"
				"Connection: close\n"
				;

		printf("200: %s\n", freenet_key);

		// Send back initial 200 response
		send(sock, http_200, strlen(http_200), 0);

		// Notify the content type - it IS there, ISN'T IT!!!!!!
		if (hfcp->meta != NULL && hfcp->meta->data != NULL)
		{
			cont = hfcp->meta->data->body.contenttype->type;
			sprintf(buf, "Content-Type: %s\n\n", cont);
			send(sock, buf, strlen(buf), 0);
		}
		else
		{
			// no friggin metadata :(
			sprintf(buf, "Content-Type: text/html\n\n");
			send(sock, buf, strlen(buf), 0);
		}

		// Pass data from Freenet to browser
		while ((len = fcpReadKey(hfcp, buf, 1024)) > 0)
			send(sock, buf, len, 0);
	}

	// Clean up and get out
	fcpDestroyHandle(hfcp);
	return 1;

}		// 'fwproxy_fproxy_req()


//
// fwproxy_reply_webblock()
//
// This function gets called if browser attempts to retrieve anything from the mainstream web
// while web blocking is enabled
//
// Simply writes an error page back to browser
//

static int fwproxy_reply_webblock(sock)
{
	char *http_reply =
				"HTTP/1.0 403 Access Denied\n"
				"Connection: close\n"
				"Content-Type: text/html\n\n"
				"<html>"
				"<head>"
				"<title>FreeWeb Security Alert</title>\n"
				"</head>"
				"<body bgcolor=\"#000000\" text=#ffa0ff link=\"#66FF66\""
				"vlink=\"#66FF66\" alink=\"#66FF66\">\n"
				"<font face=\"Arial, Helvetica, sans-serif\">"
				"<h1>FreeWeb Security Alert!</h1>\n"
				"<font size=3>You're seeing this page because the\n"
				"<b>FreeWeb Anonymity Filter</b> is <b>on</b>\n"
				"and your browser attempted to retrieve\n"
				"a resource outside of Freenet.<br><br>\n"
				"While this is generally safe, there are certain individuals who are placing\n"
				"<i>web bugs</i> and other tracing mechanisms into Freenet pages, which can\n"
				"cause your Freenet browsing activities to be logged to a computer on\n"
				"the mainstream web - a total violation of your anonymity.<br><br>\n"
				"Unless you know a Freenet web page and all its links to be safe, the only way to\n"
				"guarantee your privacy is to leave this filter <b>on</b> while you are surfing\n"
				"within Freenet, and only turn if <b>off</b> when you are surfing the mainstream web.<br><br>\n"
				"To disable or re-enable this filter, double-click\n"
				"on the FreeWeb icon in your task tray, then click on 'Configure'<br><br>\n"
				"<b>A Final Warning</b> - for the anonymity filter to be effective, you'll\n"
				"need to regularly clear or at least prune your browser's cache</font>\n"
				"</body></html>\n";
	int reply_len = strlen(http_reply);

	send(sock, http_reply, reply_len, 0);
	return 1;
}


void fwproxy_sethtl(int htl)
{
	fwproxy_htl = htl;
}


static void fwproxy_getpath(char *headers, char *path)
{
	char pathtmp[1024];
	char *docpath;
	char *s;

	// grab first line of header
	strncpy(pathtmp, headers, 1023);
	pathtmp[1023] = '\0';		// just in case

	// point to path in header
	*strchr(strchr(pathtmp, ' ') + 1, ' ') = '\0';	// chop after pathname
	docpath = strchr(pathtmp, ' ') + 1;
	if (!strncmp(docpath, "http://", 7))
	{
		docpath += 7;
		docpath = strchr(docpath, '/');
	}

	// skip leading slash in path
	docpath++;

	// copy to caller's desired location
	if (*docpath == '\0')
	{
		*path = '\0';
		return;
	}

	// now the tricky bit - need to copy the path, and perform URL decoding
	s = path;
	while (*docpath)
	{
		if (*docpath == '%')
		{
			*s++ = hexdig_to_int(docpath[1]) * 16 + hexdig_to_int(docpath[2]);
			docpath += 3;
		}
		else if (*docpath == '+')
			*s++ = ' ';
		else
			*s++ = *docpath++;
	}
	*s = '\0';

}		// 'fwproxy_getpath()'


static int hexdig_to_int(char dig)
{
	if (dig >= '0' && dig <= '9')
		return dig - '0';
	else if (dig >= 'A' && dig <= 'F')
		return dig - 'A' + 10;
	else if (dig >= 'a' && dig <= 'f')
		return dig - 'a' + 10;
	else
		return 0;
}
