/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <string.h>

#include "protocol.h"
#include "client.h"
#include "twofish.h"

int e2e_twofish_init (freenet_transfer_state *state, unsigned char *key)
{
  int status;
  unsigned char ov[FN_KEY_BYTES];

  state->tsys.fbpos = 0;

  status = twofish_setkey(&(state->tsys.ctx), key, FN_KEY_BYTES);
  if (status != 0) {
    return FNS_MAKEKEY_FAILED;
  }

  memset(state->tsys.fb, 0, FN_KEY_BYTES);

  twofish_encrypt(&(state->tsys.ctx), ov, state->tsys.fb);
  memcpy(state->tsys.fb, ov, FN_KEY_BYTES);

  return FNS_SUCCESS;
}


int e2e_twofish_encrypt (freenet_transfer_state *state,
                         unsigned char *inbuffer,
                         unsigned char *outbuffer, int len)
{
  int i;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (state->tsys.fbpos >= FN_KEY_BYTES) {

      twofish_encrypt(&(state->tsys.ctx), ov, state->tsys.fb);
      memcpy(state->tsys.fb, ov, FN_KEY_BYTES);

      state->tsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ state->tsys.fb[state->tsys.fbpos];
    state->tsys.fb[state->tsys.fbpos] = outbuffer[i];

    state->tsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int e2e_twofish_decrypt (freenet_transfer_state *state,
                         unsigned char *inbuffer,
                         unsigned char *outbuffer, int len)
{
  int i;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (state->tsys.fbpos >= FN_KEY_BYTES) {

      twofish_encrypt(&(state->tsys.ctx), ov, state->tsys.fb);
      memcpy(state->tsys.fb, ov, FN_KEY_BYTES);

      state->tsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ state->tsys.fb[state->tsys.fbpos];
    state->tsys.fb[state->tsys.fbpos] = inbuffer[i];

    state->tsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int rijndael_init (freenet_transfer_state *state, unsigned char *key)
{
  int status;
  unsigned char ov[FN_KEY_BYTES];

  status = rijndael_makeKey(&(state->rsys.key), DIR_ENCRYPT, FN_KEY_BITS, key);
  if (status != TRUE) {
    return FNS_MAKEKEY_FAILED;
  }

  status = rijndael_cipherInit(&(state->rsys.cipher), MODE_ECB, NULL);
  if (status != TRUE) {
    return FNS_CIPHERINIT_FAILED;
  }

  memset(state->rsys.fb, 0, FN_KEY_BYTES);

  status = rijndael_blockEncrypt(&(state->rsys.cipher), &(state->rsys.key),
                                 state->rsys.fb, FN_KEY_BITS, ov);
  if (status < 1) {
    return FNS_ENCRYPT_FAILED;
  }
  memcpy(state->rsys.fb, ov, FN_KEY_BYTES);

  return FNS_SUCCESS;
}


int rijndael_encrypt (freenet_transfer_state *state, unsigned char *inbuffer,
                      unsigned char *outbuffer, int len)
{
  int i, status;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (state->rsys.fbpos >= FN_KEY_BYTES) {

      status = rijndael_blockEncrypt(&(state->rsys.cipher), &(state->rsys.key),
                                     state->rsys.fb, FN_KEY_BITS, ov);
      if (status < 1) {
        return FNS_ENCRYPT_FAILED;
      }
      memcpy(state->rsys.fb, ov, FN_KEY_BYTES);
      state->rsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ state->rsys.fb[state->rsys.fbpos];
    state->rsys.fb[state->rsys.fbpos] = outbuffer[i];

    state->rsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int rijndael_decrypt (freenet_transfer_state *state, unsigned char *inbuffer,
                      unsigned char *outbuffer, int len)
{
  int i, status;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (state->rsys.fbpos >= FN_KEY_BYTES) {

      status = rijndael_blockEncrypt(&(state->rsys.cipher), &(state->rsys.key),
                                     state->rsys.fb, FN_KEY_BITS, ov);
      if (status < 1) {
        return FNS_ENCRYPT_FAILED;
      }
      memcpy(state->rsys.fb, ov, FN_KEY_BYTES);
      state->rsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ state->rsys.fb[state->rsys.fbpos];
    state->rsys.fb[state->rsys.fbpos] = inbuffer[i];

    state->rsys.fbpos++;

  }

  return FNS_SUCCESS;
}


__inline__ int endtoend_init (freenet_transfer_state *state,
                              unsigned char *key)
{
  if (state->endtoend == FN_TWOFISH) {
    return e2e_twofish_init(state, key);
  } else if (state->endtoend == FN_RIJNDAEL) {
    return rijndael_init(state, key);
  } else {
    return FNS_INVALID_ENDTOEND;
  }
}


__inline__ int endtoend_encrypt (freenet_transfer_state *state,
                                 unsigned char *inbuffer,
                                 unsigned char *outbuffer, int len)
{
  if (state->endtoend == FN_TWOFISH) {
    return e2e_twofish_encrypt(state, inbuffer, outbuffer, len);
  } else if (state->endtoend == FN_RIJNDAEL) {
    return rijndael_encrypt(state, inbuffer, outbuffer, len);
  } else {
    return FNS_INVALID_E2E;
  }
}


__inline__ int endtoend_decrypt (freenet_transfer_state *state,
                                 unsigned char *inbuffer,
                                 unsigned char *outbuffer, int len)
{
  if (state->endtoend == FN_TWOFISH) {
    return e2e_twofish_decrypt(state, inbuffer, outbuffer, len);
  } else if (state->endtoend == FN_RIJNDAEL) {
    return rijndael_decrypt(state, inbuffer, outbuffer, len);
  } else {
    return FNS_INVALID_E2E;
  }
}

