/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <values.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "protocol.h"
#include "client.h"
#include "uri.h"
#include "base64.h"

#define REQUEST 1
#define INSERT 2

extern char *optarg;
extern int optind;

int main (int argc, char **argv)
{
  int c;
  int i;
  int cmd = 0;
  int errflg = 0;
  char host[128] = "localhost";
  char port[32] = "19114";
  char file[PATH_MAX];
  int htl = 5;
  unsigned char uri[512] = "";
  unsigned char keystring[512];
  int status;
  freenet_transfer_state state;
  unsigned char in;
  int zoop;
  unsigned char out;
  FILE *instream;
  unsigned char buffer[66666];
  int usebuffer = 0;
  freenet_key key;

  while ((c = getopt(argc, argv, "brif:a:p:h:")) != EOF) {
    switch (c) {
    case 'r':
      if (cmd == 0) {
        cmd = REQUEST;
      } else {
        errflg++;
      }
      break;
    case 'i':
      if (cmd == 0) {
        cmd = INSERT;
      } else {
        errflg++;
      }
      break;
    case 'b':
      usebuffer = 1;
      break;
    case 'f':
      strncpy(file, optarg, PATH_MAX);
      break;
    case 'a':
      strncpy(host, optarg, 128);
      break;
    case 'p':
      strncpy(port, optarg, 32);
      break;
    case 'h':
      sscanf(optarg, "%d", &htl);
      break;
    case '?':
      errflg++;
    }
  }

  if (cmd == 0) {
    errflg++;
  }

  if (optind < argc) {
    strcpy(uri, argv[optind]);

    if (cmd == INSERT) {
      status = freenet_parse_insert_uri(&key, uri);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "could not parse key: %d\n", status);
        return -1;
      }
    } else {
      status = freenet_parse_request_uri(&key, uri);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "could not parse key: %d\n", status);
        return -1;
      }
    }
    optind++;
  } else {
    errflg++;
  }

  if (errflg) {
    fprintf(stderr,
            "usage: client [-r|-i] [-a <host address>] [-p <port>]\n");
    fprintf(stderr,
            "[-h <hopstolive>] key\n");
    exit (2);
  }


  status = freenet_init();
  if (status != FNS_SUCCESS) {
    fprintf(stderr, "Freenet initialization failed\n");
    return -1;
  }

  srand(time(NULL));

  if (cmd == REQUEST) {
    if (usebuffer == 0) {
      status = freenet_request_stream(&state, host, port, htl, &key);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "request failed: %d\n", status);
        return -1;
      }

      i=0;
      while (state.pos < state.size) {

        status = freenet_read_stream(&state, &in, 1);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "stream read error at byte %d of %d: %d\n",
                  state.pos, state.size, status);
          return -1;
        }

        if (i >= state.metadata_len) {
          printf("%c", in);
        }
        i++;
      }
    } else {
      status = freenet_request_buffer(&state, host, port, htl, &key,
                                      buffer, 66666);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "request failed: %d\n", status);
        return -1;
      }

      for (i = state.metadata_len; i < state.datasize; i++) {
        printf("%c", buffer[i]);
      }

    }

  } else if (cmd == INSERT) {
    if (usebuffer == 0) {
      instream = fopen(file, "r");
      if (instream == NULL) {
        printf("couldn't open file %s:\n", file);
        perror("fopen");
        return -1;
      }

      if (key.type == FN_CHK_TYPE) {
        status = freenet_insert_CHK_stream(&state, host, port, htl, FN_TWOFISH,
                                           0, &key, instream);
      } else {
        status = freenet_insert_SVK_stream(&state, host, port, htl,
                                           FN_TWOFISH, 0, &key, instream);
      }
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "insert failed: %d\n", status);
        return -1;
      }

      do {
        zoop = fgetc(instream);
        if (zoop == EOF) {
          printf("reached EOF\n");
        } else {
          out = (unsigned char)zoop;
          status = freenet_write_stream(&state, &out, 1);
          if (status != FNS_SUCCESS) {
            fprintf(stderr, "stream write error: %d\n", status);
            return -1;
          }
        }
      } while (zoop != EOF);
    } else {
      instream = fopen(file, "r");
      if (instream == NULL) {
        printf("couldn't open file %s:\n", file);
        perror("fopen");
        return -1;
      }

      i = 0;
      do {
        zoop = fgetc(instream);
        if (zoop == EOF) {
          printf("reached EOF\n");
        } else {
          buffer[i] = zoop;
          i++;
        }
      } while (zoop != EOF);


      if (key.type == FN_CHK_TYPE) {
        status = freenet_insert_CHK_buffer(&state, host, port, htl, FN_TWOFISH,
                                           0, &key, buffer, i);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "insert failed: %d\n", status);
          return -1;
        }
      } else {
        status = freenet_insert_SVK_buffer(&state, host, port, htl,
                                           FN_TWOFISH, 0, &key, buffer, i);
      }
    }

    if (key.type == FN_SVK_TYPE) {
      base64_encode(key.priv, keystring, FN_HASH_BYTES, 0);
      printf("private key: %s\n", keystring);
    }

    if (key.type != FN_KSK_TYPE) {
      freenet_get_uri(&key, uri);
    }
    printf("key:\n%s\n", uri);

  }

  return 1;
}
