/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <openssl/sha.h>

#include "protocol.h"
#include "util.h"

/* FIXME -- these functions aren't useful when they return an error
   (even EOF!), becuase it's impossible to tell what portion of the
   data was read or written.  Fix that (by returning the amount read
   in a parameter).  Right now, they can be used properly by only
   calling them with len > 1 if EOF is considered a real error (that
   is, if you are expecting an EOF at any time, read one byte at a
   time) */

int readall(int sock, unsigned char *buffer, int len)
{
  int br;
  int rlen = 0;

  while (rlen < len) {
    br = read(sock, &(buffer[rlen]), len-rlen);
    if ((br == 0) && (rlen == 0)) {
      return FNS_EOF;
    } else if (br < 1) {
      return FNS_READ_FAILED;
    }
    rlen += br;
  }

  return FNS_SUCCESS;
}


int writeall(int sock, unsigned char *buffer, int len)
{
  int bw;
  int wlen = 0;

  while (wlen < len) {
    bw = write(sock, &(buffer[wlen]), len-wlen);
    if (bw < 1) {
      return FNS_WRITE_FAILED;
    }
    wlen += bw;
  }

  return FNS_SUCCESS;
}


/* given some input data, generate a key in the usual freenet manner */
int keygen (const unsigned char *k, int klen, unsigned char *key,
            int keybytes)
{
  SHA_CTX td;
  unsigned char hash[FN_HASH_BYTES];

  if (keybytes > FN_HASH_BYTES) {
    return FNS_HASH_TOO_SMALL;  /* FIXME -- actually, if I implemented all of
                                   this algorithm, I wouldn't have this
                                   problem */
  }

  SHA1_Init(&td);
  SHA1_Update(&td, "\0", 1);
  SHA1_Update(&td, k, klen);
  SHA1_Final(hash, &td);

  memcpy(key, hash, keybytes);

  return FNS_SUCCESS;
}


int generate_random(unsigned char *buffer, int len)
{
#ifdef FN_RANDOM_FILE
  int status;
  int source;
  source = open(FN_RANDOM_FILE, O_RDONLY);
  if (source == -1) {
    return FNS_OPEN_FAILED;
  }
  status = readall(source, buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  close(source);

  return FNS_SUCCESS;
#else
  int status;

  status = RAND_bytes(buffer, len);
  if (status != 1) {
    return FNS_RANDOM_ERROR;
  }

  return FNS_SUCCESS;
#endif
}


int raw_to_bagbiting_freenet_mpi(unsigned char *x, int xlen,
                                 unsigned char *bfmpi)
{
  int16_t mpilen;
  int i, j;

  /* mpilen is the length in bits */
  mpilen = xlen * 8;

  /* Freenet's MPI format is brain-damaged, so we have to give the
     length in bits starting with the most signficant non-zero bit. */
  for (i = 0; ((i < xlen) && (x[i] == 0)); i++) {
    mpilen-=8;
  }

  if (x[i] >= 0x80) {
    j = 0;
  } else if (x[i] >= 0x40) {
    j = 1;
  } else if (x[i] >= 0x20) {
    j = 2;
  } else if (x[i] >= 0x10) {
    j = 3;
  } else if (x[i] >= 0x08) {
    j = 4;
  } else if (x[i] >= 0x04) {
    j = 5;
  } else if (x[i] >= 0x02) {
    j = 6;
  } else if (x[i] >= 0x01) {
    j = 7;
  } else {
    j = 8;
  }
  mpilen -= j;

  mpilen = htons(mpilen);
  *(int16_t *)bfmpi = mpilen;

  /* append the buffer to bfmpi */
  memcpy(&(bfmpi[2]), x, xlen);

  return FNS_SUCCESS;
}


