/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <malloc.h>
#include <string.h>
#include <openssl/dsa.h>

#include "protocol.h"
#include "util.h"
#include "client.h"
#include "endtoend.h"
#include "base64.h"

#define FN_MAX_RESTARTS 3

#define FN_HEADER_SIZE (FN_KEY_BYTES + 2)
#define FN_TRAILER_LEN  (FN_HASH_BYTES + 1)
#define FN_PARTSIZE 0x4000

int generate_DSA (unsigned char *searchkey,
                  unsigned char *pub, unsigned char *priv,
                  const unsigned char *docname,
                  freenet_group *group);

int generate_CHK_enckey_from_buffer (unsigned char *enckey,
                                     unsigned char *buffer, int buflen);

int generate_CHK_enckey_from_stream (unsigned char *enckey,
                                     FILE *instream);

int generate_hashes_from_buffer (freenet_transfer_state *state,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 unsigned char *buffer, int len,
                                 int *num_parts);

int generate_hashes_from_stream (freenet_transfer_state *state,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 int len, FILE *instream, int *num_parts);

int sign (freenet_group *group, unsigned char *priv, unsigned char *pub,
          unsigned char *buf, int len, unsigned char *sigstr);

int verify (freenet_group *group, unsigned char *pub,
            unsigned char *buf, int len, unsigned char *sigstr);

int stream_read (freenet_transfer_state *state,
                 unsigned char *buffer, int len);

int stream_readdata (freenet_transfer_state *state,
                     unsigned char *buffer, int len);

int stream_read_trailer (freenet_transfer_state *state);

int stream_read_control(freenet_transfer_state *state);

int stream_write(freenet_transfer_state *state,
                 unsigned char *buffer, int len);

int stream_writedata(freenet_transfer_state *state,
                     unsigned char *buffer, int len);

int stream_write_trailer(freenet_transfer_state *state);

int stream_storedata (freenet_transfer_state *state);

int request_insert (freenet_transfer_state *state,
                    unsigned char *searchkey, int htl);



/* sets searchkey and pub given priv, docname, and group */
int generate_DSA (unsigned char *searchkey,
                  unsigned char *pub, unsigned char *priv,
                  const unsigned char *docname,
                  freenet_group *group)
{
  int status;
  BIGNUM *x = NULL;
  BIGNUM *g;
  BIGNUM *p;
  BIGNUM *y;
  BN_CTX *ctx = NULL;
  unsigned char bfmpipub[FN_DHK_MPI_LEN];
  unsigned char hashes[FN_HASH_BYTES * 2];
  int pub_len;
  int retval;

  g = BN_new();
  p = BN_new();
  y = BN_new();

  status = BN_hex2bn(&g, group->g);
  if (status != strlen(group->g)) {
    retval = FNS_BN_FAILED;
    goto exit;
  }

  status = BN_hex2bn(&p, group->p);
  if (status != strlen(group->p)) {
    retval = FNS_BN_FAILED;
    goto exit;
  }

  x = BN_bin2bn(priv, FN_HASH_BYTES, NULL);
  if (x == NULL) {
    retval = FNS_BN_FAILED;
    goto exit;
  }

  ctx = BN_CTX_new();

  BN_mod_exp(y, g, x, p, ctx);

  pub_len = BN_num_bytes(y);
  if (pub_len != FN_DHK_LEN) {
    return FNS_INVALID_DHK;
  }

  status = BN_bn2bin(y, pub);
  if (status != pub_len) {
    retval = FNS_BN_FAILED;
    goto exit;
  }

  status = raw_to_bagbiting_freenet_mpi(pub, pub_len, bfmpipub);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto exit;
  }

  if (strlen(docname) == 0) {
    SHA1(bfmpipub, pub_len + 2, searchkey);
  } else {
    SHA1(bfmpipub, pub_len + 2, hashes);
    SHA1(docname, strlen(docname), &(hashes[FN_HASH_BYTES]));
    SHA1(hashes, FN_HASH_BYTES * 2, searchkey);
  }

  retval = FNS_SUCCESS;

 exit:
  BN_free(g);
  BN_free(p);
  BN_free(y);
  BN_free(x);
  BN_CTX_free(ctx);

  return retval;
}


/* sets the key given the keystr */
int freenet_generate_KSK (const unsigned char *keystr, freenet_key *key)
{
  int keylen;
  int status;

  keylen = strlen(keystr);

  SHA1(keystr, keylen, key->priv);

  strcpy(key->group.p, FN_GROUP_A_P);
  strcpy(key->group.q, FN_GROUP_A_Q);
  strcpy(key->group.g, FN_GROUP_A_G);

  status = generate_DSA(key->searchkey, key->pub, key->priv, "",
                        &(key->group));
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->searchkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_KSK_TYPE);

  status = keygen(keystr, keylen, key->enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->type = FN_KSK_TYPE;
  key->docname[0] = 0;

  return FNS_SUCCESS;

}


/* sets the key given priv and docname */
int freenet_generate_SVK (unsigned char *priv, const unsigned char *docname,
                          freenet_key *key)
{
  int status;

  strcpy(key->group.p, FN_GROUP_B_P);
  strcpy(key->group.q, FN_GROUP_B_Q);
  strcpy(key->group.g, FN_GROUP_B_G);

  status = generate_DSA(key->searchkey, key->pub, priv, docname,
                        &(key->group));
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->searchkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_SVK_TYPE);

  if (strlen(docname) > 0) {
    status = keygen(docname, strlen(docname), key->enckey, FN_KEY_BYTES);
    if (status != FNS_SUCCESS) {
      return status;
    }
  } else {
    status = generate_random(key->enckey, FN_KEY_BYTES);
    if (status != FNS_SUCCESS) {
      return status;
    }
  }

  key->type = FN_SVK_TYPE;
  strncpy(key->docname, docname, FN_MAX_DOCNAME);
  memcpy(key->priv, priv, FN_HASH_BYTES);

  return FNS_SUCCESS;
}


/* The generate_CHK_enckey_ functions hash the data and send it to
   keygen() (which will perform a second hash).  This first hash is
   unnecessary, and we're only doing it because Oskar Sandberg was too
   lazy to do this correctly in the reference implementation.  Calls
   to these functions should really be calls to keygen(). */
int generate_CHK_enckey_from_buffer (unsigned char *enckey,
                                     unsigned char *buffer, int buflen)
{
  int status;
  unsigned char hash[FN_HASH_BYTES];

  SHA1(buffer, buflen, hash);

  status = keygen(hash, FN_HASH_BYTES, enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int generate_CHK_enckey_from_stream (unsigned char *enckey,
                                     FILE *instream)
{
  int status;
  int c;
  unsigned char byte;
  SHA_CTX td;
  unsigned char hash[FN_HASH_BYTES];

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  SHA1_Init(&td);

  c=0;
  while (c!=EOF) {
    c = fgetc(instream);
    if (c!=EOF) {
      byte = (unsigned char)c;
      SHA1_Update(&td, &byte, 1);
    }
  }

  SHA1_Final(hash, &td);

  status = keygen(hash, FN_HASH_BYTES, enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}

/* returns num_parts and sets hashes in state */
int generate_hashes_from_buffer (freenet_transfer_state *state,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 unsigned char *buffer, int len,
                                 int *num_parts)
{
  int c, i, current_part, distance, status, content_len;
  unsigned char enc_header[FN_HEADER_SIZE];
  unsigned char byte, enc_byte;
  unsigned char dnhash[FN_HASH_BYTES];
  SHA_CTX hash_handle[FN_MAX_PARTS];

  status = endtoend_init(state, enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = endtoend_encrypt(state, header, enc_header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  current_part = 0;
  SHA1_Init(&hash_handle[current_part]);

  if (strlen(docname) > 0) {
    SHA1(docname, strlen(docname), dnhash);
    SHA1_Update(&hash_handle[current_part], dnhash, FN_HASH_BYTES);
  }

  SHA1_Update(&hash_handle[current_part], enc_header, FN_HEADER_SIZE);
  distance =  FN_HEADER_SIZE;

  content_len = len + FN_HEADER_SIZE;

  c=0;
  for (i=0; i<len; i++) {
    c = buffer[i];

    if ((content_len > state->partsize) &&
        (distance == (state->partsize - FN_HASH_BYTES))) {
      distance = 0;
      current_part++;
      SHA1_Init(&hash_handle[current_part]);
    }

    byte = (unsigned char)c;

    status = endtoend_encrypt(state, &byte, &enc_byte, 1);
    if (status != FNS_SUCCESS) {
      return status;
    }

    SHA1_Update(&hash_handle[current_part], &enc_byte, 1);
    distance++;
  }

  for (i = current_part; i > 0; i--) {
    SHA1_Final(state->hash[i], &hash_handle[i]);
    SHA1_Update(&hash_handle[i-1], state->hash[i], FN_HASH_BYTES);
  }
  SHA1_Final(state->hash[0], &hash_handle[0]);

  *num_parts = current_part;

  return FNS_SUCCESS;
}


/* returns num_parts and sets hashes in state */
int generate_hashes_from_stream (freenet_transfer_state *state,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 int len, FILE *instream, int *num_parts)
{
  int c, i, current_part, distance, status, content_len;
  unsigned char enc_header[FN_HEADER_SIZE];
  unsigned char byte, enc_byte;
  unsigned char dnhash[FN_HASH_BYTES];
  SHA_CTX hash_handle[FN_MAX_PARTS];

  status = endtoend_init(state, enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = endtoend_encrypt(state, header, enc_header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  current_part = 0;
  SHA1_Init(&hash_handle[current_part]);

  if (strlen(docname) > 0) {
    SHA1(docname, strlen(docname), dnhash);
    SHA1_Update(&hash_handle[current_part], dnhash, FN_HASH_BYTES);
  }

  SHA1_Update(&hash_handle[current_part], enc_header, FN_HEADER_SIZE);
  distance =  FN_HEADER_SIZE;

  content_len = len + FN_HEADER_SIZE;

  c=0;
  for (i=0; c!=EOF; i++) {
    c = fgetc(instream);

    if (c!=EOF) {


      if ((content_len > state->partsize) &&
          (distance == (state->partsize - FN_HASH_BYTES))) {
        distance = 0;
        current_part++;
        SHA1_Init(&hash_handle[current_part]);
      }

      byte = (unsigned char)c;

      status = endtoend_encrypt(state, &byte, &enc_byte, 1);
      if (status != FNS_SUCCESS) {
        return status;
      }

      SHA1_Update(&hash_handle[current_part], &enc_byte, 1);
      distance++;

    }

  }

  for (i = current_part; i > 0; i--) {
    SHA1_Final(state->hash[i], &hash_handle[i]);
    SHA1_Update(&hash_handle[i-1], state->hash[i], FN_HASH_BYTES);
  }
  SHA1_Final(state->hash[0], &hash_handle[0]);

  *num_parts = current_part;

  return FNS_SUCCESS;
}


/* Generate a DSA signature given key info.  This should probably take
   a freenet_key */
int sign (freenet_group *group, unsigned char *priv, unsigned char *pub,
          unsigned char *buf, int len, unsigned char *sigstr)
{
  int status, retval, rlen, slen, i, j;
  unsigned int siglen;
  DSA *dsa;
  DSA_SIG *sig = NULL;
  unsigned char *sigptr;
  unsigned char sigasn[FN_DHK_LEN];
  unsigned char sigrbuf[FN_DHK_LEN];
  unsigned char sigsbuf[FN_DHK_LEN];

  dsa = DSA_new();

  status = BN_hex2bn(&(dsa->p), group->p);
  if (status != strlen(group->p)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&(dsa->q), group->q);
  if (status != strlen(group->q)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&(dsa->g), group->g);
  if (status != strlen(group->g)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  dsa->priv_key = BN_bin2bn(priv, FN_HASH_BYTES, NULL);
  if (dsa->priv_key == NULL) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  dsa->pub_key = BN_bin2bn(pub, FN_DHK_LEN, NULL);
  if (dsa->pub_key == NULL) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = DSA_sign(0, buf, len, sigasn, &siglen, dsa);
  if (status != 1) {
    retval = FNS_DSA_FAILED;
    goto end;
  }

  sigptr = sigasn;

  sig = d2i_DSA_SIG(NULL, &sigptr, siglen);
  if (sig == NULL) {
    retval = FNS_DSA_FAILED;
    goto end;
  }

  rlen = BN_num_bytes(sig->r);
  slen = BN_num_bytes(sig->s);

  status = BN_bn2bin(sig->r, sigrbuf);
  if (status != rlen) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_bn2bin(sig->s, sigsbuf);
  if (status != slen) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  for (i = 0; i < rlen; i++) {
    sprintf(&(sigstr[i*2]), "%.2x", sigrbuf[i]);
  }
  sigstr[(i*2)] = ',';

  for (j = 0; j < slen; j++) {
    sprintf(&(sigstr[(i*2) + 1 + (j*2)]), "%.2x", sigsbuf[j]);
  }
  sigstr[(i*2) + 1 + (j*2)] = 0;

  retval = FNS_SUCCESS;

 end:
  DSA_free(dsa);
  DSA_SIG_free(sig);
  return retval;

}


/* Verify a DSA signature given key info.  This should probably take a
   freenet_key */
int verify (freenet_group *group, unsigned char *pub,
            unsigned char *buf, int len, unsigned char *sigstr)
{
  int status, retval, rlen, slen;
  int siglen;
  DSA *dsa;
  DSA_SIG *sig;
  unsigned char *comma;
  unsigned char *sigptr;
  unsigned char sigasn[FN_DHK_LEN];
  unsigned char sigvalstr[(FN_DHK_LEN * 2) + 1];

  dsa = DSA_new();
  sig = DSA_SIG_new();

  status = BN_hex2bn(&(dsa->p), group->p);
  if (status != strlen(group->p)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&(dsa->q), group->q);
  if (status != strlen(group->q)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&(dsa->g), group->g);
  if (status != strlen(group->g)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  dsa->pub_key = BN_bin2bn(pub, FN_DHK_LEN, NULL);
  if (dsa->pub_key == NULL) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  comma = (unsigned char *)strchr(sigstr, ',');
  if (comma == NULL) {
    retval = FNS_INVALID_SIG;
    goto end;
  }

  rlen = (strlen(sigstr) - strlen(comma));
  slen = (strlen(comma) - 1);

  if ((rlen > (FN_DHK_LEN*2)) ||
      (slen > (FN_DHK_LEN*2))) {
    retval = FNS_INVALID_SIG;
    goto end;
  }

  strncpy(sigvalstr, sigstr, rlen);
  sigvalstr[rlen] = 0;

  status = BN_hex2bn(&(sig->r), sigvalstr);
  if (status != strlen(sigvalstr)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  strncpy(sigvalstr, &(comma[1]), slen);
  sigvalstr[slen] = 0;

  status = BN_hex2bn(&(sig->s), sigvalstr);
  if (status != strlen(sigvalstr)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  siglen = i2d_DSA_SIG(sig, NULL);

  if (siglen > FN_DHK_LEN) {
    retval = FNS_DSA_FAILED;
    goto end;
  }

  sigptr = sigasn;

  i2d_DSA_SIG(sig, &sigptr);

  status = DSA_verify(0, buf, len, sigasn, siglen, dsa);
  if (status == 0) {
    retval = FNS_INVALID_SIG;
    goto end;
  } else if (status == -1) {
    retval = FNS_DSA_FAILED;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  DSA_free(dsa);
  DSA_SIG_free(sig);
  return retval;

}


int stream_read(freenet_transfer_state *state,
                unsigned char *buffer, int len)
{
  int status;

  status = freenet_readdata(&(state->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->pos += len;

  return FNS_SUCCESS;
}


int stream_readdata(freenet_transfer_state *state,
                    unsigned char *buffer, int len)
{
  int status;
  int retval;
  unsigned char *data;

  if (state->pos + len > state->size) {
    return FNS_OVERREAD;
  }

  data = malloc(len);
  if (data == NULL) {
    return FNS_MALLOC_FAILED;
  }

  status = freenet_readdata(&(state->connection), data, len);
  if (status == FNS_SUCCESS) {

    SHA1_Update(&(state->hash_handle), data, len);

    status = endtoend_decrypt(state, data, buffer, len);
    if (status != FNS_SUCCESS) {
      return status;
    }

    state->pos += len;
    state->contentpos += len;

    retval = FNS_SUCCESS;

  } else {
    retval = status;
  }

  free(data);

  return retval;
}


int stream_read_trailer(freenet_transfer_state *state)
{
  int status;

  status = stream_read(state, state->hash[state->current_part + 1],
                       FN_HASH_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  SHA1_Update(&(state->hash_handle), state->hash[state->current_part + 1],
              FN_HASH_BYTES);

  return FNS_SUCCESS;
}


int stream_read_control(freenet_transfer_state *state)
{
  int status;
  unsigned char ctrl;

  status = stream_read(state, &ctrl, 1);
  if (status != FNS_SUCCESS) {
    return status;
  }
  if (ctrl != FN_CB_OK) {
    return FNS_BAD_CTRL;
  }

  return FNS_SUCCESS;
}


int stream_write(freenet_transfer_state *state,
                 unsigned char *buffer, int len)
{
  int status;

  status = freenet_senddata(&(state->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->pos += len;

  return FNS_SUCCESS;
}


int stream_writedata(freenet_transfer_state *state,
                     unsigned char *buffer, int len)
{
  int status;
  int retval;
  unsigned char *encdata = NULL;

  if (state->pos + len > state->size) {
     retval = FNS_OVERWRITE;
     goto end;
  }

  encdata = malloc(len);
  if (encdata == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = endtoend_encrypt(state, buffer, encdata, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }


  status = freenet_senddata(&(state->connection), encdata, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  state->pos += len;
  state->contentpos += len;

  retval = FNS_SUCCESS;

 end:
  free(encdata);
  return retval;
}


int stream_write_trailer(freenet_transfer_state *state)
{
  int status;

  status = stream_write(state, state->hash[state->current_part + 1],
                        FN_HASH_BYTES);
  return status;
}


int stream_storedata (freenet_transfer_state *state)
{
  int status;
  freenet_message request;

  request.type = FN_STOREDATA_MSG;
  request.uniqueid = state->uniqueid;
  strcpy(request.field[0], "Depth=0");
  sprintf(request.field[1], "HopsToLive=%x", state->htl);
  request.numfields=2;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


/* read len bytes from a freenet stream (given by "state") into buffer */
int freenet_read_stream(freenet_transfer_state *state,
                        unsigned char *buffer, int len)
{
  int status;
  int distance;
  int partdata;
  int bufpos = 0;
  unsigned char temp_hash[FN_HASH_BYTES];
  freenet_message message;
  int i;

  if (state->pos + len > state->size) {
    return FNS_OVERREAD;
  }

  partdata = state->partsize - FN_HASH_BYTES;

  distance = partdata - (state->contentpos % partdata);

  while ((len >= distance) &&
         (state->current_part != state->last_part)) {

    if (distance > 0) {
      status = stream_readdata(state, &buffer[bufpos], distance);
      if (status != FNS_SUCCESS) {
        return status;
      }

      bufpos += distance;
      len -= distance;
    }

    status = stream_read_trailer(state);
    if (status != FNS_SUCCESS) {
      return status;
    }

    SHA1_Final(temp_hash, &(state->hash_handle));
    if (memcmp(state->hash[state->current_part], temp_hash, FN_HASH_BYTES)
        != 0) {
      return FNS_BAD_HASH;
    }

    SHA1_Init(&(state->hash_handle));

    status = stream_read_control(state);
    if (status != FNS_SUCCESS) {
      return status;
    }

    state->current_part++;

    distance = partdata - (state->contentpos % partdata);

  }

  if (len > 0) {
    status = stream_readdata(state, &buffer[bufpos], len);
    if (status != FNS_SUCCESS) {
      return status;
    }
    if (state->pos == state->size - 1) {
      status = stream_read_control(state);
      if (status != FNS_SUCCESS) {
        return status;
      }

      SHA1_Final(temp_hash, &(state->hash_handle));
      if (state->keytype == FN_CHK_TYPE) {
        if (memcmp(state->hash[state->current_part], temp_hash,
                   FN_HASH_BYTES)
            != 0) {
          return FNS_BAD_HASH;
        }
      } else {
        status = verify(&(state->group), state->pub, temp_hash, FN_HASH_BYTES,
                        state->sigstr);
        if (status != FNS_SUCCESS) {
          return status;
        }
      }

      /* get the storedata message, and make sure it looks okay */
      status = freenet_recvmsg(&(state->connection), &message);
      if (status != FNS_SUCCESS) {
        return status;
      }

      if (message.uniqueid != state->uniqueid) {
        return FNS_WRONG_ID;
      }

      if (message.type != FN_STOREDATA_MSG) {
        return FNS_UNEXPECTED_MESSAGE;
      }

      for (i = 0; i < message.numfields; i++) {
        if (strncmp(message.field[i], "DataSource=", 11) == 0) {
          strncpy(state->datasource, &(message.field[i][11]),
                  strlen(&(message.field[i][11])));
        }
      }

    }
  }

  return FNS_SUCCESS;
}


/* write len bytes of buffer to a freenet stream (given by "state") */
int freenet_write_stream(freenet_transfer_state *state,
                         unsigned char *buffer, int len)
{
  int status;
  int distance;
  int partdata;
  unsigned char ctrl;
  int bufpos = 0;

  if (state->pos + len > state->size) {
    return FNS_OVERWRITE;
  }

  partdata = state->partsize - FN_HASH_BYTES;

  distance = partdata - (state->contentpos % partdata);

  while ((len >= distance) &&
         (state->current_part != state->last_part)) {

    if (distance > 0) {
      status = stream_writedata(state, buffer, distance);
      if (status != FNS_SUCCESS) {
        return status;
      }

      bufpos += distance;
      len -= distance;
    }

    status = stream_write_trailer(state);
    if (status != FNS_SUCCESS) {
      return status;
    }

    ctrl = FN_CB_OK;
    status = stream_write(state, &ctrl, 1);
    if (status != FNS_SUCCESS) {
      return status;
    }

    state->current_part++;

    distance = partdata - (state->contentpos % partdata);
  }


  if (len > 0) {
    status = stream_writedata(state, &buffer[bufpos], len);
    if (status != FNS_SUCCESS) {
      return status;
    }
    if (state->pos == state->size - 1) {
      ctrl = FN_CB_OK;
      status = stream_write(state, &ctrl, 1);
      if (status != FNS_SUCCESS) {
        return status;
      }

      status = stream_storedata(state);
      if (status != FNS_SUCCESS) {
        return status;
      }
    }
  }

  return FNS_SUCCESS;
}


/* request key from the node at address:port with hops-to-live htl,
   and fill state with info on the resulting stream */
int freenet_request_stream (freenet_transfer_state *state, char *address,
                            char *port, int htl, freenet_key *key)
{
  freenet_message request;
  freenet_message reply;
  int i, j, len, metadlen, startpos;
  int depth;
  unsigned char header[FN_KEY_BYTES+2];
  int status;
  unsigned char tmp[3];
  unsigned char pubstr[(FN_DHK_LEN*2) + 1]; // null term
  unsigned char dnstr[(FN_HASH_BYTES*2) + 1]; // null term
  unsigned char dnhash[FN_HASH_BYTES];

  dnstr[0] = 0;

  status = freenet_connect(&(state->connection), address, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = generate_random((unsigned char *)&(state->uniqueid),
                           sizeof(u_int64_t));
  if (status != FNS_SUCCESS) {
    return status;
  }

  request.uniqueid = state->uniqueid;

  request.type = FN_DATAREQUEST_MSG;
  sprintf(request.field[0], "SearchKey=");
  for (i=0; i<FN_HASH_BYTES + 3; i++) {
    sprintf(&(request.field[0][10+(i*2)]), "%.2X", key->searchkey[i]);
  }
  /* generate a random int between FN_DL and FN_DH */
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[1], "Depth=%x", depth);
  sprintf(request.field[2], "HopsToLive=%x", htl);
  strcpy(request.field[3], "Source=tcp/localhost:6666");
  request.numfields = 4;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  i=0;
  do {
    status = freenet_recvmsg(&(state->connection), &reply);
    if (status != FNS_SUCCESS) {
      return status;
    }

    if (reply.uniqueid != request.uniqueid) {
      return FNS_WRONG_ID;
    }

    i++;
  } while ((reply.type == FN_QUERYRESTARTED_MSG) &&
           (i < FN_MAX_RESTARTS));

  if (reply.type == FN_DATAREPLY_MSG) {
    len = 0;

    for (i=0; i<reply.numfields; i++) {
      if (strncmp(reply.field[i], "Storable.Symmetric-cipher=", 26) == 0) {
        if (strcmp(reply.field[i],
                   "Storable.Symmetric-cipher=Twofish") == 0) {
          state->endtoend = FN_TWOFISH;
        } else if (strcmp(reply.field[i],
                          "Storable.Symmetric-cipher=Rijndael") == 0) {
          state->endtoend = FN_RIJNDAEL;
        } else {
          return FNS_INVALID_ENDTOEND;
        }
      } else if (strncmp(reply.field[i],
                         "Storable.Metadata-length=", 25) == 0) {
        sscanf(reply.field[i], "Storable.Metadata-length=%d", &metadlen);
      } else if (strncmp(reply.field[i],
                         "Storable.PartSize=", 18) == 0) {
        sscanf(reply.field[i], "Storable.PartSize=%x", &state->partsize);
      } else if (strncmp(reply.field[i], "Storable.Signature=", 19)==0) {
        strncpy(state->sigstr, &(reply.field[i][19]), ((FN_DHK_LEN*2) + 2));
      } else if (strncmp(reply.field[i], "Storable.Public-key=", 20)==0) {

        /* skip some leading zeros.  this really sucks. */
        startpos = 20;
        if ((strlen(&(reply.field[i][startpos])) > FN_DHK_LEN * 2)
            && ((startpos+1) < strlen(reply.field[i]))
            && (reply.field[i][startpos] == '0')
            && (reply.field[i][startpos+1] == '0')) {
          startpos += 2;
        }

        strncpy(pubstr, &(reply.field[i][startpos]), FN_DHK_LEN * 2);
        pubstr[(FN_DHK_LEN*2) + 1] = 0;
        for (j = 0; j < strlen(pubstr) / 2; j++) {
          strncpy(tmp, &(pubstr[j*2]), 2);
          tmp[2] = 0;
          state->pub[j] = (unsigned char)strtol(tmp, NULL, 16);
        }
      } else if (strncmp(reply.field[i], "Storable.Document-name=", 23)==0) {
        strncpy(dnstr, &(reply.field[i][23]), FN_HASH_BYTES * 2);
        dnstr[(FN_HASH_BYTES*2) + 1] = 0;
        for (j = 0; j < strlen(dnstr) / 2; j++) {
          strncpy(tmp, &(dnstr[j*2]), 2);
          tmp[2] = 0;
          dnhash[j] = (unsigned char)strtol(tmp, NULL, 16);
        }
      } else if (strncmp(reply.field[i], "DataLength=", 11)==0) {
        sscanf(reply.field[i], "DataLength=%x", &len);
      }
    }

    if (len > 0) {

      if (metadlen > len) {
        return FNS_INVALID_METADATA_LEN;
      }

      state->size = len;

      if (state->partsize == 0) {
        state->partsize = len - 1;
      }

      /* "len - 1" below excludes the final control byte from the
         stream length.  That fixes the case where the partsize (plus
         a control byte for every part) divides evenly into the stream
         length.  We don't want that to equal the next whole number. */
      state->last_part = ((len - 1) / (state->partsize + 1));
      state->datasize = state->size - (state->last_part * (FN_HASH_BYTES + 1));
      state->datasize--;
      state->datasize -= FN_HEADER_SIZE;

      state->pos = 0;
      state->contentpos = 0;
      state->current_part = 0;
      state->metadata_len = metadlen;

      state->keytype = key->type;

      memcpy(&(state->group), &(key->group), sizeof(freenet_group));

      if ((key->type == FN_KSK_TYPE) &&
          (memcmp(state->pub, key->pub, FN_DHK_LEN) != 0)) {
        return FNS_INVALID_PUBKEY;
      }

      memcpy(state->hash[0], key->searchkey, FN_HASH_BYTES);
      SHA1_Init(&(state->hash_handle));
      if (strlen(dnstr) != 0) {
        SHA1_Update(&(state->hash_handle), dnhash, FN_HASH_BYTES);
      }

      status = endtoend_init(state, key->enckey);
      if (status != FNS_SUCCESS) {
        return status;
      }

      status = freenet_read_stream(state, header, FN_HEADER_SIZE);
      if (status != FNS_SUCCESS) {
        return status;
      }

      if (header[0]!=0) {
        return FNS_INVALID_DATA;
      } else if (header[1] != FN_KEY_BYTES) {
        return FNS_UNEXPECTED_KEY_LENGTH;
      } else if (memcmp(key->enckey, &(header[2]), FN_KEY_BYTES)!=0) {
        return FNS_BAD_KEY;
      }

    }

  } else if (reply.type == FN_TIMEDOUT_MSG) {
    return FNS_TIMEOUT;
  } else if (reply.type == FN_REQUESTFAILED_MSG) {
    return FNS_REQUEST_FAILED;
  } else {
    return FNS_UNEXPECTED_MESSAGE;
  }

  return FNS_SUCCESS;
}


/* request key from the node at address:port with hops-to-live htl,
   and put the resulting state->datasize bytes of data in buffer
   (assuming the maxlen of the buffer is large enough to hold them) */
int freenet_request_buffer (freenet_transfer_state *state, char *address,
                            char *port, int htl, freenet_key *key,
                            unsigned char *buffer, int maxlen)
{
  int status;

  status = freenet_request_stream(state, address, port, htl, key);
  if (status != FNS_SUCCESS) {
    return status;
  }

  if (state->datasize > maxlen) {
    return FNS_BUFFER_TOO_SMALL;
  }


  status = freenet_read_stream(state, buffer, state->datasize);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


/* send an insertrequest and handle the reply */
int request_insert (freenet_transfer_state *state,
                    unsigned char *searchkey, int htl)
{
  char searchkeystr[128];
  freenet_message request, reply;
  int i, depth, status;

  for (i = 0; i < FN_HASH_BYTES + 3; i++) {
    sprintf(&(searchkeystr[i*2]), "%.2x", searchkey[i]);
  }

  status = generate_random((unsigned char *)&(request.uniqueid),
                           sizeof(u_int64_t));
  if (status != FNS_SUCCESS) {
    return status;
  }
  state->uniqueid = request.uniqueid;

  request.type = FN_INSERTREQUEST_MSG;
  sprintf(request.field[0], "SearchKey=%s", searchkeystr);
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[1], "Depth=%x", depth);
  sprintf(request.field[2], "HopsToLive=%x", htl);
  strcpy(request.field[3], "Source=tcp/localhost:6666");
  request.numfields=4;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  i=0;
  do {
    status = freenet_recvmsg(&(state->connection), &reply);
    if (status != FNS_SUCCESS) {
      return status;
    }

    if (reply.uniqueid != request.uniqueid) {
      return FNS_WRONG_ID;
    }

    i++;
  } while ((reply.type == FN_QUERYRESTARTED_MSG) &&
           (i < FN_MAX_RESTARTS));


  if (reply.type == FN_INSERTREPLY_MSG) {
    return FNS_SUCCESS;
  } else if (reply.type == FN_TIMEDOUT_MSG) {
    return FNS_TIMEOUT;
  } else if (reply.type == FN_REQUESTFAILED_MSG) {
    return FNS_REQUEST_FAILED;
  } else if (reply.type == FN_DATAREPLY_MSG) {
    return FNS_DATA_REPLY;
  } else {
    return FNS_UNEXPECTED_MESSAGE;
  }

}


int freenet_generate_CHK_buffer (freenet_transfer_state *state,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, unsigned char *fbuf,
                                 int len)
{
  int status;
  unsigned char header[FN_HEADER_SIZE];
  int content_len, logtwo;

  key->type = FN_CHK_TYPE;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  state->endtoend = endtoend;


  status = generate_CHK_enckey_from_buffer(key->enckey, fbuf, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  content_len = len + FN_HEADER_SIZE;

  if (content_len > FN_PARTSIZE) {
    state->partsize = FN_PARTSIZE;
  } else {
    state->partsize = content_len;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_buffer (state, "", key->enckey, header, fbuf,
                                        len, num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(key->searchkey, state->hash[0], FN_HASH_BYTES);

  /* set the third-to-last byte of the searchkey to the lowest power of two
     greater than log2(partsize) */
  logtwo = 1;
  key->searchkey[FN_HASH_BYTES] = 0;
  while (logtwo < state->partsize) {
    logtwo *= 2;
    key->searchkey[FN_HASH_BYTES]++;
  }


  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_CHK_TYPE);

  return FNS_SUCCESS;
}


/* Open a connection to the node at address:port and insert as  CHK
   len bytes of fbuf (with metadata_len of those bytes as metadata) at
   hops-to-live htl, with end-to-end encryption scheme endtoend.
   Fill key with the CHK info.  */
int freenet_insert_CHK_buffer (freenet_transfer_state *state, char *address,
                               char *port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *fbuf, int len)
{
  unsigned char header[FN_HEADER_SIZE];
  freenet_message request;
  int total_len, content_len;
  int depth;
  int num_parts;
  int status;

  status = freenet_generate_CHK_buffer(state, endtoend, key, &num_parts,
                                       fbuf, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  content_len = len + FN_HEADER_SIZE;
  total_len = content_len + (num_parts * FN_TRAILER_LEN) + 1;

  status = freenet_connect(&(state->connection), address, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = request_insert(state, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    return status;
  }

  request.uniqueid = state->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  if (num_parts == 0) {
    strcpy(request.field[4], "Storable.PartSize=0");
  } else {
    sprintf(request.field[4], "Storable.PartSize=%x", state->partsize);
  }
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  request.numfields=6;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->htl = htl;
  state->size = total_len;
  state->pos = 0;
  state->contentpos = 0;
  state->current_part = 0;
  state->last_part = num_parts;

  status = endtoend_init(state, key->enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_write_stream(state, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_write_stream(state, fbuf, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_generate_CHK_stream (freenet_transfer_state *state,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, int *len,
                                 FILE *instream)
{
  int status;
  unsigned char header[FN_HEADER_SIZE];
  int content_len, logtwo;

  key->type = FN_CHK_TYPE;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  state->endtoend = endtoend;

  status = generate_CHK_enckey_from_stream(key->enckey, instream);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = fseek(instream, 0, SEEK_END);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  *len = ftell(instream);
  if (*len == -1) {
    return FNS_FSEEK_FAILED;
  }

  status = fseek(instream, 0, SEEK_SET);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  content_len = *len + FN_HEADER_SIZE;

  if (content_len > FN_PARTSIZE) {
    state->partsize = FN_PARTSIZE;
  } else {
    state->partsize = content_len;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_stream (state, "", key->enckey, header, *len,
                                        instream, num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(key->searchkey, state->hash[0], FN_HASH_BYTES);

  /* set the third-to-last byte of the searchkey to the lowest power of two
     greater than log2(partsize) */
  logtwo = 1;
  key->searchkey[FN_HASH_BYTES] = 0;
  while (logtwo < state->partsize) {
    logtwo *= 2;
    key->searchkey[FN_HASH_BYTES]++;
  }

  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_CHK_TYPE);

  return FNS_SUCCESS;
}


/* Open a connection to the node at address:port and negotiate the CHK
   insert of rewindable file stream instream (with metadata_len of
   bytes of it as metadata) at hops-to-live htl, with end-to-end
   encryption scheme endtoend.  Fill key with the CHK info. */
int freenet_insert_CHK_stream (freenet_transfer_state *state, char *address,
                               char *port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream)
{
  unsigned char header[FN_HEADER_SIZE];
  int len, content_len, total_len;
  freenet_message request;
  int depth;
  int status;
  int num_parts;

  status = freenet_generate_CHK_stream(state, endtoend, key, &num_parts,
                                       &len, instream);
  if (status != FNS_SUCCESS) {
    return status;
  }

  content_len = len + FN_HEADER_SIZE;
  total_len = content_len + (num_parts * FN_TRAILER_LEN) + 1;

  status = freenet_connect(&(state->connection), address, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = request_insert(state, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    return status;
  }

  request.uniqueid = state->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  if (num_parts == 0) {
    strcpy(request.field[4], "Storable.PartSize=0");
  } else {
    sprintf(request.field[4], "Storable.PartSize=%x", state->partsize);
  }
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  request.numfields=6;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->htl = htl;
  state->size = total_len;
  state->pos = 0;
  state->contentpos = 0;
  state->current_part = 0;
  state->last_part = num_parts;

  status = endtoend_init(state, key->enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_write_stream(state, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  return FNS_SUCCESS;
}


/* open a connection to the node at address:port and insert len bytes
   of fbuf (with metadata_len of those bytes as metadata) at
   hops-to-live htl, with end-to-end encryption scheme endtoend under
   the SVK (or KSK or SSK) described by key */
int freenet_insert_SVK_buffer (freenet_transfer_state *state, char *address,
                               char *port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *buffer, int len)
{
  unsigned char header[FN_HEADER_SIZE];
  int i, content_len, total_len;
  unsigned char hash[FN_HASH_BYTES];
  unsigned char sigstr[(FN_DHK_LEN*2) +2]; //comma and null term
  unsigned char pubstr[(FN_DHK_LEN*2) +1]; //null term
  unsigned char dnhash[FN_HASH_BYTES];
  unsigned char dnhashstr[(FN_HASH_BYTES*2) +1]; //null term
  freenet_message request;
  int depth;
  int status;
  int num_parts;
  int fieldnum;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  state->endtoend = endtoend;

  content_len = len + FN_HEADER_SIZE;
  state->partsize = content_len;

  if (state->partsize > FN_SVK_MAXSIZE) {
    return FNS_SVK_TOO_LARGE;
  }

  total_len = content_len + 1;

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_buffer (state, key->docname, key->enckey,
                                        header, buffer, len, &num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(hash, state->hash[0], FN_HASH_BYTES);

  status = sign(&(key->group), key->priv, key->pub, hash, FN_HASH_BYTES,
                sigstr);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_connect(&(state->connection), address, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = request_insert(state, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    return status;
  }

  for (i = 0; i < FN_DHK_LEN; i++) {
    sprintf(&(pubstr[i*2]), "%.2x", key->pub[i]);
  }
  pubstr[i*2] = 0;

  request.uniqueid = state->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  strcpy(request.field[4], "Storable.PartSize=0");
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  sprintf(request.field[6], "Storable.Signature=%s", sigstr);
  sprintf(request.field[7], "Storable.Public-key=%s", pubstr);
  if (strlen(key->docname) > 0) {
    SHA1(key->docname, strlen(key->docname), dnhash);
    for (i = 0; i < FN_HASH_BYTES; i++) {
      sprintf(&(dnhashstr[i*2]), "%.2x", dnhash[i]);
    }
    dnhashstr[i*2] = 0;
    sprintf(request.field[8], "Storable.Document-name=%s", dnhashstr);
    fieldnum = 9;
  } else {
    fieldnum = 8;
  }
  request.numfields=fieldnum;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->htl = htl;
  state->size = total_len;
  state->pos = 0;
  state->contentpos = 0;
  state->current_part = 0;
  state->last_part = 0;

  status = endtoend_init(state, key->enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_write_stream(state, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_write_stream(state, buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


/* open a connection to the node at address:port and negotiate the
   insert of rewindable file stream instream (with metadata_len of
   bytes of it as metadata) at hops-to-live htl, with end-to-end
   encryption scheme endtoend under the SVK (or KSK or SSK) described
   by key */
int freenet_insert_SVK_stream (freenet_transfer_state *state, char *address,
                               char *port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream)
{
  unsigned char header[FN_HEADER_SIZE];
  int i, len, content_len, total_len;
  unsigned char hash[FN_HASH_BYTES];
  unsigned char sigstr[(FN_DHK_LEN*2) +2]; //comma and null term
  unsigned char pubstr[(FN_DHK_LEN*2) +1]; //null term
  unsigned char dnhash[FN_HASH_BYTES];
  unsigned char dnhashstr[(FN_HASH_BYTES*2) +1]; //null term
  freenet_message request;
  int depth;
  int status;
  int num_parts;
  int fieldnum;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  state->endtoend = endtoend;

  status = fseek(instream, 0, SEEK_END);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  len = ftell(instream);
  if (len == -1) {
    return FNS_FSEEK_FAILED;
  }

  status = fseek(instream, 0, SEEK_SET);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  content_len = len + FN_HEADER_SIZE;
  state->partsize = content_len;

  if (state->partsize > FN_SVK_MAXSIZE) {
    return FNS_SVK_TOO_LARGE;
  }

  total_len = content_len + 1;

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_stream (state, key->docname, key->enckey,
                                        header, len, instream,
                                        &num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(hash, state->hash[0], FN_HASH_BYTES);

  status = sign(&(key->group), key->priv, key->pub, hash, FN_HASH_BYTES,
                sigstr);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_connect(&(state->connection), address, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = request_insert(state, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    return status;
  }

  for (i = 0; i < FN_DHK_LEN; i++) {
    sprintf(&(pubstr[i*2]), "%.2x", key->pub[i]);
  }
  pubstr[i*2] = 0;

  request.uniqueid = state->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  strcpy(request.field[4], "Storable.PartSize=0");
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  sprintf(request.field[6], "Storable.Signature=%s", sigstr);
  sprintf(request.field[7], "Storable.Public-key=%s", pubstr);
  if (strlen(key->docname) > 0) {
    SHA1(key->docname, strlen(key->docname), dnhash);
    for (i = 0; i < FN_HASH_BYTES; i++) {
      sprintf(&(dnhashstr[i*2]), "%.2x", dnhash[i]);
    }
    dnhashstr[i*2] = 0;
    sprintf(request.field[8], "Storable.Document-name=%s", dnhashstr);
    fieldnum = 9;
  } else {
    fieldnum = 8;
  }
  request.numfields=fieldnum;

  status = freenet_sendmsg(&(state->connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  state->htl = htl;
  state->size = total_len;
  state->pos = 0;
  state->contentpos = 0;
  state->current_part = 0;
  state->last_part = 0;

  status = endtoend_init(state, key->enckey);
  if (status != FNS_SUCCESS) {
    return status;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_write_stream(state, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  return FNS_SUCCESS;
}
