/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <getopt.h>

#include "protocol.h"
#include "client.h"
#include "uri.h"
#include "base64.h"
#include "client_util.h"

#define REQUEST 1
#define INSERT 2

extern char *optarg;
extern int optind;

int main (int argc, char **argv)
{
  int c;
  int i;
  int cmd = 0;
  int errflg = 0;
  char host[128] = "localhost";
  char port[32] = "19114";
  char file[PATH_MAX];
  int htl = 5;
  unsigned char uri[512] = "";
  unsigned char mskuri[512] = "";
  unsigned char docname[512] = "";
  unsigned char keystring[512];
  int status;
  freenet_transfer_state state;
  unsigned char in;
  int zoop;
  unsigned char out;
  FILE *instream;
  unsigned char buffer[66666];
  unsigned char metadata[33333 + 1];
  int usebuffer = 0;
  freenet_key key;
  int metadata_len = 0;
  int show_metadata = 0;
  int max_redirects = 5;
  int redirect_count;


  while ((c = getopt(argc, argv, "brdif:a:p:m:h:")) != EOF) {
    switch (c) {
    case 'r':
      if (cmd == 0) {
        cmd = REQUEST;
      } else {
        errflg++;
      }
      break;
    case 'i':
      if (cmd == 0) {
        cmd = INSERT;
      } else {
        errflg++;
      }
      break;
    case 'b':
      usebuffer = 1;
      break;
    case 'f':
      strncpy(file, optarg, PATH_MAX);
      break;
    case 'a':
      strncpy(host, optarg, 128);
      break;
    case 'p':
      strncpy(port, optarg, 32);
      break;
    case 'm':
      sscanf(optarg, "%d", &metadata_len);
    case 'd':
      show_metadata = 1;
      break;
    case 'h':
      sscanf(optarg, "%d", &htl);
      break;
    case '?':
      errflg++;
    }
  }

  if (cmd == 0) {
    errflg++;
  }

  if (optind < argc) {
    strncpy(uri, argv[optind], strlen(argv[optind]));

    if (cmd == INSERT) {
      status = freenet_parse_insert_uri(&key, uri);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "could not parse key: %d\n", status);
        return -1;
      }
    } else {
      strncpy(mskuri, uri, strlen(uri));
      status = freenet_parse_msk(mskuri, uri, docname);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "could not parse key: %d\n", status);
        return -1;
      }
      status = freenet_parse_request_uri(&key, uri);
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "could not parse key: %d\n", status);
        return -1;
      }
    }
    optind++;
  } else {
    errflg++;
  }

  if (errflg) {
    fprintf(stderr,
            "usage: client [-r|-i] [-a <host address>] [-p <port>]\n");
    fprintf(stderr,
            "[-h <hopstolive>] [-f file] [-m metadata length]\n");
    fprintf(stderr,
            "[-d] key\n");
    exit (2);
  }


  srand(time(NULL));

  if (cmd == REQUEST) {
    if (usebuffer == 0) {

      redirect_count = -1;

      do {

        redirect_count++;

        status = freenet_request_stream(&state, host, port, htl, &key);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "request failed: %d\n", status);
          return -1;
        }

        for (i=0; ((i < state.metadata_len) && (i < 33333)); i++) {

          status = freenet_read_stream(&state, &in, 1);
          if (status != FNS_SUCCESS) {
            fprintf(stderr, "stream read error at byte %d of %d: %d\n",
                    state.pos, state.size, status);
            return -1;
          }

          metadata[i] = in;
        }

        metadata[i] = 0;

        if (state.metadata_len == state.datasize) {
          status = freenet_parse_redirect(&key, metadata);
          if ((status != FNS_SUCCESS) && (status != FNS_NOT_A_REDIRECT)) {
            fprintf(stderr, "redirect handling failed: %d\n", status);
            return -1;
          }

          if (status == FNS_NOT_A_REDIRECT) {
            status = freenet_parse_mapfile(&key, metadata, docname);
            if ((status != FNS_SUCCESS) && (status != FNS_NOT_A_MAPFILE)) {
              fprintf(stderr, "redirect handling failed: %d\n", status);
              return -1;
            }
          }
        }

      } while ((state.metadata_len == state.datasize) &&
               (status != FNS_NOT_A_MAPFILE) &&
               (redirect_count < max_redirects));

      if (show_metadata == 1) {
        printf("%s", metadata);
      }

      i=0;
      while (state.pos < state.size) {

        status = freenet_read_stream(&state, &in, 1);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "stream read error at byte %d of %d: %d\n",
                  state.pos, state.size, status);
          return -1;
        }

        printf("%c", in);

        i++;
      }

    } else {

      redirect_count = -1;

      do {

        redirect_count++;

        status = freenet_request_buffer(&state, host, port, htl, &key,
                                        buffer, 66666);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "request failed: %d\n", status);
          return -1;
        }

        for (i=0; ((i<state.metadata_len) && (i<33333)); i++) {
          metadata[i] = buffer[i];
        }

        metadata[i] = 0;

        if (state.metadata_len == state.datasize) {
          status = freenet_parse_redirect(&key, metadata);
          if ((status != FNS_SUCCESS) && (status != FNS_NOT_A_REDIRECT)) {
            fprintf(stderr, "redirect handling failed: %d\n", status);
            return -1;
          }

          if (status == FNS_NOT_A_REDIRECT) {
            status = freenet_parse_mapfile(&key, metadata, docname);
            if ((status != FNS_SUCCESS) && (status != FNS_NOT_A_MAPFILE)) {
              fprintf(stderr, "redirect handling failed: %d\n", status);
              return -1;
            }
          }
        }

      } while ((state.metadata_len == state.datasize) &&
               (status != FNS_NOT_A_MAPFILE) &&
               (redirect_count < max_redirects));

      if (show_metadata == 1) {
        printf("%s", metadata);
      }

      for (i = state.metadata_len; i < state.datasize; i++) {
        printf("%c", buffer[i]);
      }

    }

  } else if (cmd == INSERT) {
    if (usebuffer == 0) {
      instream = fopen(file, "r");
      if (instream == NULL) {
        printf("couldn't open file %s:\n", file);
        perror("fopen");
        return -1;
      }

      if (key.type == FN_CHK_TYPE) {
        status = freenet_insert_CHK_stream(&state, host, port, htl, FN_TWOFISH,
                                           metadata_len, &key, instream);
      } else {
        status = freenet_insert_SVK_stream(&state, host, port, htl,
                                           FN_TWOFISH, metadata_len, &key,
                                           instream);
      }
      if (status != FNS_SUCCESS) {
        fprintf(stderr, "insert failed: %d\n", status);
        return -1;
      }

      do {
        zoop = fgetc(instream);
        if (zoop == EOF) {
          printf("reached EOF\n");
        } else {
          out = (unsigned char)zoop;
          status = freenet_write_stream(&state, &out, 1);
          if (status != FNS_SUCCESS) {
            fprintf(stderr, "stream write error: %d\n", status);
            return -1;
          }
        }
      } while (zoop != EOF);
    } else {
      instream = fopen(file, "r");
      if (instream == NULL) {
        printf("couldn't open file %s:\n", file);
        perror("fopen");
        return -1;
      }

      i = 0;
      do {
        zoop = fgetc(instream);
        if (zoop == EOF) {
          printf("reached EOF\n");
        } else {
          buffer[i] = zoop;
          i++;
        }
      } while (zoop != EOF);


      if (key.type == FN_CHK_TYPE) {
        status = freenet_insert_CHK_buffer(&state, host, port, htl, FN_TWOFISH,
                                           metadata_len, &key, buffer, i);
        if (status != FNS_SUCCESS) {
          fprintf(stderr, "insert failed: %d\n", status);
          return -1;
        }
      } else {
        status = freenet_insert_SVK_buffer(&state, host, port, htl,
                                           FN_TWOFISH, metadata_len, &key,
                                           buffer, i);
      }
    }

    if (key.type == FN_SVK_TYPE) {
      base64_encode(key.priv, keystring, FN_HASH_BYTES, 0);
      printf("private key: %s\n", keystring);
    }

    if (key.type != FN_KSK_TYPE) {
      freenet_get_uri(&key, uri);
    }
    printf("key:\n%s\n", uri);

  }

  return 1;
}
