/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <string.h>

#include "protocol.h"
#include "util.h"
#include "client.h"
#include "base64.h"

int parse_CHK_or_SVK (unsigned char *searchkey, unsigned char *enckey,
                      unsigned char *uri)
{
  int urilen;
  unsigned char *comma;
  int commapos;
  int length;

  urilen = strlen(uri);

  comma = (unsigned char *)strchr(uri, ',');
  if (comma == NULL) {
    return FNS_INVALID_URI;
  }

  commapos = (int)(comma - uri);

  length = base64_decode_bytes(uri, commapos);
  if (length != FN_HASH_BYTES + 3) {
    return FNS_INVALID_SEARCHKEY;
  }

  length = base64_decode(uri, searchkey, commapos);
  if (length != FN_HASH_BYTES + 3) {
    return FNS_INVALID_SEARCHKEY;
  }

  length = base64_decode_bytes(&(uri[commapos+1]), urilen-(commapos+1));
  if (length != FN_KEY_BYTES) {
    return FNS_INVALID_ENCKEY;
  }

  length = base64_decode(&(uri[commapos+1]), enckey, urilen-(commapos+1));
  if (length != FN_KEY_BYTES) {
    return FNS_INVALID_ENCKEY;
  }

  return FNS_SUCCESS;
}


int parse_insert_SSK (unsigned char *priv, unsigned char *docname,
                      unsigned char *uri)
{
  int urilen;
  unsigned char *slash;
  int slashpos;
  int length;

  urilen = strlen(uri);

  slash = (unsigned char *)strchr(uri, '/');
  if (slash == NULL) {
    return FNS_INVALID_URI;
  }

  slashpos = (int)(slash - uri);

  length = base64_decode_bytes(uri, slashpos);
  if (length != FN_HASH_BYTES) {
    return FNS_INVALID_PRIVATE_KEY;
  }

  length = base64_decode(uri, priv, slashpos);
  if (length != FN_HASH_BYTES) {
    return FNS_INVALID_PRIVATE_KEY;
  }

  strncpy(docname, &(uri[slashpos+1]), FN_MAX_DOCNAME);

  return FNS_SUCCESS;
}


int parse_request_SSK (unsigned char *searchkey, unsigned char *enckey,
                       unsigned char *uri)
{
  int urilen;
  unsigned char *slash;
  int slashpos;
  int length;
  int status;
  unsigned char hashes[FN_HASH_BYTES * 2];

  urilen = strlen(uri);

  slash = (unsigned char *)strchr(uri, '/');
  if (slash == NULL) {
    return FNS_INVALID_URI;
  }

  slashpos = (int)(slash - uri);

  length = base64_decode_bytes(uri, slashpos);
  if (length != FN_HASH_BYTES + 3) {
    return FNS_INVALID_PRIVATE_KEY;
  }

  length = base64_decode(uri, hashes, slashpos);
  if (length != FN_HASH_BYTES + 3) {
    return FNS_INVALID_PRIVATE_KEY;
  }

  SHA1(&(uri[slashpos+1]), urilen-(slashpos+1), &(hashes[FN_HASH_BYTES]));
  SHA1(hashes, FN_HASH_BYTES * 2, searchkey);

  searchkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
  *((u_int16_t *)&(searchkey[FN_HASH_BYTES+1])) = htons(FN_SVK_TYPE);

  status = keygen(&(uri[slashpos+1]), urilen-(slashpos+1), enckey,
                  FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_parse_request_uri (freenet_key *key, unsigned char *uri)
{
  int urilen;
  int status;

  urilen = strlen(uri);

  if (urilen < FN_MIN_URI_LEN) {
    return FNS_INVALID_URI;
  }

  if (strncmp(uri, "freenet:", 8) != 0) {
    return FNS_INVALID_URI;
  }

  if (strncmp(&(uri[8]), "CHK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    key->type = FN_CHK_TYPE;

    status = parse_CHK_or_SVK(key->searchkey, key->enckey, &(uri[12]));
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else if (strncmp(&(uri[8]), "SVK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    key->type = FN_SVK_TYPE;

    strcpy(key->group.p, FN_GROUP_B_P);
    strcpy(key->group.q, FN_GROUP_B_Q);
    strcpy(key->group.g, FN_GROUP_B_G);

    status = parse_CHK_or_SVK(key->searchkey, key->enckey, &(uri[12]));
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else if (strncmp(&(uri[8]), "SSK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    key->type = FN_SVK_TYPE;

    strcpy(key->group.p, FN_GROUP_B_P);
    strcpy(key->group.q, FN_GROUP_B_Q);
    strcpy(key->group.g, FN_GROUP_B_G);

    status = parse_request_SSK(key->searchkey, key->enckey, &(uri[12]));
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else if (strncmp(&(uri[8]), "KSK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    status = freenet_generate_KSK(&(uri[12]), key);
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else {
    return FNS_UNKNOWN_KEYTYPE;
  }

  return FNS_SUCCESS;
}


int freenet_parse_insert_uri (freenet_key *key, unsigned char *uri)
{
  int urilen;
  int status;
  unsigned char priv[FN_HASH_BYTES];
  unsigned char docname[FN_MAX_DOCNAME];

  urilen = strlen(uri);

  if (urilen < FN_MIN_URI_LEN) {
    return FNS_INVALID_URI;
  }

  if (strncmp(uri, "freenet:", 8) != 0) {
    return FNS_INVALID_URI;
  }

  if (strncmp(&(uri[8]), "CHK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    key->type = FN_CHK_TYPE;

  } else if (strncmp(&(uri[8]), "SVK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    status = generate_random(priv, FN_HASH_BYTES);
    if (status != FNS_SUCCESS) {
      return status;
    }

    status = freenet_generate_SVK (priv, "", key);
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else if (strncmp(&(uri[8]), "SSK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    status = parse_insert_SSK(priv, docname, &(uri[12]));
    if (status != FNS_SUCCESS) {
      return status;
    }

    status = freenet_generate_SVK (priv, docname, key);
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else if (strncmp(&(uri[8]), "KSK", 3) == 0) {
    if (uri[11] != '@') {
      return FNS_INVALID_URI;
    }

    status = freenet_generate_KSK(&(uri[12]), key);
    if (status != FNS_SUCCESS) {
      return status;
    }

  } else {
    return FNS_UNKNOWN_KEYTYPE;
  }

  return FNS_SUCCESS;
}


int freenet_parse_msk (unsigned char *mskuri, unsigned char *uri,
                       unsigned char *docname)
{
  int urilen;
  char *slashes;

  urilen = strlen(mskuri);

  if (urilen < FN_MIN_URI_LEN) {
    return FNS_INVALID_URI;
  }

  if (strncmp(mskuri, "freenet:", 8) != 0) {
    return FNS_INVALID_URI;
  }

  if (strncmp(&(mskuri[8]), "MSK", 3) == 0) {
    if (mskuri[11] != '@') {
      return FNS_INVALID_URI;
    }

    strncpy(uri, "freenet:", 8);
    strncpy(&(uri[8]), &(mskuri[12]), urilen);
    slashes = strstr(uri, "//");
    if (slashes != NULL) {
      slashes[0] = 0;
    }

    docname[0] = 0;

    slashes = strstr(mskuri, "//");
    if (slashes != NULL) {
      strncpy(docname, &(slashes[2]), urilen);
    }

  } else {
    strncpy(uri, mskuri, urilen);
    docname[0] = 0;
  }

  return FNS_SUCCESS;
}


void freenet_get_uri (freenet_key *key, unsigned char *uri)
{
  unsigned char searchkeystr[(FN_HASH_BYTES + 3)*5];
  unsigned char enckeystr[FN_KEY_BYTES * 5];
  unsigned char bfmpipub[FN_DHK_LEN + 2];
  unsigned char pubhashkey[FN_HASH_BYTES + 3];


  if (key->type == FN_CHK_TYPE) {

    base64_encode(key->searchkey, searchkeystr, FN_HASH_BYTES + 3, 0);
    base64_encode(key->enckey, enckeystr, FN_KEY_BYTES, 0);

    sprintf(uri, "freenet:CHK@%s,%s\n", searchkeystr, enckeystr);

  } else if (key->type == FN_SVK_TYPE) {

    if (strlen(key->docname) > 0) {

      raw_to_bagbiting_freenet_mpi(key->pub, FN_DHK_LEN, bfmpipub);
      SHA1(bfmpipub, FN_DHK_LEN + 2, pubhashkey);
      pubhashkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
      *((u_int16_t *)&(pubhashkey[FN_HASH_BYTES+1])) = htons(FN_SVK_TYPE);

      base64_encode(pubhashkey, searchkeystr, FN_HASH_BYTES + 3, 0);

      sprintf(uri, "freenet:SSK@%s/%s\n", searchkeystr, key->docname);

    } else {

      base64_encode(key->searchkey, searchkeystr, FN_HASH_BYTES + 3, 0);
      base64_encode(key->enckey, enckeystr, FN_KEY_BYTES, 0);

      sprintf(uri, "freenet:SVK@%s,%s\n", searchkeystr, enckeystr);

    }

  }

}
