/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <malloc.h>
#include <string.h>
#include <openssl/dsa.h>

#include "protocol.h"
#include "util.h"
#include "client.h"
#include "endtoend.h"
#include "base64.h"
#include "stream.h"

#define FN_MAX_RESTARTS 3

#define FN_TRAILER_LEN  (FN_HASH_BYTES + 1)
#define FN_PARTSIZE 0x4000

int generate_DSA (unsigned char *searchkey,
                  unsigned char *pub, unsigned char *priv,
                  const unsigned char *docname,
                  freenet_group *group);

int generate_CHK_enckey_from_buffer (unsigned char *enckey,
                                     unsigned char *buffer, int buflen);

int generate_CHK_enckey_from_stream (unsigned char *enckey,
                                     FILE *instream);

int generate_hashes_from_buffer (freenet_client_stream *cstream,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 unsigned char *buffer, int len,
                                 int *num_parts);

int generate_hashes_from_stream (freenet_client_stream *cstream,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 int len, FILE *instream, int *num_parts);

int request_insert (freenet_client_stream *cstream,
                    unsigned char *searchkey, int htl);

int send_storedata (freenet_client_stream *cstream);


int send_storedata (freenet_client_stream *cstream)
{
  int status;
  freenet_message request;

  request.type = FN_STOREDATA_MSG;
  request.uniqueid = cstream->uniqueid;
  strcpy(request.field[0], "Depth=0");
  sprintf(request.field[1], "HopsToLive=%x", cstream->htl);
  request.numfields=2;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


/* read len bytes from a freenet stream (given by "cstream") into buffer */
int freenet_client_read_stream(freenet_client_stream *cstream,
                               unsigned char *buffer, int len)
{
  int status;
  int retval;
  int i;
  unsigned char *data;
  freenet_message message;

  data = malloc(len);
  if (data == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = freenet_read_stream(&(cstream->stream), data, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = endtoend_decrypt(cstream, data, buffer, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  if (cstream->stream.pos == cstream->stream.size) {
    /* get the storedata message, and make sure it looks okay */
    status = freenet_recvmsg(&(cstream->stream.connection), &message);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    if (message.uniqueid != cstream->uniqueid) {
      retval = FNS_WRONG_ID;
      goto end;
    }

    if (message.type != FN_STOREDATA_MSG) {
      retval = FNS_UNEXPECTED_MESSAGE;
      goto end;
    }

    for (i = 0; i < message.numfields; i++) {
      if (strncmp(message.field[i], "DataSource=", 11) == 0) {
        strncpy(cstream->datasource, &(message.field[i][11]),
                strlen(&(message.field[i][11])));
      }
    }
  }

  retval = FNS_SUCCESS;

 end:
  free(data);
  return retval;
}


/* write len bytes of buffer to a freenet stream (given by "cstream") */
int freenet_client_write_stream(freenet_client_stream *cstream,
                                unsigned char *buffer, int len)
{
  int status;
  int retval;
  unsigned char *data;

  data = malloc(len);
  if (data == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = endtoend_encrypt(cstream, buffer, data, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_write_stream(&(cstream->stream), data, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  if (cstream->stream.pos == cstream->stream.size) {
    status = send_storedata(cstream);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }
  }

  retval = FNS_SUCCESS;

 end:
  free(data);
  return retval;
}


/* sets searchkey and pub given priv, docname, and group */
int generate_DSA (unsigned char *searchkey,
                  unsigned char *pub, unsigned char *priv,
                  const unsigned char *docname,
                  freenet_group *group)
{
  int status;
  BIGNUM *x = NULL;
  BIGNUM *g;
  BIGNUM *p;
  BIGNUM *y;
  BN_CTX *ctx = NULL;
  unsigned char bfmpipub[FN_DHK_MPI_LEN];
  unsigned char hashes[FN_HASH_BYTES * 2];
  int pub_len;
  int retval;

  g = BN_new();
  p = BN_new();
  y = BN_new();

  status = BN_hex2bn(&g, group->g);
  if (status != strlen(group->g)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&p, group->p);
  if (status != strlen(group->p)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  x = BN_bin2bn(priv, FN_HASH_BYTES, NULL);
  if (x == NULL) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  ctx = BN_CTX_new();

  BN_mod_exp(y, g, x, p, ctx);

  pub_len = BN_num_bytes(y);
  if (pub_len != FN_DHK_LEN) {
    retval = FNS_INVALID_DHK;
    goto end;
  }

  status = BN_bn2bin(y, pub);
  if (status != pub_len) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = raw_to_bagbiting_freenet_mpi(pub, pub_len, bfmpipub);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  if (strlen(docname) == 0) {
    SHA1(bfmpipub, pub_len + 2, searchkey);
  } else {
    SHA1(bfmpipub, pub_len + 2, hashes);
    SHA1(docname, strlen(docname), &(hashes[FN_HASH_BYTES]));
    SHA1(hashes, FN_HASH_BYTES * 2, searchkey);
  }

  retval = FNS_SUCCESS;

 end:
  BN_free(g);
  BN_free(p);
  BN_free(y);
  BN_free(x);
  BN_CTX_free(ctx);

  return retval;
}


/* sets the key given the keystr */
int freenet_generate_KSK (const unsigned char *keystr, freenet_key *key)
{
  int keylen;
  int status;

  keylen = strlen(keystr);

  SHA1(keystr, keylen, key->priv);

  strcpy(key->group.p, FN_GROUP_A_P);
  strcpy(key->group.q, FN_GROUP_A_Q);
  strcpy(key->group.g, FN_GROUP_A_G);

  status = generate_DSA(key->searchkey, key->pub, key->priv, "",
                        &(key->group));
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->searchkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_KSK_TYPE);

  status = keygen(keystr, keylen, key->enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->type = FN_KSK_TYPE;
  key->docname[0] = 0;

  return FNS_SUCCESS;

}


/* sets the key given priv and docname */
int freenet_generate_SVK (unsigned char *priv, const unsigned char *docname,
                          freenet_key *key)
{
  int status;

  strcpy(key->group.p, FN_GROUP_B_P);
  strcpy(key->group.q, FN_GROUP_B_Q);
  strcpy(key->group.g, FN_GROUP_B_G);

  status = generate_DSA(key->searchkey, key->pub, priv, docname,
                        &(key->group));
  if (status != FNS_SUCCESS) {
    return status;
  }

  key->searchkey[FN_HASH_BYTES] = FN_SVK_PARTSIZE_BYTE;
  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_SVK_TYPE);

  if (strlen(docname) > 0) {
    status = keygen(docname, strlen(docname), key->enckey, FN_KEY_BYTES);
    if (status != FNS_SUCCESS) {
      return status;
    }
  } else {
    status = generate_random(key->enckey, FN_KEY_BYTES);
    if (status != FNS_SUCCESS) {
      return status;
    }
  }

  key->type = FN_SVK_TYPE;
  strncpy(key->docname, docname, FN_MAX_DOCNAME);
  memcpy(key->priv, priv, FN_HASH_BYTES);

  return FNS_SUCCESS;
}


/* The generate_CHK_enckey_ functions hash the data and send it to
   keygen() (which will perform a second hash).  This first hash is
   unnecessary, and we're only doing it because Oskar Sandberg was too
   lazy to do this correctly in the reference implementation.  Calls
   to these functions should really be calls to keygen(). */
int generate_CHK_enckey_from_buffer (unsigned char *enckey,
                                     unsigned char *buffer, int buflen)
{
  int status;
  unsigned char hash[FN_HASH_BYTES];

  SHA1(buffer, buflen, hash);

  status = keygen(hash, FN_HASH_BYTES, enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int generate_CHK_enckey_from_stream (unsigned char *enckey,
                                     FILE *instream)
{
  int status;
  int c;
  unsigned char byte;
  SHA_CTX td;
  unsigned char hash[FN_HASH_BYTES];

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  SHA1_Init(&td);

  c=0;
  while (c!=EOF) {
    c = fgetc(instream);
    if (c!=EOF) {
      byte = (unsigned char)c;
      SHA1_Update(&td, &byte, 1);
    }
  }

  SHA1_Final(hash, &td);

  status = keygen(hash, FN_HASH_BYTES, enckey, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}

/* returns num_parts and sets hashes in cstream */
int generate_hashes_from_buffer (freenet_client_stream *cstream,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 unsigned char *buffer, int len,
                                 int *num_parts)
{
  int c, i, current_part, distance, status, content_len, retval;
  unsigned char enc_header[FN_HEADER_SIZE];
  unsigned char byte, enc_byte;
  unsigned char dnhash[FN_HASH_BYTES];
  SHA_CTX *hash_handle;

  hash_handle = NULL;

  status = endtoend_init(cstream, enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = endtoend_encrypt(cstream, header, enc_header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  current_part = 0;

  hash_handle = malloc(sizeof(SHA_CTX));
  if (hash_handle == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  SHA1_Init(&hash_handle[current_part]);

  if (strlen(docname) > 0) {
    SHA1(docname, strlen(docname), dnhash);
    SHA1_Update(&hash_handle[current_part], dnhash, FN_HASH_BYTES);
  }

  SHA1_Update(&hash_handle[current_part], enc_header, FN_HEADER_SIZE);
  distance =  FN_HEADER_SIZE;

  content_len = len + FN_HEADER_SIZE;

  c=0;
  for (i=0; i<len; i++) {
    c = buffer[i];

    if ((content_len > cstream->stream.partsize) &&
        (distance == (cstream->stream.partsize - FN_HASH_BYTES))) {
      distance = 0;
      current_part++;

      hash_handle = realloc(hash_handle,
                            sizeof(SHA_CTX) * (current_part + 1));
      if (hash_handle == NULL) {
        retval = FNS_MALLOC_FAILED;
        goto end;
      }

      SHA1_Init(&hash_handle[current_part]);
    }

    byte = (unsigned char)c;

    status = endtoend_encrypt(cstream, &byte, &enc_byte, 1);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    SHA1_Update(&hash_handle[current_part], &enc_byte, 1);
    distance++;
  }

  for (i = current_part; i > 0; i--) {
    SHA1_Final(cstream->stream.hash[i], &hash_handle[i]);
    SHA1_Update(&hash_handle[i-1], cstream->stream.hash[i], FN_HASH_BYTES);
  }
  SHA1_Final(cstream->stream.hash[0], &hash_handle[0]);

  *num_parts = current_part;

  retval = FNS_SUCCESS;

 end:
  free(hash_handle);
  return FNS_SUCCESS;
}


/* returns num_parts and sets hashes in cstream */
int generate_hashes_from_stream (freenet_client_stream *cstream,
                                 const unsigned char *docname,
                                 unsigned char *enckey,
                                 unsigned char *header,
                                 int len, FILE *instream, int *num_parts)
{
  int c, i, current_part, distance, status, content_len, retval;
  unsigned char enc_header[FN_HEADER_SIZE];
  unsigned char byte, enc_byte;
  unsigned char dnhash[FN_HASH_BYTES];
  SHA_CTX *hash_handle;

  hash_handle = NULL;

  status = endtoend_init(cstream, enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = endtoend_encrypt(cstream, header, enc_header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  current_part = 0;

  hash_handle = malloc(sizeof(SHA_CTX));
  if (hash_handle == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  SHA1_Init(&hash_handle[current_part]);

  if (strlen(docname) > 0) {
    SHA1(docname, strlen(docname), dnhash);
    SHA1_Update(&hash_handle[current_part], dnhash, FN_HASH_BYTES);
  }

  SHA1_Update(&hash_handle[current_part], enc_header, FN_HEADER_SIZE);
  distance =  FN_HEADER_SIZE;

  content_len = len + FN_HEADER_SIZE;

  c=0;
  for (i=0; c!=EOF; i++) {
    c = fgetc(instream);

    if (c!=EOF) {


      if ((content_len > cstream->stream.partsize) &&
          (distance == (cstream->stream.partsize - FN_HASH_BYTES))) {
        distance = 0;
        current_part++;

        hash_handle = realloc(hash_handle,
                              sizeof(SHA_CTX) * (current_part + 1));
        if (hash_handle == NULL) {
          retval = FNS_MALLOC_FAILED;
          goto end;
        }

        SHA1_Init(&hash_handle[current_part]);
      }

      byte = (unsigned char)c;

      status = endtoend_encrypt(cstream, &byte, &enc_byte, 1);
      if (status != FNS_SUCCESS) {
        retval = status;
        goto end;
      }

      SHA1_Update(&hash_handle[current_part], &enc_byte, 1);
      distance++;

    }

  }

  for (i = current_part; i > 0; i--) {
    SHA1_Final(cstream->stream.hash[i], &hash_handle[i]);
    SHA1_Update(&hash_handle[i-1], cstream->stream.hash[i], FN_HASH_BYTES);
  }
  SHA1_Final(cstream->stream.hash[0], &hash_handle[0]);

  *num_parts = current_part;

  retval = FNS_SUCCESS;

 end:
  free(hash_handle);
  return FNS_SUCCESS;
}


/* request key from the node at address:port with hops-to-live htl,
   and fill cstream with info on the resulting stream */
int freenet_request_stream (freenet_client_stream *cstream, char *address,
                            int port, int htl, freenet_key *key)
{
  freenet_message request;
  freenet_message reply;
  int metadlen;
  int i;
  int depth;
  int status;
  int retval;
  unsigned char header[FN_KEY_BYTES+2];

  status = freenet_connect(&(cstream->stream.connection), address, port);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = generate_random((unsigned char *)&(cstream->uniqueid),
                           sizeof(u_int64_t));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  request.uniqueid = cstream->uniqueid;

  request.type = FN_DATAREQUEST_MSG;
  sprintf(request.field[0], "SearchKey=");
  for (i=0; i<FN_HASH_BYTES + 3; i++) {
    sprintf(&(request.field[0][10+(i*2)]), "%.2X", key->searchkey[i]);
  }
  /* generate a random int between FN_DL and FN_DH */
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[1], "Depth=%x", depth);
  sprintf(request.field[2], "HopsToLive=%x", htl);
  strcpy(request.field[3], "Source=tcp/localhost:6666");
  request.numfields = 4;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  i=0;
  do {
    status = freenet_recvmsg(&(cstream->stream.connection), &reply);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    if (reply.uniqueid != request.uniqueid) {
      retval = FNS_WRONG_ID;
      goto end;
    }

    i++;
  } while ((reply.type == FN_QUERYRESTARTED_MSG) &&
           (i < FN_MAX_RESTARTS));

  if (reply.type == FN_DATAREPLY_MSG) {

    /* FIXME -- ensure that all required fields are present */
    for (i=0; i<reply.numfields; i++) {
      if (strncmp(reply.field[i], "Storable.Symmetric-cipher=", 26) == 0) {
        if (strcmp(reply.field[i],
                   "Storable.Symmetric-cipher=Twofish") == 0) {
          cstream->endtoend = FN_TWOFISH;
        } else if (strcmp(reply.field[i],
                          "Storable.Symmetric-cipher=Rijndael") == 0) {
          cstream->endtoend = FN_RIJNDAEL;
        } else {
          retval = FNS_INVALID_ENDTOEND;
          goto end;
        }
      } else if (strncmp(reply.field[i],
                         "Storable.Metadata-length=", 25) == 0) {
        sscanf(reply.field[i], "Storable.Metadata-length=%d", &metadlen);
      }
    }

    cstream->metadata_len = metadlen;

    status = freenet_init_incoming_stream(&(cstream->stream), key, &reply);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    cstream->data_len = cstream->stream.payload_len - FN_HEADER_SIZE;

    if (cstream->metadata_len > cstream->stream.size) {
      retval = FNS_INVALID_METADATA_LEN;
      goto end;
    }

    if ((key->type == FN_KSK_TYPE) &&
        (memcmp(cstream->stream.pub, key->pub, FN_DHK_LEN) != 0)) {
      retval = FNS_INVALID_PUBKEY;
      goto end;
    }

    status = endtoend_init(cstream, key->enckey);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    status = freenet_client_read_stream(cstream, header, FN_HEADER_SIZE);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }

    if (header[0]!=0) {
      retval = FNS_INVALID_DATA;
      goto end;
    } else if (header[1] != FN_KEY_BYTES) {
      retval = FNS_UNEXPECTED_KEY_LENGTH;
      goto end;
    } else if (memcmp(key->enckey, &(header[2]), FN_KEY_BYTES)!=0) {
      retval = FNS_BAD_KEY;
      goto end;
    }

  } else if (reply.type == FN_TIMEDOUT_MSG) {
    retval = FNS_TIMEOUT;
    goto end;
  } else if (reply.type == FN_REQUESTFAILED_MSG) {
    retval = FNS_REQUEST_FAILED;
    goto end;
  } else {
    retval = FNS_UNEXPECTED_MESSAGE;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}


/* request key from the node at address:port with hops-to-live htl,
   and put the resulting cstream->data_len bytes of data in buffer
   (assuming the maxlen of the buffer is large enough to hold them) */
int freenet_request_buffer (freenet_client_stream *cstream, char *address,
                            int port, int htl, freenet_key *key,
                            unsigned char *buffer, int maxlen)
{
  int retval;
  int status;

  status = freenet_request_stream(cstream, address, port, htl, key);
  if (status != FNS_SUCCESS) {
    return status;
  }

  if (cstream->data_len > maxlen) {
    retval = FNS_BUFFER_TOO_SMALL;
    goto end;
  }

  status = freenet_client_read_stream(cstream, buffer, cstream->data_len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_close_connection(&(cstream->stream.connection));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}


/* send an insertrequest and handle the reply */
int request_insert (freenet_client_stream *cstream,
                    unsigned char *searchkey, int htl)
{
  char searchkeystr[128];
  freenet_message request, reply;
  int i, depth, status;

  for (i = 0; i < FN_HASH_BYTES + 3; i++) {
    sprintf(&(searchkeystr[i*2]), "%.2x", searchkey[i]);
  }

  status = generate_random((unsigned char *)&(request.uniqueid),
                           sizeof(u_int64_t));
  if (status != FNS_SUCCESS) {
    return status;
  }
  cstream->uniqueid = request.uniqueid;

  request.type = FN_INSERTREQUEST_MSG;
  sprintf(request.field[0], "SearchKey=%s", searchkeystr);
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[1], "Depth=%x", depth);
  sprintf(request.field[2], "HopsToLive=%x", htl);
  strcpy(request.field[3], "Source=tcp/localhost:6666");
  request.numfields=4;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    return status;
  }

  i=0;
  do {
    status = freenet_recvmsg(&(cstream->stream.connection), &reply);
    if (status != FNS_SUCCESS) {
      return status;
    }

    if (reply.uniqueid != request.uniqueid) {
      return FNS_WRONG_ID;
    }

    i++;
  } while ((reply.type == FN_QUERYRESTARTED_MSG) &&
           (i < FN_MAX_RESTARTS));


  if (reply.type == FN_INSERTREPLY_MSG) {
    return FNS_SUCCESS;
  } else if (reply.type == FN_TIMEDOUT_MSG) {
    return FNS_TIMEOUT;
  } else if (reply.type == FN_REQUESTFAILED_MSG) {
    return FNS_REQUEST_FAILED;
  } else if (reply.type == FN_DATAREPLY_MSG) {
    return FNS_DATA_REPLY;
  } else {
    return FNS_UNEXPECTED_MESSAGE;
  }

}


int freenet_generate_CHK_buffer (freenet_client_stream *cstream,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, unsigned char *fbuf,
                                 int len)
{
  int status;
  unsigned char header[FN_HEADER_SIZE];
  int content_len, logtwo;

  key->type = FN_CHK_TYPE;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  cstream->endtoend = endtoend;


  status = generate_CHK_enckey_from_buffer(key->enckey, fbuf, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  content_len = len + FN_HEADER_SIZE;

  if (content_len > FN_PARTSIZE) {
    cstream->stream.partsize = FN_PARTSIZE;
  } else {
    cstream->stream.partsize = content_len;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_buffer (cstream, "", key->enckey, header, fbuf,
                                        len, num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(key->searchkey, cstream->stream.hash[0], FN_HASH_BYTES);

  /* set the third-to-last byte of the searchkey to the lowest power of two
     greater than log2(partsize) */
  logtwo = 1;
  key->searchkey[FN_HASH_BYTES] = 0;
  while (logtwo < cstream->stream.partsize) {
    logtwo *= 2;
    key->searchkey[FN_HASH_BYTES]++;
  }


  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_CHK_TYPE);

  return FNS_SUCCESS;
}


/* Open a connection to the node at address:port and insert as  CHK
   len bytes of fbuf (with metadata_len of those bytes as metadata) at
   hops-to-live htl, with end-to-end encryption scheme endtoend.
   Fill key with the CHK info.  */
int freenet_insert_CHK_buffer (freenet_client_stream *cstream, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *fbuf, int len)
{
  unsigned char header[FN_HEADER_SIZE];
  freenet_message request;
  int total_len, content_len;
  int depth;
  int num_parts;
  int status;
  int retval;

  status = freenet_generate_CHK_buffer(cstream, endtoend, key, &num_parts,
                                       fbuf, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  content_len = len + FN_HEADER_SIZE;
  total_len = content_len + (num_parts * FN_TRAILER_LEN) + 1;

  status = freenet_connect(&(cstream->stream.connection), address, port);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = request_insert(cstream, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  request.uniqueid = cstream->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  if (num_parts == 0) {
    strcpy(request.field[4], "Storable.PartSize=0");
  } else {
    sprintf(request.field[4], "Storable.PartSize=%x", cstream->stream.partsize);
  }
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  request.numfields=6;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_init_outgoing_stream(&(cstream->stream));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  cstream->htl = htl;
  cstream->stream.size = total_len;
  cstream->stream.pos = 0;
  cstream->stream.contentpos = 0;
  cstream->stream.current_part = 0;
  cstream->stream.last_part = num_parts;

  status = endtoend_init(cstream, key->enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_client_write_stream(cstream, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_client_write_stream(cstream, fbuf, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_close_connection(&(cstream->stream.connection));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}


int freenet_generate_CHK_stream (freenet_client_stream *cstream,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, int *len,
                                 FILE *instream)
{
  int status;
  unsigned char header[FN_HEADER_SIZE];
  int content_len, logtwo;

  key->type = FN_CHK_TYPE;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    return FNS_INVALID_ENDTOEND;
  }
  cstream->endtoend = endtoend;

  status = generate_CHK_enckey_from_stream(key->enckey, instream);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = fseek(instream, 0, SEEK_END);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  *len = ftell(instream);
  if (*len == -1) {
    return FNS_FSEEK_FAILED;
  }

  status = fseek(instream, 0, SEEK_SET);
  if (status == -1) {
    return FNS_FSEEK_FAILED;
  }

  content_len = *len + FN_HEADER_SIZE;

  if (content_len > FN_PARTSIZE) {
    cstream->stream.partsize = FN_PARTSIZE;
  } else {
    cstream->stream.partsize = content_len;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_stream (cstream, "", key->enckey, header, *len,
                                        instream, num_parts);
  if (status != FNS_SUCCESS) {
    return status;
  }

  memcpy(key->searchkey, cstream->stream.hash[0], FN_HASH_BYTES);

  /* set the third-to-last byte of the searchkey to the lowest power of two
     greater than log2(partsize) */
  logtwo = 1;
  key->searchkey[FN_HASH_BYTES] = 0;
  while (logtwo < cstream->stream.partsize) {
    logtwo *= 2;
    key->searchkey[FN_HASH_BYTES]++;
  }

  *((u_int16_t *)&(key->searchkey[FN_HASH_BYTES+1])) = htons(FN_CHK_TYPE);

  return FNS_SUCCESS;
}


/* Open a connection to the node at address:port and negotiate the CHK
   insert of rewindable file stream instream (with metadata_len of
   bytes of it as metadata) at hops-to-live htl, with end-to-end
   encryption scheme endtoend.  Fill key with the CHK info. */
int freenet_insert_CHK_stream (freenet_client_stream *cstream, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream)
{
  unsigned char header[FN_HEADER_SIZE];
  int len, content_len, total_len;
  freenet_message request;
  int depth;
  int status;
  int num_parts;
  int retval;

  status = freenet_generate_CHK_stream(cstream, endtoend, key, &num_parts,
                                       &len, instream);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  content_len = len + FN_HEADER_SIZE;
  total_len = content_len + (num_parts * FN_TRAILER_LEN) + 1;

  status = freenet_connect(&(cstream->stream.connection), address, port);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = request_insert(cstream, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  request.uniqueid = cstream->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  if (num_parts == 0) {
    strcpy(request.field[4], "Storable.PartSize=0");
  } else {
    sprintf(request.field[4], "Storable.PartSize=%x", cstream->stream.partsize);
  }
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  request.numfields=6;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_init_outgoing_stream(&(cstream->stream));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  cstream->htl = htl;
  cstream->stream.size = total_len;
  cstream->stream.pos = 0;
  cstream->stream.contentpos = 0;
  cstream->stream.current_part = 0;
  cstream->stream.last_part = num_parts;

  status = endtoend_init(cstream, key->enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_client_write_stream(cstream, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    retval = FNS_FSEEK_FAILED;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}


/* open a connection to the node at address:port and insert len bytes
   of fbuf (with metadata_len of those bytes as metadata) at
   hops-to-live htl, with end-to-end encryption scheme endtoend under
   the SVK (or KSK or SSK) described by key */
int freenet_insert_SVK_buffer (freenet_client_stream *cstream, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *buffer, int len)
{
  unsigned char header[FN_HEADER_SIZE];
  int i, content_len, total_len;
  unsigned char hash[FN_HASH_BYTES];
  unsigned char sigstr[(FN_DHK_LEN*2) +2]; //comma and null term
  unsigned char pubstr[(FN_DHK_LEN*2) +1]; //null term
  unsigned char dnhash[FN_HASH_BYTES];
  unsigned char dnhashstr[(FN_HASH_BYTES*2) +1]; //null term
  freenet_message request;
  int depth;
  int status;
  int num_parts;
  int fieldnum;
  int retval;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    retval = FNS_INVALID_ENDTOEND;
    goto end;
  }
  cstream->endtoend = endtoend;

  content_len = len + FN_HEADER_SIZE;
  cstream->stream.partsize = content_len;

  if (cstream->stream.partsize > FN_SVK_MAXSIZE) {
    retval = FNS_SVK_TOO_LARGE;
    goto end;
  }

  total_len = content_len + 1;

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_buffer (cstream, key->docname, key->enckey,
                                        header, buffer, len, &num_parts);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  memcpy(hash, cstream->stream.hash[0], FN_HASH_BYTES);

  status = sign(&(key->group), key->priv, key->pub, hash, FN_HASH_BYTES,
                sigstr);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_connect(&(cstream->stream.connection), address, port);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = request_insert(cstream, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  for (i = 0; i < FN_DHK_LEN; i++) {
    sprintf(&(pubstr[i*2]), "%.2x", key->pub[i]);
  }
  pubstr[i*2] = 0;

  request.uniqueid = cstream->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  strcpy(request.field[4], "Storable.PartSize=0");
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  sprintf(request.field[6], "Storable.Signature=%s", sigstr);
  sprintf(request.field[7], "Storable.Public-key=%s", pubstr);
  if (strlen(key->docname) > 0) {
    SHA1(key->docname, strlen(key->docname), dnhash);
    for (i = 0; i < FN_HASH_BYTES; i++) {
      sprintf(&(dnhashstr[i*2]), "%.2x", dnhash[i]);
    }
    dnhashstr[i*2] = 0;
    sprintf(request.field[8], "Storable.Document-name=%s", dnhashstr);
    fieldnum = 9;
  } else {
    fieldnum = 8;
  }
  request.numfields=fieldnum;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_init_outgoing_stream(&(cstream->stream));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  cstream->htl = htl;
  cstream->stream.size = total_len;
  cstream->stream.pos = 0;
  cstream->stream.contentpos = 0;
  cstream->stream.current_part = 0;
  cstream->stream.last_part = 0;

  status = endtoend_init(cstream, key->enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_client_write_stream(cstream, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_client_write_stream(cstream, buffer, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_close_connection(&(cstream->stream.connection));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}


/* open a connection to the node at address:port and negotiate the
   insert of rewindable file stream instream (with metadata_len of
   bytes of it as metadata) at hops-to-live htl, with end-to-end
   encryption scheme endtoend under the SVK (or KSK or SSK) described
   by key */
int freenet_insert_SVK_stream (freenet_client_stream *cstream, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream)
{
  unsigned char header[FN_HEADER_SIZE];
  int i, len, content_len, total_len;
  unsigned char hash[FN_HASH_BYTES];
  unsigned char sigstr[(FN_DHK_LEN*2) +2]; //comma and null term
  unsigned char pubstr[(FN_DHK_LEN*2) +1]; //null term
  unsigned char dnhash[FN_HASH_BYTES];
  unsigned char dnhashstr[(FN_HASH_BYTES*2) +1]; //null term
  freenet_message request;
  int depth;
  int status;
  int num_parts;
  int fieldnum;
  int retval;

  if ((endtoend != FN_TWOFISH) &&
      (endtoend != FN_RIJNDAEL)) {
    retval = FNS_INVALID_ENDTOEND;
    goto end;
  }
  cstream->endtoend = endtoend;

  status = fseek(instream, 0, SEEK_END);
  if (status == -1) {
    retval = FNS_FSEEK_FAILED;
  }

  len = ftell(instream);
  if (len == -1) {
    retval = FNS_FSEEK_FAILED;
    goto end;
  }

  status = fseek(instream, 0, SEEK_SET);
  if (status == -1) {
    retval = FNS_FSEEK_FAILED;
    goto end;
  }

  content_len = len + FN_HEADER_SIZE;
  cstream->stream.partsize = content_len;

  if (cstream->stream.partsize > FN_SVK_MAXSIZE) {
    retval = FNS_SVK_TOO_LARGE;
    goto end;
  }

  total_len = content_len + 1;

  header[0] = 0;
  header[1] = FN_KEY_BYTES;
  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = generate_hashes_from_stream (cstream, key->docname, key->enckey,
                                        header, len, instream,
                                        &num_parts);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  memcpy(hash, cstream->stream.hash[0], FN_HASH_BYTES);

  status = sign(&(key->group), key->priv, key->pub, hash, FN_HASH_BYTES,
                sigstr);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_connect(&(cstream->stream.connection), address, port);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = request_insert(cstream, key->searchkey, htl);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  for (i = 0; i < FN_DHK_LEN; i++) {
    sprintf(&(pubstr[i*2]), "%.2x", key->pub[i]);
  }
  pubstr[i*2] = 0;

  request.uniqueid = cstream->uniqueid;

  request.type = FN_DATAINSERT_MSG;
  depth = (((float)rand() / (float)RAND_MAX) * (FN_DH - FN_DL)) + FN_DL;
  sprintf(request.field[0], "Depth=%x", depth);
  sprintf(request.field[1], "HopsToLive=%x", htl);
  sprintf(request.field[2], "DataLength=%x", total_len);
  sprintf(request.field[3], "Storable.Metadata-length=%d", metadata_len);
  strcpy(request.field[4], "Storable.PartSize=0");
  if (endtoend == FN_TWOFISH) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Twofish");
  } else if (endtoend == FN_RIJNDAEL) {
    strcpy(request.field[5], "Storable.Symmetric-cipher=Rijndael");
  }
  sprintf(request.field[6], "Storable.Signature=%s", sigstr);
  sprintf(request.field[7], "Storable.Public-key=%s", pubstr);
  if (strlen(key->docname) > 0) {
    SHA1(key->docname, strlen(key->docname), dnhash);
    for (i = 0; i < FN_HASH_BYTES; i++) {
      sprintf(&(dnhashstr[i*2]), "%.2x", dnhash[i]);
    }
    dnhashstr[i*2] = 0;
    sprintf(request.field[8], "Storable.Document-name=%s", dnhashstr);
    fieldnum = 9;
  } else {
    fieldnum = 8;
  }
  request.numfields=fieldnum;

  status = freenet_sendmsg(&(cstream->stream.connection), &request);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = freenet_init_outgoing_stream(&(cstream->stream));
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  cstream->htl = htl;
  cstream->stream.size = total_len;
  cstream->stream.pos = 0;
  cstream->stream.contentpos = 0;
  cstream->stream.current_part = 0;
  cstream->stream.last_part = 0;

  status = endtoend_init(cstream, key->enckey);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  header[0] = 0;
  header[1] = FN_KEY_BYTES;

  memcpy(&(header[2]), key->enckey, FN_KEY_BYTES);

  status = freenet_client_write_stream(cstream, header, FN_HEADER_SIZE);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  status = fseek(instream, 0, SEEK_SET);  /* rewind */
  if (status == -1) {
    retval = FNS_FSEEK_FAILED;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  if (retval != FNS_SUCCESS) {
    freenet_close_connection(&(cstream->stream.connection));
  }
  return retval;
}
