/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <time.h>

#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#endif

#include "uri.h"
#include "client_util.h"

/* On cygwin, we don't have timezone for some lame reason.  I found
   _timezone in time.h, maybe that will work!  As it turns out, I have
   to subtract the daylight savings offset (_daylight, also found by
   reading time.h) from that value. */
#ifdef __CYGWIN__
#define FN_TIMEZONE (_timezone - _daylight)
#else
#define FN_TIMEZONE timezone
#endif


long blstr2time(char *baseline);
int timediff();


time_t blstr2time (char *baseline)
{
    struct tm bline;

    /*strptime(baseline, "%Y%m%d%H%M%S", &bline);*/
    sscanf(baseline, "%4d%2d%2d%2d%2d%2d", &bline.tm_year, &bline.tm_mon,
      &bline.tm_mday, &bline.tm_hour, &bline.tm_min, &bline.tm_sec);
    bline.tm_year -= 1900;

    /* Is this always a valid way to adjust the timezone?  What if
       we're crossing a day (and month? year?) boundry? mktime's man
       page suggests that this is okay, but I'm not totally sure. */
    bline.tm_hour -= timediff();
    return mktime(&bline);
}


int timediff ()
{
    long t = time(NULL);

    /* We are calling localtime here for its side-effect of setting
       the timezone global appropriately.  I have no proof that this
       even happens on cygwin.  Please, just let me die. */
    localtime(&t);
    return (FN_TIMEZONE / 3600);
}


void time2blstr (time_t time, char *baseline)
{
    strftime(baseline, 16, "%Y%m%d%H%M%S", gmtime(&time));
    return;
}


/* I fucking hate dealing with strings in C. */
int freenet_parse_redirect (freenet_key *key, const char *metadata)
{
  char *uri;
  char *tokstr;
  char *token;
  char *ptrptr;
  int urilen;
  int status;
  int retval;
  int hasequal;
  time_t increment = 0;
  time_t baseline = 0;
  long timediff;
  char timestr[16];
  int hasincrement = 0;
  int hasbaseline = 0;
  char *val;
  time_t temp;

#ifdef DEBUG
  fprintf(stderr, "parsing for redirects:\n");
  fprintf(stderr, metadata);
#endif

  uri = NULL;

  tokstr = malloc(strlen(metadata)+1);
  if (tokstr == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  strncpy(tokstr, metadata, strlen(metadata));

  token = strtok_r(tokstr, "\n", &ptrptr);
  if (token == NULL) {
    retval = FNS_NOT_A_REDIRECT;
    goto end;
  }

  if (strncmp(token, "Redirect", 8) != 0) {
    retval = FNS_NOT_A_REDIRECT;
    goto end;
  }

  do {
    hasequal = 0;

    token = strtok_r(NULL, "\n", &ptrptr);
    if (token == NULL) {
      retval = FNS_INVALID_REDIRECT;
      goto end;
    }

    if (strchr(token, '=') != NULL) {
      hasequal = 1;
      if (strncmp(token, "increment=", 10) == 0) {
        increment = atol(&(token[10]));
        hasincrement = 1;
      } else if (strncmp(token, "baseline=", 9) == 0) {
        baseline = blstr2time(&(token[9]));
        if (baseline == -1) {
          retval = FNS_MKTIME_FAILED;
          goto end;
        }
        hasbaseline = 1;
      }
    }
  } while (hasequal == 1);

  if (strcmp(token, "End") != 0) {
    retval = FNS_INVALID_REDIRECT;
    goto end;
  }

  token = strtok_r(NULL, "\n", &ptrptr);
  if (token == NULL) {
    retval = FNS_INVALID_REDIRECT;
    goto end;
  }

  if ((hasincrement == 1) &&
      (hasbaseline == 1)) {

    /* find out how much time has elapsed since the baseline */
    timediff = time(NULL) - baseline;
    /* adjust down to the most recently passed increment */
    timediff -= timediff % increment;
    if ((timediff % increment) != 0) {
      return -timediff;
    }

    temp = baseline + timediff;

    time2blstr(temp, timestr);
    strcat(timestr, "-");
  } else {
    timestr[0] = 0;
  }

  urilen = strlen(token) + strlen(timestr);

  uri = malloc(urilen + 1);
  if (uri == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  if (strncmp(token, "freenet:KSK@", 12) == 0) {
    strncpy(uri, token, 12);
    strncat(uri, timestr, strlen(timestr));
    strncat(uri, &(token[12]), strlen(&(token[12])));
  } else if (strncmp(token, "freenet:SSK@", 12) == 0) {
    val = strchr(token, '/');
    if (val == NULL) {
      retval = FNS_INVALID_URI;
      goto end;
    }
    strncpy(uri, token, strlen(token) - strlen(val));
    strcat(uri, "/");
    strncat(uri, timestr, strlen(timestr));
    val++; /* advance val past the / */
    strncat(uri, val, strlen(val));
  } else {
    if (strlen(timestr) != 0) {
      retval = FNS_INVALID_REDIRECT;
      goto end;
    }
    strncpy(uri, token, urilen);
  }

  uri[urilen] = 0;

#ifdef DEBUG
  fprintf(stderr, "decided on redirect URI: %s\n", uri);
#endif

  status = freenet_parse_request_uri(key, uri);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:
  free(tokstr);
  free(uri);
  return retval;
}


int freenet_parse_mapfile (freenet_key *key, const char *metadata,
                           const char *docname)
{
  char *tokstr;
  char *token;
  char *ptrptr;
  int status;
  int retval;
  int hasequal;
  int hasdefault = 0;
  char uri[512];
  char default_docname[512];
  char final_docname[512];
  int doclen;
  int found;

#ifdef DEBUG
  fprintf(stderr, "parsing for mapfile:\n");
  fprintf(stderr, metadata);
#endif

  tokstr = malloc(strlen(metadata)+1);
  if (tokstr == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  strncpy(tokstr, metadata, strlen(metadata));

  token = strtok_r(tokstr, "\n", &ptrptr);
  if (token == NULL) {
    retval = FNS_NOT_A_MAPFILE;
    goto end;
  }

  if (strncmp(token, "Mapfile", 8) != 0) {
    retval = FNS_NOT_A_MAPFILE;
    goto end;
  }

  do {
    hasequal = 0;

    token = strtok_r(NULL, "\n", &ptrptr);
    if (token == NULL) {
      retval = FNS_INVALID_MAPFILE;
      goto end;
    }

    if (strchr(token, '=') != NULL) {
      hasequal = 1;
      if (strncmp(token, "default=", 8) == 0) {
        strncpy(default_docname, &(token[8]), 510);
        hasdefault = 1;
      }
    }
  } while (hasequal == 1);

  if (strcmp(token, "End") != 0) {
    retval = FNS_INVALID_MAPFILE;
    goto end;
  }

  if (docname[0] == 0) {
    if (hasdefault != 1) {
      retval = FNS_DOCNAME_NOT_FOUND;
      goto end;
    }

    strncpy(final_docname, default_docname, 510);
  } else {
    strncpy(final_docname, docname, 510);
  }

  doclen = strlen(final_docname);

  found = 0;
  do {
    token = strtok_r(NULL, "\n", &ptrptr);
    if (token != NULL) {
      if ((strncmp(token, final_docname, doclen) == 0)
          && (token[doclen] == '=')) {
        strncpy(uri, &(token[doclen+1]), 510);
        found = 1;
      }
    }
  } while ((found == 0) && (token != NULL));


  if (found != 1) {
    retval = FNS_DOCNAME_NOT_FOUND;
    goto end;
  } else {
#ifdef DEBUG
    fprintf(stderr, "decided on mapfile URI: %s\n", uri);
#endif

    status = freenet_parse_request_uri(key, uri);
    if (status != FNS_SUCCESS) {
      retval = status;
      goto end;
    }
  }

  retval = FNS_SUCCESS;

  end:
  free(tokstr);
  return retval;
}
