/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <string.h>

#include "protocol.h"
#include "client.h"
#include "twofish.h"

int e2e_twofish_init (freenet_client_stream *cstream, unsigned char *key)
{
  int status;
  unsigned char ov[FN_KEY_BYTES];

  cstream->tsys.fbpos = 0;

  status = twofish_setkey(&(cstream->tsys.ctx), key, FN_KEY_BYTES);
  if (status != 0) {
    return FNS_MAKEKEY_FAILED;
  }

  memset(cstream->tsys.fb, 0, FN_KEY_BYTES);

  twofish_encrypt(&(cstream->tsys.ctx), ov, cstream->tsys.fb);
  memcpy(cstream->tsys.fb, ov, FN_KEY_BYTES);

  return FNS_SUCCESS;
}


int e2e_twofish_encrypt (freenet_client_stream *cstream,
                         unsigned char *inbuffer,
                         unsigned char *outbuffer, int len)
{
  int i;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (cstream->tsys.fbpos >= FN_KEY_BYTES) {

      twofish_encrypt(&(cstream->tsys.ctx), ov, cstream->tsys.fb);
      memcpy(cstream->tsys.fb, ov, FN_KEY_BYTES);

      cstream->tsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ cstream->tsys.fb[cstream->tsys.fbpos];
    cstream->tsys.fb[cstream->tsys.fbpos] = outbuffer[i];

    cstream->tsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int e2e_twofish_decrypt (freenet_client_stream *cstream,
                         unsigned char *inbuffer,
                         unsigned char *outbuffer, int len)
{
  int i;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (cstream->tsys.fbpos >= FN_KEY_BYTES) {

      twofish_encrypt(&(cstream->tsys.ctx), ov, cstream->tsys.fb);
      memcpy(cstream->tsys.fb, ov, FN_KEY_BYTES);

      cstream->tsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ cstream->tsys.fb[cstream->tsys.fbpos];
    cstream->tsys.fb[cstream->tsys.fbpos] = inbuffer[i];

    cstream->tsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int rijndael_init (freenet_client_stream *cstream, unsigned char *key)
{
  int status;
  unsigned char ov[FN_KEY_BYTES];

  status = rijndael_makeKey(&(cstream->rsys.key), DIR_ENCRYPT, FN_KEY_BITS, key);
  if (status != TRUE) {
    return FNS_MAKEKEY_FAILED;
  }

  status = rijndael_cipherInit(&(cstream->rsys.cipher), MODE_ECB, NULL);
  if (status != TRUE) {
    return FNS_CIPHERINIT_FAILED;
  }

  memset(cstream->rsys.fb, 0, FN_KEY_BYTES);

  status = rijndael_blockEncrypt(&(cstream->rsys.cipher), &(cstream->rsys.key),
                                 cstream->rsys.fb, FN_KEY_BITS, ov);
  if (status < 1) {
    return FNS_ENCRYPT_FAILED;
  }
  memcpy(cstream->rsys.fb, ov, FN_KEY_BYTES);

  return FNS_SUCCESS;
}


int rijndael_encrypt (freenet_client_stream *cstream, unsigned char *inbuffer,
                      unsigned char *outbuffer, int len)
{
  int i, status;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (cstream->rsys.fbpos >= FN_KEY_BYTES) {

      status = rijndael_blockEncrypt(&(cstream->rsys.cipher), &(cstream->rsys.key),
                                     cstream->rsys.fb, FN_KEY_BITS, ov);
      if (status < 1) {
        return FNS_ENCRYPT_FAILED;
      }
      memcpy(cstream->rsys.fb, ov, FN_KEY_BYTES);
      cstream->rsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ cstream->rsys.fb[cstream->rsys.fbpos];
    cstream->rsys.fb[cstream->rsys.fbpos] = outbuffer[i];

    cstream->rsys.fbpos++;

  }

  return FNS_SUCCESS;
}


int rijndael_decrypt (freenet_client_stream *cstream, unsigned char *inbuffer,
                      unsigned char *outbuffer, int len)
{
  int i, status;
  unsigned char ov[FN_KEY_BYTES];

  for (i=0; i<len; i++) {

    if (cstream->rsys.fbpos >= FN_KEY_BYTES) {

      status = rijndael_blockEncrypt(&(cstream->rsys.cipher), &(cstream->rsys.key),
                                     cstream->rsys.fb, FN_KEY_BITS, ov);
      if (status < 1) {
        return FNS_ENCRYPT_FAILED;
      }
      memcpy(cstream->rsys.fb, ov, FN_KEY_BYTES);
      cstream->rsys.fbpos = 0;

    }


    outbuffer[i] = inbuffer[i] ^ cstream->rsys.fb[cstream->rsys.fbpos];
    cstream->rsys.fb[cstream->rsys.fbpos] = inbuffer[i];

    cstream->rsys.fbpos++;

  }

  return FNS_SUCCESS;
}


__inline__ int endtoend_init (freenet_client_stream *cstream,
                              unsigned char *key)
{
  if (cstream->endtoend == FN_TWOFISH) {
    return e2e_twofish_init(cstream, key);
  } else if (cstream->endtoend == FN_RIJNDAEL) {
    return rijndael_init(cstream, key);
  } else {
    return FNS_INVALID_ENDTOEND;
  }
}


__inline__ int endtoend_encrypt (freenet_client_stream *cstream,
                                 unsigned char *inbuffer,
                                 unsigned char *outbuffer, int len)
{
  if (cstream->endtoend == FN_TWOFISH) {
    return e2e_twofish_encrypt(cstream, inbuffer, outbuffer, len);
  } else if (cstream->endtoend == FN_RIJNDAEL) {
    return rijndael_encrypt(cstream, inbuffer, outbuffer, len);
  } else {
    return FNS_INVALID_E2E;
  }
}


__inline__ int endtoend_decrypt (freenet_client_stream *cstream,
                                 unsigned char *inbuffer,
                                 unsigned char *outbuffer, int len)
{
  if (cstream->endtoend == FN_TWOFISH) {
    return e2e_twofish_decrypt(cstream, inbuffer, outbuffer, len);
  } else if (cstream->endtoend == FN_RIJNDAEL) {
    return rijndael_decrypt(cstream, inbuffer, outbuffer, len);
  } else {
    return FNS_INVALID_E2E;
  }
}

