/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef __FN_CLIENT_H__
#define __FN_CLIENT_H__

#include <stdlib.h>
#include <sys/types.h>

#include <openssl/sha.h>

#include <protocol.h>
#include <rijndael-api-fst.h>
#include <twofish.h>

#include "protocol.h"
#include "stream.h"

#define FN_HEADER_SIZE (FN_KEY_BYTES + 2)

/* FN_SVK_MAXSIZE == 2 ** FN_SVK_PARTSIZE_BYTE */
#define FN_SVK_MAXSIZE 65536
#define FN_SVK_PARTSIZE_BYTE 0x10

#define FN_MIN_URI_LEN 12

#define FN_TWOFISH  1
#define FN_RIJNDAEL 2

typedef struct {
  int fbpos;
  unsigned char fb[FN_KEY_BYTES];
  TWOFISH_context ctx;
} freenet_twofish_system;

typedef struct {

  freenet_stream stream;

  /* common for endtoend */
  freenet_rijndael_system rsys;
  freenet_twofish_system tsys;
  int endtoend;

  /* request */
  int metadata_len;
  int data_len;
  unsigned char datasource[512];

  /* insert */
  u_int64_t uniqueid;
  int htl;

} freenet_client_stream;

int freenet_client_read_stream(freenet_client_stream *state,
                               unsigned char *buffer, int len);

int freenet_client_write_stream(freenet_client_stream *state,
                                unsigned char *buffer, int len);

int freenet_generate_KSK (const unsigned char *keystr, freenet_key *key);

int freenet_generate_SVK (unsigned char *priv, const unsigned char *docname,
                          freenet_key *key);

int freenet_request_stream (freenet_client_stream *state, char *address,
                            int port, int htl, freenet_key *key);

int freenet_request_buffer (freenet_client_stream *state, char *address,
                            int port, int htl, freenet_key *key,
                            unsigned char *buffer, int maxlen);

int freenet_generate_CHK_buffer (freenet_client_stream *state,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, unsigned char *fbuf,
                                 int len);

int freenet_generate_CHK_stream (freenet_client_stream *state,
                                 int endtoend, freenet_key *key,
                                 int *num_parts, int *len,
                                 FILE *instream);

int freenet_insert_CHK_buffer (freenet_client_stream *state, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *fbuf, int len);

int freenet_insert_CHK_stream (freenet_client_stream *state, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream);

int freenet_insert_SVK_buffer (freenet_client_stream *state, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               unsigned char *buffer, int len);

int freenet_insert_SVK_stream (freenet_client_stream *state, char *address,
                               int port, int htl, int endtoend,
                               int metadata_len, freenet_key *key,
                               FILE *instream);

#endif /* __FN_CLIENT_H__ */
