/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <malloc.h>
#include <sys/socket.h>
#include <netdb.h>
#include <string.h>
#include <stddef.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <ctype.h>
#include <time.h>

#include <openssl/sha.h>
#include <openssl/dh.h>
#include <openssl/rand.h>

#include "protocol.h"
#include "util.h"

int readall(int sock, unsigned char *buffer, int len);

int writeall(int sock, unsigned char *buffer, int len);

int open_connection (freenet_connection *connection, char *hostaddress,
                     int port);

int get_connection(freenet_connection *connection, u_short port,
                   time_t timeout);

int diffie_hellman (freenet_connection *connection, unsigned char *k,
                    int len);

int start_ciphers (freenet_connection *connection, unsigned char *key);

int request_handshake(freenet_connection *connection);

int reply_handshake (freenet_connection *connection, freenet_message *request);

int writeline(freenet_connection *connection, const char *string);

int readto(freenet_connection *connection, char *buffer, int buflen, int term);

int internal_recvmsg (freenet_connection *connection,
                      freenet_message *message);


static const struct {
        const char *str;
        int type;
} msgtypes[] = {
        { "StoreData", FN_STOREDATA_MSG },
        { "HandshakeRequest", FN_HANDSHAKEREQUEST_MSG },
        { "HandshakeReply", FN_HANDSHAKEREPLY_MSG },
        { "QueryRestarted", FN_QUERYRESTARTED_MSG },
        { "DataRequest", FN_DATAREQUEST_MSG },
        { "DataReply", FN_DATAREPLY_MSG },
        { "TimedOut", FN_TIMEDOUT_MSG },
        { "RequestFailed", FN_REQUESTFAILED_MSG },
        { "InsertRequest", FN_INSERTREQUEST_MSG },
        { "InsertReply", FN_INSERTREPLY_MSG },
        { "DataInsert", FN_DATAINSERT_MSG },
        { NULL, FN_UNKNOWN_MSG }
};


static struct {
  const char *str;
  int status;
} errstrs[] = {
  { "Success", FNS_SUCCESS },
  { "Invalid port", FNS_INVALID_PORT },
  { "Invalud address", FNS_INVALID_ADDY },
  { "connect(2) failed", FNS_CONNECT_FAILED },
  { "socket(2) failed", FNS_SOCKET_FAILED },
  { "bind(2) failed", FNS_BIND_FAILED },
  { "listen(2) failed", FNS_LISTEN_FAILED },
  { "accept(2) failed", FNS_LISTEN_FAILED },
  { "write(2) failed", FNS_WRITE_FAILED },
  { "read(2) failed", FNS_READ_FAILED },
  { "malloc(3) failed", FNS_MALLOC_FAILED },
  { "Invalid MPint", FNS_INVALID_MPINT },
  { "Invalid DHK", FNS_INVALID_DHK },
  { "mhash failed", FNS_MHASH_FAILED },
  { "Hash too small", FNS_HASH_TOO_SMALL },
  { "makekey failed", FNS_MAKEKEY_FAILED },
  { "Cipher init failed", FNS_CIPHERINIT_FAILED },
  { "Encrypt failed", FNS_ENCRYPT_FAILED },
  { "Invalid reply", FNS_INVALID_REPLY },
  { "Invalid version", FNS_INVALID_VERSION },
  { "Wrong ID", FNS_WRONG_ID },
  { "Random error", FNS_RANDOM_ERROR },
  { "open(2) failed", FNS_OPEN_FAILED },
  { "Overread", FNS_OVERREAD },
  { "Invalid request", FNS_INVALID_REQUEST },
  { "Connection gone", FNS_CONNECTION_GONE },
  { "End of file", FNS_EOF },
  { "Overwrite", FNS_OVERWRITE },
  { "Invalid data", FNS_INVALID_DATA },
  { "Unexpected key length", FNS_UNEXPECTED_KEY_LENGTH },
  { "Bad key", FNS_BAD_KEY },
  { "Zero data length", FNS_ZERO_DATALEN },
  { "Timeout", FNS_TIMEOUT },
  { "Request failed", FNS_REQUEST_FAILED },
  { "Unexpected message", FNS_UNEXPECTED_MESSAGE },
  { "Buffer too small", FNS_BUFFER_TOO_SMALL },
  { "Invalid end-to-end", FNS_INVALID_ENDTOEND },
  { "Invalid metadata length", FNS_INVALID_METADATA_LEN },
  { "BN failed", FNS_BN_FAILED },
  { "DH failed", FNS_DH_FAILED },
  { "BAD_CTRL", FNS_BAD_CTRL },
  { "Bad hash", FNS_BAD_HASH },
  { "Invalid URI", FNS_INVALID_URI },
  { "Unknown keytype", FNS_UNKNOWN_KEYTYPE },
  { "Invalid search key", FNS_INVALID_SEARCHKEY },
  { "Invalid enc key", FNS_INVALID_ENCKEY },
  { "DSA failed", FNS_DSA_FAILED },
  { "Invalid private key", FNS_INVALID_PRIVATE_KEY },
  { "fseek(3) failed", FNS_FSEEK_FAILED },
  { "Data reply", FNS_DATA_REPLY },
  { "SVK too large", FNS_SVK_TOO_LARGE },
  { "Invalid signature", FNS_INVALID_SIG },
  { "Invalid public key", FNS_INVALID_PUBKEY },
  { "Invalid end2end", FNS_INVALID_E2E },
  { "Unknown message type", FNS_UNKNOWN_MSG_TYPE },
  { "Bad trailing field", FNS_BAD_TRAILING_FIELD },
  { "Not a redirect", FNS_NOT_A_REDIRECT },
  { "Invalid redirect", FNS_INVALID_REDIRECT },
  { "Invalid 64-bit hexlen", FNS_INVALID_64BIT_HEXLEN },
  { "Invalid hex character", FNS_INVALID_HEX_CHAR },
  { "fcntl(2) failed", FNS_FCNTL_FAILED },
  { "accept(2) timed out", FNS_ACCEPT_TIMEDOUT },
  { "mktime(3) failed", FNS_MKTIME_FAILED },
  { "Not a mapfile", FNS_NOT_A_MAPFILE },
  { "Invalid mapfile", FNS_INVALID_MAPFILE },
  { "Docname not found", FNS_DOCNAME_NOT_FOUND },
  { NULL, 0 }
};


/* Convert a hex string to a 64-bit unsigned int.  I hate my life. */
__inline__ int hex2uint64(const char *string, u_int64_t *val)
{
  int i;
  int len;
  char c;
  u_int64_t mult[16] = {
    0x0000000000000001,
    0x0000000000000010,
    0x0000000000000100,
    0x0000000000001000,
    0x0000000000010000,
    0x0000000000100000,
    0x0000000001000000,
    0x0000000010000000,
    0x0000000100000000,
    0x0000001000000000,
    0x0000010000000000,
    0x0000100000000000,
    0x0001000000000000,
    0x0010000000000000,
    0x0100000000000000,
    0x1000000000000000};

  *val = 0;

  len = strlen(string);

  if ((len < 0) || (len > 16)) {
    return FNS_INVALID_64BIT_HEXLEN;
  }

  for (i=0; i<len; i++) {
    c = tolower(string[(len-1)-i]);
    if ((c >= 'a') && (c <= 'f')) {
      *val += (((c - 'a') + 10) * mult[i]);
    } else if ((c >= '0') && (c <= '9')) {
      *val += ((c - '0') * mult[i]);
    } else {
      return FNS_INVALID_HEX_CHAR;
    }
  }

  return FNS_SUCCESS;

}


__inline__ int string_to_msgtype (const char *typestr)
{
  int result = FN_UNKNOWN_MSG;
  int i;

  for(i = 0; msgtypes[i].str != NULL; i++) {
    if(strcmp(msgtypes[i].str, typestr) == 0) {
      result=msgtypes[i].type;
      break;
    }
  }

  return result;
}


__inline__ int msgtype_to_string (int msgtype, char *typestr)
{
  int i;

  for(i = 0; msgtypes[i].str != NULL; i++) {
    if(msgtypes[i].type == msgtype) {
      strcpy(typestr, msgtypes[i].str);
      return FNS_SUCCESS;
    }
  }
  return FNS_UNKNOWN_MSG_TYPE;
}


/* just open a tcp connection */
int open_connection (freenet_connection *connection, char *hostaddress,
                     int port)
{
  struct in_addr addr;
  int connected_socket, connected;
  struct sockaddr_in address;
  struct hostent *host;

  if ( (port < 1) || (port > 65535) ) {
    return FNS_INVALID_PORT;
  }

  addr.s_addr = inet_addr(hostaddress);
  if (addr.s_addr == -1) {
    host = gethostbyname(hostaddress);
    if (host == NULL) {
      return FNS_INVALID_ADDY;
    }
    addr.s_addr = ((struct in_addr *) *host->h_addr_list)->s_addr;
  }

  memset((char *) &address, 0, sizeof(address));
  address.sin_family = AF_INET;
  address.sin_port = htons(port);
  address.sin_addr.s_addr = addr.s_addr;

  connected_socket = socket(AF_INET, SOCK_STREAM, 0);

  connected = connect(connected_socket, (struct sockaddr *) &address,
    sizeof(address));
  if (connected < 0) {
    return FNS_CONNECT_FAILED;
  }

  connection->socket = connected_socket;

  return FNS_SUCCESS;
}


/* wait for a tcp connection */
int get_connection(freenet_connection *connection, u_short port,
                   time_t timeout)
{
  struct sockaddr_in address;
  int listening_socket;
  int connected_socket = -1;
  int reuse_addr = 1;
  struct sockaddr_in incoming;
  int b;
  int flags;
  int status;
  time_t starttime;

  /* Setup internet address information.
     This is used with the bind() call */
  memset((char *) &address, 0, sizeof(address));
  address.sin_family = AF_INET;
  address.sin_port = htons(port);
  address.sin_addr.s_addr = htonl(INADDR_ANY);

  listening_socket = socket(AF_INET, SOCK_STREAM, 0);
  if (listening_socket < 0) {
    return FNS_SOCKET_FAILED;
  }

  setsockopt(listening_socket, SOL_SOCKET, SO_REUSEADDR, (char *)&reuse_addr,
    sizeof(reuse_addr));

  if (bind(listening_socket, (struct sockaddr *)&address,
    sizeof(address)) < 0) {
    close(listening_socket);
    return FNS_BIND_FAILED;
  }

  if (listen(listening_socket, 1) < 0) {
    return FNS_LISTEN_FAILED;
  }

  if (timeout != 0) {
    flags = fcntl(listening_socket, F_GETFL);
    if (flags == -1) {
      return FNS_FCNTL_FAILED;
    }

    status = fcntl(listening_socket, F_SETFL, flags | O_NONBLOCK);
    if (flags == -1) {
      return FNS_FCNTL_FAILED;
    }
  }

  starttime = time(NULL);
  while(connected_socket < 0) {
    b = sizeof(incoming);
    connected_socket = accept(listening_socket,
          (struct sockaddr *)&incoming, &b);
    if (connected_socket < 0) {
      /* Either a real error occured, or blocking was interrupted for
         some reason.  Only abort execution if a real error occured. */
      if (errno == EINTR) {
        continue;    /* don't quit - do the accept again */
      } else if ((errno == EAGAIN) && (timeout != 0)) {
        if ((time(NULL) - starttime) < timeout) {
          sleep(100);
          continue;
        } else {
          close(listening_socket);
          return FNS_ACCEPT_TIMEDOUT;
        }
      } else {
        close(listening_socket);
        return FNS_ACCEPT_FAILED;
      }
    }

    status = close(listening_socket); /* Close this socket */
    if (status != 0) {
      return FNS_CLOSE_FAILED;
    }

  }

  connection->socket = connected_socket;

  return FNS_SUCCESS;
}


int diffie_hellman (freenet_connection *connection, unsigned char *k, int len)
{
  int16_t mpilen;
  unsigned char *y = NULL;
  unsigned char *ybuf = NULL;
  unsigned char *xbuf = NULL;
  DH *dhsys;
  BIGNUM *pub_key = NULL;
  int ylen, klen;
  int status;
  int retval;

  dhsys = DH_new();
  if (dhsys == NULL) {
    return FNS_MALLOC_FAILED;
  }

  /* g and n are fixed for all of freenet */
  status = BN_hex2bn(&(dhsys->g), FN_DH_G);
  if (status != strlen(FN_DH_G)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  status = BN_hex2bn(&(dhsys->p), FN_DH_N);
  if (status != strlen(FN_DH_N)) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  /* generate a random y */
  status = DH_generate_key(dhsys);
  if (status != 1) {
    retval = FNS_DH_FAILED;
    goto end;
  }

  ylen = BN_num_bytes(dhsys->pub_key);

  y = malloc(ylen);
  if (y == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = BN_bn2bin(dhsys->pub_key, y);
  if (status != ylen) {
    retval = FNS_BN_FAILED;
    goto end;
  }

  ybuf = malloc(ylen + 2);
  if (ybuf == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = raw_to_bagbiting_freenet_mpi(y, ylen, ybuf);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }
  status = writeall(connection->socket, ybuf, ylen + 2);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }


  status = readall(connection->socket, (char *)&mpilen, (size_t)2);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  mpilen = htons(mpilen);
  mpilen = (mpilen+7)/8;  /* convert to bytes */

  xbuf = malloc(mpilen);
  if (xbuf == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = readall(connection->socket, xbuf, (size_t)mpilen);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  pub_key = BN_bin2bn(xbuf, mpilen, NULL);
  if (pub_key == NULL) {
    retval = FNS_BN_FAILED;
    goto end;
  }


  if (len != DH_size(dhsys)) {
    retval = FNS_DH_FAILED;
    goto end;
  }

  klen = DH_compute_key(k, pub_key, dhsys);
  if (klen == -1) {
    retval = FNS_DH_FAILED;
    goto end;
  }

  if (len != klen) {
    retval = FNS_DH_FAILED;
    goto end;
  }

  retval = FNS_SUCCESS;


 end:

  DH_free(dhsys);
  BN_free(pub_key);
  free(y);
  free(ybuf);
  free(xbuf);
  return retval;
}


/* initialize the incoming and outgoing rijndael ciphers */
int start_ciphers (freenet_connection *connection, unsigned char *key)
{
  unsigned char oiv[FN_KEY_BYTES];
  unsigned char iiv[FN_KEY_BYTES];
  int status;

  status = generate_random(oiv, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = writeall(connection->socket, oiv, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = readall(connection->socket, iiv, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = rijndael_makeKey(&(connection->sendsys.key), DIR_ENCRYPT, FN_KEY_BITS,
                            key);
  if (status != TRUE) {
    return FNS_MAKEKEY_FAILED;
  }
  status = rijndael_makeKey(&(connection->recvsys.key), DIR_ENCRYPT, FN_KEY_BITS,
                            key);
  if (status != TRUE) {
    return FNS_MAKEKEY_FAILED;
  }

  status = rijndael_cipherInit(&(connection->sendsys.cipher), MODE_ECB, NULL);
  if (status != TRUE) {
    return FNS_CIPHERINIT_FAILED;
  }
  status = rijndael_cipherInit(&(connection->recvsys.cipher), MODE_ECB, NULL);
  if (status != TRUE) {
    return FNS_CIPHERINIT_FAILED;
  }

  status = rijndael_blockEncrypt(&(connection->sendsys.cipher), &(connection->sendsys.key),
                                 oiv, FN_KEY_BITS, connection->sendsys.fb);
  if (status < 1) {
    return FNS_ENCRYPT_FAILED;
  }
  status = rijndael_blockEncrypt(&(connection->recvsys.cipher), &(connection->recvsys.key),
                                 iiv, FN_KEY_BITS, connection->recvsys.fb);
  if (status < 1) {
    return FNS_ENCRYPT_FAILED;
  }

  connection->sendsys.fbpos = 0;
  connection->recvsys.fbpos = 0;

  return FNS_SUCCESS;
}


/* initiate a handshake (and complete it) */
int request_handshake (freenet_connection *connection)
{
  freenet_message message;
  float version;
  freenet_message reply;
  int i, status;


  status = generate_random((unsigned char *)&(message.uniqueid),
                           sizeof(u_int64_t));
  if (status != FNS_SUCCESS) {
    return status;
  }

  message.type = FN_HANDSHAKEREQUEST_MSG;
  strcpy(message.field[0], "Depth=1");
  strcpy(message.field[1], "HopsToLive=1");
  strcpy(message.field[2], "KeepAlive=true");
  message.numfields = 3;

  status = freenet_sendmsg(connection, &message);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = internal_recvmsg(connection, &reply);
  if (status != FNS_SUCCESS) {
    return status;
  }

  if (reply.type != FN_HANDSHAKEREPLY_MSG) {
    return FNS_INVALID_REPLY;
  }

  if (reply.uniqueid != message.uniqueid) {
    return FNS_WRONG_ID;
  }

  for (i = 0; i < reply.numfields; i++) {
    if (strncmp(reply.field[i], "Version=", 8) == 0) {
      sscanf(reply.field[i], "Version=%f", &version);
      if (version<FN_VERSION) {
        return FNS_INVALID_VERSION;
      }
    }
  }

  return FNS_SUCCESS;
}


/* handle a handshake */
int reply_handshake (freenet_connection *connection, freenet_message *request)
{
  freenet_message reply;
  float version;
  int i, status;

  if (request->type != FN_HANDSHAKEREQUEST_MSG) {
    return FNS_INVALID_REQUEST;
  }

  for (i = 0; i < request->numfields; i++) {
    if (strncmp(request->field[i], "Version=", 8) == 0) {
      sscanf(request->field[i], "Version=%f", &version);
      if (version > FN_VERSION) {
        return FNS_INVALID_VERSION;
      }
    }
  }

  reply.type = FN_HANDSHAKEREPLY_MSG;
  reply.uniqueid = request->uniqueid;
  sprintf(reply.field[0], "Version=%.3f", FN_VERSION);
  sprintf(reply.field[1], "Depth=1");
  sprintf(reply.field[2], "HopsToLive=1");
  sprintf(reply.field[3], "Revision=%.3f", FN_REVISION);
  sprintf(reply.field[4], "Build=%d", FN_BUILD);
  reply.numfields = 5;

  status = freenet_sendmsg(connection, &reply);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_senddata(freenet_connection *connection, const char *buffer,
                     int len)
{
  unsigned char *outbuffer;
  unsigned char newsfb[FN_KEY_BYTES];
  int i, status, retval;

  outbuffer = malloc(len);
  if (outbuffer == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }


  for (i = 0; i < len; i++) {
    if (connection->sendsys.fbpos >= FN_KEY_BYTES) {
      status = rijndael_blockEncrypt(&(connection->sendsys.cipher),
                                     &(connection->sendsys.key), connection->sendsys.fb,
                                     FN_KEY_BITS, newsfb);
      if (status < 1) {
        retval = FNS_ENCRYPT_FAILED;
        goto end;
      }
      memcpy(connection->sendsys.fb, newsfb, FN_KEY_BYTES);
      connection->sendsys.fbpos = 0;
    }

    /* combine the cypher and the plaintext to get the cyphertext */
    outbuffer[i] = buffer[i] ^ connection->sendsys.fb[connection->sendsys.fbpos];
    /* put the cyphertext into the feedback buffer */
    connection->sendsys.fb[connection->sendsys.fbpos] = outbuffer[i];

    connection->sendsys.fbpos++;
  }

  status = writeall(connection->socket, outbuffer, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  retval = FNS_SUCCESS;

 end:

  free(outbuffer);
  return retval;
}


int writeline(freenet_connection *connection, const char *string)
{
  int status, len;

  len = strlen(string);

  status = freenet_senddata(connection, string, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_senddata(connection, "\n", 1);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}



int freenet_sendmsg (freenet_connection *connection, freenet_message *message)
{
  int i, status;
  unsigned char typeline[512];
  unsigned char uniqline[512];
  u_int32_t luniqueid, runiqueid;

  status = msgtype_to_string(message->type, typeline);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = writeline(connection, typeline);
  if (status != FNS_SUCCESS) {
    return status;
  }

  /* sprintf(uniqline, "UniqueID=%Lx", message->uniqueid); */
  /* all this shifting and weird formatting is here to get the effect of
     "%Lx" without the GNU extension for long long "L" */
  luniqueid = (message->uniqueid) / 0x0000000100000000;
  runiqueid = ((message->uniqueid) * 0x0000000100000000) / 0x0000000100000000;
  sprintf(uniqline, "UniqueID=%lx%.8lx", luniqueid, runiqueid);

  status = writeline(connection, uniqline);
  if (status != FNS_SUCCESS) {
    return status;
  }

#ifdef DEBUG
  fprintf(stderr, "sending = \n");
  fprintf(stderr, "%s\n", typeline);
  fprintf(stderr, "%s\n", uniqline);
  for (i = 0; i < message->numfields; i++) {
    fprintf(stderr, "%s\n", message->field[i]);
  }
#endif

  for (i = 0; i < message->numfields; i++) {
    status = writeline(connection, message->field[i]);
    if (status != FNS_SUCCESS) {
      return status;
    }
  }

  if ((message->type != FN_DATAREPLY_MSG) &&
      (message->type != FN_DATAINSERT_MSG)) {
    status = writeline(connection, "EndMessage");
    if (status != FNS_SUCCESS) {
      return status;
    }
  } else {
    status = writeline(connection, "Data");
    if (status != FNS_SUCCESS) {
      return status;
    }
  }

  return FNS_SUCCESS;
}


int freenet_readdata(freenet_connection *connection, char *buffer, int len)
{
  unsigned char *inbuffer;
  unsigned char newrfb[FN_KEY_BYTES];
  int i, status, retval;

  inbuffer = malloc(len);
  if (inbuffer == NULL) {
    retval = FNS_MALLOC_FAILED;
    goto end;
  }

  status = readall(connection->socket, inbuffer, len);
  if (status != FNS_SUCCESS) {
    retval = status;
    goto end;
  }

  for (i = 0; i < len; i++) {

    if (connection->recvsys.fbpos >= FN_KEY_BYTES) {
      status = rijndael_blockEncrypt(&(connection->recvsys.cipher),
                                     &(connection->recvsys.key), connection->recvsys.fb,
                                     FN_KEY_BITS, newrfb);
      if (status < 1) {
        retval = FNS_ENCRYPT_FAILED;
        goto end;
      }
      memcpy(connection->recvsys.fb, newrfb, FN_KEY_BYTES);
      connection->recvsys.fbpos = 0;
    }

    /* combine the cyphertext and the cypher to get the plaintext */
    buffer[i] = inbuffer[i] ^ connection->recvsys.fb[connection->recvsys.fbpos];
    /* put the cyphertext into the feedback buffer */
    connection->recvsys.fb[connection->recvsys.fbpos] = inbuffer[i];

    connection->recvsys.fbpos++;

  }

  retval = FNS_SUCCESS;


 end:

  free(inbuffer);
  return retval;
}


int readto(freenet_connection *connection, char *buffer, int buflen, int term)
{
  int i, status;
  char in;

  i = 0;
  do {

    status = freenet_readdata(connection, &in, 1);
    if (status != FNS_SUCCESS) {
      return status;
    }

    buffer[i] = in;
    i++;

  } while ((buffer[i-1] != term) && (i < buflen));

  i--; /* back up to the token terminator */

  buffer[i] = 0;  /* replace the token terminator with a string terminator */

  return FNS_SUCCESS;
}


int internal_recvmsg (freenet_connection *connection, freenet_message *message)
{
  int status;
  unsigned char typeline[512];
  unsigned char line[512];

#ifdef DEBUG
  int i;
  u_int32_t luniqueid, runiqueid;
#endif

  message->numfields = 0;
  status = readto(connection, typeline, 510, '\n');
  if (status == FNS_EOF) {
    return FNS_CONNECTION_GONE;
  } else if (status != FNS_SUCCESS) {
    return status;
  }

  message->type = string_to_msgtype(typeline);

  do {

    status = readto(connection, line, 510, '\n');
    if (status != FNS_SUCCESS) {
      return status;
    }

    if (strncmp(line, "UniqueID=", 9) == 0) {
      /* sscanf(&(line[9]), "%Lx", &(message->uniqueid)); */
      status = hex2uint64(&(line[9]), &(message->uniqueid));
      if (status != FNS_SUCCESS) {
        return status;
      }
    } else if (strchr(line, '=') != NULL) {
      strncpy(message->field[message->numfields], line, 510);
      message->numfields++;
    }

  } while (strchr(line, '=') != NULL);


  if ((message->type != FN_DATAREPLY_MSG) &&
      (message->type != FN_DATAINSERT_MSG)) {
    if (strcmp(line, "EndMessage") != 0) {
      return FNS_BAD_TRAILING_FIELD;
    }
  } else {
    if (strcmp(line, "Data") != 0) {
      return FNS_BAD_TRAILING_FIELD;
    }
  }


#ifdef DEBUG
  fprintf(stderr, "received = \n");
  fprintf(stderr, "%s\n", typeline);
  /* fprintf(stderr, "UniqueID=%Lx", message->uniqueid); */
  /* all this shifting and weird formatting is here to get the effect of
     "%Lx" without the GNU extension for long long "L" */
  luniqueid = (message->uniqueid) / 0x0000000100000000;
  runiqueid = ((message->uniqueid) * 0x0000000100000000) / 0x0000000100000000;
  fprintf(stderr, "UniqueID=%lx%.8lx\n", luniqueid, runiqueid);
  for (i = 0; i < message->numfields; i++) {
    fprintf(stderr, "%s\n", message->field[i]);
  }
#endif

  return FNS_SUCCESS;
}


int freenet_recvmsg (freenet_connection *connection, freenet_message *message)
{
  int status;

  status = internal_recvmsg(connection, message);
  if ((status == FNS_CONNECTION_GONE) && (connection->expect_callback)) {
    status = close(connection->socket);
    if (status != 0) {
      return FNS_CLOSE_FAILED;
    }
    status = freenet_get_connection(connection, 6666);
    if (status != FNS_SUCCESS) {
      return status;
    }
    status = internal_recvmsg(connection, message);
    if (status != FNS_SUCCESS) {
      return status;
    }
  } else if (status != FNS_SUCCESS) {
    return status;
  }

  if (message->type == FN_HANDSHAKEREQUEST_MSG) {
    status = reply_handshake(connection, message);
    if (status != FNS_SUCCESS) {
      return status;
    }

    status = internal_recvmsg(connection, message);
    if (status != FNS_SUCCESS) {
      return status;
    }
  }

  return FNS_SUCCESS;
}


int freenet_connect (freenet_connection *connection, char *hostaddress,
                     int port)
{
  int status;

  connection->expect_callback = FN_TRUE;
  connection->timeout = 30;

  status = open_connection(connection, hostaddress, port);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_auth_connection(connection);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = request_handshake(connection);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_get_connection (freenet_connection *connection, u_short port)
{
  int status;

  connection->expect_callback = FN_TRUE;

  status = get_connection(connection, port, connection->timeout);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = freenet_auth_connection(connection);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_auth_connection (freenet_connection *connection)
{
  unsigned char k[FN_DHK_LEN];
  unsigned char bfmpik[130];
  unsigned char key[FN_KEY_BYTES];
  int status;

  status = diffie_hellman(connection, k, FN_DHK_LEN);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = raw_to_bagbiting_freenet_mpi(k, FN_DHK_LEN, bfmpik);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = keygen(bfmpik, FN_DHK_MPI_LEN, key, FN_KEY_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  status = start_ciphers(connection, key);
  if (status != FNS_SUCCESS) {
    return status;
  }

  return FNS_SUCCESS;
}


int freenet_close_connection (freenet_connection *connection)
{
  int status;

  status = close(connection->socket);
  if (status != 0) {
    return FNS_CLOSE_FAILED;
  }

  return FNS_SUCCESS;
}


const char *freenet_strerror(int status)
{
  int i;

  for(i = 0; errstrs[i].str != NULL; i++) {
    if(errstrs[i].status == status) {
      return errstrs[i].str;
    }
  }

  return "Unknown status code";
}
