/*  libfreenet
 *  Copyright 2001 Steven Hazel <sah@thalassocracy.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <string.h>
#include <openssl/sha.h>
#include <openssl/dsa.h>

#include "protocol.h"
#include "util.h"
#include "stream.h"

int stream_read(freenet_stream *stream,
                unsigned char *buffer, int len)
{
  int status;

  status = freenet_readdata(&(stream->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  stream->pos += len;

  return FNS_SUCCESS;
}


int stream_readdata(freenet_stream *stream,
                    unsigned char *buffer, int len)
{
  int status;

  if (stream->pos + len > stream->size) {
    return FNS_OVERREAD;
  }

  status = freenet_readdata(&(stream->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  SHA1_Update(&(stream->hash_handle), buffer, len);

  stream->pos += len;
  stream->contentpos += len;

  return FNS_SUCCESS;
}


int stream_read_trailer(freenet_stream *stream)
{
  int status;

  status = stream_read(stream, stream->hash[stream->current_part + 1],
                       FN_HASH_BYTES);
  if (status != FNS_SUCCESS) {
    return status;
  }

  SHA1_Update(&(stream->hash_handle), stream->hash[stream->current_part + 1],
              FN_HASH_BYTES);

  return FNS_SUCCESS;
}


int stream_read_control(freenet_stream *stream)
{
  int status;
  unsigned char ctrl;

  status = stream_read(stream, &ctrl, 1);
  if (status != FNS_SUCCESS) {
    return status;
  }
  if (ctrl != FN_CB_OK) {
    return FNS_BAD_CTRL;
  }

  return FNS_SUCCESS;
}


int stream_write(freenet_stream *stream,
                 unsigned char *buffer, int len)
{
  int status;

  status = freenet_senddata(&(stream->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  stream->pos += len;

  return FNS_SUCCESS;
}


int stream_writedata(freenet_stream *stream,
                     unsigned char *buffer, int len)
{
  int status;

  if (stream->pos + len > stream->size) {
     return FNS_OVERWRITE;
  }

  status = freenet_senddata(&(stream->connection), buffer, len);
  if (status != FNS_SUCCESS) {
    return status;
  }

  stream->pos += len;
  stream->contentpos += len;

  return FNS_SUCCESS;
}


int stream_write_trailer(freenet_stream *stream)
{
  int status;

  status = stream_write(stream, stream->hash[stream->current_part + 1],
                        FN_HASH_BYTES);
  return status;
}


/* read len bytes from a freenet raw (encrypted) stream into buffer */
int freenet_read_stream(freenet_stream *stream, unsigned char *buffer, int len)
{
  int status;
  int distance;
  int partdata;
  int bufpos = 0;
  unsigned char temp_hash[FN_HASH_BYTES];

  if (stream->direction != FN_INCOMING_STREAM) {
    return FNS_BAD_STREAM_DIRECTION;
  }

  if (stream->pos + len > stream->size) {
    return FNS_OVERREAD;
  }

  partdata = stream->partsize - FN_HASH_BYTES;

  distance = partdata - (stream->contentpos % partdata);

  while ((len >= distance) &&
         (stream->current_part != stream->last_part)) {

    if (distance > 0) {
      status = stream_readdata(stream, &buffer[bufpos], distance);
      if (status != FNS_SUCCESS) {
        return status;
      }

      bufpos += distance;
      len -= distance;
    }

    status = stream_read_trailer(stream);
    if (status != FNS_SUCCESS) {
      return status;
    }

    SHA1_Final(temp_hash, &(stream->hash_handle));
    if (memcmp(stream->hash[stream->current_part], temp_hash, FN_HASH_BYTES)
        != 0) {
      return FNS_BAD_HASH;
    }

    SHA1_Init(&(stream->hash_handle));

    status = stream_read_control(stream);
    if (status != FNS_SUCCESS) {
      return status;
    }

    stream->current_part++;

    distance = partdata - (stream->contentpos % partdata);

  }

  if (len > 0) {
    status = stream_readdata(stream, &buffer[bufpos], len);
    if (status != FNS_SUCCESS) {
      return status;
    }
    if (stream->pos == stream->size - 1) {

      status = stream_read_control(stream);
      if (status != FNS_SUCCESS) {
        return status;
      }

      SHA1_Final(temp_hash, &(stream->hash_handle));
      if (stream->keytype == FN_CHK_TYPE) {
        if (memcmp(stream->hash[stream->current_part], temp_hash,
                   FN_HASH_BYTES)
            != 0) {
          return FNS_BAD_HASH;
        }
      } else {

        status = verify(&(stream->group), stream->pub, temp_hash,
                        FN_HASH_BYTES, stream->sigstr);
        if (status != FNS_SUCCESS) {
          return status;
        }

      }
    }
  }

  return FNS_SUCCESS;
}


/* write len bytes of buffer to a raw (encrypted) freenet stream */
int freenet_write_stream(freenet_stream *stream, unsigned char *buffer,
                         int len)
{
  int status;
  int distance;
  int partdata;
  unsigned char ctrl;
  int bufpos = 0;

  if (stream->direction != FN_OUTGOING_STREAM) {
    return FNS_BAD_STREAM_DIRECTION;
  }

  if (stream->pos + len > stream->size) {
    return FNS_OVERWRITE;
  }

  partdata = stream->partsize - FN_HASH_BYTES;

  distance = partdata - (stream->contentpos % partdata);

  while ((len >= distance) &&
         (stream->current_part != stream->last_part)) {

    if (distance > 0) {
      status = stream_writedata(stream, buffer, distance);
      if (status != FNS_SUCCESS) {
        return status;
      }

      bufpos += distance;
      len -= distance;
    }

    status = stream_write_trailer(stream);
    if (status != FNS_SUCCESS) {
      return status;
    }

    ctrl = FN_CB_OK;
    status = stream_write(stream, &ctrl, 1);
    if (status != FNS_SUCCESS) {
      return status;
    }

    stream->current_part++;

    distance = partdata - (stream->contentpos % partdata);
  }


  if (len > 0) {
    status = stream_writedata(stream, &buffer[bufpos], len);
    if (status != FNS_SUCCESS) {
      return status;
    }
    if (stream->pos == stream->size - 1) {
      ctrl = FN_CB_OK;
      status = stream_write(stream, &ctrl, 1);
      if (status != FNS_SUCCESS) {
        return status;
      }
    }
  }

  return FNS_SUCCESS;
}


/* fill stream with info on the incoming stream described by key and message */
int freenet_init_incoming_stream (freenet_stream *stream, freenet_key *key,
                                  freenet_message *message)
{
  int i, j, len, startpos;
  unsigned char tmp[3];
  unsigned char pubstr[(FN_DHK_LEN*2) + 1]; // null term
  unsigned char dnstr[(FN_HASH_BYTES*2) + 1]; // null term
  unsigned char dnhash[FN_HASH_BYTES];

  stream->direction = FN_INCOMING_STREAM;

  dnstr[0] = 0;

  len = 0;

  /* FIXME -- ensure that all required fields are present */
  for (i=0; i<message->numfields; i++) {
    if (strncmp(message->field[i],
                "Storable.PartSize=", 18) == 0) {
      sscanf(message->field[i], "Storable.PartSize=%x", &stream->partsize);
    } else if (strncmp(message->field[i], "Storable.Signature=", 19)==0) {
      strncpy(stream->sigstr, &(message->field[i][19]), ((FN_DHK_LEN*2) + 2));
    } else if (strncmp(message->field[i], "Storable.Public-key=", 20)==0) {

      /* skip some leading zeros.  this really sucks. */
      startpos = 20;
      if ((strlen(&(message->field[i][startpos])) > FN_DHK_LEN * 2)
          && ((startpos+1) < strlen(message->field[i]))
          && (message->field[i][startpos] == '0')
          && (message->field[i][startpos+1] == '0')) {
        startpos += 2;
      }

      strncpy(pubstr, &(message->field[i][startpos]), FN_DHK_LEN * 2);
      pubstr[(FN_DHK_LEN*2) + 1] = 0;
      for (j = 0; j < strlen(pubstr) / 2; j++) {
        strncpy(tmp, &(pubstr[j*2]), 2);
        tmp[2] = 0;
        stream->pub[j] = (unsigned char)strtol(tmp, NULL, 16);
      }
    } else if (strncmp(message->field[i], "Storable.Document-name=", 23)==0) {
      strncpy(dnstr, &(message->field[i][23]), FN_HASH_BYTES * 2);
      dnstr[(FN_HASH_BYTES*2) + 1] = 0;
      for (j = 0; j < strlen(dnstr) / 2; j++) {
        strncpy(tmp, &(dnstr[j*2]), 2);
        tmp[2] = 0;
        dnhash[j] = (unsigned char)strtol(tmp, NULL, 16);
      }
    } else if (strncmp(message->field[i], "DataLength=", 11)==0) {
      sscanf(message->field[i], "DataLength=%x", &len);
    }
  }

  if (len > 0) {

    stream->size = len;

    if (stream->partsize == 0) {
      stream->partsize = len - 1;
    }

    /* "len - 1" below excludes the final control byte from the
       stream length.  That fixes the case where the partsize (plus
       a control byte for every part) divides evenly into the stream
       length.  We don't want that to equal the next whole number. */
    stream->last_part = ((len - 1) / (stream->partsize + 1));

    stream->payload_len = stream->size -
      (stream->last_part * (FN_HASH_BYTES + 1));
    stream->payload_len--;

    stream->pos = 0;
    stream->contentpos = 0;
    stream->current_part = 0;

    stream->keytype = key->type;

    memcpy(&(stream->group), &(key->group), sizeof(freenet_group));

    memcpy(stream->hash[0], key->searchkey, FN_HASH_BYTES);
    SHA1_Init(&(stream->hash_handle));
    if (strlen(dnstr) != 0) {
      SHA1_Update(&(stream->hash_handle), dnhash, FN_HASH_BYTES);
    }

  } else {
    return FNS_ZERO_DATALEN;
  }

  return FNS_SUCCESS;
}


/* fill stream with info on the outgoing stream */
int freenet_init_outgoing_stream (freenet_stream *stream)
{

  stream->direction = FN_OUTGOING_STREAM;

  return FNS_SUCCESS;

}
