/*
 * Copyright (c) 2004, 2005, 2006 Robin J Carey. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

# include  "IBAA.h"

# include  <sys/types.h>	/* For:	boolean_t, u_char, size_t	*/
				/*	u_int32_t			*/
# include  <sys/param.h>	/* For:	FALSE, TRUE			*/


/*-------------------------- IBAA CSPRNG -------------------------------*/

/* NOTE: The original source code from which this source code (IBAA)
 *       was taken has no copyright/license. The algorithm has no patent
 *       and is freely/publicly available from:
 *
 *           http://www.burtleburtle.net/bob/rand/isaac.html
 */

/*
 * ^ means XOR, & means bitwise AND, a<<b means shift a by b.
 * barrel(a) shifts a 19 bits to the left, and bits wrap around
 * ind(x) is (x AND 255), or (x mod 256)
 */
typedef	u_int32_t	u4;   /* unsigned four bytes, 32 bits */

# define  ALPHA		(8)
# define  SIZE		(1 << ALPHA)
# define  ind(x)	((x) & (SIZE - 1))
# define  barrel(a)	(((a) << 19) ^ ((a) >> 13))  /* beta=32,shift=19 */
 
static void IBAA
(
	u4 * m,		/* Memory: array of SIZE ALPHA-bit terms */
	u4 * r,		/* Results: the sequence, same size as m */
	u4 * aa,	/* Accumulator: a single value */
	u4 * bb		/* the previous result */
)
{
  u4	a, b, x, y, i;
 
  a = *aa; b = *bb;
  for (i = 0; i < SIZE; ++i)
  {
    x = m [ i ];  
    a = barrel(a) + m [ ind(i + (SIZE / 2)) ];	/* set a */
    m [ i ] = y = m [ ind(x) ] + a + b;		/* set m */
    r [ i ] = b = m [ ind(y >> ALPHA) ] + x;	/* set r */
  }
  *bb = b; *aa = a;
}

/*-------------------------- IBAA CSPRNG -------------------------------*/


static u4		IBAA_memory	[ SIZE ];
static u4		IBAA_results	[ SIZE ];
static u4		IBAA_aa;
static u4		IBAA_bb;

static boolean_t	IBAA_called;	/* FALSE until first IBAA_Byte() */

static const u_char *	IBAA_results_byte_ptr;
static size_t		IBAA_bytes_extracted;			/* bytes */


static void		IBAA_Call (void);


/* Initialize IBAA. */
void
IBAA_Init (void)
{
  size_t	i;
  for (i = 0; i < IBAA_Size (); ++i) {
    IBAA_memory [ i ] = i;
  }
  IBAA_aa = IBAA_bb = 0;
  IBAA_called = FALSE;
}

size_t
IBAA_Size (void)
{
  return SIZE;
}

/* PRIVATE: Call IBAA to produce 256 32-bit u4 results. */
static void
IBAA_Call (void)
{
  IBAA (IBAA_memory, IBAA_results, &IBAA_aa, &IBAA_bb);
  IBAA_results_byte_ptr = (const u_char *) IBAA_results;
  IBAA_bytes_extracted = 0;
}

/* Add a 32-bit u4 seed value into IBAAs memory. */
void
IBAA_Seed (const u_int32_t val)
{
  static u_char		memIndex = 0;

  IBAA_memory [ memIndex++ ] = val;
}

/* Extract a byte from IBAAs 256 32-bit u4 results array. */
u_char
IBAA_Byte (void)
{
  if (!IBAA_called) {		/* Not called for the first time. */
    IBAA_Call ();		/* Call for the first time. */
    IBAA_called = TRUE;
  }
  {
    const u_char	result = *IBAA_results_byte_ptr;

    ++IBAA_results_byte_ptr;
    if ((++IBAA_bytes_extracted) >= sizeof (IBAA_results)) {
      IBAA_Call ();
    }
    return result;
  }
}

/* EOF */
