/*
 * Copyright (c) 2004, 2005, 2006 Robin J Carey. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

# include  "L15.h"

# define L15_STATE_SIZE	256

static LByteType	x, y, z;
static LByteType	start_x;
static LByteType	state [ L15_STATE_SIZE ];


/* PRIVATE FUNCS:
 */

static void		Swap (const LByteType pos1, const LByteType pos2);
static void		InitState (void);
static void		KSA (const LByteType * const key, const size_t keyLen);
static void		Discard (const LByteType numCalls);


static void
Swap (const LByteType pos1, const LByteType pos2)
{
  const LByteType	save1 = state [ pos1 ];
  state [ pos1 ] = state [ pos2 ];
  state [ pos2 ] = save1;
}

static void
InitState (void)
{
  size_t	i;
  for (i = 0; i < L15_STATE_SIZE; ++i) {
    state [ i ] = i;
  }
}

static void
KSA (const LByteType * const key, const size_t keyLen)
{
# define  L_SCHEDULE(xx)					\
								\
for (i = 0; i < L15_STATE_SIZE; ++i) {				\
  Swap (i, (stateIndex += (state [ i ] + (xx))));		\
}
  size_t	i, keyIndex;
  LByteType	stateIndex = 0;
  L_SCHEDULE(keyLen);
  for (keyIndex = 0; keyIndex < keyLen; ++keyIndex) {
    L_SCHEDULE(key [ keyIndex ]);
  }
}

static void
Discard (const LByteType numCalls)
{
  LByteType	i;
  for (i = 0; i < numCalls; ++i) {
    (void) L15_Byte ();
  }
}


/* PUBLIC INTERFACE:
 */

void
L15 (const LByteType * const key, const size_t keyLen)
{
  x = y = start_x = 0;
  InitState ();
  KSA (key, keyLen);
  Discard (L15_Byte ());
}

LByteType
L15_Byte (void)
{
  Swap (state [ x ], y);
  z = state [ x++ ] + state [ y-- ];
  if (x == start_x) {
    --y;
  }
  return state [ z ];
}

void
L15_Vector (const LByteType * const key, const size_t keyLen)
{
  KSA (key, keyLen);
}

size_t
L15_QueryStateSize (void)
{
  return L15_STATE_SIZE;
}

/* EOF */
