/*
 * Copyright (c) 2004, 2005, 2006 Robin J Carey. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

# include  <sys/types.h>	/* For:	size_t, u_int32_t, u_char,	*/
				/*	boolean_t, dev_t		*/
# include  <sys/param.h>	/* For:	FALSE, TRUE			*/

# include  <sys/random.h>	/* For: Original interface.		*/
# include  <sys/time.h>		/* For:	nanotime(), nanouptime(),	*/
				/*	struct timespec			*/
# include  <sys/poll.h>		/* For:	POLL*				*/
# include  <sys/proc.h>		/* For:	struct proc			*/


/* Portability note: The u_char/unsigned char type is used where
 * uint8_t from <stdint.h> or u_int8_t from <sys/types.h> should really
 * be being used. On FreeBSD, it is safe to make the assumption that these
 * different types are equivalent (on all architectures).
 * The FreeBSD <sys/crypto/rc4> module also makes this assumption.
 */

/*------------------------------ IBAA ----------------------------------*/

/*-------------------------- IBAA CSPRNG -------------------------------*/

/* NOTE: The original source code from which this source code (IBAA)
 *       was taken has no copyright/license. The algorithm has no patent
 *       and is freely/publicly available from:
 *
 *           http://www.burtleburtle.net/bob/rand/isaac.html
 */

/*
 * ^ means XOR, & means bitwise AND, a<<b means shift a by b.
 * barrel(a) shifts a 19 bits to the left, and bits wrap around
 * ind(x) is (x AND 255), or (x mod 256)
 */
typedef	u_int32_t	u4;   /* unsigned four bytes, 32 bits */

# define  ALPHA		(8)
# define  SIZE		(1 << ALPHA)
# define  ind(x)	((x) & (SIZE - 1))
# define  barrel(a)	(((a) << 19) ^ ((a) >> 13))  /* beta=32,shift=19 */
 
static void IBAA
(
	u4 * m,		/* Memory: array of SIZE ALPHA-bit terms */
	u4 * r,		/* Results: the sequence, same size as m */
	u4 * aa,	/* Accumulator: a single value */
	u4 * bb		/* the previous result */
)
{
  u4	a, b, x, y, i;
 
  a = *aa; b = *bb;
  for (i = 0; i < SIZE; ++i)
  {
    x = m [ i ];  
    a = barrel(a) + m [ ind(i + (SIZE / 2)) ];	/* set a */
    m [ i ] = y = m [ ind(x) ] + a + b;		/* set m */
    r [ i ] = b = m [ ind(y >> ALPHA) ] + x;	/* set r */
  }
  *bb = b; *aa = a;
}

/*-------------------------- IBAA CSPRNG -------------------------------*/


static u4		IBAA_memory	[ SIZE ];
static u4		IBAA_results	[ SIZE ];
static u4		IBAA_aa;
static u4		IBAA_bb;

static boolean_t	IBAA_called;	/* FALSE until first IBAA_Byte() */

static const u_char *	IBAA_results_byte_ptr;
static size_t		IBAA_bytes_extracted;			/* bytes */


static void		IBAA_Init (void);
static size_t		IBAA_Size (void);
static void		IBAA_Call (void);
static void		IBAA_Seed (const u_int32_t val);
static u_char		IBAA_Byte (void);


/* Initialize IBAA. */
static void
IBAA_Init (void)
{
  size_t	i;
  for (i = 0; i < IBAA_Size (); ++i) {
    IBAA_memory [ i ] = i;
  }
  IBAA_aa = IBAA_bb = 0;
  IBAA_called = FALSE;
}

static size_t
IBAA_Size (void)
{
  return SIZE;
}

/* PRIVATE: Call IBAA to produce 256 32-bit u4 results. */
static void
IBAA_Call (void)
{
  IBAA (IBAA_memory, IBAA_results, &IBAA_aa, &IBAA_bb);
  IBAA_results_byte_ptr = (const u_char *) IBAA_results;
  IBAA_bytes_extracted = 0;
}

/* Add a 32-bit u4 seed value into IBAAs memory. */
static void
IBAA_Seed (const u_int32_t val)
{
  static u_char		memIndex = 0;

  IBAA_memory [ memIndex++ ] = val;
}

/* Extract a byte from IBAAs 256 32-bit u4 results array. */
static u_char
IBAA_Byte (void)
{
  if (!IBAA_called) {		/* Not called for the first time. */
    IBAA_Call ();		/* Call for the first time. */
    IBAA_called = TRUE;
  }
  {
    const u_char	result = *IBAA_results_byte_ptr;

    ++IBAA_results_byte_ptr;
    if ((++IBAA_bytes_extracted) >= sizeof (IBAA_results)) {
      IBAA_Call ();
    }
    return result;
  }
}

/*------------------------------ IBAA ----------------------------------*/


/*------------------------------- L15 ----------------------------------*/

/* NOTE: The L15/Leopard website is:
 *
 *		http://www.leopard.uk.com
 */

/* IMPORTANT NOTE: LByteType must be exactly 8-bits in size or this software
 * will not function correctly.
 */
typedef unsigned char	LByteType;

# define  L15_STATE_SIZE	256

static LByteType	L15_x, L15_y, L15_z;
static LByteType	L15_start_x;
static LByteType	L15_state [ L15_STATE_SIZE ];


/* PRIVATE FUNCS:
 */

static void		Swap (const LByteType pos1, const LByteType pos2);
static void		InitState (void);
static void		KSA (const LByteType * const key, const size_t keyLen);
static void		Discard (const LByteType numCalls);

/* PUBLIC INTERFACE:
 */
static void		L15 (const LByteType * const key, const size_t keyLen);
static LByteType	L15_Byte (void);
static void		L15_Vector (const LByteType * const key,
							const size_t keyLen);
static size_t		L15_QueryStateSize (void);



static void
Swap (const LByteType pos1, const LByteType pos2)
{
  const LByteType	save1 = L15_state [ pos1 ];
  L15_state [ pos1 ] = L15_state [ pos2 ];
  L15_state [ pos2 ] = save1;
}

static void
InitState (void)
{
  size_t	i;
  for (i = 0; i < L15_STATE_SIZE; ++i) {
    L15_state [ i ] = i;
  }
}

static void
KSA (const LByteType * const key, const size_t keyLen)
{
# define  L_SCHEDULE(xx)					\
								\
for (i = 0; i < L15_STATE_SIZE; ++i) {				\
  Swap (i, (stateIndex += (L15_state [ i ] + (xx))));		\
}
  size_t	i, keyIndex;
  LByteType	stateIndex = 0;
  L_SCHEDULE(keyLen);
  for (keyIndex = 0; keyIndex < keyLen; ++keyIndex) {
    L_SCHEDULE(key [ keyIndex ]);
  }
}

static void
Discard (const LByteType numCalls)
{
  LByteType	i;
  for (i = 0; i < numCalls; ++i) {
    (void) L15_Byte ();
  }
}


/* PUBLIC INTERFACE:
 */

static void
L15 (const LByteType * const key, const size_t keyLen)
{
  L15_x = L15_y = L15_start_x = 0;
  InitState ();
  KSA (key, keyLen);
  Discard (L15_Byte ());
}

static LByteType
L15_Byte (void)
{
  Swap (L15_state [ L15_x ], L15_y);
  L15_z = (L15_state [ L15_x++ ] + L15_state [ L15_y-- ]);
  if (L15_x == L15_start_x) {
    --L15_y;
  }
  return L15_state [ L15_z ];
}

static void
L15_Vector (const LByteType * const key, const size_t keyLen)
{
  KSA (key, keyLen);
}

static size_t
L15_QueryStateSize (void)
{
  return L15_STATE_SIZE;
}

/*------------------------------- L15 ----------------------------------*/


/*---------------------------- KERNEL I/FACE ---------------------------------*/
/*------------------------ INTERNAL PROTOTYPES -------------------------------*/
static void		InitSeed (const struct timespec * const nt);
static void		rnddev_initialize (void);
static void		NANOUP_EVENT (void);
static void		rnddev_add_keyboard (const u_char scancode);
static void		rnddev_add_true_randomness (const u_int32_t value);
static void		rnddev_add_interrupt_OR_read (void);
static size_t		RnddevRead (void * const buf, const size_t nbytes,
						u_char (*rndfunc)(void));
static size_t		rnddev_read_random (void * const buf,
							const size_t nbytes);
static size_t		rnddev_read_urandom (void * const buf,
							const size_t nbytes);
static int		rnddev_random_poll (dev_t dev, int events,
							struct proc * p);
/*------------------------ INTERNAL PROTOTYPES -------------------------------*/


/*--------------------------- INITIALIZATION ---------------------------------*/
static void
InitSeed (const struct timespec * const nt)
{
  IBAA_Seed (nt->tv_nsec);
  L15_Vector ((const LByteType *) &nt->tv_nsec, sizeof (nt->tv_nsec));
}

static void
rnddev_initialize (void)
{
  static boolean_t	RNDDEVinitialized = FALSE;

  if (!RNDDEVinitialized) {
    IBAA_Init ();		/* Initialize IBAA. */
    {				/* Initialize L15. */
      struct timespec	now;
      nanouptime (&now);
      L15 ((const LByteType *) &now.tv_nsec, sizeof (now.tv_nsec));
    }
    {				/* Fully seed IBAA and L15. */
      size_t		i;
      struct timespec	now;

      for (i = 0; i < (IBAA_Size () / 2); ++i) {
	nanotime (&now);	InitSeed (&now);
	nanouptime (&now);	InitSeed (&now);
      }
    }
    RNDDEVinitialized = TRUE;
  }
}
/*--------------------------- INITIALIZATION ---------------------------------*/


/* Time-buffered event time-stamping. This is necessary to cutoff higher
 * event frequencies, e.g. an interrupt occuring at 25Hz. In such cases
 * the CPU is being chewed and the timestamps are skewed (minimal variation).
 * Use a nano-second time-delay to limit how many times an Event can occur
 * in one second; <= 5Hz. Note that this doesn't prevent time-stamp skewing.
 * This implementation randmoises the time-delay between events, which adds
 * a layer of security/unpredictability with regard to read-events (a user
 * controlled input).
 *
 * Note: now.tv_nsec should range [ 0 - 1000,000,000 ].
 * Note: "ACCUM" is a security measure (result = capped-unknown + unknown),
 *       and also produces an uncapped (>=32-bit) value.
 */
static void
NANOUP_EVENT (void)
{
  static struct timespec	ACCUM		= { 0, 0 };
  static struct timespec	NEXT		= { 0, 0 };
  struct timespec		now;

  nanouptime (&now);
  if ((now.tv_nsec > NEXT.tv_nsec) || (now.tv_sec != NEXT.tv_sec)) {
    /* Randomised time-delay: 200e6 - 350e6 ns; 5 - 2.86 Hz. */
    const unsigned long		one_mil		= 1000000UL;	/* 0.001 s */
    const unsigned long		timeDelay	=
			(one_mil * 200) + ((ACCUM.tv_nsec % 151) * one_mil);
    NEXT.tv_nsec = now.tv_nsec + timeDelay;
    NEXT.tv_sec = now.tv_sec;
    ACCUM.tv_nsec += now.tv_nsec;
    IBAA_Seed (ACCUM.tv_nsec);
    L15_Vector ((const LByteType *) &ACCUM.tv_nsec, sizeof (ACCUM.tv_nsec));
  }
}

static void
rnddev_add_keyboard (const u_char scancode)
{
  rnddev_initialize ();

  L15_Vector ((const LByteType *) &scancode, sizeof (scancode));
  NANOUP_EVENT ();
}

static void
rnddev_add_true_randomness (const u_int32_t value)
{
  rnddev_initialize ();

  IBAA_Seed (value);
  L15_Vector ((const LByteType *) &value, sizeof (value));
}

static void
rnddev_add_interrupt_OR_read (void)
{
  rnddev_initialize ();

  NANOUP_EVENT ();
}

/* Internal read function to avoid code duplication. */
static size_t
RnddevRead (void * const buf, const size_t nbytes, u_char (*rndfunc)(void))
{
  rnddev_initialize ();

  {
    u_char * const	outBuf	= (u_char *) buf;
    size_t		i;

    for (i = 0; i < nbytes; ++i) {
      outBuf [ i ] = rndfunc ();
    }
    rnddev_add_interrupt_OR_read ();	/* After the data-copy. */
    return i;
  }
}

/* IBAA exported as /dev/random */
static size_t
rnddev_read_random (void * const buf, const size_t nbytes)
{
  return RnddevRead (buf, nbytes, IBAA_Byte);
}

/* L15 exported as /dev/urandom */
static size_t
rnddev_read_urandom (void * const buf, const size_t nbytes)
{
  return RnddevRead (buf, nbytes, L15_Byte);
}

/* /dev/[u]random is read-only.
 * /dev/[u]random never blocks.
 */
static int
rnddev_random_poll (dev_t dev, int events, struct proc * p)
{
  rnddev_initialize ();

  {
    const int	READflags = (POLLIN | POLLRDNORM);
    int		revents = 0;

    if (events & READflags) {
      revents |= READflags;	/* Always Readable. */
    }
    return revents;
  }
}
/*---------------------------- KERNEL I/FACE ---------------------------------*/


/* ----------- *** FreeBSD-4.11 Kernel interface routines. *** ----------- */

void		rand_initialize (void)
	{ }

void		add_keyboard_randomness (u_char scancode)
{
  rnddev_add_keyboard (scancode);
}

void		add_interrupt_randomness (void * vsc)
{
  struct random_softc * const	sc = (struct random_softc *) vsc;

  (sc->sc_handler)(sc->sc_arg);
  rnddev_add_interrupt_OR_read ();
}

void		add_true_randomness (int val)
			/* XXX u_int32_t; int is old i/face XXX */
{
  rnddev_add_true_randomness (val);
}

u_int		read_random (void * buf, u_int nbytes)
{
  return rnddev_read_random (buf, nbytes);
}

u_int		read_random_unlimited (void * buf, u_int nbytes)
{
  return rnddev_read_urandom (buf, nbytes);
}

int		random_poll (dev_t dev, int events, struct proc * p)
{
  return rnddev_random_poll (dev, events, p);
}

/* ----------- *** FreeBSD-4.11 Kernel interface routines. *** ----------- */


/*			   --- NOTES ---
 *
 * Note: The word "entropy" is often incorrectly used to describe
 * random data. The word "entropy" originates from the science of
 * Physics. The correct descriptive definition would be something
 * along the lines of "seed", "unpredictable numbers" or
 * "unpredictable data".
 *
 * Note: Some /dev/[u]random implementations save "seed" between
 * boots which represents a security hazard since an adversary
 * could acquire this data (since it is stored in a file). If
 * the unpredictable data used in the above routines is only
 * generated during Kernel operation, then an adversary can only
 * acquire that data through a Kernel security compromise and/or
 * a cryptographic algorithm failure/cryptanalysis.
 *
 * Note: On FreeBSD-4.11, interrupts have to be manually enabled
 * using the rndcontrol(8) command.
 *
 *		--- DESIGN (FreeBSD-4.11 based) ---
 *
 *   The rnddev module automatically initializes itself the first time
 * it is used (client calls any public rnddev_*() interface routine).
 * Both CSPRNGs are initially seeded from the precise nano[up]time() routines.
 * Tests show this method produces good enough results, suitable for intended
 * use. It is necessary for both CSPRNGs to be completely seeded, initially.
 *
 *   After initialization and during Kernel operation the only suitable
 * unpredictable data available is:
 *
 *	(1) Keyboard scan-codes.
 *	(2) Nanouptime acquired by a Keyboard/Read-Event.
 *	(3) Suitable interrupt source; hard-disk/ATA-device.
 *
 *      (X) Mouse-event (xyz-data unsuitable); NOT IMPLEMENTED.
 *
 *   This data is added to both CSPRNGs in real-time as it happens/
 * becomes-available. Additionally, unpredictable (?) data may be
 * acquired from a true-random number generator if such a device is
 * available to the system (not advisable !).
 *   Nanouptime() acquired by a Read-Event is a very important aspect of
 * this design, since it ensures that unpredictable data is added to
 * the CSPRNGs even if there are no other sources.
 *   The nanouptime() Kernel routine is used since time relative to
 * boot is less adversary-known than time itself.
 *
 *   This design has been thoroughly tested with debug logging
 * and the output from both /dev/random and /dev/urandom has
 * been tested with the DIEHARD test-suite; both pass.
 *
 * MODIFICATIONS MADE TO ORIGINAL "kern_random.c":
 * o ...
 * 16th July 2005:
 * o SEED_NANOUP() -> NANOUP_EVENT() function rename.
 * o Make NANOUP_EVENT() handle the time-buffering directly so that all
 *   time-stamp-events use this single time-buffer (including keyboard).
 *   This removes dependancy on "time_second" Kernel variable.
 * o Removed second-time-buffer code in rnddev_add_interrupt_OR_read (void).
 * o Rewrote the time-buffering algorithm in NANOUP_EVENT() to use a
 *   randomised time-delay range.
 * 12th Dec 2005:
 * o Updated to (hopefully final) L15 algorithm.
 * 12th-24th June 2006:
 * o Added missing (u_char *) cast in RnddevRead() function.
 * o Changed copyright to 2-clause BSD license and cleaned up the layout
 *   of this file.
 */

/* EOF */
