// Copyright (c) 2003, 2006 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# ifndef  OOO__FileCrypt_h__OOO
# define  OOO__FileCrypt_h__OOO

# include  "ByteType.h"
# include  "Password.h"
# include  "FileUtil.h"
# include  "CipherPacket.h"

# include  <limits.h>		// For: PATH_MAX
# include  <stdio.h>		// For: snprintf(3), perror(3)
# include  <stdlib.h>		// For: exit(3), malloc(3), free(3), size_t
# include  <string.h>		// For: strdup(3), strlen(3), strcmp(3),
				//      memset(3)


static const char * const	EXTENSION = ".L15";

class FileCrypt {
  private:
    char * const	fileName;
    ByteType *		data;

    char *		Prompt (const char * const prompt) const {
      const size_t	pAlloc = strlen (prompt) + strlen (fileName) + 2 + 1;
      char * const	p = (char *) malloc (pAlloc);
      if (p == NULL) {
	perror ("malloc(3)");
	exit (EXIT_FAILURE);
      }
      snprintf (p, pAlloc, "%s%s: ", prompt, fileName);
      return p;
    }

  public:
    FileCrypt (const char * const Filename)
	:	fileName (strdup (Filename)),
		data (NULL) {
      if (fileName == NULL) {
	FileUtil::PError ("strdup(3)", Filename);
	exit (EXIT_FAILURE);
      }
    }

    ~FileCrypt (void) {
      free (fileName);
      if (data != NULL) {
	free (data);
      }
    }

    static bool		ExtensionPresent (const char * const s) {
      const size_t	l = strlen (s);
      if (l > strlen (EXTENSION) &&
		strcmp (&s [ l - strlen (EXTENSION) ], EXTENSION) == 0) {
	return true;
      }
      return false;
    }

    static const char *	WithoutExtension (const char * const s) {
      static char	S [ PATH_MAX + 1 ];
      if (!FileCrypt::ExtensionPresent (s)) {
	return NULL;
      }
      snprintf (S, sizeof (S), "%s", s);
      S [ strlen (S) - strlen (EXTENSION) ] = '\0';
      return S;
    }

    static const char *	WithExtension (const char * const s) {
      static char	S [ PATH_MAX + 1 ];
      snprintf (S, sizeof (S), "%s%s", s, EXTENSION);
      return S;
    }

    // Return NULL on error.
    //
    ByteType *		Decrypt (size_t * const Datasize) {
      if (!ExtensionPresent (fileName)) {
	return NULL;
      }
      FILE * const	fIn = FileUtil::FOpenEmptyCheck (fileName, "rb");
      off_t		inSize;
      ASSERT (FileUtil::FileSize (fileName, &inSize) == 0);
      char * const	prompt = Prompt (PROMPT);
      ByteType		password [ MAX_PASS_LEN ];
      const size_t	passLen =
		PROMPT_PASSWORD(prompt, password, sizeof (password));
      free (prompt);
      CipherPacket	cipher (password, passLen);
      memset (password, 0, sizeof (password));
      if (data != NULL) {
	free (data);
      }
      data = (ByteType *) malloc (inSize);
      if (data == NULL) {
	perror ("malloc(3)");
	exit (EXIT_FAILURE);
      }
      const int		dataSize = cipher.DataDecrypt (data, fIn);
      FileUtil::FCloseErrCheck (fIn);
      if (dataSize < 0) {
	return NULL;
      } else {
	*Datasize = dataSize;
	return data;
      }
    }

    void		Encrypt (const ByteType * const Data,
						const size_t Datasize) const {
      const char * const	outputFile = WithExtension (fileName);
      FILE * const		fOut =
			FileUtil::FOpenErrCheck (outputFile, "wb");
      char * const	prompt = Prompt (CHOOSE_PROMPT);
      ByteType		password [ MAX_PASS_LEN ];
      const size_t	passLen =
		PROMPT_PASSWORD_CONFIRM(prompt, password, sizeof (password));
      free (prompt);
      CipherPacket	cipher (password, passLen);
      memset (password, 0, sizeof (password));
      cipher.DataEncrypt (Data, Datasize, fOut);
      FileUtil::FCloseErrCheck (fOut);
    }
};

# endif		// !OOO__FileCrypt_h__OOO
