// Copyright (c) 2003 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# ifndef  OOO__Key_h__OOO
# define  OOO__Key_h__OOO

# include  "ASSERT.h"
# include  "RSA.h"
# include  "ByteType.h"
# include  "FileUtil.h"
# include  "FileCrypt.h"
# include  "Terminal.h"

# include  <stdio.h>		// For: printf(3), fprintf(3), fflush(3),
				//      fwrite(3), fread(3), getchar(3)
# include  <stdlib.h>		// For: exit(3), size_t
# include  <string.h>		// For: memset(3)

# include  <openssl/bn.h>
# include  <openssl/crypto.h>

enum KeyType {
  PRIVATE_KEY, PUBLIC_KEY
};

class Key {
  public:
    static bool		ReadKey (const char * const file, BIGNUM * const key) {
      bool		encrypted = false;
      const char *	path = FileUtil::HomeDirFile (PROGDIR, file);
      ASSERT (path != NULL);
      off_t		fileSize;
      int		retVal = FileUtil::FileSize (path, &fileSize);
      if (retVal == -1) {
	path = FileCrypt::WithExtension (path);
	retVal = FileUtil::FileSize (path, &fileSize);
	if (retVal == -1) {
	  FileUtil::PError ("stat(2)", path);
	  exit (EXIT_FAILURE);
	} else {
	  encrypted = true;
	}
      }
      if (fileSize <= 0) {
	fprintf (stderr, "%s: Empty file.\n", path);
	exit (EXIT_FAILURE);
      }
      ByteType		bin [ fileSize ];
      size_t		dataSize;
      ByteType *	data;
      if (encrypted) {
	FileCrypt		crypt (path);
	data = crypt.Decrypt (&dataSize);
	ASSERT (data != NULL && dataSize > 0);
      } else {
	FILE * const	fIn = FileUtil::FOpenErrCheck (path, "rb");
	ASSERT (fread (bin, sizeof (ByteType),
					sizeof (bin), fIn) == sizeof (bin));
	FileUtil::FCloseErrCheck (fIn);
	data = bin;
	dataSize = sizeof (bin);
      }
      ASSERT (BN_bin2bn (data, dataSize, key) != NULL);
      memset (data, 0, dataSize);
      ASSERT (BN_num_bytes (key) > 0);
      return encrypted;
    }

    static void		WriteKey (const KeyType k, BIGNUM * const key) {
      printf ("Writing %s-key to disk:\n",
			(k == PRIVATE_KEY) ? "private" : "public");
      fflush (stdout);
      bool	encrypt = false;
      if (k == PUBLIC_KEY) {
	bool		gotYesNo = false;
	int		YesNo;
	Terminal	term;
	term.RawInput ();
	do {
	  printf ("Encrypt public-key (y/n) ? ");
	  fflush (stdout);
	  YesNo = getchar ();
	  if (YesNo == 'y') {
	    encrypt = gotYesNo = true;
	  } else if (YesNo == 'n') {
	    gotYesNo = true;
	  }
	  printf ("\n");
	} while (!gotYesNo);
	term.Restore ();
      } else {
	encrypt = true;
      }
      const char * const	fileName = FileUtil::HomeDirFile (PROGDIR,
					(k == PRIVATE_KEY) ? PRIVKEY : PUBKEY);
      ASSERT (fileName != NULL);
      ASSERT (BN_num_bytes (key) > 0);
      ByteType		bin [ BN_num_bytes (key) ];
      ASSERT (BN_bn2bin (key, bin) == (int) sizeof (bin));
      if (encrypt) {
	FileCrypt	crypt (fileName);
	crypt.Encrypt (bin, sizeof (bin));
      } else {
	FILE * const	fOut = FileUtil::FOpenErrCheck (fileName, "wb");
	ASSERT (fwrite (bin, sizeof (ByteType),
				sizeof (bin), fOut) == sizeof (bin));
	FileUtil::FCloseErrCheck (fOut);
      }
      memset (bin, 0, sizeof (bin));
    }
};

# endif		// !OOO__Key_h__OOO
