// Copyright (c) 2003 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# ifndef  OOO__CipherPacket_h__OOO
# define  OOO__CipherPacket_h__OOO

# include  "ByteType.h"
# include  "L14.h"
# include  "Random.h"
# include  "IV.h"

# include  <stdio.h>		// For: fwrite(3), fread(3), ferror(3), feof(3)
# include  <stdlib.h>		// For: size_t, NULL
# include  <string.h>		// For: memcpy(3), memset(3)


// The PacketType structure is used to "sandwich" ciphertext.
// Ciphertext is padded on "both sides" by random data.
//
// This method thwarts "known-plaintext" attacks, since an
// adversary is denied knowledge of the ciphertext correlating
// to his/her "known-plaintext". It also prevents cryptanalysis
// of the ciphertext since an adversary is unable to distinguish
// the ciphertext from padding.
//
// A Cipher-Packet is composed of 4 logically distinct parts:
//
//	start-pad (0-255)	[ random ]
//	ciphertext-length (1)	[ encrypted ]
//	ciphertext (0-255)	[ encrypted ]
//	end-pad (0-255)		[ random ]
//
struct PacketType {
  ByteType		data [ 255 + 1 + 255 + 255 ];
  size_t		dataLen;
};

struct DataType {
  ByteType		data [ 255 ];
  ByteType		dataLen;
};

class CipherPacket {
  private:
    L14			theCipher;
    Random		entropy;

    enum StateType {
      STATE_INIT,
      STATE_SP_SKIP,
      STATE_MSG_LEN,
      STATE_MSG,
      STATE_EP_SKIP
    };
    StateType		state;
    ByteType		startPadLen;
    DataType		plaintext;
    ByteType		msgIndex;
    ByteType		endPadLen;

    void		StartPad (void) {
      startPadLen = theCipher.Byte ();
      if (startPadLen == 0) {
	state = STATE_MSG_LEN;
      } else {
	state = STATE_SP_SKIP;
      }
    }

    DataType *		EndPad (void) {
      endPadLen = theCipher.Byte ();
      if (endPadLen == 0) {
	state = STATE_INIT;
	return &plaintext;
      } else {
	state = STATE_EP_SKIP;
      }
      return NULL;
    }

  public:
    CipherPacket (const ByteType * const key, const size_t keyLen)
	:	theCipher (key, keyLen, L_UNKNOWN_STATE),
		entropy (RND_DEVICE),
		state (STATE_INIT) {
    }

    void		PacketEncrypt (const DataType * const in,
						PacketType * const out) {
      size_t		outIndex = 0;

      const ByteType	spLen = theCipher.Byte ();
      for (size_t i = 0; i < spLen; ++i, ++outIndex) {
	out->data [ outIndex ] = entropy.Byte ();
      }

      out->data [ outIndex++ ] = in->dataLen ^ theCipher.Byte ();
      for (size_t i = 0; i < in->dataLen; ++i, ++outIndex) {
	out->data [ outIndex ] = in->data [ i ] ^ theCipher.Byte ();
      }

      const ByteType	epLen = theCipher.Byte ();
      for (size_t i = 0; i < epLen; ++i, ++outIndex) {
	out->data [ outIndex ] = entropy.Byte ();
      }
      out->dataLen = outIndex;
    }

    DataType *				StreamDecrypt (const ByteType in) {
      switch (state) {
	case STATE_INIT:
	  StartPad ();
	  return StreamDecrypt (in);
	  break;
	case STATE_SP_SKIP:
	  if (--startPadLen == 0) {
	    state = STATE_MSG_LEN;
	  }
	  break;
	case STATE_MSG_LEN:
	  plaintext.dataLen = in ^ theCipher.Byte ();
	  if (plaintext.dataLen > 0) {
	    msgIndex = 0;
	    state = STATE_MSG;
	  } else {
	    return EndPad ();
	  }
	  break;
	case STATE_MSG:
	  plaintext.data [ msgIndex++ ] = in ^ theCipher.Byte ();
	  if (msgIndex == plaintext.dataLen) {
	    return EndPad ();
	  }
	  break;
	case STATE_EP_SKIP:
	  if (--endPadLen == 0) {
	    state = STATE_INIT;
	    return &plaintext;
	  }
	  break;
      }
      return NULL;
    }

    void			DataEncrypt (const ByteType * const data,
			const size_t dataSize, FILE * const fOut) {
      ByteType		pad [ theCipher.Byte () ];
      entropy.Bytes (pad, sizeof (pad));
      ASSERT (fwrite (pad, sizeof (ByteType), sizeof (pad),
						fOut) == sizeof (pad));
      ByteType		iv [ IVsize ];
      entropy.Bytes (iv, sizeof (iv));
      ASSERT (fwrite (iv, sizeof (ByteType), sizeof (iv), fOut) == sizeof (iv));
      theCipher.InitVector (iv, sizeof (iv));
      PacketType	Packet;
      DataType		Data;
      size_t		left;
      for (size_t accum = 0; accum < dataSize; accum += Data.dataLen) {
	left = dataSize - accum;
	Data.dataLen = (left > 255) ? entropy.Byte () : left;
	memcpy (Data.data, &data [ accum ], Data.dataLen);
	PacketEncrypt (&Data, &Packet);
	ASSERT (fwrite (Packet.data, sizeof (ByteType),
				Packet.dataLen, fOut) == Packet.dataLen);
      }
      memset (&Data, 0, sizeof (Data));
      memset (iv, 0, sizeof (iv));
    }

    // NOTE: "data" must point to enough space.
    int				DataDecrypt (ByteType * const data,
					FILE * const fIn) {
      ByteType		pad [ theCipher.Byte () ];
      ASSERT (fread (pad, sizeof (ByteType), sizeof (pad),
						fIn) == sizeof (pad));
      ByteType		iv [ IVsize ];
      ASSERT (fread (iv, sizeof (ByteType), sizeof (iv),
						fIn) == sizeof (iv));
      theCipher.InitVector (iv, sizeof (iv));
      const DataType *	Data;
      ByteType		buf [ 1024 ];
      size_t		got;
      size_t		dataIndex	= 0;
      do {
	got = fread (buf, sizeof (ByteType), sizeof (buf), fIn);
	if (ferror (fIn)) {
	  return -1;
	}
	if (got > 0) {
	  for (size_t i = 0; i < got; ++i) {
	    Data = StreamDecrypt (buf [ i ]);
	    if (Data != NULL) {
	      memcpy (&data [ dataIndex ], Data->data, Data->dataLen);
	      dataIndex += Data->dataLen;
	    }
	  }
	}
      } while (!feof (fIn));
      memset (&plaintext, 0, sizeof (plaintext));
      memset (iv, 0, sizeof (iv));
      return dataIndex;
    }
};

# endif		// !OOO__CipherPacket_h__OOO
