// Copyright (c) 2003 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# ifndef  OOO__FileUtil_h__OOO
# define  OOO__FileUtil_h__OOO

# include  "ASSERT.h"

// ANSI C/BSD
# include  <errno.h>		// For: errno, EINTR
# include  <limits.h>		// For: PATH_MAX
# include  <stdio.h>		// For: fopen(3), fclose(3), fgetc(3),
				//      ferror(3),
				//      snprintf(3), perror(3)
				//      NULL, EOF, FILE
# include  <stdlib.h>		// For: getenv(3), exit(3)
# include  <string.h>		// For: strlen(3)

// POSIX/BSD
# include  <sys/types.h>
# include  <sys/stat.h>		// For: stat(2), mkdir(2), umask(2)
# include  <fcntl.h>		// For: open(2), O_RDONLY, ...
# include  <unistd.h>		// For: close(2), ssize_t



// C++ Front-End to C stdio library calls, e.g. fopen(3), open(2).
//
class FileUtil {
  public:
    static void			PError (const char * const sysCall,
					const char * const fileName) {
      char		buf [ strlen (sysCall) + 2 + strlen (fileName) + 1 ];
      snprintf (buf, sizeof (buf), "%s: %s", sysCall, fileName);
      perror (buf);
    }

    // Return 0 on success, -1 on failure.
    static int			FileSize (const char * const fName,
						off_t * const size) {
      struct stat	sb;
      const int		retVal = stat (fName, &sb);
      if (retVal == 0) {
	*size = sb.st_size;
      }
      return retVal;
    }

    // Return NULL on error, or a pointer to a statically stored string
    // of the form "/home/robin/<file>".
    static const char *		HomeFile (const char * const file) {
      static char		path [ PATH_MAX + 1 ];
      const char * const	home = getenv ("HOME");

      if (home == NULL || strlen (home) == 0) {
	return NULL;
      }
      if (home [ strlen (home) - 1 ] == '/') {
	snprintf (path, sizeof (path), "%s%s", home, file);
      } else {
	snprintf (path, sizeof (path), "%s/%s", home, file);
      }
      return path;
    }

    // Return NULL on error, or a pointer to a statically stored string
    // of the form "/home/robin/<dir>/<file>".
    static const char *		HomeDirFile (const char * const dir,
			const char * const file) {
      static char		path [ PATH_MAX + 1 ];
      const char * const	home = getenv ("HOME");

      if (home == NULL || strlen (home) == 0) {
	return NULL;
      }
      if (home [ strlen (home) - 1 ] == '/') {
	snprintf (path, sizeof (path), "%s%s/%s", home, dir, file);
      } else {
	snprintf (path, sizeof (path), "%s/%s/%s", home, dir, file);
      }
      return path;
    }

    // Return -1 on error, 0 on success.
    static int			MkDir (const char * const path,
			const mode_t mode = S_IRWXU) {
      return mkdir (path, mode);
    }

    // Return NULL on error.
    static FILE *		FOpen (const char * const file,
					const char * const flags) {
      FILE *		F;
      do {
	F = fopen (file, flags);
      } while (F == NULL && errno == EINTR);
      return F;
    }

    static FILE *		FOpenErrCheck (const char * const file,
					const char * const flags) {
      FILE * const	F = FileUtil::FOpen (file, flags);
      if (F == NULL) {
	FileUtil::PError ("fopen(3)", file);
	exit (EXIT_FAILURE);
      }
      return F;
    }

    static FILE *		FOpenEmptyCheck (const char * const file,
					const char * const flags) {
      off_t		fileSize;
      const int		retVal = FileUtil::FileSize (file, &fileSize);
      if (retVal == -1) {
	FileUtil::PError ("stat(2)", file);
	exit (EXIT_FAILURE);
      } else if (fileSize <= 0) {
	fprintf (stderr, "%s: Empty file.\n", file);
	exit (EXIT_FAILURE);
      }
      return FileUtil::FOpenErrCheck (file, flags);
    }

    // Return EOF on error, 0 on success.
    static int			FClose (FILE * const F) {
      int		retVal;
      do {
	retVal = fclose (F);
      } while (retVal == EOF && errno == EINTR);
      return retVal;
    }

    static void			FCloseErrCheck (FILE * const F) {
      if (FileUtil::FClose (F) == EOF) {
	perror ("fclose(3)");
	exit (EXIT_FAILURE);
      }
    }

    // Reading "f" continues until a newline, EOF, error, or resultSize - 1
    // bytes have been read. Any newline read is discarded.
    // Return -1 on error, or the number of bytes read.
    // In all cases the "result" string is NUL terminated.
    static ssize_t		FReadCharNewline (FILE * const f,
		char * const result, const size_t resultSize) {
      ASSERT (resultSize >= 2);
      int		retVal;
      size_t		count	= 0;
      do {
	retVal = fgetc (f);
	if (retVal != EOF && retVal != '\n') {
	  result [ count++ ] = retVal;
	}
      } while (retVal != EOF && retVal != '\n' && count < (resultSize - 1));
      result [ count ] = '\0';
      if (ferror (f)) {
	return -1;
      } else {
	return count;
      }
    }

    static int			Open (const char * const fName,
						int flags = O_RDONLY) {
      int		fd;
      do {
	fd = open (fName, flags);
      } while (fd < 0 && errno == EINTR);
      return fd;
    }

    static int			Close (const int fd) {
      int		retVal;
      do {
	retVal = close (fd);
      } while (retVal < 0 && errno == EINTR);
      return retVal;
    }

    // Create files: -rw-------
    static void			SecureMask (void) {
      umask (S_IRWXG|S_IRWXO);
    }
};

# endif		// !OOO__FileUtil_h__OOO
