// Copyright (c) 2003 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# include  "ASSERT.h"
# include  "ByteType.h"
# include  "Startup.h"
# include  "RSA.h"
# include  "FileUtil.h"
# include  "Random.h"
# include  "Key.h"
# include  "Password.h"

# include  <limits.h>		// For: PATH_MAX
# include  <stdio.h>		// For: printf(3), fprintf(3), snprintf(3),
				//      perror(3), fflush(3), fread(3),
				//      ferror(3), feof(3)
# include  <stdlib.h>		// For: exit(3), atexit(3)
# include  <string.h>		// For: memset(3), memcpy(3)
# include  <unistd.h>		// For: getopt(3)

// OpenSSL
# include  <openssl/bn.h>
# include  <openssl/crypto.h>
# include  <openssl/sha.h>


# define  ASSIGN_LEN(xx)						\
(Over511) ? ((xx) = entropy.Byte ()) : ((xx) = (entropy.Byte () % maxDataLen))

// NOTE: Ideally, saltLen should be in the range 0 -> 255. However, the
//       BN_bin2bn() function in the OpenSSL BIGNUM library renders the
//       same BIGNUM value for the two distinct binary representations of:
//       { 0, 1, 2, 3 } and { 1, 2, 3 }. This is a bug as far as I'm
//       concerned. Thus, saltLen has to be in the range 1 -> 255.
//
# define  ENCRYPT_MSG							\
									\
do {									\
  do {									\
    ASSIGN_LEN(saltLen);						\
  } while (saltLen == 0);						\
  messageBin [ 0 ] = saltLen;						\
  entropy.Bytes (&messageBin [ 1 ], saltLen);				\
  memcpy (&messageBin [ 1 + saltLen ], Data, dataLen);			\
  ASSERT (BN_bin2bn (messageBin, (1 + saltLen + dataLen),		\
						message) != NULL);	\
  ASSERT (BN_mod_exp (ciphertext, message, e, to_n, ctx) == 1);		\
} while ((secretPubkeys &&						\
		(BN_num_bytes (ciphertext) >= BN_num_bytes (to_n))) ||	\
		BN_cmp (ciphertext, message) == 0)

# define  HEX_BEGIN							\
  hexStr = BN_bn2hex (ciphertext);					\
  ASSERT (hexStr != NULL && strlen (hexStr) > 0)

# define  HEX_END							\
  fflush (fOut);							\
  OPENSSL_free (hexStr)

# define  HEXPRINT							\
  HEX_BEGIN;								\
  fprintf (fOut, "%s\n", hexStr);					\
  HEX_END

# define  SIGHEXPRINT							\
  HEX_BEGIN;								\
  fprintf (fOut, "%c%s\n", SIGNATURE, hexStr);				\
  HEX_END

static const char * const	OUTPUT_FILE = CIPHERTEXT_FILE;

static BIGNUM * const	to_n	= BN_new ();
static BIGNUM * const	n	= BN_new ();
static BIGNUM * const	d	= BN_new ();
static BIGNUM * const	message	= BN_new ();
static ByteType		plaintext	[ 16384 ];
static size_t		plaintextLen;
static size_t		plaintextIndex;
static ByteType		messageBin	[ max_msg_size ];
static ByteType		Data		[ max_msg_data_size ];
static ByteType		dataLen, saltLen;

static void
ClearMemory (void)
{
  BN_clear (to_n);
  BN_clear (n);
  BN_clear (d);
  BN_clear (message);
  memset (plaintext, 0, sizeof (plaintext));
  plaintextLen = plaintextIndex = 0;
  memset (messageBin, 0, sizeof (messageBin));
  memset (Data, 0, sizeof (Data));
  dataLen = saltLen = 0;
}

static void
Usage (const char * const argv0)
{
  static const char * const str = "[ -o output-file ] -t to-public-key";
  fprintf (stderr, "%s [ -s ] -i input-file %s\n", argv0, str);
  fprintf (stderr, "%s [ -s ] -m %s\n", argv0, str);
  exit (EXIT_FAILURE);
}

int
main (int argc, char * argv [])
{
  Startup	startup;
  startup.Useless ();

  char		inputFile	[ PATH_MAX + 1 ];
  char		outputFile	[ PATH_MAX + 1 ];
  char		toKeyFile	[ PATH_MAX + 1 ];
  bool		gotInputFile	= false;
  bool		gotToKeyFile	= false;
  bool		secretPubkeys	= false;
  bool		directInput	= false;
  int		ch;

  snprintf (outputFile, sizeof (outputFile), "%s", OUTPUT_FILE);
  while ((ch = getopt (argc, argv, "t:i:o:sm")) != -1) {
    switch (ch) {
      case 't':
	snprintf (toKeyFile, sizeof (toKeyFile), "%s", optarg);
	gotToKeyFile = true;
	break;
      case 'i':
	snprintf (inputFile, sizeof (inputFile), "%s", optarg);
	gotInputFile = true;
	break;
      case 'o':
	snprintf (outputFile, sizeof (outputFile), "%s", optarg);
	break;
      case 's':
	secretPubkeys = true;
	break;
      case 'm':
	directInput = true;
	break;
      case '?':
      default:
	Usage (argv [0]);
    }
  }
  if (!gotToKeyFile || (!gotInputFile && !directInput)
				|| (gotInputFile && directInput)) {
    Usage (argv[0]);
  }
  if (atexit (&ClearMemory) < 0) {
    perror ("atexit(3)");
    exit (EXIT_FAILURE);
  }

  FILE *	fIn;
  if (!directInput) {
    fIn = FileUtil::FOpenEmptyCheck (inputFile, "rb");
  }

  // Read in targets public-key:
  //
  if (!secretPubkeys && Key::ReadKey (toKeyFile, to_n)) {
    printf ("Entering secret public-key mode.\n");
    secretPubkeys = true;
  }
  const size_t		to_nSize	= BN_num_bytes (to_n);

  // Read in owners public and private keys for signing:
  //
  Key::ReadKey (PUBKEY, n);
  Key::ReadKey (PRIVKEY, d);

  // Open "outputFile":
  //
  FILE * const	fOut = FileUtil::FOpenErrCheck (outputFile, "wb");

  if (directInput) {
    const char * prompt = "Enter message (^D to finish): ";
    plaintextLen = Password (false, EOF, prompt, plaintext, sizeof (plaintext));
    plaintextIndex = 0;
  }

  // Do the RSA Encryption: c=m^e mod to_n
  //
  printf ("Encrypting plaintext:");
  fflush (stdout);
  Random		entropy (RND_DEVICE);
  const bool		Over511		= (to_nSize > 511);
  const ByteType	maxDataLen	= (to_nSize / 2);
  ASSERT (maxDataLen > 0);
  BN_CTX * const	ctx		= BN_CTX_new ();
  BIGNUM * const	ciphertext	= BN_new ();
  BIGNUM * const	e		= BN_new ();
  ASSERT (BN_set_word (e, encExp) == 1);
  char *		hexStr;
  SHA_CTX		SHA1context;
  SHA1_Init (&SHA1context);
  do {
    ASSIGN_LEN(dataLen);
    if (dataLen > 0) {
      if (directInput) {
	const size_t	left = plaintextLen - plaintextIndex;
	if (dataLen > left) {
	  dataLen = left;
	}
	memcpy (Data, &plaintext [ plaintextIndex ], dataLen);
	plaintextIndex += dataLen;
      } else {
	dataLen = fread (Data, sizeof (ByteType), dataLen, fIn);
	ASSERT (!ferror (fIn));
	if (dataLen == 0) {		// EOF
	  break;
	}
      }
      SHA1_Update (&SHA1context, Data, dataLen);
    }
    ENCRYPT_MSG;
    HEXPRINT;
  } while ((!directInput && !feof (fIn)) ||
			(directInput && (plaintextIndex < plaintextLen)));
  printf (" done\n");


  // RSA Encrypted: RSA Signature with appendix:
  //
  //   My solution to the "Reblocking problem" is to generate the digital
  // signature in the normal fashion, convert this signature to binary
  // and then encrypt this as normal data in the normal way. But the
  // encrypted signature data is given a special marking to identify it as
  // the digital signature.
  //
  printf ("Generating digital signature:");
  fflush (stdout);
  //
  // Generate M (1 + SHA1 + salt):
  //
  unsigned char		SHA1digest [ SHA_DIGEST_LENGTH ];
  SHA1_Final (SHA1digest, &SHA1context);
  ByteType		Mbin [ BN_num_bytes (n) ];
  ASSERT (sizeof (Mbin) > (1 + sizeof (SHA1digest)));
  const size_t		MsaltMaxLen	=
				sizeof (Mbin) - 1 - sizeof (SHA1digest);
  const size_t		MsaltLen	= entropy.Ulong() % MsaltMaxLen;
  //
  // NOTE: Due to the "bug" in OpenSSL BN_bin2bn(), Mbin must begin with
  // a non-zero byte. Since it is possible that the first byte of the SHA1
  // digest may be zero, we have to use a non-zero "startPad" byte.
  //
  do {
    Mbin [ 0 ] = entropy.Byte ();
  } while (Mbin [ 0 ] == 0);
  memcpy (&Mbin [ 1 ], SHA1digest, sizeof (SHA1digest));
  memset (SHA1digest, 0, sizeof (SHA1digest));
  entropy.Bytes (&Mbin [ 1 + sizeof (SHA1digest) ], MsaltLen);
  BIGNUM * const	M	= BN_new ();
  ASSERT (BN_bin2bn (Mbin, (1 + sizeof (SHA1digest) + MsaltLen), M) != NULL);
  memset (Mbin, 0, sizeof (Mbin));
  //
  // Compute RSA signature s=M^d mod n:
  //
  BIGNUM * const	s	= BN_new ();
  ASSERT (BN_mod_exp (s, M, d, n, ctx) == 1);
  BN_clear (M);
  //
  // Convert s to binary:
  //
  ASSERT (BN_num_bytes (s) >= 0);
  ByteType		sBin [ BN_num_bytes (s) ];
  ASSERT (BN_bn2bin (s, sBin) == BN_num_bytes (s));
  BN_clear (s);
  printf (" done\n");

  printf ("Encrypting digital signature:");
  fflush (stdout);
  //
  // Treat sBin as normal data to be encrypted:
  //
  size_t	i	= 0;
  size_t	left;
  do {
    ASSIGN_LEN(dataLen);
    left = sizeof (sBin) - i;
    if (dataLen > left) {
      dataLen = left;
    }
    memcpy (Data, &sBin [ i ], dataLen);
    i += dataLen;
    ENCRYPT_MSG;
    SIGHEXPRINT;
  } while (left);
  memset (sBin, 0, sizeof (sBin));
  printf (" done\n");

  // Let O/S take care of allocated resources: Just exit ...

  exit (EXIT_SUCCESS);
}
