// Copyright (c) 2003 Robin J Carey. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author, Robin J Carey, may not be used to endorse or
//    promote products derived from this software without specific prior
//    written permission.
// 4. This software may not be used for terrorism, paedophilia or crimes
//    against humanity.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
//

# include  "Startup.h"
# include  "ASSERT.h"
# include  "ByteType.h"
# include  "Key.h"
# include  "RSA.h"
# include  "Random.h"

# include  <stdio.h>		// For: printf(3), fprintf(3), fflush(3)
# include  <stdlib.h>		// For: atol(3), exit(3)
# include  <string.h>		// For: memset(3)
# include  <unistd.h>		// For: getopt(3)

// OpenSSL
# include  <openssl/bn.h>
# include  <openssl/crypto.h>


static void
Usage (const char * const argv0)
{
  fprintf (stderr, "%s [ -b key-size ]\n", argv0);
  exit (EXIT_FAILURE);
}

// Operations:
//
// 1) RSA public/private key generation.
// 2) RSA encryption/decryption verification.
// 3) Write RSA public-key to disk.
// 4) Write (and encrypt) RSA private-key to disk.
//
int
main (int argc, char * argv [])
{
  Startup	startup;
  startup.Useless ();

  unsigned long		pq_bitlen = p_q_min_bits;
  int			ch;

  while ((ch = getopt (argc, argv, "b:")) != -1) {
    switch (ch) {
      case 'b':
	pq_bitlen = atol (optarg);
	break;
      case '?':
      default:
	Usage (argv[0]);
    }
  }
  if (pq_bitlen < p_q_min_bits) {
    Usage (argv[0]);
  }

  BN_CTX * const	ctx	= BN_CTX_new ();
  Random		entropy (RND_DEVICE);
  int			retVal;

  printf ("Generating keys: ");
  fflush (stdout);
  //
  // Generate p:
  //
  BIGNUM * const	p		= BN_new ();
  unsigned long		p_bytelen	= pq_bitlen / 8;
  unsigned long		p_add		= entropy.Ulong () % p_bytelen;
  p_bytelen += p_add;
  ASSERT (p_bytelen >= p_q_min_bytes);
  ByteType	P [ p_bytelen ];
  entropy.Bytes (P, sizeof (P));
  ASSERT (BN_bin2bn (P, sizeof (P), p) != NULL);
  while ((retVal = BN_is_prime (p, BN_prime_checks, NULL, ctx, NULL)) != 1) {
    ASSERT (retVal != -1);
    ASSERT (BN_add_word (p, entropy.Ulong ()) == 1);
  }
  //
  // Generate q:
  //
  BIGNUM * const	q		= BN_new ();
  unsigned long		q_bytelen	= pq_bitlen / 8;
  unsigned long		q_add;
  do {
    q_add = entropy.Ulong () % q_bytelen;
  } while (q_add == p_add);		// Ensure p_add != q_add:
  q_bytelen += q_add;
  ASSERT (q_bytelen >= p_q_min_bytes);
  ByteType	Q [ q_bytelen ];
  entropy.Bytes (Q, sizeof (Q));
  ASSERT (BN_bin2bn (Q, sizeof (Q), q) != NULL);
  while ((retVal = BN_is_prime (q, BN_prime_checks, NULL, ctx, NULL)) != 1) {
    ASSERT (retVal != -1);
    ASSERT (BN_add_word (q, entropy.Ulong ()) == 1);
  }
  //
  // Wipe memory:
  //
  memset (P, 0, sizeof (P));
  memset (Q, 0, sizeof (Q));
  p_bytelen = q_bytelen = q_add = p_add = 0;

  //
  // Compute RSA public-key n = p*q:
  //
  BIGNUM * const	n	= BN_new ();
  ASSERT (BN_mul (n, p, q, ctx) == 1);
  ASSERT (BN_num_bytes (n) > 0);
  ASSERT (((unsigned long) BN_num_bytes (n)) >= n_min_bytes);
  //
  // Compute RSA phi = (p - 1) * (q - 1):
  //
  BIGNUM * const	pp	= BN_new ();	// p - 1
  BIGNUM * const	qq	= BN_new ();	// q - 1
  BIGNUM * const	one	= BN_new ();
  BIGNUM * const	phi	= BN_new ();
  ASSERT (BN_set_word (one, 1) == 1);
  ASSERT (BN_sub (pp, p, one) == 1);
  ASSERT (BN_sub (qq, q, one) == 1);
  ASSERT (BN_mul (phi, pp, qq, ctx) == 1);
  //
  // Compute RSA private-key d:
  //
  BIGNUM * const	e	= BN_new ();
  BIGNUM * const	d	= BN_new ();
  ASSERT (BN_set_word (e, encExp) == 1);
  ASSERT (BN_mod_inverse (d, e, phi, ctx) != NULL);
  //
  // Verify decryption exponent d is sufficiently large:
  //
  if (BN_num_bytes (d) < (BN_num_bytes (n) / 2)) {
    fprintf (stderr, "\n\n\aDecryption exponent too small; "
					"run %s again.\n", argv[0]);
    exit (EXIT_FAILURE);
  }
  printf ("done.\n");

  // Verify RSA encryption and decryption works correctly:
  //
  printf ("Verifying encryption/decryption: ");
  fflush (stdout);
  {
    BIGNUM * const	message		= BN_new ();
    BIGNUM * const	ciphertext	= BN_new ();
    BIGNUM * const	plaintext	= BN_new ();

    ASSERT (BN_set_word (message, encExp) == 1);
    ASSERT (BN_mod_exp (ciphertext, message, e, n, ctx) == 1);
    ASSERT (BN_mod_exp (plaintext, ciphertext, d, n, ctx) == 1);
    ASSERT (BN_cmp (message, plaintext) == 0);

    BN_free (message);
    BN_free (ciphertext);
    BN_free (plaintext);
  }
  printf ("done.\n");

  // Write RSA public-key n to disk:
  //
  printf ("\n");
  Key::WriteKey (PUBLIC_KEY, n);

  // Write RSA private-key d to disk:
  //
  printf ("\n");
  Key::WriteKey (PRIVATE_KEY, d);

  BN_clear (p);
  BN_clear (q);
  BN_clear (n);
  BN_clear (pp);
  BN_clear (qq);
  BN_clear (phi);
  BN_clear (d);

  // Let O/S take care of allocated resources: Just exit ...

  exit (EXIT_SUCCESS);
}
