# outgoing.py = deal with outgoing email
#
##########################################################
# Copyright (c) 2001 Philip Hunt. See COPYING for details
# of licensing information.
##########################################################

# Last altered: 17-Jul-2001
# History:
# 9-Jun-2001 PH: created

"""***
This module deals with processing outgoing email.

Outgoing email has two things which need doing to it:

(1) add X-Herbivore...: headers to indicate the outgoing email
system is herbivore-aware, and to broadcast the sender's public
key.

(2) Do a lookup on the PGP public keyring to see if there is
a Herbivore key for the recipient. (Note that if there is more
than one recipient, don't for now attempt to encrypt -- this
will have to be dealt with later). If there is a public key, 
get the relevant headers that need to be encrypted, put them
in a file together with the message body, encrypt them with GPG,
and append the result to the message headers as determined by 
this program.

***"""

#***** python standard libraries:
import rfc822   # understands mail headers
import sys
import os
from string import *

#***** part of Herbrip:
import utility

from herb_globals import *
from mailheader import *
import pubkeydict
import startup
import encwrap

debug = 0  # debugging this module?


#---------------------------------------------------------------------

# process outgoing email, reading from filename (fnEmailOut) and
# writing herbivorized result to filename (fnEmailHerb)

def processOut(fnEmailOut, fnEmailHerb):
   # get a file handle for the incoming mail
   try:
      mailOut = open(fnEmailOut, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting" % fnEmailOut
      sys.exit(1)   

   # read the header lines
   mh = MailHeader()
   mh.readFromFileHandle(mailOut)
   
   # read the body lines, too:
   mailBody = mailOut.read()
   mailOut.close()
   
   # find out who the message is to  
   # for now, assume one recipient. If more than one,
   # use the first. This is an assumption that **MUST**
   # be changed later
   if debug:
      print "** mh:", mh
   recipients = mh.mess.getaddrlist("To")
   numRecipients = len(recipients)
   if numRecipients != 1:
      print "!! There are %d recipients; there should be exactly 1 !!"\
         % numRecipients
      print "!!***** They are: %s" % recipients 
      
   # get the email address if the initial recipient     
   recipient = recipients[0][1]     
   
   # do we have a public key for this recipient?
   # If so, we must encrypt this message
   encrypt = startup.publicKeys.has_key(recipient)
   encTxt = X_HV_PLAINTEXT # value in X-Herbivore header for non-encreyped
   if encrypt: encTxt = X_HV_ENCRYPTED
   
   # add Herbivore headers to the email:
   mh.put(X_HV, encTxt)
   mh.put(X_HV_VERSION, HERBIVORE_VERSION)
   
   # broadcast our key:
   myPublicKey = encwrap.stripHF(startup.secretKeys['public'])
   mh.put(X_HV_KEY, myPublicKey)
   
   # encrypt if we can
   if encrypt:
      recipientPubKey = startup.publicKeys[recipient]['key']
      cipherText = encwrap.pkEnc(recipientPubKey, mailBody)
      mailBody = cipherText
   
   # write result to (fnEmailHerb)
   fhEmailOut = open(fnEmailHerb, "w")
   fhEmailOut.write(mh.asString())
   fhEmailOut.write("\n") # separate header from body
   fhEmailOut.write(mailBody)
   fhEmailOut.close()
   
   
#---------------------------------------------------------------------

#---------------------------------------------------------------------

#---------------------------------------------------------------------

#---------------------------------------------------------------------


#end outgoing.py
