# outgoing.py = deal with outgoing email
#
##########################################################
# Copyright (c) 2001 Philip Hunt. See COPYING for details
# of licensing information.
##########################################################

# Last altered: 9-Aug-2001
# History:
# 9-Jun-2001 PH: created
# 9-Aug-2001 PH: puts Subject: and some of the other header lines
# in the unencrpted header into a "plaintext-header" which becomes
# part of the encrypted plaintext; this enhances security because 
# an eavesdropper can no longer see the Subject in clear.

"""***
This module deals with processing outgoing email.

Outgoing email has two things which need doing to it:

(1) add X-Herbivore...: headers to indicate the outgoing email
system is herbivore-aware, and to broadcast the sender's public
key.

(2) Do a lookup on the public keyring to see if there is
a Herbivore key for the recipient. (Note that if there is more
than one recipient, don't for now attempt to encrypt -- this
will have to be dealt with later). If there is a public key, 
get the relevant headers that need to be encrypted, put them
in a file together with the message body, encrypt them,
and append the result to the message headers as determined by 
this program.

***"""

#***** python standard libraries:
import rfc822   # understands mail headers
import sys
import os
from string import *

#***** PH library:
import utility
from mailheader import *

#***** part of Herbrip:
from herb_globals import *
import pubkeydict
import startup
import encwrap

debug = 0  # debugging this module?


#---------------------------------------------------------------------


# process outgoing email, reading from filename (fnEmailOut) and
# writing herbivorized result to filename (fnEmailHerb)

def processOut(fnEmailOut, fnEmailHerb):
   # get a file handle for the incoming mail
   try:
      mailOut = open(fnEmailOut, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting" % fnEmailOut
      sys.exit(1)   

   # read the header lines
   m = Mail()
   m.readFromFileHandle(mailOut)
      
   # find out who the message is to  
   # for now, assume one recipient. If more than one,
   # use the first. This is an assumption that **MUST**
   # be changed later
   if debug:
      print "*** m.header: %s ***" % m.header
   recipients = m.header.getaddrlist("To")
   numRecipients = len(recipients)
   if numRecipients != 1:
      print "!! Warning: There are %d recipients; there should be exactly 1 !!"\
         % numRecipients
      print "!!***** They are: %s" % recipients 
      
   # get the email address if the initial recipient     
   recipient = recipients[0][1]     
   
   mProcessed = processForRecipient(m, recipient)
   
   # write result to (fnEmailHerb)
   fhEmailOut = open(fnEmailHerb, "w")
   fhEmailOut.write(mProcessed.asString())
   fhEmailOut.close()
   

#---------------------------------------------------------------------

"""***
Process an email message for one recipient.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) the mail message, before processing

recipient (a string) the recipient's email address

Return value
~~~~~~~~~~~~
A mailheader.Mail containing the mail message to be output.

***"""

def processForRecipient(m, recipient):

   # do we have a public key for this recipient?
   # If so, we must encrypt this message
   encrypt = startup.publicKeys.has_key(recipient)
   encTxt = X_HV_PLAINTEXT # value in X-Herbivore header for non-encrypted
   if encrypt: encTxt = X_HV_ENCRYPTED
   
   # Build up the processed version of this email:
   mProcessed = Mail()
   mProcessed.header.readFromString(m.header.asString())
   
   # add Herbivore headers to the processed email:
   mProcessed.header.put(X_HV, encTxt)
   mProcessed.header.put(X_HV_VERSION, HERBIVORE_VERSION)
   
   # broadcast our key:
   myPublicKey = encwrap.stripHF(startup.secretKeys['public'])
   mProcessed.header.put(X_HV_KEY, myPublicKey)
   
   # !!!!! TODO: make encryption a subroutine !!!!!
   # encrypt if we can
   if encrypt:      
      encryptForOutput(m, mProcessed, recipient)
   else:
      mProcessed.body = m.body # don't process the body   
   
   return mProcessed
   
   
#---------------------------------------------------------------------

"""***
Encrypt an email message for output.

This includes putting some of the headers from the message into
a header-in-plaintext text block at the start of the plaintext to
be encrypted.

On exit, the value of (mProcessed) is suitably changed; it follows
that this is passed-by-reference.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) contains the input message (before processing)

mProcessed (a mailheader.Mail) contains the output message being worked
   on. When this function is invoked, (mProcessed.header) contains the 
   headers from (m) with the X-Herbivore etc headers added, and 
   (mProcessed.body) is empty; it will contain the encrypted body.
  
recipient (a string). The email address of the recipient of this
   message.  

Local variables
~~~~~~~~~~~~~~~

headerInPT (a mailheader.MailHeader) vartious "header" lines that go
   at the start of the palin text.
   
plainText (a string) the plain text, ie. the stuff that gets encrypted.

cipherText (a string) what you get when you encrypt (plainText).   

***"""

def encryptForOutput(m, mProcessed, recipient):
   recipientPubKey = startup.publicKeys[recipient]['key']
   headerInPT = MailHeader()
      
   # transfer "Subject:" to header-in-body:
   subj = m.header.get("Subject", "")
   if debug: print "@@@@ subj is '%s'" % subj
   headerInPT.put("Subject", subj)
   mProcessed.header.put("Subject", "(herbivore-encrypted mail)")
      
   # transfer other fields, removing the field from mProcessed's header   
   fields = ["In-Reply-To", "References", "Keywords", "Comments"]
   for f in fields:
      v = m.header.get(f)
      if v == None: continue
      headerInPT.put(f, v)
      mProcessed.header.removeheader(f)   
      
   # testing for now...   
   headerInPT.put("Herb-Type", "normal")
   headerInPT.put("Herb-X-Type", "SOAP")
     
   plainText = headerInPT.asString() + "\n" + m.body.asString()
   cipherText = encwrap.pkEnc(recipientPubKey, plainText)
   mProcessed.body.readFromString(cipherText)
      

#---------------------------------------------------------------------

#---------------------------------------------------------------------


#end outgoing.py
